/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/logging_config.hpp
 *
 * @brief [LEVEL: beta] The @ref diagnostics::Logging_Config facade
 *
 * $Id: logging_config.hpp,v 1.16 2005/06/23 09:54:19 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @test library/interface/frame/logging_config.t.cpp
 */

#ifndef DIAGNOSTICS__FRAME__LOGGING_CONFIG_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__LOGGING_CONFIG_HPP__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;

// used in the interface by reference
// <diagnostics/frame/logger.hpp>
class Logger;

/**
 * @class Logging_Config
 *
 * @brief The Logging_Config facade is used to configure the logging
 * framework: It offers the client the possibility to
 * register/unregister loggers dynamically.
 *
 * All calls to the Logging_Config and to @ref Logging_Facility::log
 * are mutexed. Nesting is not allowed, but is checked if @ref
 * DIAGNOSTICS_SWITCH_CHECK_FOR_NESTING is set to 1. Nesting could only
 * occur, if a @ref Logger would attempt to call methods of
 * Logging_Config.
 *
 * @attention The first call to one of the methods of Logging_Config
 * or to @ref Logging_Facility::log causes the framework to
 * initialize itself. If initialization fails, ABORT is called.
 *
 * @note The user of this library should call @ref init() at the
 * beginning of the main function to ensure thast the framework is
 * properly initialized.
 *
 * @nosubgrouping
 */
class Logging_Config
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Static Modifiers
     * @{
     */
public:
    /**
     * @brief Registers a logger in the logging framework.
     *
     * First, the @a logger receives a @ref TYPE_LOG_OPEN message
     * immediately. If an exception occurs during logging this
     * message, @a logger is not registered. However, the message
     * might be sent already.
	 *
	 * @pre @a logger != NULL
	 * @pre @a logger is not registered at the time of calling
	 * @pre the call is not nested, i.e., orginates not from within
	 * diagsnotic-land
	 * @post the logger is registered 
     *
     * @throw Logging_Config_Exception 
     */
    static void register_logger(Logger * const logger);

    /**
     * @brief Unregisters a logger in the logging framework.
     *
     * The @a logger receives a @ref TYPE_LOG_CLOSE message
     * beforehand. 
     *
     * @attention If an exception occurs, the logger might be
     * unregistered or not. More specifically, if the exception
     * originates from final log message, then the @a logger is
     * already unregistered.
     *
	 * @pre @a logger != NULL
	 * @pre @a logger is registered
	 * @pre the call is not nested, i.e., orginates not from within
	 * diagsnotic-land
	 * @post the logger is unregistered (and can be registered again)
	 *
     * @throw Logging_Config_Exception 
     */
    static void unregister_logger(Logger * const logger);

    /**
     * @brief Enforces the initialization of the logging
     * framework. This call is a noop, once the logging framework is
     * initialized.
	 *
	 * @attention If init fails to set up its internal data
	 * structures, it will report via @ref DIAGNOSTICS_PANIC_LOG and
	 * then call ::std::abort.
	 *
	 * @attention Any call to @ref Logging_Config or @ref
	 * Logging_Facility prior to the first call to init, causes an
	 * implicit call to init.
	 *
	 * @pre the call is not nested, i.e., orginates not from within
	 * diagsnotic-land
     *
     * @throw Logging_Config_Exception
     */
    static void init();
    // @}
};

DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
