/*
 * CvsGraph graphical representation generator of brances and revisions
 * of a file in cvs/rcs.
 *
 * Copyright (C) 2001  B. Stultiens
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

%x STR

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "utils.h"
#include "readconf.h"
#include "y.tab.h"

static void reset_str(void);
static void add_str(const char *s, int l);
static char *get_str(void);

int line_number;

#define YY_NO_UNPUT 1

%}

%%
branch_bgcolor		return tBRANCH_BGCOLOR;
branch_bspace		return tBRANCH_BSPACE;
branch_color		return tBRANCH_COLOR;
branch_font		return tBRANCH_FONT;
branch_lspace		return tBRANCH_LSPACE;
branch_rspace		return tBRANCH_RSPACE;
branch_tspace		return tBRANCH_TSPACE;
branch_connect		return tBRANCH_CONNECT;
branch_margin		return tBRANCH_MARGIN;
box_shadow		return tBOX_SHADOW;
color_bg		return tCOLOR_BG;
cvsmodule		return tCVSMODULE;
cvsroot			return tCVSROOT;
date_format		return tDATE_FORMAT;
rev_color		return tREV_COLOR;
rev_bgcolor		return tREV_BGCOLOR;
rev_font		return tREV_FONT;
rev_separator		return tREV_SEPARATOR;
rev_minline		return tREV_MINLINE;
rev_maxline		return tREV_MAXLINE;
rev_lspace		return tREV_LSPACE;
rev_rspace		return tREV_RSPACE;
rev_tspace		return tREV_TSPACE;
rev_bspace		return tREV_BSPACE;
rev_text		return tREV_TEXT;
rev_text_color		return tREV_TEXT_COLOR;
rev_text_font		return tREV_TEXT_FONT;
tag_color		return tTAG_COLOR;
tag_font		return tTAG_FONT;
title			return tTITLE;
title_x			return tTITLE_X;
title_y			return tTITLE_Y;
title_font		return tTITLE_FONT;
title_align		return tTITLE_ALIGN;
title_color		return tTITLE_COLOR;
margin_top		return tMARGIN_TOP;
margin_bottom		return tMARGIN_BOTTOM;
margin_left		return tMARGIN_LEFT;
margin_right		return tMARGIN_RIGHT;
image_type		return tIMAGE_TYPE;
image_quality		return tIMAGE_QUALITY;
map_name		return tMAP_NAME;
map_branch_href		return tMAP_BRANCH_HREF;
map_branch_alt		return tMAP_BRANCH_ALT;
map_rev_href		return tMAP_REV_HREF;
map_rev_alt		return tMAP_REV_ALT;
map_diff_href		return tMAP_DIFF_HREF;
map_diff_alt		return tMAP_DIFF_ALT;

jpeg			yylval.num = IMAGE_JPEG; return tNUMBER;
png			yylval.num = IMAGE_PNG; return tNUMBER;
gif			yylval.num = IMAGE_GIF; return tNUMBER;

true			yylval.num = 1; return tNUMBER;
false			yylval.num = 0; return tNUMBER;

left			yylval.num = 0; return tNUMBER;
center			yylval.num = 1; return tNUMBER;
right			yylval.num = 2; return tNUMBER;

tiny			yylval.num = 0; return tNUMBER;
small			yylval.num = 1; return tNUMBER;
medium			yylval.num = 2; return tNUMBER;
large			yylval.num = 3; return tNUMBER;
giant			yylval.num = 4; return tNUMBER;

[a-zA-Z_][a-zA-Z0-9_]*	yylval.str = xstrdup(yytext); return tUNKNOWN;

[+-]?0[0-7]*		yylval.num = strtol(yytext, NULL, 8); return tNUMBER;
[+-]?[1-9][0-9]*	yylval.num = strtol(yytext, NULL, 10); return tNUMBER;
[+-]?0[xX][0-9a-fA-F]+	yylval.num = strtol(yytext, NULL, 16); return tNUMBER;
[+-]?0[xX]		yyerror("Invalid hex number");

=			return *yytext;
;			return *yytext;

\"			reset_str(); BEGIN(STR);
<STR>[^\\\"\n]*		add_str(yytext, yyleng);
<STR>\\[abfnrtv]	{
				char ch;
				switch(yytext[1])
				{
				case 'a': ch = '\a'; break;
				case 'b': ch = '\b'; break;
				case 'f': ch = '\f'; break;
				case 'n': ch = '\n'; break;
				case 'r': ch = '\r'; break;
				case 't': ch = '\t'; break;
				case 'v': ch = '\v'; break;
				}
				add_str(&ch, 1);
			}
<STR>\\[012][0-7]{0,2}		{ char ch = strtol(yytext+1, NULL, 8); add_str(&ch, 1); }
<STR>\\[xX][a-fA-F0-9]{2}	{ char ch = strtol(yytext+1, NULL, 8); add_str(&ch, 1); }
<STR>\\[xX][a-fA-F0-9]	yyerror("Invalid hex escape");
<STR>\\.		add_str(yytext+1, 1);
<STR>\n			yyerror("Newline in string");
<STR>\"			yylval.str = get_str(); BEGIN(INITIAL); return tSTRING;

[ \t\f\v\r\b]+		;	/* Ignore whitespace */
#[^\n]*			;	/* Ignore comments */

\n			line_number++;
.			yyerror("Unknown char/unmatched text '%c' (0x%02x)", isprint(*yytext) ? *yytext : ' ', *yytext);

%%

int yywrap(void)
{
	return 1;
}

#define STRALLOCSIZE	128
static char *str;
static int nstr;
static int nastr;

static void reset_str(void)
{
	nstr = 0;
}

static void add_str(const char *s, int l)
{
	if(nstr + l + 1 > nastr)
	{
		str = xrealloc(str, nastr+STRALLOCSIZE);
		nastr += STRALLOCSIZE;
	}
	memcpy(str+nstr, s, l);
	nstr += l;
}

static char *get_str(void)
{
	str[nstr] = '\0';
	return xstrdup(str);
}

