/*
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codenarc.rule

import org.codehaus.groovy.ast.ClassNode
import org.codehaus.groovy.ast.PropertyNode
import org.codehaus.groovy.ast.FieldNode
import org.codehaus.groovy.ast.MethodNode
import org.codehaus.groovy.ast.ConstructorNode

/**
 * This class tests the @SuppressWarnings functionality.
 *
 * @author Hamlet D'Arcy
 * @version $Revision: 609 $ - $Date: 2011-02-24 13:48:43 -0500 (Thu, 24 Feb 2011) $
 */
class SupressWarningsTest extends AbstractRuleTestCase {

    static boolean failOnClass = false
    static boolean failOnConstructor = false
    static boolean failOnMethod = false
    static boolean failOnProperty = false
    static boolean failOnField = false

    def void setUp() {
        super.setUp()
        failOnClass = false
        failOnConstructor = false
        failOnMethod = false
        failOnProperty = false    
        failOnField = false
    }

    void testRuleProperties() {
        assert rule.priority == 2
        assert rule.name == 'ForceViolations'
    }

    @Override
    @SuppressWarnings('JUnitTestMethodWithoutAssert')
    def void testThatUnrelatedCodeHasNoViolations() {
        // make sure parent does not run
    }

    void testSuppressOnClass() {
        failOnClass = true

        final SOURCE = '''
            @SuppressWarnings('ForceViolations') class MyClass1 {}
            @SuppressWarnings(["ForceViolations", "SomethingElse"]) class MyClass2 {}
            @SuppressWarnings class MyClass3 {}
            class MyClass4 {}
        '''

        assertTwoViolations SOURCE, 4, 'class MyClass3 {}', 5, 'class MyClass4 {}'
    }

    void testSuppressOnConstructor() {
        failOnConstructor = true
        final SOURCE = '''
            class MyClass1 {
                @SuppressWarnings('ForceViolations')
                MyClass1() {}
            }

            class MyClass2 {
                @SuppressWarnings(["ForceViolations", "SomethingElse"])
                MyClass2() {}
            }

            class MyClass3 {
                @SuppressWarnings MyClass3() {}
            }
            class MyClass4 {
                MyClass4() {}
            }
        '''

        assertTwoViolations SOURCE,
                13, '@SuppressWarnings MyClass3() {}',
                16, 'MyClass4() {}'
    }

    void testVisitProperty() {
        failOnProperty = true
        final SOURCE = '''
            class MyClass1 {
                @SuppressWarnings('ForceViolations')
                def someProperty
            }

            class MyClass2 {
                @SuppressWarnings(["ForceViolations", "SomethingElse"])
                def someProperty
            }

            class MyClass3 {
                @SuppressWarnings def someProperty
            }
            class MyClass4 {
                def someProperty
            }
        '''

        assertTwoViolations SOURCE,
                13, 'def someProperty',
                16, 'def someProperty'
    }

    void testVisitField() {
        failOnField = true
        final SOURCE = '''
            class MyClass1 {
                @SuppressWarnings('ForceViolations')
                def someProperty
            }

            class MyClass2 {
                @SuppressWarnings(["ForceViolations", "SomethingElse"])
                def someProperty
            }

            class MyClass3 {
                @SuppressWarnings def someProperty
            }
            class MyClass4 {
                def someProperty
            }
        '''

        assertTwoViolations SOURCE,
                13, 'def someProperty',
                16, 'def someProperty'
    }

    void testVisitMethod() {
        failOnMethod = true
        final SOURCE = '''
            class MyClass {
                @SuppressWarnings('ForceViolations')
                def method1() {}

                @SuppressWarnings(["ForceViolations", "SomethingElse"])
                def method2() {}

                @SuppressWarnings def method3() {}

                def method4() {}
            }
        '''

        assertTwoViolations SOURCE,
                9, 'def method3()',
                11, 'def method4()'
    }

    protected Rule createRule() {
        new ForceViolationsRule()
    }
}

class ForceViolationsRule extends AbstractAstVisitorRule {
    String name = 'ForceViolations'
    int priority = 2
    Class astVisitorClass = ForceViolationsRuleAstVisitor
}

class ForceViolationsRuleAstVisitor extends AbstractAstVisitor {

    protected void visitClassEx(ClassNode node) {
        if (SupressWarningsTest.failOnClass && isFirstVisit(node)) {
            addViolation node, 'visitClassEx'
        }
    }

    def void visitPropertyEx(PropertyNode node) {
        if (SupressWarningsTest.failOnProperty && isFirstVisit(node)) {
            addViolation node, 'visitPropertyEx'
        }
    }

    def void visitFieldEx(FieldNode node) {
        if (SupressWarningsTest.failOnField && isFirstVisit(node)) {
            addViolation node, 'visitFieldEx'
        }
    }

    def void visitMethodEx(MethodNode node) {
        if (SupressWarningsTest.failOnMethod && isFirstVisit(node)) {
            addViolation node, 'visitMethodEx'
        }
    }

    def void visitConstructorEx(ConstructorNode node) {
        if (SupressWarningsTest.failOnConstructor && isFirstVisit(node)) {
            addViolation node, 'visitConstructorEx'
        }
    }

    protected void visitConstructorOrMethodEx(MethodNode node, boolean isConstructor) {
        if (isConstructor && SupressWarningsTest.failOnConstructor && isFirstVisit(node)) {
            addViolation node, 'visitConstructorOrMethodEx'
        }
        if (!isConstructor && SupressWarningsTest.failOnMethod && isFirstVisit(node)) {
            addViolation node, 'visitConstructorOrMethodEx'
        }
    }


}
