#include "cp_types.h"
#include "cp_proto.h"

/* Compute radii of packing to meet angle_sum targets, specified as
"aims". A negative "aim" means that that circle should not have its
radius adjusted. aim of zero is possible only for boundary 
circles in hyp setting. 

All routines are iterative, inspired by routine suggested
by Thurston. This file includes recent algorithms, old reliable
versions, and experimental stuff being tried in spherical case.

Radii are of type r,h, or s (for special): r is euclidean radius,
h is hyperbolic, and s=exp(-h). Other relationships are:
h=log((1+r)/(1-r)), s=(1-r)/(1+r), r=(1-s)/(1+s).
Will be storing radii as s_radii. If h is infinite, then will store
eucl radius with negative value as the s_radius so it can be
used in graphing the appropriate circle. 

There currently is no packing algorithm for spherical geometry. Only
approach is to force packing into unit disc, do maximal packing
there and project back to sphere. Often, this involves a "puncture"
of the complex; missing circle is replaced as northern hemisphere
after max packing is projected back to sph. 

Overlap angles specified > Pi/2 can lead to incompatibilities. To
avoid domain errors, routines computing angles in such cases return
angle Pi. */

extern int repack_activity_msg();

int h_riffle(struct p_data *p,int passes)
/* CRC - modified 5/28/97 -- uses model with super steps, 
safety checks. Anglesum calculations are in-line. Adjust 
hyperbolic s-radii to meet curvature targets in 'aim'. 
Neg. aim means that vertex is free - not subject to adjustment. */
{
  int i,j,k,j1,j2,aimnum=0,*index,count=0,key=0,key0,N,flag;
  double r, r1,r2,r3,fbest,faim,del,bet;
  double ttoler,cut=100,fact=-1,cut0,fact0,denom;
  double ftol=0.05,lmax,rat,twor,*R0;
  double m2,m3,sr,t1,t2,t3,tr,o1,o2,o3;
  struct R_data *pR_ptr;
  struct K_data *pK_ptr;

  index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
  R0 = (double *)malloc((p->nodecount+1)*sizeof(double));
  pR_ptr=p->packR_ptr;
  pK_ptr=p->packK_ptr;
  /* list radii to be adjusted, store squared radii */
  for (i=1;i<=p->nodecount;i++)
    {
      if (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0 
	  && pR_ptr[i].aim<.001)
	pR_ptr[i].rad = (-.2);
      else if (pR_ptr[i].aim>0)
	{
	  if (pR_ptr[i].rad<=0 && pR_ptr[i].aim>.00001)
	    pR_ptr[i].rad = .01;
	  index[aimnum]=i;
	  aimnum++;
	}
      if (pR_ptr[i].rad>0)
	pR_ptr[i].rad = pR_ptr[i].rad*pR_ptr[i].rad;
    }
  if (aimnum==0) 
    {free(index);free(R0);return count;}
  ttoler = 3*sqrt((double)aimnum)*toler;		/* adjust tolerance */
  cut = ttoler+1;
  /*	cut =3*aimnum*10;*/
  while (cut >ttoler && count<passes)         /* Begin Main Loop */
    {
      cut0 = cut;key0 = key;fact0 = fact;	/* save prev values */
      for (i=1;i<=p->nodecount;i++) R0[i] = pR_ptr[i].rad;
      cut = 0;				    
      /* Begin Iteration Loop */
      for (j=0;j<aimnum;j++) 
	{
	  /* anglesum calc */
	  fbest = 0;
	  i = index[j];
	  if ((r = pR_ptr[i].rad)<0) r = 0;
	  sr = sqrt(r);
	  N = pK_ptr[i].num;
	  if (!p->overlap_status) /* no overlaps */
	    { 			
	      twor = 2*r; 
	      r2 = pR_ptr[pK_ptr[i].flower[0]].rad;
	      m2 = (r2>0) ? (1-r2)/(1-r*r2) : 1;
	      for (k=1;k<=N;k++)	/* loop through petals */
		{
		  r3 = pR_ptr[pK_ptr[i].flower[k]].rad;
		  m3 = (r3>0) ? (1-r3)/(1-r*r3) : 1;
		  fbest += acos(1-twor*m2*m3); /* angle calc */
		  m2 = m3;
		} 
	    }
	  else /* with overlaps, old routine */
	    {
	      j2 = pK_ptr[i].flower[0];
	      ((r2 = pR_ptr[j2].rad) > 0) ? r2=sqrt(r2) : 0;
	      o2=pK_ptr[i].overlaps[0];
	      for (k=1;k<=N;k++)
		{
		  r1=r2;
		  o1=o2;
		  j1=j2;
		  j2=pK_ptr[i].flower[k];
		  ((r2 = pR_ptr[j2].rad) >0) ? r2=sqrt(r2) : 0;
		  o2=pK_ptr[i].overlaps[k];
		  o3=pK_ptr[j1].overlaps[nghb(p,j1,j2)];
		  fbest += 
		    acos(h_cos_overlap(sr,r1,r2,o3,o2,o1,&flag));
		}
	    }
	  faim = pR_ptr[i].aim;	/* get target sum */
	  /* set up for model */
	  denom = 1/(2*((double)N));
	  del = sin(faim*denom);
	  bet = sin(fbest*denom);
	  r2 = (bet-sr)/(bet*r-sr);	/* reference radius */
	  if (r2>0) 			/* calc new label */
	    {
	      t1 = 1 - r2;
	      t2 = 2*del;
	      t3 = t2/(sqrt(t1*t1+t2*t2*r2)+t1);
	      r2 = t3*t3; 
	    }
	  else
	    r2 = del*del;		/* use lower limit */
	  pR_ptr[i].rad = r2;		/* store new label */ 
	  pR_ptr[i].curv = fbest;		/* store new anglesum */
	  fbest = fbest-faim;
	  cut += fbest*fbest;		/* accumulate error */
	}
      /* End Iteration Loop */

      cut = sqrt(cut);	
      key = 1;
      fact = cut/cut0;
      if (key0==1 && fact < 1.0)		/* try to extrapolate */
	{
	  cut *= fact;
	  if (fabs(fact-fact0)<ftol)	/* use a super step */
	    fact = fact/(1-fact);
	  lmax = 1000;
	  for (j=0;j<aimnum;j++)		/* find max allowable step*/
	    {
	      i = index[j];
	      r = pR_ptr[i].rad;
	      rat = r - R0[i];
	      if (rat>0)			    
		lmax = (lmax < (tr=((1-r)/rat))) ? lmax : tr;    
	      /* to keep R<1 */
	      else if (rat < 0)
		lmax = (lmax < (tr= (-r/rat))) ? lmax : tr;      
	      /* to keep R>0 */
	    }
	  /* compute new step */
	  fact = (fact < 0.5*lmax) ? fact : 0.5*lmax;
	  /* interpolate new labels */						
	  for (j=0;j<aimnum;j++)
	    {
	      i = index[j];
	      pR_ptr[i].rad += fact*(pR_ptr[i].rad-R0[i]);
	    }
	  key = 0;
	}
		
      /* show activity */
      repack_activity_msg("*");
		
      count++;
    } /* end of main while loop */

  for (i=1;i<=p->nodecount;i++)               /* reset labels */
    if (pR_ptr[i].rad>0)
      pR_ptr[i].rad = sqrt(pR_ptr[i].rad);
  free(index);
  free(R0);
  return count;
} /* h_riffle */

int old_h_riffle(struct p_data *p,int passes)
     /* pre 1996 riffle routine (slow, but dependable) -- */
{
  int i,j,aimnum=0,*index,count=0,dummy,flag;
  double recip,accum,verr,err,cut;
  struct K_data *pK_ptr;
  struct R_data *pR_ptr;

  index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
  pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
  accum=0;
  for (i=1;i<=p->nodecount;i++)
    {
      if (pK_ptr[i].bdry_flag &&
	  pR_ptr[i].aim>=0  && pR_ptr[i].aim<.001)
	pR_ptr[i].rad=(-.2);
      else if (pR_ptr[i].aim>0)
	{
	  if (pR_ptr[i].rad<=0 && pR_ptr[i].aim>.00001) 
	    pR_ptr[i].rad=.01;
	  index[aimnum]=i;
	  aimnum++;
	  err=pR_ptr[i].curv-pR_ptr[i].aim;
	  accum += (err<0) ? (-err) : err;
	}
    }
  if (aimnum==0) {free(index);return count;}
  recip=.333333/aimnum;
  cut=accum*recip;
  while (cut >toler && count<passes)
    {
      for (j=0;j<aimnum;j++)
	{
	  i=index[j];
	  h_anglesum_overlap(p,i,pR_ptr[i].rad,
			     &pR_ptr[i].curv,&flag);
	  verr=pR_ptr[i].curv-pR_ptr[i].aim;
	  if (fabs(verr)>cut)
	    {
	      pR_ptr[i].rad=h_radcalc(p,i,
				      pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
	      count++;
	    }
	}
      accum=0;
      for (j=0;j<aimnum;j++)
	{
	  i=index[j];
	  err=pR_ptr[i].curv-pR_ptr[i].aim;
	  accum += (err<0) ? (-err) : err;
	}
      cut=accum*recip;

      /* show activity */
      repack_activity_msg("o");
	
    } /* end of while */
  free(index);
  return count;
} /* old_h_riffle */

int h_riffle_vert(struct p_data *p,int i)
{
  int n=0,dummy,flag;
  double diff;
  struct R_data *pR_ptr;

  pR_ptr=p->packR_ptr;
  h_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
  diff=pR_ptr[i].curv-pR_ptr[i].aim;
  while (n<20 && (diff>okerr || diff<(-okerr)) )
    /* limit number of attempts */
    {
      pR_ptr[i].rad=h_radcalc(p,i,
			      pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
      h_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
      diff=pR_ptr[i].curv-pR_ptr[i].aim;
      n++;
    }
  return n;
} /* h_riffle_vert */

