#include "cp_types.h"
#include "cp_proto.h"
#include <ctype.h>

/* Read circle packing (or data for packing) into p from 
open file fp. Return 0 on error. Key "NODECOUNT:" indicates
new packing, in which case basic combinatorics are read;
"CHECKCOUNT:" indicates new data for current packing. 
Return return bit-coding for what was read as follows: 

    1: 00001     basic combinatorics (new pack)
    2: 00002     aims
    3: 00004     inv_dist

    4: 00010     radii 
    5: 00020     centers
    6: 00040     angle sums

    7: 00100     vertex_map 
    8: 00200     lists of verts/faces/edges 
    9: 00400     colors (circle or face)

    10: 01000   vertex plot_flags

CAUTION: responsibility of calling routine to update pack info
(eg., aims, centers, etc) based on bits set in return value. 
For a new pack, reading vertex_map, lists, or colors causes 
defaults to be set before info is read in. When pack is
not new pack, the info read in supercedes that in pack. */

int readpack(FILE *fp,struct p_data *p)
{
  int i,j,k,n,ind,v,v1,v2,count,return_flag=0,K_flag=0,N_flag=0;
  int f_col_flag=0,c_col_flag=0;
  double dum,*newrad;
  char command[NAME_MAX],geometry[64],filename[NAME_MAX],*name;
  char buf[BUFSIZE];
  struct Edgelist *trace,*clobber;
  struct K_data *pK_ptr;
  struct R_data *pR_ptr;

  pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
  while ( (fscanf(fp,"%127s",command)!=EOF)
	  && (strcmp(command,"NODECOUNT:")!=0)
	  && (strcmp(command,"CHECKCOUNT:")!=0)
	  && (strcmp(command,"END")!=0) )
    {		/* in case PACKNAME apprears first */
      if (strcmp(command,"PACKNAME:")==0)
	{
	  fgets(buf,1024,fp);
	  if (sscanf(buf,"%127s",filename)) N_flag=1;
	}
    }
  if (!strcmp(command,"") || !strcmp(command,"END")) /* no data */
      return 0; 
  if (strcmp(command,"NODECOUNT:")==0) /* new packing */
    {
      return_flag |= 0001;
      p->file_name[0]='\0';
      p->hes=0; /* default */
      if (fscanf(fp,"%d",&count)!=1
	  || count<3 || !alloc_pack_space(p,count,0) )
	{     
	  sprintf(msgbuf,"Read aborted: out of memory");
	  emsg(); 
	  return 0;
	}
      pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr; /* update ptr's */
      p->nodecount=count;
      edge_free(&(p->vertex_map));
      for (i=1;i<=p->nodecount;i++) pK_ptr[i].plot_flag=0;
    }
  else if (strcmp(command,"CHECKCOUNT:")==0) /* partial data */
    {
      if (!p->status || fscanf(fp,"%d",&count)!=1 
	  || count!=p->nodecount)
	{
	  sprintf(msgbuf,
		  "Read aborted: CHECKCOUNT not equal nodecount.");
	  emsg(); 
	  return 0;
	}
    }
  else return 0; /* didn't find key words NODECOUNT or CHECKCOUNT */
  while ( (fscanf(fp,"%63s",command)!=EOF) 
	  && (strcmp(command,"END")!=0) )
    {
      if (strcmp(command,"ALPHA/BETA/GAMMA:")==0)
	fscanf(fp,"%d %d %d",&p->alpha,
	       &p->beta,&p->gamma);
      else if (strcmp(command,"GEOMETRY:")==0)
	{
	  fscanf(fp,"%63s",geometry);
	  if (toupper(geometry[0])=='H') p->hes=-1;
	  else if (toupper(geometry[0])=='E') p->hes=0;
	  else if (toupper(geometry[0])=='S') p->hes=1;
	  else 
	    {
	      dum=atof(geometry);
	      if (dum>okerr) p->hes=1;
	      else if (dum<(-1.0)*okerr) p->hes=-1;
	      else p->hes=0;
	    }
	}
      else if (strcmp(command,"FLOWERS:")==0)
	{
	  K_flag=1;
	  if (!(return_flag & 0001)) 
	    {
	      sprintf(msgbuf,
		      "Read aborted: FLOWERS given w/o NODECOUNT.");
	      emsg();
	      return 0;
	    }
	  for (i=1;i<=count && (return_flag & 0001);i++)
	    {
	      if (!fscanf(fp,"%d",&k)) return_flag &= 7776;
	      if ( !fscanf(fp,"%d",&pK_ptr[k].num)
		   || pK_ptr[k].num>MAX_PETALS ) 
		return_flag &= 7776;
	      if (pK_ptr[k].flower) free(pK_ptr[k].flower);
	      pK_ptr[k].flower=(int *)
		calloc((size_t)(pK_ptr[k].num+1),sizeof(int));
	      for (j=0;(return_flag & 0001) && j<=pK_ptr[k].num;j++)
		if (!fscanf(fp,"%d",&pK_ptr[k].flower[j])
		    || pK_ptr[k].flower[j]<1
		    || pK_ptr[k].flower[j]>count) 
		  return_flag &= 7776;
	      if ( (return_flag & 0001)
		   && pK_ptr[k].flower[0]
		   !=pK_ptr[k].flower[pK_ptr[k].num] 
		   && pK_ptr[k].num>(MAX_PETALS-1) ) 
		return_flag &= 7776; 
	    }
	  if (!(return_flag & 0001) || !complex_count(p,1))
	    {
	      sprintf(msgbuf,
		      "Read aborted: error in reading complex, v%d.",
		      k);
	      emsg();
	      return 0;
	    }
	  p->status=1;
	  /* need to run complex_count before free_overlaps */
	  /* fixup: don't understand why free_overlaps was here? */
	  /* free_overlaps(p);*/
	}
      else if (strcmp(command,"CIRCLE_PLOT_FLAGS:")==0)
	/* If pack is new, only these plot_flags will be set;
	   if not, only these are changed */
	{
	  return_flag |= 01000;
	  while (fscanf(fp,"%d %d",&i,&j)==2
		 && i>0 && i<=count)
	    pK_ptr[i].plot_flag=j;
	}
      else if (strcmp(command,"PACKNAME:")==0)
	{
	  fgets(buf,1024,fp);
	  if (sscanf(buf,"%127s",filename)) N_flag=1;
	}
      else if (strcmp(command,"RADII:")==0)
	{
	  return_flag |= 0010;
	  if (p->hes<0) for (i=1;i<=count;i++)
	    {
	      fscanf(fp,"%lf",&dum);
	      if (dum>0) pR_ptr[i].rad=exp(-dum);
	      else pR_ptr[i].rad=dum;
	    }
	  else for (i=1;i<=count;i++)
	    fscanf(fp,"%lf",&pR_ptr[i].rad);
	}
      else if (!(return_flag & 00001) /* only allowed with CHECKCOUNT file */
	       && strcmp(command,"SELECT_RADII:")==0)
	{
	  return_flag |= 0010;
	  if (p->hes<0)
	    while ((fscanf(fp,"%d %lf",&i,&dum)==2)
		   && i>0 && i<=count)
	      {
		if (dum>0) pR_ptr[i].rad=exp(-dum);
		else pR_ptr[i].rad=dum;
	      }
	  else
	    while ((fscanf(fp,"%d %lf",&i,&dum)==2)
		   && i>0 && i<=count && dum>0.0)
	      pR_ptr[i].rad=dum;
	}
      else if (strcmp(command,"ANGLE_AIMS:")==0)
	{
	  set_aim_default(p);
	  return_flag |= 0002;
	  while ((fscanf(fp,"%d %lf",&i,&dum)==2)
		 && i>0 && i<=count) 
	    pR_ptr[i].aim=dum;
	}
      else if (strcmp(command,"INV_DISTANCES:")==0) 
	/* NOTE: depends on FLOWERS */
	{
	  if (alloc_overlaps(p))
	    {
	      return_flag |= 0004;
	      k=1;
	      while (fscanf(fp,"%d %d",&i,&k)==2 
		     && (j=nghb(p,i,k))>=0
		     && fscanf(fp,"%lf",&dum))
		set_overlap(p,i,j,dum);
	    }
	}
      else if (strcmp(command,"CENTERS:")==0)
	{
	  return_flag |= 0020;
	  for (i=1;i<=count;i++)
	    fscanf(fp,"%lf %lf",
		   &pR_ptr[i].center.re,&pR_ptr[i].center.im);
	}
      else if (strcmp(command,"ANGLESUMS:")==0)
	{
	  return_flag |= 0040;
	  for (i=1;i<=count;i++)
	    fscanf(fp,"%lf",&pR_ptr[i].curv);
	}
      else if (strcmp(command,"CIRCLE_COLORS:")==0) 
	{
	  return_flag |= 0400;
	  c_col_flag=1;
	  while (fscanf(fp,"%d %d",&i,&j)==2
		 && i>0 && i<=count
		 && j>=0 && j<=NUM_COLORS)
	    pK_ptr[i].color=j;
	}
      else if ( strcmp(command,"VERTEX_MAP:")==0)
	{
	  if (p->vertex_map) 
	    edge_free(&(p->vertex_map));
	  p->vertex_map=trace=clobber=
	    (struct Edgelist *)calloc(1,sizeof(struct Edgelist));
	  while (fscanf(fp,"%d %d",&i,&j)==2
		 && i>0 && i<=count)
	    {
	      trace->v=i;
	      trace->w=j;
	      trace->next=
		(struct Edgelist *)calloc(1,sizeof(struct Edgelist));
	      clobber=trace;
	      trace=trace->next;
	    }
	  if (clobber==trace) /* none found */
	    edge_free(&(p->vertex_map));
	  else 
	    {
	      free(trace);
	      clobber->next=NULL;
	      return_flag |= 0100;
	    }
	}
      else if (strcmp(command,"TRI_COLORS:")==0)
	/* face colors, but given triple of verts */
	{
	  return_flag |= 0400;
	  f_col_flag=1;
	  catalog_faces(p); /* set up face_org ptrs */
	  while (fscanf(fp,"%d %d %d %d",&v,&v1,&v2,&j)==4
		 && v>0 && v<=p->nodecount
		 && v1>0 && v1<=p->nodecount
		 && v2>0 && v2<=p->nodecount
		 && j>=0 && j<=NUM_COLORS)
	    {
	      n=*(face_org[v]);
	      for (i=1;i<=n;i++)
		{
		  k=*(face_org[v]+i);
		  if (   ((ind=check_face(p,k,v,v1)) >= 0 
			  && p->faces[k].vert[(ind+2)%3]==v2)
			 || ((ind=check_face(p,k,v1,v)) >= 0
			     && p->faces[k].vert[(ind+2)%3]==v2) )
		    {
		      p->faces[k].color=j;
		      n=0; /* to stop loop */
		    }
		}
	    }
	  destroy_catalog(p->nodecount);
	}
      else if (strcmp(command,"VERT_LIST:")==0)
	{
	  struct Vertlist *trace,*clobber;
	  if (p->vlist) vert_free(&(p->vlist));
	  p->vlist=trace=clobber=(struct Vertlist *)
	    calloc(1,sizeof(struct Vertlist));
	  while (fscanf(fp,"%d",&v) && v>0 && v<=p->nodecount)
	    {
	      trace->v=v;
	      clobber=trace;
	      trace=trace->next=(struct Vertlist *)
		calloc(1,sizeof(struct Vertlist));
	    }
	  if (trace!=p->vlist) /* got something */
	    {
	      clobber->next=NULL;
	      free(trace);
	      return_flag |= 0200;
	    }
	  else
	    {
	      free(p->vlist);
	      p->flist=NULL;
	    }
	}
      else if (strcmp(command,"FACE_LIST:")==0)
	{
	  struct Vertlist *trace,*clobber;
	  if (p->flist) vert_free(&(p->flist));
	  p->flist=trace=clobber=(struct Vertlist *)
	    calloc(1,sizeof(struct Vertlist));
	  while (fscanf(fp,"%d",&v) && v>0 
		 && v<=p->facecount)
	    {
	      trace->v=v;
	      clobber=trace;
	      trace=trace->next=(struct Vertlist *)
		calloc(1,sizeof(struct Vertlist));
	    }
	  if (trace!=p->flist) /* got something */
	    {
	      clobber->next=NULL;
	      free(trace);
	      return_flag |= 0200;
	    }
	  else
	    {
	      free(p->flist);
	      p->flist=NULL;
	    }
	}
      else if (strcmp(command,"EDGE_LIST:")==0 )
	{
	  struct Edgelist *etrace,*eclobber;
	  if (p->elist) edge_free(&(p->elist));
	  etrace=eclobber=p->elist=(struct Edgelist *)
	    calloc(1,sizeof(struct Edgelist));
	  while (fscanf(fp,"%d %d",&v1,&v2)==2 
		 && v1>0 && v1<=p->nodecount
		 && v2>0 && v2<=p->nodecount
		 && (nghb(p,v1,v2))>=0)
	    {
	      etrace->v=v;
	      eclobber=etrace;
	      etrace=etrace->next=(struct Edgelist *)
		calloc(1,sizeof(struct Edgelist));
	    }
	  if (etrace!=p->elist) /* got something */
	    {
	      eclobber->next=NULL;
	      free(etrace);
	      return_flag |= 0200;
	    }
	  else 
	    {
	      free(p->elist);
	      p->elist=NULL;
	    }
	}
      else if (strcmp(command,"RADII_INTERACTIONS:")==0
	       && p->hes==0)
	/* changes eucl radii: entry "i j f" means to
	   multiply rad(j) by (rad(i)/rad(j))^f. */
	{
	  newrad=(double *)calloc(count+1,sizeof(double));
	  for (i=1;i<=count;i++) newrad[i]=pR_ptr[i].rad;
	  while (fscanf(fp,"%d %d %lf",&i,&j,&dum)==3
		 && i>0 && i<=count && j>0 && j<=count
		 && dum>0)
	    newrad[j] *= pow(pR_ptr[i].rad/pR_ptr[j].rad,dum);
	  for (i=1;i<=count;i++) pR_ptr[i].rad=newrad[i];
	}
    } /* end of while */

  if ((return_flag & 0001) && !K_flag)
    {
      sprintf(msgbuf,"Read aborted: no FLOWERS given.");
      emsg();
      return 0;
    }
  if (N_flag) /* record packname */
    {
      i=strlen(filename);
      while (i>0 && filename[i-1]!='/') i--;
      name=filename+i;  /* cut off any directory names */
      strcpy(p->file_name,name);
    }
  if ((return_flag & 0001) && !(return_flag & 0002))
    set_aim_default(p);
  if ((return_flag & 0001) && !(return_flag & 0004))
    set_default_overlaps(p);
  /* leave it for calling routine to compute centers and
     angle sums, if desired */
 
  if ((return_flag & 0001) && !(return_flag & 01000))
    for (i=1;i<=p->nodecount;i++) pK_ptr[i].plot_flag=1;

  /* new pack: colors? */
  if ((return_flag & 0001) && !c_col_flag) 
    for (i=1;i<=count;i++) pK_ptr[i].color=FG_COLOR;
  if ((return_flag & 0001) && !f_col_flag) 
    for (i=1;i<=p->facecount;i++) p->faces[i].color=FG_COLOR;

  return (return_flag);
} /* readpack */
