/* cp_comb.c   Routines to manipulate combinatorics of simplicial complexes */

#include "cp_head.h"
static char next[BUFSIZE];

struct stface
 {
	int face;
	struct stface *prev;
	struct stface *next;
 };

struct New_verts
{
	int v,w;	/* endpoints of edge, v<w */
};

int **face_org;	/* temp ptrs to list of faces by vertex */
extern int free_overlaps();

/* ================ declarations ========== */

/*OK need global MAX_COMPONENTS */
int
complex_count(p,flag) /* Identify and count the 
no. of bdry and interior
nodes, bdry and interior components,  Euler characteristic, genus, etc,
for pack p. Stores indicators of bdry and int components in packdata (at
most MAX_COMPONENTS of each). flag means to set default color codes for
circles.*/
struct p_data *p;
int flag;
{
	int i,node,count,bcount,new_vert,num_edges,m,n;
	int comp_count,ptr,k,j,fj,icount,more_flag,next_bdry;
	int max_components=MAX_COMPONENTS;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
/* some initialization and counting */
	node=p->nodecount;
	for (i=1;i<=max_components;i++) 
		p->bdry_starts[i]=p->int_starts[i]=0;
	bcount=0;
	comp_count=0;
	for (i=1;i<=node;i++) 
	 {
		pK_ptr[i].plot_flag=0;
		if (flag) pK_ptr[i].color=FG_COLOR;
		if (pK_ptr[i].flower[0]!=pK_ptr[i].flower[pK_ptr[i].num]) 
		 {
			bcount++;
			pK_ptr[i].bdry_flag=1;
		 }
		else pK_ptr[i].bdry_flag=0;
	 }
	p->intnode=node-bcount;

/* identify bdry components, give pointers; note: index starts with 1 */
	count=0;
	new_vert=0;
	do
	 {
		new_vert++;
		if (pK_ptr[new_vert].bdry_flag && !pK_ptr[new_vert].plot_flag)
		 {
		   comp_count++;
		   if (comp_count>max_components)
		    {
			sprintf(msgbuf,
				"This complex has too many bdry components.");
			emsg();
			return 0;
		    }
		   p->bdry_starts[comp_count]=new_vert;
		   next_bdry=new_vert;
		   do
		    {
			pK_ptr[next_bdry].plot_flag=1;
			next_bdry=pK_ptr[next_bdry].flower[0];
			count++;
			if (count>p->nodecount)
			 {
				sprintf(msgbuf,"Problem tracing a boundary path; combinatoric error.");
				emsg();
				return 0;
			 }
		    }
		   while (next_bdry!=new_vert);
		 }
	 }
	while (count<bcount);
	p->num_bdry_comp=comp_count;

/* identify int components and pointers to them */
	icount=node-bcount; /* number of interiors */
	if (icount>0)
	 {
		comp_count=0;
		do
 {
	ptr=0;
	do {ptr++;}
	while (pK_ptr[ptr].plot_flag); 
		/* find smallest index not plotted (hence int node) */
	comp_count++;
	if (comp_count>max_components)
	 {
		sprintf(msgbuf,
			"This complex has too many interior components.");
		emsg();
		return 0;
	 }
	icount--;
	p->int_starts[comp_count]=ptr;
	pK_ptr[ptr].plot_flag=ptr;
	do
	 {
	   more_flag=0; /* will tell me whether I am still finding anything */
	   for (k=ptr;k<=node;k++)
	    {
	      if (pK_ptr[k].plot_flag==ptr)
	       {
		for (j=0;j<=pK_ptr[k].num;j++) 
		 {
			fj=pK_ptr[k].flower[j];
			if (!pK_ptr[fj].plot_flag) 
			 {
				pK_ptr[fj].plot_flag=ptr;
				icount--;
				more_flag=1;
			 }
		 }
	       }
	    }
	 } /* end of inside do */
	while (more_flag && icount>0);
 } /* end of outside do */
		while (icount>0);
		p->num_int_comp=comp_count;
	 } /* end of 'if icount' */
/* find Euler characteristic and genus */
	count=0;
	for (k=1;k<=node;k++) count += pK_ptr[k].num;
	p->facecount=count/3;
	num_edges=(count + bcount)/2;
	p->euler=node-num_edges+p->facecount;
	p->genus=(2-p->euler-p->num_bdry_comp)/2;
/* check if geom is appropriate */
	if (bcount==0 && p->genus==0 && p->hes<=0)
	 {
		strcpy(msgbuf,
		   "Note: This complex is a topological sphere.");
		msg();
	 } 
/* allocate space for face data, store ordered triples of verts. */
	if (!alloc_faces_space(p))
	 {
		sprintf(msgbuf,"Error allocating face data space for p%d.",
			pack_num(p));
		emsg();
		return 0;
	 }
	count=1;
	i=1;
	while (count<=p->facecount && i<=p->nodecount)
	{
		for (j=0;j<pK_ptr[i].num;j++)
		 {
			m=pK_ptr[i].flower[j];
			n=pK_ptr[i].flower[j+1];
			if (m>i && n>i) 
			 {
				p->faces[count].vert[0]=i;
				p->faces[count].vert[1]=m;
				p->faces[count].vert[2]=n;
				p->faces[count].color=FG_COLOR;
				count++;
			 }
		 }
		i++;
	 }
	return 1;
} /* complex_count */

/*OK*/
int
nghb(p,v,w) /* return -1 if w is not neighbor of v, else returns index of
w in flower of v. */
int v,w;
struct p_data *p;
{
	int j;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	if (v<1 || v>p->nodecount || w<1 || w>p->nodecount) return -1;
	for (j=0;j<=pK_ptr[v].num;j++)
		if (pK_ptr[v].flower[j]==w) return j;
	return (-1);
} /* nghb */

/*OK*/
int
edge_count(p,w) /* return count of edges in edge path containing w. If
in error, return -1. */
struct p_data *p;
int w;
{
	int next,count;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	if (!p->status || w<1 || w>p->nodecount || !pK_ptr[w].bdry_flag)
		return -1;
	count=1;
	next=pK_ptr[w].flower[0];
	while (next!=w && count<p->nodecount)
	 {
		next=pK_ptr[next].flower[0];
		count++;
	 }
	if (count>=p->nodecount) return -1;
	return count;
} /* edge_count */

/*OK*/
int
find_index(p,f,v) /* return -1 if v not in face f; else return index of v
in vertices of f. */
struct p_data *p;
int f,v;
{
	int m=0;

	while (m<3) 
	 {if (p->faces[f].vert[m]==v) return m; m++;}
	return -1;
} /* find_index */

/*OK*/
int
check_face(p,f,v,w) /* Return -1 if ordered edge <v,w> not in face f. 
Else, return index of v in verts of f. */
struct p_data *p;
int f,v,w;
{
	int m=0;

	while (m<3)
	 {if (p->faces[f].vert[m]==v && p->faces[f].vert[(m+1)%3]==w)
		return m; m++;}
	return -1;
} /* check_face */
	
/*OK*/
int
nghb_tri(p,m,n) /* -1 if face n doesn't share edge e with m; else gives 
index of begin vertex of e in data of n. */
int m,n;
struct p_data *p;
{
	int nj,mj,v1,v2;

	if (m<1 || m > p->facecount || n<1 || n > p->facecount) return -1;
	for (nj=0;nj<=2;nj++)
	 {
		v1=p->faces[n].vert[nj];
		v2=p->faces[n].vert[(nj+1)%3];
		for (mj=0;mj<=2;mj++)
			if ( (v1==p->faces[m].vert[mj]) && 
			 (v2==p->faces[m].vert[(mj+2)%3]) )
				return nj;
	 }
	return -1;
} /* nghb_tri */

int
bdry_comp_count(p,v) /* retrn count of edges in bdry component of v. 
0 if v not bdry or some comb error. */
struct p_data *p;
int v;
{
	int next,count=1,maxcount;

	if (!p->packK_ptr[v].bdry_flag) return 0;
	maxcount=p->nodecount-p->intnode;
	next=v;
	while ((next=p->packK_ptr[next].flower[0])!=v 
		&& count<= maxcount) count++;
	if (count>maxcount) return 0;
	return count;
} /* bdry_comp_count */

/*OK*/
int
cross_edge_vert(p,v,k) /* Return vert across edge from v. Edge is that 
from flower[k] to flower[k+1]. Return 0 on failure. */
struct p_data *p;
int v,k;
{
	int N,ind,w;
	struct K_data *pK_ptr;
	extern int nghb();

	pK_ptr=p->packK_ptr;
	if (v<1 || v>p->nodecount || k<0 || k>(N=pK_ptr[v].num)
	   || (k==N && pK_ptr[v].bdry_flag) ) return 0;
	w=pK_ptr[v].flower[k];
	ind=nghb(p,w,v);
	if (pK_ptr[w].bdry_flag)
	 {
		if (ind<2) return 0;
		else return (pK_ptr[w].flower[ind-2]);
	 }
	return (pK_ptr[w].flower[(ind+pK_ptr[w].num-2)%(pK_ptr[w].num)]);
} /* cross_edge_vert */

int
hex_proj(p,v,w) /* If v is interior and hex, w is petal, return
petal u opposite w. If v bdry, 3 faces, w bdry, return petal
on bdry opposite w. Return 0 on failure. */
struct p_data *p;
int v,w;
{
	if (v==w || (p->packK_ptr[v].bdry_flag && p->packK_ptr[v].num!=3)
	   || (!p->packK_ptr[v].bdry_flag && p->packK_ptr[v].num!=6))
		return 0;
	if (p->packK_ptr[v].bdry_flag)
	 {
		if (p->packK_ptr[v].flower[0]==w) 
			return p->packK_ptr[v].flower[3];
		if (p->packK_ptr[v].flower[3]==w)
			return p->packK_ptr[v].flower[0];
		return 0;
	 }
	return p->packK_ptr[v].flower[(nghb(p,v,w)+3)%6];
} /* hex_proj */ 

/*OK*/
int
remove_edge(p,v1,v2) /* remove a bdry edge. Already checked that v2 is
counterclockwise neighbor of v1 and removal is okay */
int v1,v2;
struct p_data *p;
{
	int i,v,n,m,*newflower;
	float *newoverlaps;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int facedraworder(),complex_count();

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
/* verify data */
	if (v1<1 || v1>p->nodecount || v2<1 || v2>p->nodecount
		|| !pK_ptr[v1].bdry_flag || !pK_ptr[v2].bdry_flag
		|| (v2!=pK_ptr[v1].flower[0] && v1!=pK_ptr[v2].flower[0]) )
	 {
		strcpy(msgbuf,"An appropriate edge was not specified.");
		emsg();
		return 0;
	 }
	if (v1==pK_ptr[v2].flower[0]) {n=v1;v1=v2;v2=n;}
	if (pK_ptr[v1].num<2 || pK_ptr[v2].num<2 
		|| pK_ptr[pK_ptr[v1].flower[1]].bdry_flag )
	 {
		strcpy(msgbuf,"An appropriate edge was not specified.");
		emsg();
		return 0;
	 }
/* fix up v1 */
	newflower=(int *)malloc(pK_ptr[v1].num*sizeof(int));
	if (p->overlap_status)
		newoverlaps=(float *)malloc(pK_ptr[v1].num*sizeof(float));
	for (i=1;i<=pK_ptr[v1].num;i++)
	 {
		newflower[i-1]=pK_ptr[v1].flower[i];
		if (p->overlap_status)
			newoverlaps[i-1]=pK_ptr[v1].overlaps[i];
	 }
	free(pK_ptr[v1].flower);
	pK_ptr[v1].flower=newflower;
	if (p->overlap_status)
	 {
		free(pK_ptr[v1].overlaps);
		pK_ptr[v1].overlaps=newoverlaps;
	 }
	pK_ptr[v1].num--;
	pK_ptr[pK_ptr[v1].flower[0]].bdry_flag=1;
	pR_ptr[v1].aim=-1;

/* fix up v2 */
	newflower=(int *)malloc(pK_ptr[v2].num*sizeof(int));
	if (p->overlap_status)
		newoverlaps=(float *)malloc(pK_ptr[v2].num*sizeof(float));
	for (i=0;i<pK_ptr[v2].num;i++)
	 {
		newflower[i]=pK_ptr[v2].flower[i];
		if (p->overlap_status)
		   newoverlaps[i]=pK_ptr[v2].overlaps[i];
	 }
	free(pK_ptr[v2].flower);
	pK_ptr[v2].flower=newflower;
	if (p->overlap_status)
	 {
		free(pK_ptr[v2].overlaps);
		pK_ptr[v2].overlaps=newoverlaps;
	 }
	pK_ptr[v2].num--;
	pR_ptr[v2].aim=-1;

/* fix up common neighbor v */
	v=pK_ptr[v1].flower[0];
	newflower=(int *)malloc(pK_ptr[v].num*sizeof(int));
	if (p->overlap_status)
		newoverlaps=(float *)malloc(pK_ptr[v].num*sizeof(float));
	n=nghb(p,v,v2);
	m=pK_ptr[v].num;
	for (i=n;i<=m;i++)
	 {
		newflower[i-n]=pK_ptr[v].flower[i];
		if (p->overlap_status)
			newoverlaps[i-n]=pK_ptr[v].overlaps[i];
	 }
	for (i=1;i<n;i++)
	 {
		newflower[m-n+i]=pK_ptr[v].flower[i];
		if (p->overlap_status)
			newoverlaps[m-n+i]=pK_ptr[v].overlaps[i];
	 }
	pK_ptr[v].num--;
	free(pK_ptr[v].flower);
	pK_ptr[v].flower=newflower;
	if (p->overlap_status)
	 {
		free(pK_ptr[v].overlaps);
		pK_ptr[v].overlaps=newoverlaps;
	 }
	pR_ptr[v].aim=-1;

/* redo pack combinatorics */
	complex_count(p,FALSE);
	facedraworder(p,NULL);
	return 1;
} /* remove_edge */


/*OK*/
int
swap_nodes(p,datastr) /* interchange two vertex numbers */
struct p_data *p;
char *datastr;
{
	int i,j,v,w,vdum=0,pet;
	struct K_data *pK_ptr,holdK;
	struct R_data *pR_ptr,holdR;
	extern int facedraworder(),complex_count();

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	if (sscanf(datastr,"%d %d",&v,&w)!=2 || v<1 || w<1 
		|| v>p->nodecount || w>p->nodecount || v==w) return 0;
	 {
		holdR=pR_ptr[v];holdK=pK_ptr[v];
		pR_ptr[v]=pR_ptr[w];pK_ptr[v]=pK_ptr[w];
		pR_ptr[w]=holdR;pK_ptr[w]=holdK;
		for (i=0;i<=pK_ptr[v].num;i++)
		 {
			pet=pK_ptr[v].flower[i];
			if (pet==v) pet=w;
			for (j=0;j<=pK_ptr[pet].num;j++)
				if (pK_ptr[pet].flower[j]==w) 
					pK_ptr[pet].flower[j]=vdum;
		 }
		for (i=0;i<=pK_ptr[w].num;i++)
		 {
			pet=pK_ptr[w].flower[i];
			if (pet==vdum) pet=v;
			for (j=0;j<=pK_ptr[pet].num;j++)
				if (pK_ptr[pet].flower[j]==v) 
					pK_ptr[pet].flower[j]=w;
		 }
		for (i=0;i<=pK_ptr[v].num;i++)
		 {
			pet=pK_ptr[v].flower[i];
			for (j=0;j<=pK_ptr[pet].num;j++)
				if (pK_ptr[pet].flower[j]==vdum) 
					pK_ptr[pet].flower[j]=v;
		 }
		if (p->active_node==v) p->active_node=w;
		else if (p->active_node==w) p->active_node=v;
		if (p->alpha==v) p->alpha=w;
		else if (p->alpha==w) p->alpha=v;
		if (p->beta==v) p->beta=w;
		else if (p->beta==w) p->beta=v;
		if (p->gamma==v) p->gamma=w;
		else if (p->gamma==w) p->gamma=v;
/* fixup ?? -- try to reestablish overlap data */
		complex_count(p,FALSE);
		facedraworder(p,NULL);
	 }
	return 1;
} /* swap_nodes */

/*OK*/
int 
reverse_orient(p) /* replace complex by one of reverse orientation */
struct p_data *p;
{
	int i,num,j;
	struct K_data *pK_ptr,*newK_ptr;
	extern int handle_cmd();

	pK_ptr=p->packK_ptr;
	if ( (newK_ptr=(struct K_data *)
	   calloc((size_t)(p->nodecount+1),sizeof(struct K_data)))==NULL ) 
		return 0;
	for (i=1;i<=p->nodecount;i++)
	 {
		newK_ptr[i]=pK_ptr[i];
		num=newK_ptr[i].num;
		newK_ptr[i].flower=(int *)malloc((num+1)*sizeof(int));
		for (j=0;j<=num;j++) /* change orientation */
			newK_ptr[i].flower[j]=
			   pK_ptr[i].flower[num-j];
		if (pK_ptr[i].overlaps)
		 {
			newK_ptr[i].overlaps=(float *)
				malloc((num+1)*sizeof(float));
			for (j=0;j<=num;j++) 
			   newK_ptr[i].overlaps[j]=
				pK_ptr[i].overlaps[num-j];
		 }
	 }
	for (i=1;i<=p->nodecount;i++)
	 {
		free(pK_ptr[i].flower);
		free(pK_ptr[i].overlaps);
		pK_ptr[i]=newK_ptr[i];
	 }
	free(newK_ptr);
	sprintf(buf,"fix -p%d -K -c",pack_num(p));
	handle_cmd(buf);
	sprintf(buf,"fix -p%d",pack_num(p));
	handle_cmd(buf);
	return 1;
} /* reverse_orient */

/*OK*/
int
double_K(p,datastr) /* double packing across some or all bdry comps,
get compact complex. Error should leave pack unharmed. Lose edge data. */
struct p_data *p;
char *datastr;
{
	int i,j,jj,k,final_node_count,node,num,new,vert,hits;
	int *newflower,*holdflower;
	int v,*oldnew=NULL,*oldnewflag=NULL;
	struct K_data *pK_ptr=NULL,*newK_ptr=NULL;
	struct R_data *pR_ptr=NULL,*newR_ptr=NULL;
	extern struct Vertlist *node_link_parse();
	struct Vertlist *vertlist=NULL,*trace=NULL;
	char *endptr=NULL;
	extern int complex_count(),facedraworder(),alloc_pack_space();
	extern void fillcurves(),vert_free();

	if (p->num_bdry_comp==0) 
		return 0; /* already compact */
	node=p->nodecount;
/* new data area */
	if ((newK_ptr=(struct K_data *)
	   calloc((size_t)(2*node+1),sizeof(struct K_data)))==NULL)
		return 0;
	if ((newR_ptr=(struct R_data *)
	   malloc((2*node+1)*sizeof(struct R_data)))==NULL)
	   	{free(newK_ptr);return 0;}
	if (!alloc_pack_space(p,2*p->nodecount,1)) 
	 {free(newK_ptr);free(newR_ptr);return 0;}
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr; 
	free_overlaps(p); /* fixup ??: could try to save overlaps */
	for (i=1;i<=node;i++)
	 {
		newK_ptr[i]=newK_ptr[i+node]=pK_ptr[i];
		newR_ptr[i]=newR_ptr[i+node]=pR_ptr[i];
		num=newK_ptr[i].num;
		newflower=(int *)malloc((num+1)*sizeof(int));
		for (j=0;j<=num;j++) /* create new ptr */
			newflower[j]=pK_ptr[i].flower[j];
		newK_ptr[i].flower=newflower;
		newflower=(int *)malloc((num+1)*sizeof(int));
		for (j=0;j<=num;j++) /* change orientation */
			newflower[j]=pK_ptr[i].flower[num-j]+node;
		newK_ptr[i+node].flower=newflower;
	 }
	oldnew=(int *)malloc((2*node+1)*sizeof(int));
	oldnewflag=(int *)malloc((2*node+1)*sizeof(int));
	for (i=1;i<=2*node;i++) 
	 {
		oldnewflag[i]=0; /* know new number ? */
		oldnew[i]=i;
	 }
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) == NULL )
	 {
		datastr[0]='b';datastr[1]='\0';
		if ((vertlist=node_link_parse(p,datastr,&endptr,&hits))==NULL)
			goto ABORT;  /* some combinatoric error */
	 }
	trace=vertlist;
	while (trace!=NULL)
	 {
		if (newK_ptr[v=trace->v].bdry_flag)
		 {
		   do /* take care of this whole bdry component */
		    {
			oldnew[v+node]=v;
			oldnewflag[v+node]=1;
			if (p->hes<0 && pR_ptr[v].rad<=0) pR_ptr[v].rad=.5;
/* fix flower */
			num=pK_ptr[v].num;
			newK_ptr[v].num=2*num;
			newK_ptr[v].bdry_flag=0;
			newR_ptr[v].aim=2.0*M_PI;
			if (num<2) goto ABORT;   /* flower too small */
			newflower=(int *)malloc((2*num+1)*sizeof(int));
			for (j=0;j<=num;j++) 
			   newflower[j]=newK_ptr[v].flower[j];
			for (j=1;j<num;j++)
			   newflower[j+num]=newK_ptr[v+node].flower[j];
			free(newK_ptr[v].flower);
			newK_ptr[v].flower=newflower;
			newK_ptr[v].flower[2*num]=newK_ptr[v].flower[0];
/* remove refs to old nums; flowers get only good numbers */
			for (j=0;j<=pK_ptr[v].num;j++) 
			 {
			   jj=newK_ptr[v+node].flower[j];
			   for (k=0;k<=newK_ptr[jj].num;k++)
			      if (oldnewflag[vert=newK_ptr[jj].flower[k]])
			         newK_ptr[jj].flower[k]=oldnew[vert];
			 }
			v=pK_ptr[v].flower[0];
		    }
		   while (v!=trace->v);
		 }
		trace=trace->next;
	 }
	vert_free(&vertlist);
/* throw out unused numbs, replacing from top */
	final_node_count=2*node;
	for (i=node+1;i<=2*node;i++) if (oldnewflag[i])
	 {
		while (oldnewflag[final_node_count]) final_node_count--;
		if ((new=final_node_count)>i) 
			/* swap nums: good data 'new' goes to opening at i. */
		 {
			for (j=0;j<=newK_ptr[new].num;j++)
			 {
				jj=newK_ptr[new].flower[j];
				for (k=0;k<=newK_ptr[jj].num;k++)
				   if (newK_ptr[jj].flower[k]==new)
				      newK_ptr[jj].flower[k]=i;
			 }
			newR_ptr[i]=newR_ptr[new];
			holdflower=newK_ptr[i].flower; 
			newK_ptr[i]=newK_ptr[new];
			newK_ptr[new].flower=holdflower; 
				/* need old pointer to throw out later */
			oldnewflag[new]=1;
			oldnewflag[i]=0;
		 }
	 }
/* move packdata ptrs, clean up, and leave */
	for (i=1;i<=p->nodecount;i++) free(pK_ptr[i].flower);
	free(pK_ptr);free(pR_ptr);
	free(oldnewflag);free(oldnew);
	p->nodecount=final_node_count;
	for (i=p->nodecount+1;i<=2*node;i++) 
	 {
		free(newK_ptr[i].flower);
		newK_ptr[i].flower=NULL;
	 }
	p->packK_ptr=newK_ptr;
	p->packR_ptr=newR_ptr;
	alloc_pack_space(p,p->nodecount,1);
	complex_count(p,FALSE);
	fillcurves(p);
	facedraworder(p,NULL);
	return 1;
ABORT:
 {
	for (i=1;i<=2*node;i++) free(newK_ptr[i].flower);
	if (newK_ptr) free(newK_ptr);
	if (newR_ptr) free(newR_ptr);
	if (oldnew) free(oldnew);
	if (oldnewflag) free(oldnewflag);
	vert_free(&vertlist); 
	return 0;
 }

} /* double_K */

/*OK*/
int
adjoin(p1,p2,v1,v2,n,Oldnew) /* adjoin the pack p2 to p1, starting with
vertex v2 of p2 to vertex v1 of p1, and proceeding n additional vertices
about the boundary of p1 in the negative (clockwise) direction. Put
results in p1. If p1!=p2, then p2 should remain unchanged.
return 1 if successful. Lose overlap data. oldnew returns
conversions of indices for possible use. */
int v1,v2,n,**Oldnew;
struct p_data *p1,*p2;
{
	int i,ii,jj,j,k,kk,ll,m,mm,nn,node,newnode,endvertex,endnode;
	int b1,b2,count,dist,w,indx;
	int next,v,old_vert,new_vert;
	int *oldnew=NULL,*oldnewflag=NULL,*newflower;
	struct K_data *pK_ptr1,*pK_ptr2;
	struct R_data *pR_ptr1,*pR_ptr2;
	extern void delete(),choose_gamma(),choose_beta();
	extern int bdry_comp_count(),alloc_pack_space();

	pK_ptr2=p2->packK_ptr;pR_ptr2=p2->packR_ptr;
	pK_ptr1=p1->packK_ptr;pR_ptr1=p1->packR_ptr;
	if 
	 ( !p1->status
	   || !p2->status
	   || n<1 || v1<=0 || v2<=0 
	   || v1>p1->nodecount || v2>p2->nodecount
	   || (b1=bdry_comp_count(p1,v1))<n
	   || (b2=bdry_comp_count(p2,v2))<n
	   || p1->locks
	 )
		return 0;
/* fixup ??: could try to save overlaps */
	free_overlaps(p1); 
	if (p1!=p2) /* procedure for adjoining distinct packs */
 {
	oldnew=(int *)malloc((p2->nodecount+2)*sizeof(int));
	oldnewflag=(int *)calloc((size_t)(p2->nodecount+2),sizeof(int));

/* store new indices */
	node=v1;
	newnode=v2;
	for (i=1;i<=n;i++)
	 {
		oldnewflag[newnode]=1;
		oldnew[newnode]=node;
		node=pK_ptr1[node].flower[pK_ptr1[node].num];
		newnode=pK_ptr2[newnode].flower[0];
	 }
	endvertex=newnode; 	/* last p2 vertex */
	endnode=node; 		/* last p1 vertex */
	if (b2>n) 
	 {
		oldnewflag[endvertex]=1;
		oldnew[endvertex]=endnode;
	 }
		
	 	/* get rest of new indices */
	newnode=p1->nodecount+1;
	for (i=1;i<=p2->nodecount;i++)
	   if (!oldnewflag[i])
	    {oldnew[i]=newnode;newnode++;}
	if (!alloc_pack_space(p1,newnode+2,1)) return 0;
	pK_ptr1=p1->packK_ptr;pR_ptr1=p1->packR_ptr; /* updata ptr's */
	p1->nodecount=newnode-1;

/* fix up flowers */

	for (i=1;i<=p2->nodecount;i++)
	 {
		j=oldnew[i];

		if (i==endvertex)  /* special measures for last vertex */
		 {
		   if ((b1 > n) && (b2 > n)) 
				/* flower begins with nghbs in p2 */
		    {
			ll=pK_ptr2[i].num;
			m=pK_ptr1[j].num;
			newflower=(int *)malloc((ll+m+1)*sizeof(int));
			for (k=1;k<=m;k++) 
				newflower[ll+m-k+1]=
				pK_ptr1[j].flower[m-k+1];
			for (k=0;k<=ll;k++) 
				newflower[k]=
				oldnew[pK_ptr2[i].flower[k]]; 
			free(pK_ptr1[j].flower);
			pK_ptr1[j].flower=newflower;
 			pK_ptr1[j].num=ll+m;
		    }
		   else if ((b1==n) && (b2>n)) 
				/* identify endnode and v1 in p1;
				use parts of three flowers;
				endnode to be deleted later. */
		    {
			for (ii=0;ii<=pK_ptr1[endnode].num;ii++)
				/* remove references to endnode in p1 */
			 {
				w=pK_ptr1[endnode].flower[ii];
				indx=nghb(p1,w,node);
				if (indx==0 && !pK_ptr1[w].bdry_flag)
				   pK_ptr1[w].flower[0]=
					pK_ptr1[w].flower[pK_ptr1[w].num]=v1;
				else if (indx>=0) pK_ptr1[w].flower[indx]=v1;
			 }
			ll=pK_ptr2[endvertex].num;
			mm=pK_ptr1[v1].num;
			nn=pK_ptr2[v2].num;
			newflower=(int *)malloc((ll+mm+nn+1)*sizeof(int));
			for (k=0;k<ll;k++)
			   newflower[k]=oldnew[pK_ptr2[endvertex].flower[k]];
			for (k=ll;k<=(ll+mm);k++)
			   newflower[k]=pK_ptr1[v1].flower[k-ll];
			for (k=(ll+mm+1);k<=(ll+mm+nn);k++)
			   newflower[k]=oldnew[pK_ptr2[v2].flower[ll+mm-k]];
			free(pK_ptr1[v1].flower);
			pK_ptr1[v1].flower=newflower;
 			pK_ptr1[v1].num=ll+mm+nn;
		    }
		   else if (b1==n && b2==n)
				/* whole bdrys; all become interior */
		    {
			ll=pK_ptr2[v2].num;
			mm=pK_ptr1[v1].num;
			newflower=(int *)malloc((ll+mm+1)*sizeof(int));
			for (k=0;k<ll;k++)
			   newflower[k]=oldnew[pK_ptr2[v2].flower[k]];
			for (k=ll;k<=(ll+mm);k++)
			   newflower[k]=pK_ptr1[v1].flower[k-ll];
			free(pK_ptr1[v1].flower);
			pK_ptr1[v1].flower=newflower;
 			pK_ptr1[v1].num=ll+mm;
		    }
		 }

/* rest of flowers are more routine */

		else if (oldnewflag[i])
		 {
			ll=pK_ptr2[i].num;
			mm=pK_ptr1[j].num;
			newflower=(int *)malloc((ll+mm+1)*sizeof(int));
			for (k=0;k<=mm;k++)
				newflower[k]=pK_ptr1[j].flower[k];
			for (k=mm+1;k<=ll+mm;k++)
				newflower[k]=oldnew[pK_ptr2[i].flower[k-mm]];
			if (j>p1->nodecount) free(pK_ptr1[j].flower);
			pK_ptr1[j].flower=newflower;
			pK_ptr1[j].num = ll+mm;
		 }
		else
		 {
			pR_ptr1[j].center=pR_ptr2[i].center;
			pR_ptr1[j].rad=pR_ptr2[i].rad;
			pK_ptr1[j].num=pK_ptr2[i].num;
			newflower=(int *)
			   malloc((pK_ptr2[i].num+1)*sizeof(int));
			for (k=0;k<=pK_ptr2[i].num;k++)
				newflower[k]=
				oldnew[pK_ptr2[i].flower[k]];
			if (j>p1->nodecount) free(pK_ptr1[j].flower);
			pK_ptr1[j].flower=newflower;
		 }
	 } /* end of for loop */
	if (b1>n && b2==n) delete(p1,endnode);
		/* use all bdry comp of p2, not p1; causes identification
		of endnode and v1 in p1. References to endnode
		have been changed. */
 } /* done with case of different packs */


	else /* adjoin pack to itself */
 {
	node=p1->nodecount;
/* first check if everything is compatable. */
	next=v1;
	if (!(count=bdry_comp_count(p1,v1))) return 0;
	dist=1;
	while ((next=pK_ptr1[next].flower[0])!=v2 
		&& dist<=count) dist++;
		/* dist counts edges to v2, if on same bdry comp */
	if (count>=p1->nodecount || count<2 /* error in combinatorics */
		|| dist==1			/* leave self-id'd edge */
		|| n>count			/* not enough edges */
		|| (dist<count && ((2*n)>(count-dist)||(2*n)==(count-dist-1)))
		|| (v1==v2 && (2*n==(count-1) || 2*n>count))
					/* leave self-id'd edge or
					conflicting ident's */
		) return 0;
	if ((dist=count-dist)==2*n) /* will be same as zip from vert 
		half way tween v1 and v2; indicate by setting v2=v1. */
	 {
		dist=n;
		while (dist>0)
		 {
			v1=pK_ptr1[v2].flower[0];
			v2=v1;
			dist--;
		 }
	 }
	if (count==dist && v1!=v2) 	/* v1, v2 on separate bdry comps*/
	 {
		next=v2;
		if (!(dist=bdry_comp_count(p1,v2))
			|| dist<3 		/* error in complex */
			|| n > dist		/* not long enough */
			|| (n==count && n!=dist) || (n==dist && n!=count)
				/* conflict on whether comps close up */
		) return 0;
	 }
/* now handle various possibilities */
	oldnew=(int *)malloc((node+2)*sizeof(int));
	for (kk=1;kk<=node;kk++) oldnew[kk]=kk;
	next=pK_ptr1[v1].flower[pK_ptr1[v1].num];
	v=pK_ptr1[v2].flower[0];
	if (next==v && v1!=v2) /* verts v2, next, v1 along edge */
	 {
		old_vert=pK_ptr1[next].flower[0]; /* v1 */
		new_vert=pK_ptr1[next].flower[pK_ptr1[next].num]; /* v2 */
		if (!close_up(p1,next)) return 0;
		for (kk=1;kk<=node;kk++)
		 {
			if (oldnew[kk]==old_vert) oldnew[kk]=new_vert;
			else if (oldnew[kk]>old_vert) oldnew[kk] -= 1;
		 }
	 }
	else if (v1==v2) /* zip up n edges */
	 {
	   next=v1;
	   for (j=1;j<=n;j++) 
	    {
		old_vert=pK_ptr1[next].flower[0];
		new_vert=pK_ptr1[next].flower[pK_ptr1[next].num];
		if (old_vert!=new_vert) 
		 {
			if (!close_up(p1,next)) return 0;
			for (kk=1;kk<=node;kk++)
			 {
				if (oldnew[kk]==old_vert) oldnew[kk]=new_vert;
				if (oldnew[kk]>old_vert) oldnew[kk] -= 1;
			 }
			if (new_vert>old_vert) new_vert--;
		 	next=new_vert;
		 }
		else {pK_ptr1[next].bdry_flag=1;j=n+1;}
	    }
	 }
	else  
	 {
			/* attach first edge */
/* fix up v1 (will remove v2) */
		ll=pK_ptr1[v1].num;
		mm=pK_ptr1[v2].num;
		newflower=(int *)malloc((ll+mm+1)*sizeof(int));
		for (i=0;i<=ll;i++) newflower[i]=pK_ptr1[v1].flower[i];
		for (i=ll+1;i<=(ll+mm);i++) 
			newflower[i]= pK_ptr1[v2].flower[i-ll];
		free(pK_ptr1[v1].flower);
		pK_ptr1[v1].flower=newflower;
		pK_ptr1[v1].num = ll+mm;
/* fix up 'next', clockwise from v1 (will remove ctrclk'w nghb v of v2) */
		ll=pK_ptr1[v].num;
		mm=pK_ptr1[next].num;
		newflower=(int *)malloc((ll+mm+1)*sizeof(int));
		for (i=0;i<=ll;i++) newflower[i]=pK_ptr1[v].flower[i];
		for (i=ll+1;i<=(ll+mm);i++) 
			newflower[i]= pK_ptr1[next].flower[i-ll];
		free(pK_ptr1[next].flower);
		pK_ptr1[next].flower=newflower;
		pK_ptr1[next].num = ll+mm;
/* fix up things pointed to v (which is to be removed) */
		for (i=0;i<=pK_ptr1[v].num;i++)
		 {
			ii=pK_ptr1[v].flower[i];
			for (j=0;j<=pK_ptr1[ii].num;j++)
			 {
				jj=pK_ptr1[ii].flower[j];
				if (jj==v) pK_ptr1[ii].flower[j]=next;
			 }
		 }
/* fix up things pointed to v2 (which is to be removed )*/
		for (i=0;i<=pK_ptr1[v2].num;i++)
		 {
			ii=pK_ptr1[v2].flower[i];
			for (j=0;j<=pK_ptr1[ii].num;j++)
			 {
				jj=pK_ptr1[ii].flower[j];
				if (jj==v2) pK_ptr1[ii].flower[j]=v1;
			 }
		 }
/* now to remove v and v2, replaced by next and v1, respectively. 
Old names should no longer be in any flowers.*/
		delete(p1,v);
		for (kk=1;kk<=node;kk++)
		 {
			if (oldnew[kk]==v) oldnew[kk]=next;
			if (oldnew[kk]>v) oldnew[kk]--;
		 }
		if (v2>v) v2--;
		if (next>v) next--;
		if (v1>v) v1--;
		delete(p1,v2);
		for (kk=1;kk<=node;kk++)
		 {
			if (oldnew[kk]==v2) oldnew[kk]=v1;
			if (oldnew[kk]>v2) oldnew[kk]--;
		 }
		if (next>v2) next--;
/* now zip up chain of edges for rest of attachments */
	   if (n>1) for (j=1;j<=n-1;j++) 
	    {
		old_vert=pK_ptr1[next].flower[0];
		new_vert=pK_ptr1[next].flower[pK_ptr1[next].num];
		if (old_vert!=new_vert) 
		 {
			if (!close_up(p1,next)) return 0;
			for (kk=1;kk<=node;kk++)
			 {
				if (oldnew[kk]==old_vert) oldnew[kk]=new_vert;
				if (oldnew[kk]>old_vert) oldnew[kk]--;
			 }
			if (new_vert>old_vert) new_vert--;
		 	next=new_vert;
		 }
		else {pK_ptr1[next].bdry_flag=1;j=n+1;}
	    }
	 }
 } /* done with case of self-adjoin. */

/* finish up */
	for (i=1;i<=p1->nodecount;i++) /* set bdry_flag data */
	 {
		if (pK_ptr1[i].flower[0]==pK_ptr1[i].flower[pK_ptr1[i].num])
		 {
			pK_ptr1[i].bdry_flag=0;
			if (pR_ptr1[i].rad<=0) pR_ptr1[i].rad=.7;
			/* so ones now in interior don't have infinite radius*/
		 }
		else pK_ptr1[i].bdry_flag=1;
	 }
	choose_beta(p1);choose_gamma(p1);
	if (oldnewflag) free(oldnewflag);
	*Oldnew=oldnew;
		/* size might have decreased */
	alloc_pack_space(p1,p1->nodecount+1,1); 
	return 1;
} /* adjoin */

/*OK*/
int
close_up(p,vert) /* identify edges/vert on opposite sides of 
bdry vert 'vert'. */
int vert;
struct p_data *p;
{
	int next,v,i,ii,j,jj,w,*newflower,newnum,hold;
	float vert_next_angle,next_w_angle,*newoverlaps;
	struct K_data *pK_ptr;
	extern void delete();

	pK_ptr=p->packK_ptr;
/* fix up 'next', vertex after vert */
	if (pK_ptr[vert].num<3) return 0; /* too few faces */
	v=pK_ptr[vert].flower[0];
	next=pK_ptr[vert].flower[pK_ptr[vert].num];
/* if next==v, then vert and next should be interior; nothing more to do */
	if (next==v)
	 {
		pK_ptr[vert].bdry_flag=pK_ptr[next].bdry_flag=0;
		p->packR_ptr[vert].aim=p->packR_ptr[next].aim=2*M_PI; 
		return 1;
	 } 
/* fix up vert */
	pK_ptr[vert].flower[0]=next;
	pK_ptr[vert].bdry_flag=0;
	p->packR_ptr[vert].aim=2*M_PI;
	if (p->overlap_status)
	 {
		vert_next_angle=(0.5)*(pK_ptr[vert].overlaps[0]+
			pK_ptr[vert].overlaps[pK_ptr[vert].num]);
		pK_ptr[vert].overlaps[0]=
			pK_ptr[vert].overlaps[pK_ptr[vert].num]=
				vert_next_angle;
	 }

/* if next and v share common neighbors: either <v,vert,next> or
<v,vert,next,w> is a closed bdry comp; in former case, modify w
to put in latter case (one face disappears). */

	else if (pK_ptr[v].flower[0]==next
		|| (pK_ptr[v].flower[0]==
		(w=pK_ptr[next].flower[pK_ptr[next].num])) )
	 {
		if (pK_ptr[v].flower[0]==next) remove_edge(p,v,next);
		newnum=pK_ptr[v].num+pK_ptr[next].num;
		newflower=(int *)malloc((newnum+1)*sizeof(int));
		if (p->overlap_status) 
		 {
		   newoverlaps=(float *)malloc((newnum+1)*sizeof(float));
		   next_w_angle=(0.5)*
			(pK_ptr[w].overlaps[0]+
			pK_ptr[w].overlaps[pK_ptr[w].num]);
		 }
		hold=pK_ptr[next].num;
		for (i=0;i<=pK_ptr[next].num;i++)
		 {
		   newflower[i]=pK_ptr[next].flower[i];
		   if (p->overlap_status) newoverlaps[i]=
			pK_ptr[next].overlaps[i];
		 }
		for (i=1;i<=pK_ptr[v].num;i++)
		 {
		   newflower[i+pK_ptr[next].num]=pK_ptr[v].flower[i];
		   if (p->overlap_status) newoverlaps[i+pK_ptr[next].num]=
			pK_ptr[next].overlaps[i];
		 }
		free(pK_ptr[next].flower);
		pK_ptr[next].flower=newflower;
		pK_ptr[next].num=newnum;
		if (p->overlap_status)
		 {
			free(pK_ptr[next].overlaps);
			pK_ptr[next].overlaps=newoverlaps;
			pK_ptr[next].overlaps[0]=
			   pK_ptr[next].overlaps[newnum]=vert_next_angle;
			pK_ptr[next].overlaps[hold]=next_w_angle;	
		 }
			/* fix up w */
		pK_ptr[w].flower[pK_ptr[w].num]=next;
		if (p->overlap_status)
		   pK_ptr[w].overlaps[0]=pK_ptr[w].overlaps[pK_ptr[w].num]
			= vert_next_angle;
		pK_ptr[next].bdry_flag=	pK_ptr[w].bdry_flag=0;
		p->packR_ptr[next].aim=p->packR_ptr[w].aim=2*M_PI;
	 }

/* otherwise, next remains boundary vertex */
	else
	 {
		newnum=pK_ptr[v].num+pK_ptr[next].num;
		newflower=(int *)malloc((newnum+1)*sizeof(int));
		if (p->overlap_status) 
		   newoverlaps=(float *)malloc((newnum+1)*sizeof(float));
		for (i=0;i<=pK_ptr[v].num;i++)
		 {
		   newflower[i]=pK_ptr[v].flower[i];
		   if (p->overlap_status)
			newoverlaps[i]=pK_ptr[v].overlaps[i];
		 }
		for (i=1;i<=pK_ptr[next].num;i++)
		 {
		   newflower[i+pK_ptr[v].num]=pK_ptr[next].flower[i];
		   if (p->overlap_status)
			newoverlaps[i+pK_ptr[v].num]=pK_ptr[v].overlaps[i];
		 }
		if (p->overlap_status)
			newflower[pK_ptr[v].num]=vert_next_angle;
		pK_ptr[next].flower=newflower;
		pK_ptr[next].num=newnum;
		if (p->overlap_status)
		 {
			free(pK_ptr[w].overlaps);
			pK_ptr[w].overlaps=newoverlaps;
		 }
		p->packR_ptr[next].aim=-1.0;
	 }

/* fix up things with v in their flowers (v will be removed) */
	for (i=0;i<=pK_ptr[v].num;i++)
	 {
		ii=pK_ptr[v].flower[i];
		for (j=0;j<=pK_ptr[ii].num;j++)
		 {
			jj=pK_ptr[ii].flower[j];
			if (jj==v) pK_ptr[ii].flower[j]=next;
		 }
	 }
	delete(p,v);
	return 1;
} /* close_up */

/*OK*/
int
add_circles(p,datastr) /* add vertices to bdry verts in string.
Lose edge data. */
struct p_data *p;
char *datastr;
{
	int count=0,hits;
	char *endptr;
	extern struct Vertlist *node_link_parse();
	struct Vertlist *vertlist,*trace;
	extern int facedraworder(),complex_count();
	extern void vert_free();

	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
			count += add_vert(p,trace->v);
			trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&vertlist);
	 }
	if (count)
	 {
		if (!complex_count(p,FALSE)) return 0;
		facedraworder(p,NULL);
		sprintf(msgbuf,"Added %d circles to p%d.",
			count,pack_num(p));
		msg();
		return count;
	 }
	return 0;
} /* add_circles */

/*OK*/
int
add_vert(p,v) /* adds circle connected to v and clockwise bdry 
neighbor. Verify v is valid node, set needed default data. 
CAUTION: after return, be sure to update any corrupted pointers
pK_ptr and pR_ptr. */
int v;
struct p_data *p;
{
	int v2,node,i,n,*newflower,flag;
	float *newoverlaps;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int any_compcenter(),alloc_pack_space();
	extern void e_anglesum_overlap(),h_anglesum_overlap(),
		s_anglesum_overlap();

	if (v<1 || v>p->nodecount || !p->packK_ptr[v].bdry_flag) return 0;
	v2=p->packK_ptr[v].flower[p->packK_ptr[v].num];
	if ( (node=p->nodecount+1) > (p->sizelimit)
	   && !alloc_pack_space(p,node,1) )
	 {
	   sprintf(msgbuf,
	     "Space allocation problem with adding vertex.");
	   emsg();
	   return 0;
	 }
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr; 
	if (pR_ptr[v].rad<=0)  /* avoid infinite rad */
	 {
		pR_ptr[v].rad=.1;
	 }
/* add to flower of v */
	n=pK_ptr[v].num;
	newflower=(int *)malloc((n+2)*sizeof(int));
	for (i=0;i<=n;i++) newflower[i]=pK_ptr[v].flower[i];
	newflower[n+1]=node;
	free(pK_ptr[v].flower);
	pK_ptr[v].flower=newflower;
	if (pK_ptr[v].overlaps)
	 {
		newoverlaps=(float *)malloc((n+2)*sizeof(float));
		for (i=0;i<=n;i++) newoverlaps[i]=pK_ptr[v].overlaps[i];
		newoverlaps[n+1]=1.0;
		free(pK_ptr[v].overlaps);
		pK_ptr[v].overlaps=newoverlaps;
	 }
	pK_ptr[v].num=n+1;
/* add to flower of v2 */
	n=pK_ptr[v2].num;
	newflower=(int *)malloc((n+2)*sizeof(int));
	for (i=1;i<=n+1;i++) newflower[i]=pK_ptr[v2].flower[i-1];
	newflower[0]=node;
	free(pK_ptr[v2].flower);
	pK_ptr[v2].flower=newflower;
	if (pK_ptr[v2].overlaps)
	 {
		newoverlaps=(float *)malloc((n+2)*sizeof(float));
		for (i=0;i<=n;i++) newoverlaps[i]=pK_ptr[v2].overlaps[i];
		newoverlaps[n+1]=1.0;
		free(pK_ptr[v2].overlaps);
		pK_ptr[v2].overlaps=newoverlaps;
	 }
	pK_ptr[v2].num=n+1;
/* add new node */
	p->nodecount++;
	pK_ptr[node].num=1;
	pK_ptr[node].flower=(int *)malloc((2)*sizeof(int));
	pK_ptr[node].flower[0]=v;
	pK_ptr[node].flower[1]=v2;
	if (p->overlap_status)
	 {
		pK_ptr[node].overlaps=(float *)malloc(2*sizeof(float));
		pK_ptr[node].overlaps[0]=pK_ptr[node].overlaps[1]=1.0;
	 }
	pK_ptr[node].bdry_flag=1;
	pK_ptr[node].mark=0;
	pK_ptr[node].color=FG_COLOR;
	pR_ptr[node].rad=pR_ptr[v].rad;
	any_compcenter(p->hes,
		pR_ptr[v].center,pR_ptr[v2].center,&pR_ptr[node].center,
		pR_ptr[v].rad,pR_ptr[v2].rad,&pR_ptr[node].rad,1.0,1.0,1.0);
/* fix curvatures */
	if (p->hes<0)
	 {
	   h_anglesum_overlap(p,node,pR_ptr[node].rad,
		&pR_ptr[node].curv,&flag);
	   h_anglesum_overlap(p,v,pR_ptr[v].rad,&pR_ptr[v].curv,&flag);
	   h_anglesum_overlap(p,v2,pR_ptr[v2].rad,&pR_ptr[v2].curv,&flag);
	 }
	else if (p->hes==0)
	 {
	   e_anglesum_overlap(p,node,pR_ptr[node].rad,
		&pR_ptr[node].curv,&flag);
	   e_anglesum_overlap(p,v,pR_ptr[v].rad,&pR_ptr[v].curv,&flag);
	   e_anglesum_overlap(p,v2,pR_ptr[v2].rad,&pR_ptr[v2].curv,&flag);
	 }
	else if (p->hes>okerr)
	 {
	   s_anglesum_overlap(p,node,pR_ptr[node].rad,
		&pR_ptr[node].curv,&flag);
	   s_anglesum_overlap(p,v,pR_ptr[v].rad,&pR_ptr[v].curv,&flag);
	   s_anglesum_overlap(p,v2,pR_ptr[v2].rad,&pR_ptr[v2].curv,&flag);
	 }
/* set defualt aims */
	pR_ptr[node].aim=pR_ptr[v].aim=pR_ptr[v2].aim=-1;
	return 1;
} /* add_vert */

/*OK*/
int
enfold(p,p1) /* links p2 (counterclockwise node) to p3 (clockwise node),
making p1 interior. Already verified, p1 is bdry node. */
int p1;
struct p_data *p;
{
	int n,p2,p3,i,*newflower;
	float *newoverlaps;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	p2=pK_ptr[p1].flower[0]; /* first neighbor */
	p3=pK_ptr[p1].flower[pK_ptr[p1].num]; /* second neighbor */
/* adjust flower of p1 */
	n=pK_ptr[p1].num;
	newflower=(int *)malloc((n+2)*sizeof(int));
	for (i=0;i<=n;i++)
		newflower[i]=pK_ptr[p1].flower[i];
	newflower[n+1]=p2;
	free(pK_ptr[p1].flower);
	pK_ptr[p1].flower=newflower;
	if (pK_ptr[p1].overlaps)
	 {
		newoverlaps=(float *)malloc((n+2)*sizeof(float));
		for (i=0;i<=n;i++)
		   newoverlaps[i]=pK_ptr[p1].overlaps[i];
		newoverlaps[n+1]=pK_ptr[p1].overlaps[0];
		free(pK_ptr[p1].overlaps);
		pK_ptr[p1].overlaps=newoverlaps;
	 }
	pK_ptr[p1].bdry_flag=0;
	pR_ptr[p1].aim=2.0*M_PI;
	pK_ptr[p1].num++;
	if (pR_ptr[p1].rad<=0) /* avoid infinity rad */
	 {
		pR_ptr[p1].rad=.1; 
	 } 
/* add to flower of p2 */
	n=pK_ptr[p2].num;
	newflower=(int *)malloc((n+2)*sizeof(int));
	for (i=0;i<=n;i++) newflower[i]=pK_ptr[p2].flower[i];
	newflower[n+1]=p3;
	free(pK_ptr[p2].flower);
	pK_ptr[p2].flower=newflower;
	pK_ptr[p2].num++;
	if (pK_ptr[p2].overlaps)
	 {
		newoverlaps=(float *)malloc((n+2)*sizeof(float));
		newoverlaps[n+1]=1.0;
		free(pK_ptr[p2].overlaps);
		pK_ptr[p2].overlaps=newoverlaps;
	 }
	if (pK_ptr[p2].flower[0]==p3) 
	 {
		pK_ptr[p2].bdry_flag=0;
		if (pR_ptr[p2].rad<=0) /*avoid infinity*/
		 {
			pR_ptr[p2].rad=.1;
		 }
		if (pK_ptr[p2].overlaps) 	
			pK_ptr[p2].overlaps[n+1]=pK_ptr[p2].overlaps[0];
		pR_ptr[p2].aim=2.0*M_PI;
	 }
/* add to flower of p3 */
	n=pK_ptr[p3].num;
	newflower=(int *)malloc((n+2)*sizeof(int));
	for (i=1;i<=n+1;i++) newflower[i]=pK_ptr[p3].flower[i-1];
	newflower[0]=p2;
	free(pK_ptr[p3].flower);
	pK_ptr[p3].flower=newflower;
	if (pK_ptr[p3].overlaps)
	 {
		newoverlaps=(float *)malloc((n+2)*sizeof(float));
		for (i=1;i<=n+1;i++) newoverlaps[i]=pK_ptr[p3].overlaps[i-1];
		newoverlaps[0]=1.0;
		free(pK_ptr[p3].overlaps);
		pK_ptr[p3].overlaps=newoverlaps;
	 }
	pK_ptr[p3].num++;
	if (pK_ptr[p3].flower[pK_ptr[p3].num]==p2) 
	 {
		pK_ptr[p3].bdry_flag=0;
		if (pR_ptr[p3].rad<=0) /*avoid infinity*/
		 {
			pR_ptr[p3].rad=.1;
		 }
		if (pK_ptr[p3].overlaps)
			pK_ptr[p3].overlaps[0]=pK_ptr[p3].overlaps[n+1];
		pR_ptr[p2].aim=2.0*M_PI;
	 }
/* update pack structures and beta after returning */
	return 1;
} /* enfold */

/*OK*/
int
slit_complex(p,datastr) /* slit complex along edge chain, which is
created from list of vertices. Chain may start at
bdry or int, continue while interior, shouldn't disconnect complex.
Not very solid -- bad path may screw up complex. For interior cut,
need at least two edges. Lose overlap data. */
char *datastr;
struct p_data *p;
{
	int ind,back,forward,new,k,v,w,next,num,wnum,u,hits;
	int *oldflower,*newflower;
	struct Vertlist *vertlist,*vtrace,*node_link_parse();
	struct Edgelist *edgelist,*trace;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	char *endptr;
	extern void edge_free(),vert_free();
	extern int nghb(),open_edge();

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) == NULL )
		return 0; /* not valid list of vertices */
	free_overlaps(p);
/* make up edgelist */
	vtrace=vertlist;
	v=vtrace->v;
	if (vtrace->next!=NULL && nghb(p,v,(w=vtrace->next->v))>=0)
		edgelist=(struct Edgelist *)malloc(sizeof(struct Edgelist));
	else {vert_free(&vertlist);return 0;} /* no first edge */
	trace=edgelist;
	trace->v=v;
	trace->w=w;
	trace->next=NULL;
	v=w;
	vtrace=vtrace->next;
	while (vtrace->next!=NULL && nghb(p,v,(w=vtrace->next->v))>=0
		&& !pK_ptr[w].bdry_flag)
		/* stops if path disconnects or hits bdry */
	 {
		trace->next=(struct Edgelist *)malloc(sizeof(struct Edgelist));
		trace=trace->next;
		trace->next=NULL;
		trace->v=v;
		trace->w=w;
		v=w;
		vtrace=vtrace->next;
	 }
	
/* use it */
	trace=edgelist;
	v=trace->v;
	next=trace->w;
	if (pK_ptr[v].bdry_flag && open_edge(p,v,w)==0) 
		{edge_free(&edgelist);return 0;}  /* failed to open first edge */
	else if (!pK_ptr[v].bdry_flag) 	/* opening interior edge */
	 {
		if ((trace=trace->next)==NULL) /* need at least two edges to start */ 
			{edge_free(&edgelist);return 0;} 
		w=next;
		next=trace->w;
		   /* three verts are (in order) v,w,next, slitting towards next*/
	/* need one new vert (w splits into two) */
		new=p->nodecount+1;
		if (new>p->sizelimit-1 && !alloc_pack_space(p,new+1,1))
		 {
		   sprintf(msgbuf,
		     "Space allocation problem with adding vertex.");
		   emsg();
		   return 0;
		 }
	/* fix v */
		ind=nghb(p,v,w);
		newflower=(int *)malloc((pK_ptr[v].num+1)*sizeof(int));
		for (k=1;k<=pK_ptr[v].num;k++)
		   newflower[k]=pK_ptr[v].flower[(k+ind) % pK_ptr[v].num];
 		free(pK_ptr[v].flower);
		pK_ptr[v].flower=newflower;
		pK_ptr[v].flower[0]=new;
		pK_ptr[v].bdry_flag=1;
	/* fix w */
		back=nghb(p,w,v);
		forward=nghb(p,w,next);
		num=(forward+pK_ptr[w].num-back) % (wnum=pK_ptr[w].num);
		oldflower=(int *)malloc((pK_ptr[w].num+1)*sizeof(int));
		for (k=0;k<=pK_ptr[w].num;k++)
			oldflower[k]=pK_ptr[w].flower[k];
		newflower=(int *)malloc((num+1)*sizeof(int));
		for (k=0;k<=num;k++)
		   newflower[k]=oldflower[(k+back) % wnum];
		free(pK_ptr[w].flower);
		pK_ptr[w].flower=newflower;
		pK_ptr[w].num=num;
		pK_ptr[w].bdry_flag=1;
		free(oldflower);
	/* fix next */
		ind=nghb(p,next,w);
		newflower=(int *)malloc((pK_ptr[next].num+1)*sizeof(int));
		for (k=0;k<pK_ptr[next].num;k++)
		   newflower[k]=pK_ptr[next].flower
			[(k+ind) % pK_ptr[next].num];
		free(pK_ptr[next].flower);
		pK_ptr[next].flower=newflower;
		pK_ptr[next].flower[pK_ptr[next].num]=new;
		pK_ptr[next].bdry_flag=1;
	/* fix new */
		num=(back+wnum-forward) % wnum;
		pK_ptr[new].flower=(int *)malloc((num+1)*sizeof(int));
		for (k=1;k<num;k++)
		   pK_ptr[new].flower[k]=oldflower[(k+forward) % wnum];
		pK_ptr[new].flower[0]=next;
		pK_ptr[new].flower[num]=v;
		pK_ptr[new].num=num;
		pK_ptr[new].bdry_flag=1;
		for (k=1;k<num;k++)
		 {
			back=nghb(p,(u=pK_ptr[new].flower[k]),w);
			pK_ptr[u].flower[back]=new;
		 }
		pR_ptr[new].rad=pR_ptr[w].rad;
		pR_ptr[new].center=pR_ptr[w].center;
		p->nodecount++;
	 } /* end of else */
/* now, continue opening prescribed edges */
	while ((v=next) && (trace=trace->next)!=NULL
		&& open_edge(p,v,(next=trace->w)));
	edge_free(&edgelist);
	if (trace==NULL) return 1;
	else return 0;
} /* slit_complex */
			
/*OK*/
int 
open_edge(p,v,w) /* from bdry v to ngb interior w, open edge. 0 for
failure. */
struct p_data *p;
int v,w;
{
	int new,ind,back,k,u,*newflower;
	float *newoverlaps;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int nghb();

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	if (v<0 || v>p->nodecount || !pK_ptr[v].bdry_flag 
		|| (ind=nghb(p,v,w))<0 || pK_ptr[w].bdry_flag)
		return 0;
/* new vert */
	new=p->nodecount+1;
	if ( (p->nodecount+1) > (p->sizelimit)
	   && !alloc_pack_space(p,new,1) )
	 {
	   sprintf(msgbuf,
	     "Space allocation problem with adding vertex.");
	   emsg();
	   return 0;
	 }
	pK_ptr[new].num=pK_ptr[v].num-ind;
	pK_ptr[new].flower=(int *)malloc((pK_ptr[new].num+1)*sizeof(int));
	for (k=0;k<=pK_ptr[new].num;k++)
		pK_ptr[new].flower[k]=pK_ptr[v].flower[k+ind];
	if (p->overlap_status) 
	 {
		pK_ptr[new].overlaps=(float *)
			malloc((pK_ptr[new].num+1)*sizeof(float));
		for (k=0;k<=pK_ptr[new].num;k++)
			pK_ptr[new].overlaps[k]=pK_ptr[v].overlaps[k+ind];
	 }
	else pK_ptr[new].overlaps=NULL;
	pK_ptr[new].bdry_flag=1;
	pR_ptr[new].rad=pR_ptr[v].rad;
	pR_ptr[new].center=pR_ptr[v].center;
/* fix its ngb's flowers */
	for (k=1;k<=pK_ptr[new].num;k++)
	 {
		back=nghb(p,(u=pK_ptr[v].flower[k+ind]),v);
		pK_ptr[u].flower[back]=new;
	 }
/* fix v's flower */
	newflower=(int *)malloc((ind+1)*sizeof(int));
	for (k=0;k<=ind;k++) newflower[k]=pK_ptr[v].flower[k];
	free(pK_ptr[v].flower);
	pK_ptr[v].flower=newflower;
	if (pK_ptr[v].overlaps) 
	 {
		newoverlaps=(float *)malloc((ind+1)*sizeof(float));
		for (k=0;k<=ind;k++)
		   newoverlaps[k]=pK_ptr[v].overlaps[k];
		free(pK_ptr[v].overlaps);
		pK_ptr[v].overlaps=newoverlaps;
	 }
	pK_ptr[v].num=ind;
/* fix w */
	back=nghb(p,w,v);
	newflower=(int *)malloc((pK_ptr[w].num+1)*sizeof(int));
	for (k=0;k<pK_ptr[w].num;k++)
		newflower[k]=pK_ptr[w].flower[(k+back) % pK_ptr[w].num];
	newflower[pK_ptr[w].num]=new;
	free(pK_ptr[w].flower);
	pK_ptr[w].flower=newflower;
	if (pK_ptr[w].overlaps)
	 {
		newoverlaps=(float *)malloc((pK_ptr[w].num+1)*sizeof(float));
		for (k=0;k<pK_ptr[w].num;k++)
		   newoverlaps[k]=pK_ptr[w].overlaps[(k+back) % pK_ptr[w].num];
		newoverlaps[pK_ptr[w].num]=pK_ptr[w].overlaps[back];
		free(pK_ptr[w].overlaps);
		pK_ptr[w].overlaps=newoverlaps;
	 }
	pK_ptr[w].bdry_flag=1;
	p->nodecount++;
	return 1;
} /* open_edge */


/*OK*/
void
choose_alpha(p) /* Choose new alpha if needed. */
struct p_data *p;
{
	int flag=0,i;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	i=p->alpha;
	if (i>0 && i<=p->nodecount && !pK_ptr[i].bdry_flag) 
		/* alpha okay already */
	 {
		if (p->alpha==p->gamma) p->gamma=pK_ptr[p->alpha].flower[0];
		return;
	 }
	i=0;
	do
	 {
		i++;
		if (pK_ptr[i].flower[0]==pK_ptr[i].flower[pK_ptr[i].num])
			flag=1;
	 }
	while (i<p->nodecount && !flag);
	if (flag) p->alpha=i;
	else if (p->beta!=1) p->alpha=1;
	else p->alpha=pK_ptr[p->beta].flower[0];
	if (p->gamma==p->alpha) 
		p->gamma=pK_ptr[p->alpha].flower[0];
	return;
} /* choose_alpha */

/*OK*/
void
choose_beta(p) /* when need to choose new beta; avoid alpha */
struct p_data *p;
{
	int i;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	i=p->beta;
	if (i>0 && i<=p->nodecount 
		&& (pK_ptr[i].bdry_flag || p->num_bdry_comp==0)) return; /* okay */
	i=1;
	while (pK_ptr[i].bdry_flag==0 && i<p->nodecount) i++;
	if (i==p->alpha) i=pK_ptr[i].flower[0];
	p->beta=i;
}/*choose_beta*/

/*OK*/
void
choose_gamma(p) /* avoid alpha */
struct p_data *p;
{
	int i;

	i=p->gamma;
	if (i>0 && i<p->nodecount && i!=p->alpha) return; /* okay */
	if (p->alpha==1) p->gamma=p->packK_ptr[1].flower[0];
	else p->gamma=1;
	return;
}

/*OK*/
int
set_alpha(p,i)
struct p_data *p;
{
	struct K_data *pK_ptr;
	extern int facedraworder();

	pK_ptr=p->packK_ptr;
	if (p->status && i>0 && i<=p->nodecount && !pK_ptr[i].bdry_flag)
	 {
		p->alpha=i;
		if (i==p->gamma) p->gamma=pK_ptr[i].flower[0];
		facedraworder(p,NULL);
	 	return 1;
	 }
	return 0;
} /* set_alpha */

/*OK*/
int
set_beta(p,i)
struct p_data *p;
{
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	if (p->status && i>0 && i<=p->nodecount && pK_ptr[i].bdry_flag)
	 { p->beta=i; return 1; }
	return 0;
} /* set_beta */

/*OK*/
int
set_gamma(p,i)
struct p_data *p;
{
	if (p->status && i>0 && i<=p->nodecount && i != p->alpha)
	 {p->gamma=i;return 1;}
	return 0;
} /* set_gamma */


/*OK*/
int
add_layer(p,v1,v2,n) /* add layer of nodes to bdry nodes of pack p,
from v1 to v2, degree n. Return 1 if okay, 0 if at least one node
ends up with more than n neighbors. Already checked that v1 and v2 are
nodes in same bdry component.  */
struct p_data *p;
int v1,v2,n;
{
	int flag=1,need,vert,nextvert,i;
	extern int enfold(),complex_count(),facedraworder();
	extern void choose_beta(),set_aim_default();
	
	if (v2==v1) v2=p->packK_ptr[v2].flower[p->packK_ptr[v2].num];
	vert=v1;
	nextvert=p->packK_ptr[vert].flower[0];
	while (vert!=v2)	
		/* go until you get to v2 */
	 {
		if (n<p->packK_ptr[vert].num) flag=0;
		else if ((need=n-p->packK_ptr[vert].num-1)>0)
			for (i=1;i<=need;i++)
				add_vert(p,vert);
		enfold(p,vert);
		vert=nextvert;
		nextvert=p->packK_ptr[vert].flower[0];
	 }
	if (vert==v2)	/* now do v2 itself */
	 {
		if (n<p->packK_ptr[vert].num) flag=0;
		else if ((need=n-p->packK_ptr[vert].num-1)>0)
			for (i=1;i<=need;i++)
				add_vert(p,vert);
		enfold(p,vert);
	 }
	complex_count(p,FALSE);
	facedraworder(p,NULL);
	set_aim_default(p);
	choose_beta(p);
	return flag;
} /* add_layer */

/*OK*/
int 
ideal_bdry_node(p,v) /* connect all nodes of bdry containing v to
a new node (which becomes interior). Return 1 if successful. */
struct p_data *p;
int v;
{
	int i,j,next_vert,start,newnode,numb,flag,count,bdry_no;
	int *newflower,node;
	float *newoverlaps;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern void choose_alpha(),choose_beta(),
		set_aim_default();
	extern int complex_count(),facedraworder(),alloc_pack_space();
	
	if ( (node=p->nodecount+1) > (p->sizelimit)
	   && !alloc_pack_space(p,node,1) )
	 {
	   sprintf(msgbuf,
	     "Space allocation problem with adding vertex.");
	   emsg();
	   return 0;
	 }
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
/* find starting point of bdry containing v */
	flag=0;
	for (j=1;j<=p->num_bdry_comp;j++)
	 {
		if ((start=p->bdry_starts[j])==v) flag=1;
		next_vert=start;
		while (!flag && (next_vert=pK_ptr[next_vert].flower[0])!=start)
			if (next_vert==v) flag=1;
		if (flag) 
		 {
			bdry_no=j;
			break;
		 }
	 }
	if (!flag) return 0;
	next_vert=start=p->bdry_starts[bdry_no];
	count=1;
	while ((next_vert=pK_ptr[next_vert].flower[0])!=start) count++;
	if (count < 3) 
	 {
		sprintf(msgbuf,"Bdry comp has too few vertices for a flower");
		emsg();
		return 0;
	 }
	newnode=p->nodecount+1;
	pK_ptr[newnode].num=count;
	pK_ptr[newnode].bdry_flag=0;
	pK_ptr[newnode].mark=0;
	pK_ptr[newnode].color=FG_COLOR;
	next_vert=start;
	pK_ptr[newnode].flower=(int *)malloc((count+1)*sizeof(int));
	if (p->overlap_status)
	   pK_ptr[newnode].overlaps=(float *)malloc((count+1)*sizeof(float));
/* fix up orig bdry verts */
	for (j=0;j<count;j++)
	 {
		numb=pK_ptr[next_vert].num;
		newflower=(int *)malloc((numb+3)*sizeof(int));
		for (i=0;i<=numb;i++) 
			newflower[i]=pK_ptr[next_vert].flower[i];
		newflower[numb+1]=newnode;
		newflower[numb+2]=newflower[0];
		free(pK_ptr[next_vert].flower);
		pK_ptr[next_vert].flower=newflower;
		pK_ptr[next_vert].num=numb+2;
		pK_ptr[next_vert].bdry_flag=0;
		pK_ptr[newnode].flower[count-j]=next_vert;
		if (p->overlap_status)
			pK_ptr[newnode].overlaps[count-j]=1.0;
		next_vert=pK_ptr[next_vert].flower[0];
		if (pK_ptr[next_vert].overlaps)
		 {
			newoverlaps=(float *)malloc((numb+3)*sizeof(float));
			for (i=0;i<=numb;i++)
			   newoverlaps[i]=pK_ptr[next_vert].overlaps[i];
			newoverlaps[numb+1]=1.0;
			newoverlaps[numb+2]=
				pK_ptr[next_vert].overlaps[0];
			free(pK_ptr[next_vert].overlaps);
			pK_ptr[next_vert].overlaps=newoverlaps;
		 }
	 }
	pK_ptr[newnode].flower[0]=pK_ptr[newnode].flower[count];
	if (p->overlap_status)
	   pK_ptr[newnode].overlaps[0]=pK_ptr[newnode].overlaps[count];
	p->packR_ptr[newnode].rad=0.5;
	p->nodecount++;
	choose_alpha(p);
	choose_beta(p);
	complex_count(p,FALSE);
	facedraworder(p,NULL);
	set_aim_default(p);
	if (p->num_bdry_comp == 0)
	 {
	   sprintf(msgbuf,
		"Note: there are no bdry components for this complex.");
	   msg();
	 }
	pR_ptr[newnode].center.re=pR_ptr[newnode].center.im=0.0;
	pR_ptr[newnode].rad=.5;
	if (p->hes>okerr) /* sphere: set radius for hemisphere. */
		pR_ptr[newnode].rad=M_PI/2.0;
	return 1;
} /* ideal_bdry_node */

/*OK*/
int
add_barys(p,datastr) /* add trivalent verts to faces. */
struct p_data *p;
char *datastr;
{
	int count=0,hits,node;
	extern struct Vertlist *face_link_parse();
	struct Vertlist *facelist,*trace,*dups;
	char *endptr;
	extern void vert_free();
	extern int facedraworder(),complex_count();

/* fixup ??: save overlaps */
	if ( (facelist=face_link_parse(p,datastr,&endptr,&hits)) == NULL )
		return 0;
	if ( (node=p->nodecount+hits) > (p->sizelimit)
	   && !alloc_pack_space(p,node,1) )
	 {
	   sprintf(msgbuf,
	     "Space allocation problem with adding barycenters.");
	   emsg();
	   return 0;
	 }
	trace=facelist;
	while (trace!=NULL)
	 {
		dups=facelist;
		while (dups!=trace)  /* zero out any repeats */

		 {
			if (trace->v==dups->v)
			 { trace->v=0;continue;}
			dups=dups->next;
		 }
		if (trace->v) count += add_barycenter(p,trace->v);
		trace=trace->next;
	 }
	vert_free(&facelist);
	if (count)
	 {
		free_overlaps(p);
		complex_count(p,FALSE);
		facedraworder(p,NULL);
	 }
	return count;
} /* add_barys */

/*OK*/
int
add_barycenter(p,f) /* called only by 'add_barys'; put trivalent 
vert in face f; data not updated between calls. */
struct p_data *p;
int f;
{
	int i,j,v,w,new,indx,*newflower;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int nghb();
	
/* fixup ??: save overlaps */
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
/* add new vertex */
	new=p->nodecount+1;
	p->nodecount++;
	pK_ptr[new].num=3;
	pK_ptr[new].flower=(int *)malloc(4*sizeof(int));
	pK_ptr[new].flower[0]=pK_ptr[new].flower[3]=p->faces[f].vert[0];
	pK_ptr[new].flower[1]=p->faces[f].vert[1];
	pK_ptr[new].flower[2]=p->faces[f].vert[2];
	pK_ptr[new].bdry_flag=0;
	pK_ptr[new].mark=0;
	pK_ptr[new].color=FG_COLOR;
	pR_ptr[new].rad=pR_ptr[p->faces[f].vert[0]].rad;
	pR_ptr[new].aim=2.0*M_PI;
/* compute packed radius */
	if (p->hes<-okerr) h_riffle_vert(p,new);
	else if (p->hes>okerr) s_riffle_vert(p,new);
	else e_riffle_vert(p,new);
	if (p->hes<okerr)
	 {
		pR_ptr[new].center.re=
			(pR_ptr[p->faces[f].vert[0]].center.re
			+pR_ptr[p->faces[f].vert[1]].center.re
			+pR_ptr[p->faces[f].vert[2]].center.re)/3.0;
		pR_ptr[new].center.im=
			(pR_ptr[p->faces[f].vert[0]].center.im
			+pR_ptr[p->faces[f].vert[1]].center.im
			+pR_ptr[p->faces[f].vert[2]].center.im)/3.0;
	 }
	else {pR_ptr[new].center.re=0.0;pR_ptr[new].center.re=M_PI;}
/* adjust nghb flowers */
	for (i=0;i<3;i++)
	 {
		v=p->faces[f].vert[i];
		w=p->faces[f].vert[(i+1)%3];
		indx=nghb(p,v,w);
		newflower=(int *)malloc((pK_ptr[v].num+2)*sizeof(int));
		for (j=0;j<=indx;j++) newflower[j]=pK_ptr[v].flower[j];
		for (j=pK_ptr[v].num;j>indx;j--)
			newflower[j+1]=pK_ptr[v].flower[j];
		newflower[indx+1]=new;
		free(pK_ptr[v].flower);
		pK_ptr[v].flower=newflower;		
		pK_ptr[v].num++;
	 }
	return 1;
} /* add_barycenter */

/*OK*/
int
split_flips(p,datastr,flag) /* add verts to edges (flag=1) or flip
edges (flag=2). */
struct p_data *p;
char *datastr;
int flag;
{
	int count=0,hits,node;
	extern struct Edgelist *node_pair_link();
	struct Edgelist *edgelist,*trace;
	char *endptr;
	extern int complex_count(),facedraworder(),
		split_edge(),flip_edge();
	extern void fillcurves(),edge_free();

/* fixup ??: save overlaps */
	if ( (edgelist=node_pair_link(p,datastr,&endptr,&hits)) == NULL )
		return 0;
	if ( flag && (node=p->nodecount+hits) > (p->sizelimit)
	   && !alloc_pack_space(p,node,1) )
	 {
	   sprintf(msgbuf,
	     "Space allocation problem with adding vertices to edges.");
	   emsg();
	   return 0;
	 }
	trace=edgelist;
	while (trace!=NULL)
	 {
		if (flag==1) count += split_edge(p,trace->v,trace->w);
		else count +=flip_edge(p,trace->v,trace->w);
		trace=trace->next;
	 }
	edge_free(&edgelist);
	if (count)
	 {
		free_overlaps(p); 
		complex_count(p,FALSE);
		facedraworder(p,NULL);
		fillcurves(p);
	 }
	return count;
} /* split_flips */

/*OK*/
int
split_edge(p,v,w) /* called only by 'split_flips'; put deg-4 vertex in <v,w>;
face data not updated between calls. */
struct p_data *p;
int v,w;
{
	int j,ind_v,ind_w,bflag=0,new,lv,rv,*newflower;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int nghb(),e_riffle_vert(),h_riffle_vert();

/* fixup ??: save overlaps */
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	if (pK_ptr[v].bdry_flag && pK_ptr[w].bdry_flag)
	 {
		if (pK_ptr[v].flower[0]==w) bflag=1;
		else if (pK_ptr[w].flower[0]==v) /* swap v, w */
		 {
			bflag=1;
			v=w;
			w=pK_ptr[v].flower[0];
		 }
	 }
	if ((ind_v=nghb(p,v,w))<0 || (ind_w=nghb(p,w,v))<0)
		return 0;
	lv=pK_ptr[v].flower[ind_v+1];
	if (!bflag) rv=pK_ptr[w].flower[ind_w+1];
/* put in new vertex */
	new=p->nodecount+1;
	pK_ptr[new].flower=(int *)malloc((5-2*bflag)*sizeof(int));
	p->nodecount++;
	if (bflag) 
	 {
		pK_ptr[new].num=2;
		pK_ptr[new].flower[0]=w;
		pK_ptr[new].flower[1]=lv;
		pK_ptr[new].flower[2]=v;
		pK_ptr[new].bdry_flag=1;
		pK_ptr[new].mark=0;
		pK_ptr[new].color=FG_COLOR;
		pR_ptr[new].rad=pR_ptr[v].rad;
		pR_ptr[new].aim=-.1;
	 }
	else
	 {
		pK_ptr[new].num=4;
		pK_ptr[new].flower[0]=pK_ptr[new].flower[4]=v;
		pK_ptr[new].flower[1]=rv;
		pK_ptr[new].flower[2]=w;
		pK_ptr[new].flower[3]=lv;
		pK_ptr[new].bdry_flag=0;
		pK_ptr[new].mark=0;
		pK_ptr[new].color=FG_COLOR;
		pR_ptr[new].rad=.5*pR_ptr[v].rad;
		pR_ptr[new].aim=2.0*M_PI;
/* compute packed radius */
		if (p->hes<-okerr) h_riffle_vert(p,new);
		else if (p->hes>okerr) s_riffle_vert(p,new);
		else e_riffle_vert(p,new);
	 }
	if (p->hes<okerr) 
	 {
		pR_ptr[new].center.re=
			(pR_ptr[v].center.re+pR_ptr[w].center.re)/2.0;
		pR_ptr[new].center.im=
			(pR_ptr[v].center.im+pR_ptr[w].center.im)/2.0;
	 }
	else {pR_ptr[new].center.re=0.0;pR_ptr[new].center.re=M_PI;}
/* adjust nghb flowers */
	ind_v=nghb(p,lv,v);
	newflower=(int *)malloc((pK_ptr[lv].num+2)*sizeof(int));
	for (j=0;j<=ind_v;j++) newflower[j]=pK_ptr[lv].flower[j];
	for (j=pK_ptr[lv].num;j>ind_v;j--)
		newflower[j+1]=pK_ptr[lv].flower[j];
	newflower[ind_v+1]=new;
	free(pK_ptr[lv].flower);
	pK_ptr[lv].flower=newflower;		
	pK_ptr[lv].num++;
	if (!bflag)
	 {
		ind_w=nghb(p,rv,w);
		newflower=(int *)malloc((pK_ptr[rv].num+2)*sizeof(int));
		for (j=0;j<=ind_w;j++) newflower[j]=pK_ptr[rv].flower[j];
		for (j=pK_ptr[rv].num;j>ind_w;j--)
			newflower[j+1]=pK_ptr[rv].flower[j];
		newflower[ind_w+1]=new;
		free(pK_ptr[rv].flower);
		pK_ptr[rv].flower=newflower;		
		pK_ptr[rv].num++;
	 }
/* adjust v and w */
	ind_v=nghb(p,v,w);
	pK_ptr[v].flower[ind_v]=new;
	if (ind_v==0 && !pK_ptr[v].bdry_flag) 
		pK_ptr[v].flower[pK_ptr[v].num]=new;
	ind_w=nghb(p,w,v);
	pK_ptr[w].flower[ind_w]=new;
	if (ind_w==0 && !pK_ptr[w].bdry_flag) 
		pK_ptr[w].flower[pK_ptr[w].num]=new;
	return 1;
} /* split_edge */

/*OK*/
int
flip_edge(p,v,w) /* call only from 'split_flips'; for contig faces, 
replace current shared edge by edge between opposite vertices; 
some data not updated between calls. */
struct p_data *p;
int v,w;
{
	int j,ind_v,ind_w,lv,rv,*newflower;
	struct K_data *pK_ptr;
	extern int nghb();

/* fixup ??: save overlaps */
	pK_ptr=p->packK_ptr;
	if ((pK_ptr[v].bdry_flag && pK_ptr[v].flower[0]==w)
		|| (pK_ptr[w].bdry_flag && pK_ptr[w].flower[0]==v)
		|| (ind_v=nghb(p,v,w))<0 || (ind_w=nghb(p,w,v))<0 )
		return 0; /* not interior edge or error */
	lv=pK_ptr[v].flower[ind_v+1];
	rv=pK_ptr[w].flower[ind_w+1];
/* adjust flower of v */
	newflower=(int *)malloc(pK_ptr[v].num*sizeof(int));
	for (j=0;j<ind_v;j++) newflower[j]=pK_ptr[v].flower[j];
	for (j=ind_v+1;j<=pK_ptr[v].num;j++)
		newflower[j-1]=pK_ptr[v].flower[j];
	free(pK_ptr[v].flower);
	pK_ptr[v].flower=newflower;
	pK_ptr[v].num--;
	if (ind_v==0 && !pK_ptr[v].bdry_flag) 
		pK_ptr[v].flower[pK_ptr[v].num]=pK_ptr[v].flower[0];
/* adjust flower of w */
	newflower=(int *)malloc(pK_ptr[w].num*sizeof(int));
	for (j=0;j<ind_w;j++) newflower[j]=pK_ptr[w].flower[j];
	for (j=ind_w+1;j<=pK_ptr[w].num;j++)
		newflower[j-1]=pK_ptr[w].flower[j];
	free(pK_ptr[w].flower);
	pK_ptr[w].flower=newflower;
	pK_ptr[w].num--;
	if (ind_w==0 && !pK_ptr[w].bdry_flag) 
		pK_ptr[w].flower[pK_ptr[w].num]=pK_ptr[w].flower[0];
/* adjust nghb flowers */
	ind_v=nghb(p,lv,v);
	newflower=(int *)malloc((pK_ptr[lv].num+2)*sizeof(int));
	for (j=0;j<=ind_v;j++) newflower[j]=pK_ptr[lv].flower[j];
	for (j=pK_ptr[lv].num;j>ind_v;j--)
		newflower[j+1]=pK_ptr[lv].flower[j];
	newflower[ind_v+1]=rv;
	free(pK_ptr[lv].flower);
	pK_ptr[lv].flower=newflower;
	pK_ptr[lv].num++;
	
	ind_w=nghb(p,rv,w);
	newflower=(int *)malloc((pK_ptr[rv].num+2)*sizeof(int));
	for (j=0;j<=ind_w;j++) newflower[j]=pK_ptr[rv].flower[j];
	for (j=pK_ptr[rv].num;j>ind_w;j--)
		newflower[j+1]=pK_ptr[rv].flower[j];
	newflower[ind_w+1]=lv;
	free(pK_ptr[rv].flower);
	pK_ptr[rv].flower=newflower;
	pK_ptr[rv].num++;
	return 1;
} /* flip_edge */


/*OK*/
int
hex_refine(q) /* new combinatorics in pack q.
Each edge gets new vertex, each face broken into 4 faces.
Try to salvage old centers/radii. Propogate original overlaps
to new edges; new edge gets overlap of original edge it is
parallel to. (Note: orig verts end up with same vector of
overlaps they started with.) return 0 if fails.*/
struct p_data *q;
{
     int v,w,j,k,lv,rv,new,Num,count=1,j1,j2,j3,j4;
     struct New_verts *new_verts=NULL;
     struct p_data *p;
     struct K_data *pK_ptr1,*pK_ptr2;
     struct R_data *pR_ptr2;
     extern int vert_for_edge(),complex_count(),
     		facedraworder();
     extern void fillcurves(),set_aim_default();

	if (!q->status || !(pK_ptr1=(struct K_data *)
		calloc((size_t)(q->nodecount+2),sizeof(struct K_data)))
	  || !(p=(struct p_data *)malloc(sizeof(struct p_data)))
	   ) goto ABORT;
	*p=*q;
	p->packK_ptr=pK_ptr1;
	Num=(q->facecount)+(q->nodecount)-(q->euler); /* number of edges */
	if (!alloc_pack_space(q,q->nodecount+Num,1))
	 {
	   sprintf(msgbuf,"hex_refine failed; space allocation problems.");
	   emsg();
	   goto ABORT;
	 }
	pK_ptr2=q->packK_ptr;pR_ptr2=q->packR_ptr;
	for(j=1;j<=q->nodecount;j++) 
		/* copy needed as changes made. Make space twice as 
		long to hold crossref data for vert_for_edge. */
	 {
		pK_ptr1[j]=pK_ptr2[j];
		pK_ptr1[j].flower=(int *)
			calloc((size_t)(2*pK_ptr1[j].num+2),sizeof(int));
		for (k=0;k<=pK_ptr1[j].num;k++)
			pK_ptr1[j].flower[k]=pK_ptr2[j].flower[k];
		free(pK_ptr2[j].flower);
		pK_ptr2[j].flower=NULL;
	 }

/* create new vertex indices and crossref data. */
	new_verts=(struct New_verts *)malloc((Num+2)*sizeof(struct New_verts));
	for (v=1;v<q->nodecount;v++)
	for (k=0;k<(pK_ptr1[v].num+pK_ptr1[v].bdry_flag);k++)
        if ((w=pK_ptr1[v].flower[k])>v)
	 {
		new_verts[count].v=v;
		new_verts[count].w=w;
		pK_ptr1[v].flower[pK_ptr1[v].num+1+k]=q->nodecount+count;
		count++;
	 }
		/* NOTE: store new vertex number only when w>v */
	if (count!=(Num+1)) goto ABORT;

/* fix original flowers */
	for (v=1;v<=q->nodecount;v++)
	 {
		pK_ptr2[v].flower=(int *)
			malloc((pK_ptr1[v].num+1)*sizeof(int));
		for (j=0;j<=pK_ptr1[v].num;j++)
             		if ((pK_ptr2[v].flower[j]=vert_for_edge(p,v,j)) < 0) 
			goto ABORT;
        	pR_ptr2[v].rad *= 0.5; 
                    /* cut radius in half as first guess */
      }

/* fix flowers of new verts. */
			
     for (new=q->nodecount+1;new<=q->nodecount+Num;new++) 
      {
          v=new_verts[new-q->nodecount].v;
          w=new_verts[new-q->nodecount].w;
	  if ((pK_ptr1[v].bdry_flag && w==pK_ptr1[v].flower[0])
		|| (pK_ptr1[w].bdry_flag && v==pK_ptr1[w].flower[0])) 
			/* bdry edge? in one face only. */
	   {
		if (v==pK_ptr1[w].flower[0]) {j=v;v=w;w=j;}
			/* now <v,w> is ctr-clkwise bdry edge */
		pK_ptr2[new].flower=(int *)malloc(4*sizeof(int));
		pK_ptr2[new].flower[0]=w;
		pK_ptr2[new].flower[1]=vert_for_edge(p,w,pK_ptr1[w].num-1);
		pK_ptr2[new].flower[2]=vert_for_edge(p,v,1);
		pK_ptr2[new].flower[3]=v;
		pK_ptr2[new].num=3;
		pK_ptr2[new].bdry_flag=1;
		if (q->overlap_status)
		 {
		   pK_ptr2[new].overlaps=(float *)malloc(4*sizeof(float));
		   pK_ptr2[new].overlaps[0]=pK_ptr1[w].overlaps[pK_ptr1[w].num];
		   pK_ptr2[new].overlaps[1]=pK_ptr1[v].overlaps[1];
		   pK_ptr2[new].overlaps[2]=pK_ptr1[w].overlaps[pK_ptr1[w].num-1];
		   pK_ptr2[new].overlaps[3]=pK_ptr1[v].overlaps[0];
		 }
	    }
	   else
	    {
		pK_ptr2[new].flower=(int *)malloc(7*sizeof(int));
		pK_ptr2[new].flower[0]=w;
		j1=(nghb(p,w,v)-1+pK_ptr1[w].num)%(pK_ptr1[w].num);
		pK_ptr2[new].flower[1]=vert_for_edge(p,w,j1);
		j2=(nghb(p,v,w)+1);
		pK_ptr2[new].flower[2]=vert_for_edge(p,v,j2);
		pK_ptr2[new].flower[3]=v;
		j3=(j2-2+pK_ptr1[v].num)%(pK_ptr1[v].num);
		pK_ptr2[new].flower[4]=vert_for_edge(p,v,j3);
		j4=(nghb(p,w,v)+1);
		pK_ptr2[new].flower[5]=vert_for_edge(p,w,j4);
		pK_ptr2[new].flower[6]=w;
		pK_ptr2[new].num=6;
		pK_ptr2[new].bdry_flag=0;
		if (q->overlap_status)
		 {
		   pK_ptr2[new].overlaps=(float *)malloc(7*sizeof(float));
		   pK_ptr2[new].overlaps[0]=
			pK_ptr2[new].overlaps[6]=
			pK_ptr1[w].overlaps[nghb(p,w,v)];
		   pK_ptr2[new].overlaps[1]=pK_ptr1[v].overlaps[j2];
		   pK_ptr2[new].overlaps[2]=pK_ptr1[w].overlaps[j1];
		   pK_ptr2[new].overlaps[3]=pK_ptr1[v].overlaps[nghb(p,v,w)];
		   pK_ptr2[new].overlaps[4]=pK_ptr1[w].overlaps[j4];
		   pK_ptr2[new].overlaps[5]=pK_ptr1[v].overlaps[j3];
		 }
	    }
	pK_ptr2[new].mark=0;
	pK_ptr2[new].color=FG_COLOR;
        pR_ptr2[new].rad=(pR_ptr2[v].rad+pR_ptr2[w].rad)/2.0;
	pR_ptr2[new].center=pR_ptr2[v].center;
	if (q->hes <= okerr)
	 {
          pR_ptr2[new].center.re=
		(pR_ptr2[v].center.re+pR_ptr2[w].center.re)/2.0;
          pR_ptr2[new].center.im=
		(pR_ptr2[v].center.im+pR_ptr2[w].center.im)/2.0;
	 }
      }

	if (pK_ptr1) 
	 {
		for (j=1;j<=q->nodecount;j++) 
		   if (pK_ptr1[j].flower) free(pK_ptr1[j].flower);
		free(pK_ptr1);
	 }
	if (p) free(p);
	q->nodecount=q->nodecount+Num;
	complex_count(q,FALSE);
	facedraworder(q,NULL);
	fillcurves(q);
	set_aim_default(q);
	if (new_verts) free(new_verts);
	return 1;

ABORT:
	if (new_verts) free(new_verts);
	if (pK_ptr1) 
	 {
		for (j=1;j<=q->nodecount;j++) 
		   if (pK_ptr1[j].flower) free(pK_ptr1[j].flower);
		free(pK_ptr1);
	 }
	if (p) free(p);
	return 0;
} /* hex_refine */

/*OK*/
int
vert_for_edge(p,v,j) /* WARNING: use ONLY in hex_refine. Given 
edge v and index j to a petal, return new vert number for midpoint. 
New nos. have been temporarily stored as extra data in second half 
of p flower ptrs. -1 for error. */
struct p_data *p;
int v,j;
{
     int k,w;
     extern int nghb();

	if (j<0 || j>p->packK_ptr[v].num) return -1;
	if (!p->packK_ptr[v].bdry_flag && j==p->packK_ptr[v].num) j=0;
	w=p->packK_ptr[v].flower[j];
	if (v<w) return p->packK_ptr[v].flower[p->packK_ptr[v].num+1+j];
	if ((k=nghb(p,w,v))<0) return -1;
	return p->packK_ptr[w].flower[p->packK_ptr[w].num+1+k];
} /* vert_for_edge */
          

/*OK*/
int
remove_circle(p,datastr) /* remove one trivalent or bdry vert */
struct p_data *p;
char *datastr;
{
	int i,v,w,flag,count=0,hits,*newflower;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern struct Vertlist *node_link_parse();
	struct Vertlist *vertlist,*trace,*newtrace;
	char *endptr;
	extern int remove_bdry_vert(),remove_tri_vert();
	extern void vert_free(),fillcurves();

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) == NULL )
		return 0;
	trace=vertlist;
	while (trace!=NULL)
	 {
		if((v=trace->v)==0) 
			{trace=trace->next;continue;}
		flag=0;
		if (!pK_ptr[v].bdry_flag)
		 {
			if (pK_ptr[v].num!=3) flag++;
			else for (i=0;i<3;i++)
			 {
			   w=pK_ptr[v].flower[i];
			   if ((!pK_ptr[w].bdry_flag && pK_ptr[w].num<=3)
			      || (pK_ptr[w].bdry_flag && pK_ptr[w].num<2))
			      flag++;
			 }
		 }
		else if (pK_ptr[v].bdry_flag)
		 {
			if (pK_ptr[pK_ptr[v].flower[0]].num<2
			   || pK_ptr[pK_ptr[v].flower[pK_ptr[v].num]].num<2)
				flag++;
			for (i=1;i<pK_ptr[v].num;i++) 
			   if (pK_ptr[pK_ptr[v].flower[i]].bdry_flag) flag++;
		 }
		if (flag) {trace=trace->next;continue;} 
			/* vert doesn't qualify */
		if (pK_ptr[v].bdry_flag)
		 {
			for (i=0;i<=pK_ptr[v].num;i++) 
				pR_ptr[pK_ptr[v].flower[i]].aim=-.1;
			count += remove_bdry_vert(p,v);
		 }
		else
		 {
			count += remove_tri_vert(p,v);
		 }
		newtrace=trace->next;
		while (newtrace!=NULL) /* fix remainder of list */
		 {
			if (newtrace->v==v) newtrace->v=0;
			else if (newtrace->v>v) newtrace->v--;
		 	newtrace=newtrace->next;
		 }
		trace=trace->next;
	 }
	if (!count) 
	 {
		sprintf(msgbuf,"no vertices qualified for deletion.");
		emsg();
		return 0;
	 }
	vert_free(&vertlist);
	complex_count(p,FALSE);
	facedraworder(p,NULL);
	fillcurves(p);
	return 1;
} /* remove_circle */

/*OK*/
int
remove_bdry_vert(p,v) /* remove vert v from pack p; have verified 
that v is bdry vert, removal will not disconnect or leave a 
neighbor without any faces. */
int v;
struct p_data *p;
{
	int i,j,num,m,w,eflag,*newflower,indx,start;
	float *newoverlaps;
	struct K_data *pK_ptr;
	extern void delete();

	pK_ptr=p->packK_ptr;
/* fix flowers of neighbors */
	for (i=0;i<=pK_ptr[v].num;i++)
	 {
		w=pK_ptr[v].flower[i];
		num=pK_ptr[w].num;
		if (pK_ptr[w].bdry_flag)
		 {
		   newflower=(int *)malloc(pK_ptr[w].num*sizeof(int));
		   if (pK_ptr[w].flower[0]==v) 
			for (j=0;j<pK_ptr[w].num;j++) 
			   newflower[j]=pK_ptr[w].flower[j+1];
		   else if (pK_ptr[w].flower[pK_ptr[w].num]==v)
			for (j=0;j<pK_ptr[w].num;j++)
			   newflower[j]=pK_ptr[w].flower[j];
		   if (pK_ptr[w].overlaps)
		    {
			newoverlaps=(float *)
				malloc(pK_ptr[w].num*sizeof(float));
			if (pK_ptr[w].flower[0]==v) 
			   for (j=0;j<pK_ptr[w].num;j++) 
				newoverlaps[j]=pK_ptr[w].overlaps[j+1];
			else if (pK_ptr[w].flower[pK_ptr[w].num]==v)
			   for (j=0;j<pK_ptr[w].num;j++)
				newoverlaps[j]=pK_ptr[w].overlaps[j];
		   	free(pK_ptr[w].overlaps);
		   	pK_ptr[w].overlaps=newoverlaps;
		    }
		   free(pK_ptr[w].flower);
		   pK_ptr[w].flower=newflower;
		   pK_ptr[w].num--;
		 }
		else
		 {
		   indx=nghb(p,w,v);
		   newflower=(int *)
		 	malloc((pK_ptr[w].num-1)*sizeof(int));
		   start=(indx+1)%pK_ptr[w].num;
		   for (j=0;j<pK_ptr[w].num-1;j++) newflower[j]=
			pK_ptr[w].flower[(start+j)%pK_ptr[w].num];
		   if (pK_ptr[w].overlaps)
		    {
		   	newoverlaps=(float *)
		 	   malloc((pK_ptr[w].num-1)*sizeof(float));
		   	start=(indx+1)%pK_ptr[w].num;
		   	for (j=0;j<pK_ptr[w].num-1;j++) newoverlaps[j]=
			   pK_ptr[w].overlaps[(start+j)%pK_ptr[w].num];
		   	free(pK_ptr[w].overlaps);
		   	pK_ptr[w].overlaps=newoverlaps;
		    }
		   free(pK_ptr[w].flower);
		   pK_ptr[w].flower=newflower;
		   pK_ptr[w].num=num-2;
		   pK_ptr[w].bdry_flag=1;
			
		 }
	 } /* done removing references to v */
	delete(p,v);
	return 1;
} /* remove_bdry_vert */

/*OK*/
int
remove_tri_vert(p,v) /* remove a trivalent, interior vert.
Already checked that v qualifies. */
struct p_data *p;
int v;
{
	int j,k,w,indx,*newflower;
	float *newoverlaps;
	struct K_data *pK_ptr;
	extern int nghb();
	extern void delete();
	
	pK_ptr=p->packK_ptr;
/* adjust nghbs */
	for (k=0;k<3;k++)
	 {
		w=pK_ptr[v].flower[k];
		indx=nghb(p,w,v);
		newflower=(int *)malloc(pK_ptr[w].num*sizeof(int));
		for (j=0;j<indx;j++) newflower[j]=pK_ptr[w].flower[j];
		for (j=indx;j<pK_ptr[w].num;j++)
		 {
			newflower[j]=pK_ptr[w].flower[j+1];
		 }
		free(pK_ptr[w].flower);
		pK_ptr[w].flower=newflower;
		if (pK_ptr[w].overlaps)
		 {
		   newoverlaps=(float *)malloc(pK_ptr[w].num*sizeof(float));
		   for (j=0;j<indx;j++) 
			newoverlaps[j]=pK_ptr[w].overlaps[j];
		   for (j=indx;j<pK_ptr[w].num;j++)
		    {
			newoverlaps[j]=pK_ptr[w].overlaps[j+1];
		    }
		   free(pK_ptr[w].overlaps);
		   pK_ptr[w].overlaps=newoverlaps;
		 }
		pK_ptr[w].num--;
		if (indx==0 && !pK_ptr[w].bdry_flag)
		 {
			pK_ptr[w].flower[pK_ptr[w].num]=
				pK_ptr[w].flower[0];
			if (pK_ptr[w].overlaps)
				pK_ptr[w].overlaps[pK_ptr[w].num]=
					pK_ptr[w].overlaps[0];
		 }
	 }
	delete(p,v);
	return 1;
} /* remove_tri_vert */

/*OK*/
int
remove_quad_vert(p,datastr) /* remove 4-deg interior vert v, but put 
in edge between w and the opposite vert z. Only does one vert now,
and overlaps are discarded. */
struct p_data *p;
char *datastr;
{
	int v,w,z,j,x,indx,hits,*newflower;
	struct K_data *pK_ptr;
	extern struct Vertlist *node_link_parse();
	struct Vertlist *vertlist;
	char *endptr;
	extern void delete(),vert_free(),fillcurves();
	extern int complex_count(),facedraworder();
	
	pK_ptr=p->packK_ptr;
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) == NULL 
		|| vertlist->next==NULL)
		return 0;
/* fixup ?? : should try to save overlaps */
	free_overlaps(p);
	v=vertlist->v;
	w=vertlist->next->v;
	vert_free(&vertlist); /* only do one vert for now */	
	if (pK_ptr[v].bdry_flag || pK_ptr[v].num!=4
		|| (indx=nghb(p,w,v))<0) return 0;
	z=pK_ptr[v].flower[(nghb(p,v,w)+2)%pK_ptr[v].num]; 
/* fix w and z*/
	if (indx==0) 
		pK_ptr[w].flower[0]=pK_ptr[w].flower[pK_ptr[w].num]=z;
	else pK_ptr[w].flower[indx]=z;
	if ((indx=nghb(p,z,v))==0) 
		pK_ptr[z].flower[0]=pK_ptr[z].flower[pK_ptr[z].num]=w;
	else pK_ptr[z].flower[indx]=w;
/* remove v from others */
	x=pK_ptr[v].flower[(nghb(p,v,w)+1)%pK_ptr[v].num];
	indx=nghb(p,x,v);
	newflower=(int *)malloc(pK_ptr[x].num*sizeof(int));
	for (j=0;j<indx;j++) newflower[j]=pK_ptr[x].flower[j];
	for (j=indx;j<=pK_ptr[x].num;j++)
	 {
		newflower[j]=pK_ptr[x].flower[j+1];
	 }
	free(pK_ptr[x].flower);
	pK_ptr[x].flower=newflower;
	pK_ptr[x].num--;
	if (indx==0 && !pK_ptr[x].bdry_flag)
	 {
		pK_ptr[x].flower[pK_ptr[x].num]=pK_ptr[x].flower[0];
	 }
	x=pK_ptr[v].flower[(nghb(p,v,z)+1)%pK_ptr[v].num];
	indx=nghb(p,x,v);
	newflower=(int *)malloc(pK_ptr[x].num*sizeof(int));
	for (j=0;j<indx;j++) newflower[j]=pK_ptr[x].flower[j];
	for (j=indx;j<=pK_ptr[x].num;j++)
	 {
		newflower[j]=pK_ptr[x].flower[j+1];
	 }
	free(pK_ptr[x].flower);
	pK_ptr[x].flower=newflower;
	pK_ptr[x].num--;
	if (indx==0 && !pK_ptr[x].bdry_flag)
	 {
		pK_ptr[x].flower[pK_ptr[x].num]=pK_ptr[x].flower[0];
	 }
	delete(p,v);
	complex_count(p,FALSE);
	facedraworder(p,NULL);
	fillcurves(p);
	return 1;
} /* remove_quad_vert */
	
/*OK*/
int
puncture(p,datastr) /* remove one interior vert; must have at least 2 generations of interior nbhs. lose edge data. */
struct p_data *p;
char *datastr;
{
	int k,j,w,v,*newflower,start;
	struct K_data *pK_ptr;
	extern void delete(),fillcurves();
	extern int complex_count(),facedraworder(),handle_cmd(),
		pack_num();

	pK_ptr=p->packK_ptr;
/* check suitability */
	if (sscanf(datastr,"%d",&v)==0 || v<1 || v>p->nodecount
		|| pK_ptr[v].bdry_flag) return 0;
	for (j=0;j<pK_ptr[v].num;j++)
	 {
		if (pK_ptr[w=pK_ptr[v].flower[j]].num<4) return 0;
		for (k=0;k<=pK_ptr[w].num;k++)
			if (pK_ptr[pK_ptr[w].flower[k]].bdry_flag) return 0;
	 }

/* carry on */
/* fixup ?? : should try to save overlaps */
	free_overlaps(p); /* lose data */

/* adjust neighbors */
	for (j=0;j<pK_ptr[v].num;j++)
	 {
		w=pK_ptr[v].flower[j];
		if ((start=nghb(p,w,v))<0) return 0; /* data will be screwy */
		newflower=(int *)malloc((pK_ptr[w].num-1)*sizeof(int));
		for (k=0;k<pK_ptr[w].num-1;k++) 
		   newflower[k]=pK_ptr[w].flower[(start+1+k) % (pK_ptr[w].num)];
		pK_ptr[w].num -= 2;
		pK_ptr[w].bdry_flag=1;
		free(pK_ptr[w].flower);
		p->packR_ptr[w].aim=-.1;
		pK_ptr[w].flower=newflower;
	 }
	delete(p,v);
	complex_count(p,FALSE);
	facedraworder(p,NULL);
	sprintf(buf,"set_aim -p%d -d",pack_num(p));
	handle_cmd(buf);
	fillcurves(p);
	return 1;
} /* puncture */

/*OK*/
void	
delete(p,v) /* routine to remove vertex from pack. Assume flowers 
pointing to it already adjusted.  */
int v;
struct p_data *p;
{
	int i,k;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern void choose_alpha(),choose_beta();

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	for (i=1;i<=p->nodecount;i++)
		if (i!=v) for (k=0;k<=pK_ptr[i].num;k++)
			if (pK_ptr[i].flower[k]>v) pK_ptr[i].flower[k]--;
	if (pK_ptr[v].flower) free(pK_ptr[v].flower);
	if (pK_ptr[v].overlaps) free(pK_ptr[v].overlaps);
	for (i=v;i<p->nodecount;i++)
	 {
		pK_ptr[i]=pK_ptr[i+1];
		pR_ptr[i]=pR_ptr[i+1];
	 }
	pK_ptr[p->nodecount].flower=NULL;
	p->nodecount--;
	if (p->active_node>=v) p->active_node--;
	if (p->alpha>v) p->alpha--;
	else if (p->alpha==v) choose_alpha(p);
	if (p->beta>v) p->beta--;
	else if (p->beta==v) choose_beta(p);
	return;
} /* delete */

/*OK*/
int
fix_call(p,datastr) /* for "Fix" command and fix calls */
struct p_data *p;
char *datastr;
{
	int count=0,hits,v,w,n,flag=0,d;
	extern struct Vertlist *face_link_parse(),*try_hex_pg();
	struct Vertlist *facelist=NULL;
	char *nextptr,*endptr;
	extern void set_aim_default(),fillcurves(),vert_free();
	extern int comp_pack_centers(),layout_report(),
		complex_count(),facedraworder();

	nextptr=datastr;
	while (grab_next(&nextptr,next) && next[0]=='-') /* flags set */
	 {
		flag=1;
		switch(next[1])
		 {
		   case 'a': /* default aims */
		    {
			set_aim_default(p);
			count++;
			break;
		    }
		   case 'c': /* centers */
		    {
			comp_pack_centers(p);
			count++;
			break;
		    }
		   case 'd': /* layout by drawing order and report; return*/
		    {
			if (sscanf(nextptr,"%d",&v)!=1 || v<1 
			   || v>p->nodecount) 
				v=0;
			layout_report(p,v,TRUE);
			count++;
			return count;
		    }			
		   case 'F': /* redo everything */
		    {
			complex_count(p,TRUE);
			facedraworder(p,NULL);
			fillcurves(p);
			comp_pack_centers(p);
			set_aim_default(p);
			count++;
			return count;
		    }
		   case 'f': /* given face list */
		    {
			if ( (facelist=
			   face_link_parse(p,nextptr,&endptr,&hits)) )
			count +=facedraworder(p,facelist);
			vert_free(&facelist);
			return count; /* this has to be last flag */
		    }
		   case 'h': /* drawing order via hex_walk routine */
		    {
			if ( sscanf(nextptr,"%d %d %d",&v,&w,&n)!=3
			   || !(facelist=try_hex_pg(p,v,w,n,&d,&d,&d,&d)) ) 
				break;
			if (!facelist) /* some error */
			 {
				sprintf(msgbuf,"hex_walk failed.");
				emsg();
			 }
			else count +=facedraworder(p,facelist);
			vert_free(&facelist);
			break;
		    }
		   case 'K': /* combinatorics */
		    {
			complex_count(p,FALSE);
			facedraworder(p,NULL);
			count++;
			break;
		    }
		   case 'r': /* recompute centers along given facelist,
				but don't draw them */
		    {
			if ( (facelist=
			   face_link_parse(p,nextptr,&endptr,&hits)) )
			count += layout_facelist(p,0,&facelist,1,0,0,NOSHOW);
			break;
		    }
		   case 's': /* angle sums */
		    {
			fillcurves(p);
			count++;
			break;
		    }
		  } /* end of switch */
	 } /* end of while */
	if (flag) return count;
	fillcurves(p);
	comp_pack_centers(p);
	count++;
	return count;
} /* fix_call */


/* ===================  facedraworder =============== */
/* Adapted, in part, from Woodrow Johnson MS Thesis, UTK, 1994 */

/* Terminology/strategy: Faces should = "red" or "white".
Red = simple closed chain of bdry faces and ones along each
side of "cuts" needed for multip-connected case; rest are white.
Blue = tmp situation: can't fit in red chain combinatorially,
so we attempt to attach them, putting some kinks (side steps)
in red chain. Finally, end up with linked list for drawing and 
separate closed list for red faces (all kept in p->faces data).
*/

/*OK*/
int
facedraworder(p,drawlist) /* return 1 if succeed. If drawlist,
use as red chain to alter final drawing. drawlist unchanged. */
struct p_data *p;
struct Vertlist *drawlist;
{
	int i,j,loop_count=0,count=1,alpha,*face_ptr,f,num;
	int tripflag,stopface,dummy,f1,f2,ind;
	int stopvert,lastvert,stop=0,looplimit,stopflag;
	struct stface *fhandle,*chandle,*temp,*killit,*hold;
	struct Vertlist *face_order,*fdo,*trace;
	struct K_data *pK_ptr;
	extern int nghb_tri(),find_index(),catalog_faces(),zip_vert(),
		free_face();
	extern void vert_free(),add_face_order();

	pK_ptr=p->packK_ptr;
	catalog_faces(p); 

/* try to arrange so given list will be in red chain. Last face should
equal first. Mark outside edge from list of faces first to avoid crossing
it, then computed red chain should incorporate it. */

	for (j=1;j<=p->nodecount;j++) p->packK_ptr[j].plot_flag=0;
	if (drawlist && drawlist->next && drawlist->next->next) 	
	 {
		trace=drawlist;
		while (trace->next->next!=NULL) trace=trace->next;
		f=trace->v; /* find last face */
		trace=drawlist;
		while (trace->next!=NULL) /* cycle to end */
		 {
			f1=f;
			f=trace->v;
			f2=trace->next->v;
			if ((ind=nghb_tri(p,f2,f))>=0)
			   pK_ptr[p->faces[f].vert[ind]].plot_flag = -1;
			else if (nghb_tri(p,f2,f1)>=0) /* f is blue*/
			 {
			   for (j=0;j<3;j++) 
		 	     pK_ptr[p->faces[f].vert[j]].plot_flag =-1;
			 }
			trace=trace->next;
		 }
	 }
	alpha=p->alpha;
	if (pK_ptr[alpha].plot_flag==-1) /* must try new alpha */
	 {
		j=0;
		while (pK_ptr[pK_ptr[alpha].flower[j]].plot_flag==-1 
			&& j<=pK_ptr[alpha].num) j++;
		if (j>pK_ptr[alpha].num) /* abandon attempt to use drawlist */
		   for (j=1;j<=p->nodecount;j++) p->packK_ptr[j].plot_flag=0;
		else alpha=p->alpha=pK_ptr[alpha].flower[j];
	 }
		
/* initialize. Note that we now find computed red chain. */

	for (i=1;i<=p->facecount;i++)
	 {
		p->faces[i].rwb_flag=2;
		p->faces[i].plot_flag=0;
	 }
	chandle=temp=(struct stface *)malloc(sizeof(struct stface));
	face_order=fdo=(struct Vertlist *)malloc(sizeof(struct Vertlist));



/* -------------------- first, build up red chain ----------- */

/* start with faces about alpha */

	face_ptr=face_org[alpha]+1;
	num=p->packK_ptr[alpha].num;
	chandle->face=f=*(face_ptr); /* first face */
	fdo->v=f;
	p->faces[f].plot_flag=1;
	p->faces[f].rwb_flag=1;
	p->faces[f].index_flag=find_index(p,f,alpha);
	for (i=1;i<num;i++)
	 {
		f=*(face_ptr+i);
		chandle->next=(struct stface *)malloc(sizeof(struct stface));
		fhandle=chandle;
		chandle=chandle->next;
		chandle->prev=fhandle;
		chandle->face=f;
		fdo=fdo->next=
			(struct Vertlist *)malloc(sizeof(struct Vertlist));
		fdo->v=f;
		p->faces[f].plot_flag=1;
		p->faces[f].rwb_flag=1;
		p->faces[f].index_flag=find_index(p,f,alpha);
	 }
	fdo->next=NULL;
	chandle->next=temp;
	temp->prev=chandle;

/* ----------- the rest (main loop) -------------------- */
	looplimit=(100)*p->nodecount;
	while (!stop && count && loop_count<looplimit)
	 {
		count=stopface=0;
		while ( (stopflag=
		   zip_vert(p,&chandle,&fdo,stopface,&stopface,&tripflag))>0 
		   && loop_count<looplimit)
			/* go until you have a marker vert */
		 {loop_count++;count += stopflag;}
		if (stopflag<0) stop=1;/* redchain a closed flower */
		while ( !stop && (stopflag=
		   zip_vert(p,&chandle,&fdo,stopface,&dummy,&tripflag))>=0 
		   && !tripflag && loop_count<looplimit)
			/* go until you encounter stopvert or stop */
		 {loop_count++;count += stopflag;}
		if (stopflag<0) stop=1; /* redchain a closed flower */
		while ( !stop && (stopflag=
		   zip_vert(p,&chandle,&fdo,stopface,&stopface,&tripflag))>0  
		   && loop_count<looplimit)
			/* go until you have a marker vert */
		 {loop_count++;count += stopflag;}
		if (stopflag<0 || stop) /* redchain a closed flower */
		 {
			stop=1;
			p->faces[chandle->face].rwb_flag=0;
			killit=chandle=chandle->next;
			while (chandle->next!=killit)
			 {
				p->faces[chandle->next->face].rwb_flag=0;
				chandle=chandle->next;
			 }
			free_face(&chandle);
			p->first_red_face=0;
		 }
	 } /* end of while */
/* find blue faces to be put in (computed) red chain */
	if (drawlist && drawlist->next && drawlist->next->next) 
		/* tack on any missing faces (blue) from list */
	 {
		trace=drawlist;
		while (trace->next->next!=NULL) trace=trace->next;
		f=trace->v; /* find last face */
		trace=drawlist;
		while (trace->next!=NULL)
		 {
			f1=f;
			f=trace->v;
			f2=trace->next->v;
			if (p->faces[f].rwb_flag==2
				&& nghb_tri(p,f1,f)>=0 
				&& in_handle(&chandle,f1)) /* blue face */
			 {
				hold=chandle->next;
				chandle->next=(struct stface *)
					malloc(sizeof(struct stface));
				hold->next->prev=chandle->next;
				chandle->next->prev=chandle;
				chandle=chandle->next;
				chandle->face=f;
				chandle->next=hold;
				p->faces[f].rwb_flag=1;
				add_face_order(p,&fdo,f,f1);
			 }
			trace=trace->next;
		 }
	 }
	else /* normal route when no drawlist provided */
	 {				
		if (!stop) add_blue(p,&chandle,&fdo,&stopvert);
		lastvert=0;
		while (!stop && add_blue(p,&chandle,&fdo,&lastvert)>=0
		   && stopvert!=lastvert );
	 }

/* ------------------------ now get final order ----------- */

/* arrange final drawing order */
	fdo=face_order;
	if ( (f=final_order(p,&fdo,chandle)) ) 
		/* have some missing faces */
	 {
		if (f==-1) /* error, redo all by old-style drawing order */
		 {
			for (i=1;i<=p->nodecount;i++) 
				p->packK_ptr[i].plot_flag=0;
			for (i=1;i<=p->facecount;i++) 
				p->faces[i].plot_flag=0;
			p->packK_ptr[alpha].plot_flag=1;
			f=*(face_org[alpha]+1);
			for (j=0;j<3;j++) 
			   p->packK_ptr[p->faces[f].vert[j]].plot_flag=1;
			fdo=face_order; /* now, fall through */
		 } 
		if (!wrapup_order(p,fdo->v)) /* old-style order */
		 {
			sprintf(msgbuf,"A drawing order error occurred.");
			emsg();
		 }
	 }
		
/* arrange list of red faces */	

	if (chandle==NULL) /* e.g., sphere; throw away red chain */
	 {
		for (i=1;i<=p->facecount;i++) p->faces[i].rwb_flag=0;
		p->first_red_face=0;
	 }
	else
	 {
		for (i=1;i<=p->facecount;i++) 
			p->faces[i].next_red=chandle->face;
		fhandle=chandle;
		while (chandle->next!=fhandle)
		 {
			p->faces[chandle->face].next_red=
				chandle->next->face;
			chandle=chandle->next;
		 }
	 }
	free_face(&chandle);
	destroy_catalog(p);
	vert_free(&face_order);
	return 1;
} /* facedraworder */

/*OK*/
int
zip_vert(p,handle,fdo,stopface,lastface,tripflag) /* find vert and 
fan of contig red faces, reset handle downstream. If rest of faces 
blue, then zip: adjust redchain, face colors, drawing order, etc. 
Return 1 if zipped. Return 0 if not zip-able. Return -1
if redchain is around single vert. set tripflag if stopface encountered. */
struct p_data *p;
struct stface **handle;
struct Vertlist **fdo;
int stopface,*lastface,*tripflag;
{
	int cface,n,findex=0,bindex,eindex,i,j,num,*face_ptr,vert;
	struct stface *bzip,*new_seg,*tmp,*hold,*killit;
	extern void add_face_order();

	bzip=*handle;
	*tripflag=0;
	cface=(*handle)->face;
	if ( (n=nghb_tri(p,(*handle)->next->face,cface))<0 ) 
	 {*lastface=cface;return 0;} 
		/* this would imply error in red chain */
	vert=p->faces[cface].vert[n];

/* don't zip at any vertex which is on right side of redchain */

	if (p->packK_ptr[vert].plot_flag==-1) 
	 {
		*handle=(*handle)->next;
		return 0;
	 }
	
	num=p->packK_ptr[vert].num;
	face_ptr=face_org[vert]+1;
/* find index of cface */
	while (*(face_ptr+findex)!=cface && findex<(num-1)) findex++;
/* find index of downstream face */
	i=findex;
	while ( (*handle)->next->face==*(face_ptr+((i+num-1) % num))
	   && (*handle)->next->face!=cface ) 
	 {
		*handle=(*handle)->next;
		i--;
	 }
	if ((*handle)->next->face==cface) return -1; 
		/* redchain goes around single vert */
	eindex= (num+i) % num;
/* find index of upstream face */
	i=findex;
	while ( bzip->prev->face==*(face_ptr+((i+1) % num))
	   && bzip->prev->face!=cface )
	 {
		bzip=bzip->prev;
		i++;
	 }
	if (bzip->prev->face==cface) return -1; 
		/* this should have been caught before */
	bindex=(i % num);
/* did we come across stopface? */
	n=(bindex+num-eindex) % num;
	for (j=0;j<=n;j++) 
		if (*(face_ptr+(j+eindex) % num)==stopface) *tripflag=1;
	*lastface=*(face_ptr+bindex);
/* only zip interiors */
	if (p->packK_ptr[vert].bdry_flag) return 0;
	i=bindex+1;
	while ( (j=(i % num))!=eindex )
	 {
		if (p->faces[*(face_ptr+j)].rwb_flag!=2) return 0;
		i++;
	 }

/* success: push redchain over vertex */

	if ( (i=(bindex+1) % num)!=eindex ) /* new red segment */
	 {
		tmp=new_seg=(struct stface *)malloc(sizeof(struct stface));
		new_seg->prev=bzip;
		new_seg->face=*(face_ptr+i);
		p->faces[new_seg->face].rwb_flag=1;
		add_face_order(p,fdo,new_seg->face,bzip->face);
		i++;
		while ( (j=i % num)!=eindex)
		 {
			hold=tmp;
			tmp=tmp->next=
			   (struct stface *)malloc(sizeof(struct stface));
			tmp->prev=hold;
			tmp->face=*(face_ptr+j);
			p->faces[tmp->face].rwb_flag=1;
			add_face_order(p,fdo,tmp->face,hold->face);
			i++;
		 }
	 }
	else /* no new red */
	 {
		tmp=bzip;
		new_seg=*handle;
	 }
/* remove old segment (becomes white) */
	killit=bzip->next;
	while (killit!=*handle)
	 {
		hold=killit->next;
		p->faces[killit->face].rwb_flag=0;
		free(killit);
		killit=hold;
	 }		
/* insert new red segment */
	(*handle)->prev=tmp;
	tmp->next=(*handle);
	bzip->next=new_seg;
	return 1;
} /* zip_vert */

/*OK*/
int
add_blue(p,handle,fdo,vert) /* see if blue face can be added to face
in red chain. Return -1 when we've gone around until we hit a spot 
where blue face has been added (this should mean we have added all
possible blue faces. Return 1 if face is added, zero otherwise. */
struct p_data *p;
struct stface **handle;
struct Vertlist **fdo;
int *vert;
{
	int cface,n,findex=0,nindex,i,num,newface,*face_ptr;
	struct stface *hold;
	extern void add_face_order();

	cface=(*handle)->face;
	if ( (n=nghb_tri(p,(*handle)->next->face,cface))<0 ) return -1; 
		/* this is recently added face; means we've gone all the
			way around  */
	*vert=p->faces[cface].vert[n];
	num=p->packK_ptr[*vert].num;
	face_ptr=face_org[*vert]+1;
	while (*(face_ptr+findex)!=cface && findex<(num-1)) findex++;
	i=findex;
	while ( (*handle)->next->face==*(face_ptr+((i+num-1) % num))
	   && (*handle)->next->face!=cface ) 
	 {
		*handle=(*handle)->next;
		i--;
	 }
	nindex=(num+i-1) % num;
	newface=*(face_ptr+nindex);
	if ( p->faces[newface].rwb_flag!=2
	   || (p->packK_ptr[*vert].bdry_flag && nindex>findex) ) 
		return 0; /* face not blue or on wrong end of bdry flower */
	hold=(*handle)->next;
	(*handle)->next=(struct stface *)malloc(sizeof(struct stface));
	(*handle)->next->prev=*handle;
	*handle=(*handle)->next;
	(*handle)->face=newface;
	(*handle)->next=hold;
	hold->prev=*handle;
	p->faces[newface].rwb_flag=1;
	add_face_order(p,fdo,newface,(*handle)->prev->face);
	(*handle)=(*handle)->next;
	return 1;
} /* add_blue */

/*OK*/
void
add_face_order(p,fdo,face,preface)  /* Insert face in draw order next 
to preface. */
struct p_data *p;
struct Vertlist **fdo;
int face,preface;
{
	int n;
	extern int nghb_tri();

	(*fdo)=(*fdo)->next=(struct Vertlist*)malloc(sizeof(struct Vertlist));
	(*fdo)->v=face;
	(*fdo)->next=NULL;
	p->faces[face].plot_flag=1;
	n=nghb_tri(p,preface,face);
	if (n<0) p->faces[face].index_flag=0; /* shouldn't happen */ 
	else p->faces[face].index_flag=n;
} /* add_face_order */

/*OK*/
int 
final_order(p,fdo,handle)  /* strategy: Simply connected case, 
do in order without regard to face color. Non-simply connected case,
start with alpha, proceed to first red face, do whole red chain,  
come back for rest of interior. This is needed 
because some circles will be moved while drawing red faces.
Return 0 for success and all faces plotted, -1 if an error occurs
which will require use of old method, 1 if success, but some faces 
still not plotted. */
struct p_data *p;
struct Vertlist **fdo;
struct stface *handle;
{
	int i,j,blue_count,f,start_face,index,fhold;
	struct Vertlist *start,*temp;
	struct stface *hold;
	extern int nice_index(),in_handle();

	start=temp=*fdo;

/* set next_face */

	p->first_face=(*fdo)->v;
	for (i=1;i<=p->nodecount;i++) p->packK_ptr[i].plot_flag=0;
	for (i=1;i<=p->facecount;i++) p->faces[i].plot_flag=0;
	for (j=0;j<3;j++) 
		p->packK_ptr[p->faces[(*fdo)->v].vert[j]].plot_flag=1;
	p->faces[p->first_face].plot_flag=1;

/* ----------------- simply connected ------------------------- */

	if ((p->euler==1 || p->euler==2) && p->genus==0)
	 {
		f=(*fdo)->v;
		while ((*fdo)->next) /* get all rest in fdo */
		 {
			*fdo=(*fdo)->next;
			p->faces[f].next_face=(*fdo)->v;
			f=(*fdo)->v;
			index=p->faces[f].index_flag;
			p->faces[f].index_flag=nice_index(p,f,index);
			p->packK_ptr[
				p->faces[f].vert[(p->faces[f].index_flag+2) % 3]
				].plot_flag=1;
			p->faces[f].plot_flag=1;
		 }
/* set first_red_face */
		if (handle!=NULL)
		 {
			while (temp->next!=NULL 
			   && p->faces[temp->v].rwb_flag!=1)
				temp=temp->next;
			if (temp->next!=NULL) p->first_red_face=temp->v;
			else p->first_red_face=0;
		 }
/* check for remaining faces */
		for (i=1;i<=p->facecount;i++) 
		   if (!p->faces[i].plot_flag)
			return 1; /* some circles not plotted */
		p->faces[(*fdo)->v].next_face=start->v;
		return 0;
	 }

/* ------------------------ non-simply connected ----------------- */

	else
	 {
/* first interiors, from alpha */
		while (!p->faces[(*fdo)->v].rwb_flag && (*fdo)->next!=NULL)
		 {
			f=(*fdo)->v;
			index=p->faces[f].index_flag;
			p->faces[f].next_face=(*fdo)->next->v;
			p->faces[f].index_flag=nice_index(p,f,index);
			p->packK_ptr[
			   p->faces[f].vert[(p->faces[f].index_flag+2) % 3]
			   ].plot_flag=1;
			p->faces[f].plot_flag=1;
			*fdo=(*fdo)->next;
		 }
		if ((*fdo)->next==NULL) /* should not happen */
		 {
			p->faces[(*fdo)->v].next_face=start->v;
			return -1;
		 }
/* now, put whole red chain in draw order */
		p->first_red_face=f=(*fdo)->v;
		in_handle(&handle,(*fdo)->v);
		blue_count=0;
		while (handle->next->face!=(*fdo)->v 
			&& blue_count<=p->facecount)
		 {
			p->faces[handle->face].next_face=handle->next->face;
			handle=handle->next;
			blue_count++;
		 }
		if (blue_count>p->facecount)
		 {
			sprintf(msgbuf,"Draw order problem.");
			emsg();
			return -1;
		 }
/* loop thru again to reset index_flags. */
		in_handle(&handle,(*fdo)->v);
		start_face=f=handle->face;
		index=p->faces[f].index_flag;
		p->faces[f].index_flag=nice_index(p,f,index);
		p->packK_ptr[p->faces[f].vert[(p->faces[f].index_flag+2) % 3]
			].plot_flag=1;
		p->faces[f].plot_flag=1;
		while(handle->next->face!=start_face)
		 {
			blue_count=0;
			hold=handle;
			if ((index=nghb_tri(p,hold->face,
			   handle->next->face))<0 
			   && (index=nghb_tri(p,hold->prev->face,
			   handle->next->face))<0) return -1;
				/* blue face, so back up; if previous
				face isn't contiguous, error. */ 
			f=handle->next->face;
			p->faces[f].index_flag=nice_index(p,f,index); 
			p->packK_ptr[p->faces[f].
				vert[(p->faces[f].index_flag+2) % 3]
				].plot_flag=1;
			p->faces[f].plot_flag=1;
			handle=handle->next;
		 } /* end of while */
/* get rest of white in fdo */
		fhold=handle->face;
		while ((*fdo)->next!=NULL)	
		 {
			while ( p->faces[(*fdo)->v].rwb_flag 
			   && (*fdo)->next!=NULL )
				*fdo=(*fdo)->next;
					/* pass up any red/blue faces */
			if ((*fdo)->next!=NULL)
			 {
				p->faces[fhold].next_face=f=(*fdo)->v;
				p->faces[f].index_flag=
					nice_index(p,f,p->faces[f].index_flag);
				p->packK_ptr[p->faces[f].
				   vert[(p->faces[f].index_flag+2) % 3]
				   ].plot_flag=1;
				p->faces[f].plot_flag=1;
				fhold=(*fdo)->v;
				*fdo=(*fdo)->next;
			 }
		 }
/* check last face */
		if (!p->faces[(f=(*fdo)->v)].plot_flag)
		 {
			p->faces[fhold].next_face=f;
			p->faces[f].plot_flag=1;
			fhold=f;
		 } 
		p->faces[fhold].next_face=start->v;		
/* check for remaining faces */
		for (i=1;i<=p->facecount;i++) 
		   if (!p->faces[i].plot_flag) return 1;
		return 0;
	 } /* end of non-simply connected case */
 } /* final_order */

/*OK*/
int
wrapup_order(p,lastface) /* Wrap up face order. Put in remaining faces.
return 0 if fails. */
struct p_data *p;
int lastface;
{
	int k=1,hit=1,stop=0,F,i,index,count=0,n;

	F=p->facecount;
	for (i=1;i<=F;i++) if (p->faces[i].plot_flag) count++;
	while (count<F && !stop)
	 {
		if (k > F)
		 {
			k=1;
			if (!hit) stop=1; /* no new faces being added */
			hit=0;
		 }
		while (p->faces[k].plot_flag && k < F) k++;
		if (!p->faces[k].plot_flag && face_ready(p,k,&index))
		 {
			if ( (n=nghb_tri(p,lastface,k))<0 )
				p->faces[k].index_flag=index;
			else p->faces[k].index_flag=n;
			p->faces[lastface].next_face=k;
			p->faces[k].plot_flag=1;
			p->packK_ptr[p->faces[k].
				vert[(index+2) % 3]].plot_flag=1;
			lastface=k;
			count++;
			hit=1;
		 }
		k++;
	 }
	p->faces[lastface].next_face=p->first_face;
	if (count<F) return 0; /* shouldn't ever happen */
	return 1;
} /* wrapup_order */

/*OK*/
int 
face_ready(p,face,index) /* return 1 if face ready to draw, set
index of vert to draw first. */
struct p_data *p;
int face,*index;
{
	f_data *ptr;

	ptr=p->faces+face;
	if (p->packK_ptr[ptr->vert[0]].plot_flag 
		&& p->packK_ptr[ptr->vert[1]].plot_flag) 
	 {
		*index=0;
		return 1;
	 }
	if (p->packK_ptr[ptr->vert[1]].plot_flag 
		&& p->packK_ptr[ptr->vert[2]].plot_flag) 
	 {
		*index=1;
		return 1;	
	 }
	if (p->packK_ptr[ptr->vert[2]].plot_flag 
		&& p->packK_ptr[ptr->vert[0]].plot_flag) 
	 {
		*index=2;
		return 1;
	 }
	*index=0;
	return 0;
} /* face_ready */

/*OK*/
int
in_handle(handle,face) /* position closed list at face; 
return 0 if not there. */
struct stface **handle;
int face;
{
	struct stface *current;

	current=*handle;
	if ((*handle)->face==face) return 1;
	do {current=current->next;}
	while ( (current->face!=face) && ((current->next)!=*handle) );
	if (current->face!=face) return 0;
	*handle=current;
	return 1;
} /* in_handle */

/*OK*/
int
nice_index(p,f,index) /* set index_flag of face f so that it uses nghb'ing
white face using verts already drawn, if possible. "index" is first 
preference and default. plot_flags are >=0 for verts already drawn. */
struct p_data *p;
int f,index;
{
	int v,w,i,m,num,g,*face_ptr;

	for (i=0;i<3;i++)
	 {
		v=p->faces[f].vert[(index+i)%3];
		w=p->faces[f].vert[(index+i+1)%3];
		if (p->packK_ptr[v].plot_flag > 0 
		   && !p->packK_ptr[v].bdry_flag
		   && p->packK_ptr[w].plot_flag > 0 
		   && !p->packK_ptr[w].bdry_flag)
		 {
			face_ptr=face_org[w];
			num=*face_ptr;
			face_ptr++;
			m=0;
			while (*(face_ptr+m)!=f && m<(num-1)) m++;
			g=*(face_ptr+((m+1) % num));
			if (p->faces[g].rwb_flag==0) return ((index+i)%3);
				 /* yes, face is white */
		 }
	 }
	for (i=0;i<3;i++) /* fallback: try for white neighbor */
	 {
		v=p->faces[f].vert[(index+i)%3];
		w=p->faces[f].vert[(index+i+1)%3];
		if (p->packK_ptr[v].plot_flag > 0
		   && p->packK_ptr[w].plot_flag)
		 {
			face_ptr=face_org[w];
			num=*face_ptr;
			face_ptr++;
			m=0;
			while (*(face_ptr+m)!=f && m<(num-1)) m++;
			g=*(face_ptr+((m+1) % num));
			if (p->faces[g].rwb_flag==0) return ((index+i)%3);
				 /* yes, face is white */
		 }
	 }
	for (i=0;i<3;i++) /* fallback: use any two plotted ngbh's. */
	 {
		v=p->faces[f].vert[(index+i)%3];
		w=p->faces[f].vert[(index+i+1)%3];
		if (p->packK_ptr[v].plot_flag > 0
		   && p->packK_ptr[w].plot_flag) return ((index+i)%3);
	 }
	if (index<0 || index>2) return 0;
	return index;
} /* nice_index */

/*OK*/
int
free_face(handle)
struct stface **handle;
{
	struct stface *kil;

	if ((*handle)==NULL) return 0;
	while ((*handle)->next!=*handle) 
	 {
		if ((*handle)->next->prev!=(*handle)) /* malformed */
		 {
			sprintf(msgbuf,"String handling problem.");
			emsg();
			return 0;
		 }
		(*handle)->prev->next=(*handle)->next;
		(*handle)->next->prev=(*handle)->prev;
		kil=*handle;
		*handle=(*handle)->prev;
		free(kil);
	 }
	free(*handle);
	*handle=NULL;
	return 1;
} /* free_face */

/* ---------------- new routines to replace build_org 1/98 ---------- */
int
catalog_faces(p) /* Build array of ptrs, one for each vertex v. Point 
to list of face nos. v is in. In list for v, 0 entry is
num of faces for v; then face 1, face 2, .... 
Eg. first face in list contains v, flower[0], and flower[1].*/
struct p_data *p;
{
	int v,j,f,w,n,*face_ptr;
	struct K_data *pK_ptr;

	face_org=(void *)calloc((size_t)(p->nodecount+2),sizeof(int *));
	pK_ptr=p->packK_ptr;
	for (v=1;v<=p->nodecount;v++)
	 {
		face_org[v]=(int *)
		   calloc((size_t)(pK_ptr[v].num+2),sizeof(int));
		*(face_org[v])=p->packK_ptr[v].num;
	 }
	if (!p->faces) alloc_faces_space(p);
	for (f=1;f<=p->facecount;f++)
	for (j=0;j<3;j++)
	 {
		v=p->faces[f].vert[j];
		face_ptr=face_org[v];	
		w=p->faces[f].vert[(j+1)%3];
		n=nghb(p,v,w);
		*(face_ptr+(n+1))=f;
	 }
	return 1;		
} /* catalog_faces */

int
destroy_catalog(p) /* free memory from catalog_faces */
struct p_data *p;
{
	int v;

	for (v=1;v<=p->nodecount;v++)
		if (face_org[v]) free(face_org[v]);
	if (face_org) 
	 { free(face_org);face_org=NULL;}
	return 1;
} /* destroy_catalog */

/* ----------------------------------------------------- */

/*OK*/
int
hex_parallel_call(p,v,w,n,cmod,show) /* Try to draw parallelogram (treat
complex as though it were hex). show=1 means to draw faces and to compute 
complex modulus of parallelogram. */
struct p_data *p;
int v,w,n;
complex *cmod;
int show;
{
	int hold,hold1,hold2,count,v1,v2,v3,v4;
	complex z0,z1,z2,zz;
	struct Vertlist *facelist,*ftrace;
	struct Vertlist *dlist,*curve1,*ctrace1,*curve2,*ctrace2;
	extern struct Vertlist *try_hex_pg();
	extern int layout_facelist();
	extern void vert_free();
	extern complex csub(),cdiv();
	extern float cAbs();

	cmod->re=cmod->im=0.0;
	if (show) 
	 {
		dlist=try_hex_pg(p,v,w,n,&v1,&v2,&v3,&v4);
		layout_facelist(p,3,&dlist,1,0,0,SHOW); 
			/* note: clears list*/
	 }
	ftrace=facelist=try_hex_pg(p,v,w,n,&v1,&v2,&v3,&v4);
	while (ftrace && ftrace->next) ftrace=ftrace->next;
	if (ftrace && ftrace->v!=facelist->v) /* didn't close up */
	 {vert_free(&facelist); return 0;}
/* create lists to get to corners */
	ftrace=facelist;
	if (ftrace!=NULL)
	 {
		curve1=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		ctrace1=curve1;
	 	ctrace1->next=NULL;
		ctrace1->v=ftrace->v;
	 }
	else {vert_free(&facelist); return 0;}
	ftrace=ftrace->next;
	count=1;
	while (ftrace!=NULL && count < (2*n)-1)
	 {
		ctrace1=ctrace1->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
		ctrace1->next=NULL;
		ctrace1->v=ftrace->v;
		count++;
		ftrace=ftrace->next;
	 }
	if (!ftrace || !(ftrace->next)) 
	 {vert_free(&facelist);vert_free(&curve1);return 0;}
	hold1=ctrace1->v;  /* have to save this second-to-last face */
	ctrace1=ctrace1->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
	ctrace1->next=NULL;
	ctrace1->v=ftrace->v;
	ctrace1=ctrace1->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
	ctrace1->next=NULL;
	ctrace1->v=hold=curve1->v; /* close up */
	if (show)
	 {
		layout_facelist(p,3,&curve1,1,0,0,NOSHOW); /* clears curve1 */
		z0=p->packR_ptr[v1].center; /* first corner */
		z1=p->packR_ptr[v2].center; /* second corner */
	 }
/* second curve list */
	curve2=ctrace2=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
 	ctrace2->next=(struct Vertlist *)
			malloc(sizeof(struct Vertlist));
	ctrace2->next->next=NULL;
	ctrace2->v=hold; /* want start/end at same face as curve1 */
	ctrace2=ctrace2->next;
	ctrace2->v=hold2=ftrace->v;
	ftrace=ftrace->next;
	count=1;
	while (ftrace!=NULL && count < (2*n)-1)
	 {
		ctrace2=ctrace2->next=(struct Vertlist *)
				malloc(sizeof(struct Vertlist));
		ctrace2->next=NULL;
		ctrace2->v=ftrace->v;
		count++;
		ftrace=ftrace->next;
	 }
	if (!ftrace) 
	 {vert_free(&facelist);vert_free(&curve1);vert_free(&curve2);return 0;}
	ctrace2=ctrace2->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
	ctrace2->next=NULL;
	ctrace2->v=hold1;
	ctrace2=ctrace2->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
	ctrace2->next=NULL;
	ctrace2->v=hold2;
	ctrace2=ctrace2->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
	ctrace2->next=NULL;
	ctrace2->v=curve2->v; /* takes some doing to close up */
	if (show)
	 {
		layout_facelist(p,3,&curve2,1,0,0,NOSHOW); /* clears curve2*/
		z2=p->packR_ptr[v3].center; /* third corner */
	 }
/* later, may want to save curve1, curve2 as generators */
	vert_free(&curve1);vert_free(&curve2);
/* now to compute modulus */
	if (show)
	 {
		zz=csub(z1,z0);
		if (cAbs(zz)> .0000001)
			*cmod=cdiv(csub(z2,z1),zz);
		else {cmod->re=0.0;cmod->im=0.0;}
	 }
	return 1;
} /* hex_parallel_call */
	
struct Vertlist *
try_hex_pg(p,v,w,n,v1,v2,v3,v4) /* for hex_walk, called by 
hex_parallel_call and fix -h to draw parallelogram. Edge 
lengths n, start edge = v w, counterclockwise. Return corner
verts; v2,v4 "sharp" corners. */
struct p_data *p;
int v,w,n,*v1,*v2,*v3,*v4;
{
     int i,num;
     extern struct Vertlist *path_follow();
     extern struct Edgelist *node_to_edge();
     struct Edgelist *edgelist;
     struct Vertlist *vertlist,*vtrace,*facelist;
     struct K_data *pK_ptr;
     extern int nghb();
     extern void edge_free(),vert_free();

     pK_ptr=p->packK_ptr;
     facelist=NULL;
     if (nghb(p,v,w)<0) return facelist; /* must start with edge */
/* record first two vertices */
     *v1=v;
     vertlist=(struct Vertlist *)malloc(sizeof(struct Vertlist));
     vertlist->v=v;
     vtrace=vertlist->next=(struct Vertlist *)
	malloc(sizeof(struct Vertlist));
     vtrace->v=w;
     vtrace->next=NULL;
/* first edge */
     for (i=2;i<=n;i++) 
      {
          vtrace=vtrace->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
          num=pK_ptr[w].num;
          vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-3) % num];
          vtrace->next=NULL;
          v=w;
          w=vtrace->v;
      }
/* sharp left turn */
     *v2=w;
     vtrace=vtrace->next=(struct Vertlist *)
	malloc(sizeof(struct Vertlist));
     num=pK_ptr[w].num;
     vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-1) % num];
     vtrace->next=NULL;
     v=w;
     w=vtrace->v;
/* second edge */
     for (i=2;i<=n;i++) 
      {
          vtrace=vtrace->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
          num=pK_ptr[w].num;
          vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-3) % num];
          vtrace->next=NULL;
          v=w;
          w=vtrace->v;
      }
/* left turn */
     *v3=w;
     vtrace=vtrace->next=(struct Vertlist *)
	malloc(sizeof(struct Vertlist));
     num=pK_ptr[w].num;
     vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-2) % num];
     vtrace->next=NULL;
     v=w;
     w=vtrace->v;
/* third edge */
     for (i=2;i<=n;i++) 
      {
          vtrace=vtrace->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
          num=pK_ptr[w].num;
          vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-3) % num];
          vtrace->next=NULL;
          v=w;
          w=vtrace->v;
      }
/* sharp left turn */
     *v4=w;
     vtrace=vtrace->next=(struct Vertlist *)malloc(sizeof(struct Vertlist));
     num=pK_ptr[w].num;
     vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-1) % num];
     vtrace->next=NULL;
     v=w;
     w=vtrace->v;
     for (i=2;i<=n;i++) /* last edge */
      {
          vtrace=vtrace->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
          num=pK_ptr[w].num;
          vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-3) % num];
          vtrace->next=NULL;
          v=w;
          w=vtrace->v;
      }
     edgelist=node_to_edge(p,vertlist);
     vert_free(&vertlist);
     facelist=path_follow(p,edgelist,TRUE);
     edge_free(&edgelist);
     return facelist;
} /* try_hex_pg */

/*OK*/
struct Edgelist *
node_to_edge(p,vertlist) /* given vertlist, string together as many as form a
valid edgelist. */
struct p_data *p;
struct Vertlist *vertlist;
{
     int v,w;
     struct Vertlist *vtrace,*vclobber;
     struct Edgelist *edgelist,*etrace;
     extern int nghb();

     edgelist=NULL;
/* remove repeats in vertlist */
     vtrace=vertlist;
     while (vtrace!=NULL)
      {
          if (vtrace->next!=NULL && vtrace->v==vtrace->next->v)
           {
               vclobber=vtrace->next;
               vtrace->next=vtrace->next->next;
               free(vclobber);
           }
          else vtrace=vtrace->next;
      }
/* get started */
     if (vertlist==NULL || vertlist->next == NULL
          || (v=vertlist->v) <1 || v > p->nodecount
          || nghb(p,v,(w=vertlist->next->v))<0) return edgelist;
     edgelist=(struct Edgelist *)malloc(sizeof(struct Edgelist));
     edgelist->v=v;
     edgelist->w=w;
     edgelist->next=NULL;
     etrace=edgelist;
     vtrace=vertlist->next->next;
     while (vtrace!=NULL)
      {
          v=w;
          w=vtrace->v;
          if (nghb(p,v,w)<0) return edgelist; /* chain broken */
          etrace->next=(struct Edgelist *)malloc(sizeof(struct Edgelist));
          etrace=etrace->next;
          etrace->v=v;
          etrace->w=w;
          etrace->next=NULL;
          vtrace=vtrace->next;
      }
     return edgelist;
} /* node_to_edge */

/*OK*/
struct Vertlist *
path_follow(p,edgelist,flag) /* given linked list of edges, return 
list of faces on their left. flag means to close up with last face. 
On error return partial list if consistent. */
struct p_data *p;
struct Edgelist *edgelist;
int flag;
{
     int v,u,w,stopv,nextf;
     struct Edgelist *etrace;
     struct Vertlist *facelist,*ftrace;
     extern int left_face();
     
     facelist=NULL;
/* find first face */
     if (edgelist==NULL || !(nextf=
          left_face(p,(v=edgelist->v),(w=edgelist->w),&u))) return facelist;
     facelist=(struct Vertlist *)malloc(sizeof(struct Vertlist));
     facelist->v=nextf;
     facelist->next=NULL;
     etrace=edgelist->next;
     ftrace=facelist;
/* add while edges last */
     while (etrace!=NULL)
      {
          stopv=etrace->w;
               /* get clockwise faces about w */
	v=u;
          while (u!=stopv && (nextf= left_face(p,v,w,&u)))
           {
               v=u;
               ftrace=ftrace->next=(struct Vertlist *)
		  malloc(sizeof(struct Vertlist));
               ftrace->v=nextf;
               ftrace->next=NULL;
           }
          if (u!=stopv) return facelist; /* ran into boundary */
	v=etrace->v;
	w=etrace->w;
	left_face(p,v,w,&u);
	etrace=etrace->next;
      }
     if (!flag || ftrace->v==facelist->v) return facelist;
	stopv=v;
	v=u;
     while (u!=stopv && (nextf=left_face(p,u,w,&u)))
      {
          ftrace=ftrace->next=(struct Vertlist *)
		malloc(sizeof(struct Vertlist));
          ftrace->v=nextf;
          ftrace->next=NULL;
          if (nextf==facelist->v) return facelist;
      }
     return facelist;
} /* path_follow */
          
/*OK*/
int
left_face(p,v,w,u) /* return face to left of edge v w. Return 0 if no face. 
u is third vert of face. */
struct p_data *p;
int v,w,*u;
{
     int f,i;

     for (f=1;f<=p->facecount;f++)
      {if (check_face(p,f,v,w)>=0) break;}
     if (f>p->facecount) return 0; /* no face */
     for (i=0;i<3;i++) if ((*u=p->faces[f].vert[i])!=v && (*u)!=w) return f;
     return 0;
} /* left_face */

/* +++++++++++++++++++++++  Debugging help +++++++++++++++++++++++ */

/* Want to put more general debugging routines in separate file Debug.c */

/*OK*/
int
showhandle(handle) /* show and check doubly linked list of faces */
struct stface *handle;
{
	int count=0,total=1,first,last;
	struct stface *trace;

	if ((trace=handle->next)==NULL) return 0;
	first=trace->face;
	while (trace!=handle && total < 2000)
	 {
		total++;
		last=trace->face;
		trace=trace->next;
	 }
	fprintf(stderr,
		"\nCheck double linked face list: count=%d,first=%d,last=%d.\n",
		total,first,last);
	while (trace!=handle && count <= total+1)
	 {
		count++;
		fprintf(stderr,"  %d",trace->face);
		if (trace->prev->next!=trace)
			fprintf(stderr," prev->next wrong: %d %d, skips %d",
				handle->prev->face,handle->prev->next->face,
				handle->face);
		if (trace->next->prev!=trace)
			fprintf(stderr," next->prev wrong: %d %d, skips %d",
				handle->next->face,handle->next->prev->face,
				handle->face);
	 }
	return count;
} /* showhandle */

/*OK*/
int
showlist(vertlist) /* show a Vertlist */
struct Vertlist *vertlist;
{
	int count=0,first,last;
	struct Vertlist *trace;

	if ((trace=vertlist)==NULL) return 0;
	first=vertlist->v;
	while (trace)
	 {
		count++;
		last=trace->v;
		trace=trace->next;
	 }
	fprintf(stderr,"\nShow a 'Vertlist': count=%d, first=%d, last=%d.\n",
		count,first,last);
	trace=vertlist;
	while (trace)
	 {
		fprintf(stderr," %d",trace->v);
		trace=trace->next;
	 }
	fprintf(stderr,"  end\n");
} /* showlist */

/*OK*/
int
pford(p) /* print the face order as recorded in the pack */
struct p_data *p;
{
	int f,count=1,stop;

	fprintf(stderr,"Pack face order: %d \n", (stop=p->first_face));
	f=stop;
	while ( p->faces[f].next_face!=stop 
	   && count < 2*p->facecount
	   && p->faces[f].next_face >0
	   && p->faces[f].next_face <=p->facecount && count++)
		fprintf(stderr,"%d ",(f=p->faces[f].next_face)); 
	return count;
} /* pford */

/*OK*/
pfacered(p) /* print red face order according to pack */
struct p_data *p;
{
	int f,count=1,stop;
	extern int pack_num();

	fprintf(stderr,"Pack %d red face order: %d \n", pack_num(p),
		(stop=p->first_red_face));
	f=stop;
	while ( p->faces[f].next_red!=stop 
	   && count < 2*p->facecount
	   && p->faces[f].next_red >0
	   && p->faces[f].next_red <=p->facecount && count++)
		fprintf(stderr,"%d ",(f=p->faces[f].next_red)); 
	return count;
} /* pfacered */

/*OK*/
void
pfdo(fdo) /* print fdo, used in facedraworder */
struct Vertlist *fdo;
{
	fprintf(stderr,"FDO: ");
	while (fdo!=NULL)
	 {
		fprintf(stderr,"%d ",fdo->v);
		fdo=fdo->next;
	 }
	fprintf(stderr,"\n");
}

/*OK*/
void	
pblue(p) /* find blue faces, used in facedraworder */
struct p_data *p;
{
	int f;

	fprintf(stderr,"blue faces: ");
	for (f=1;f<=p->facecount;f++)
	if (p->faces[f].rwb_flag==2) fprintf(stderr,"%d ",f);
	fprintf(stderr,"\n");
}

/*OK*/
void
pred(p,handle) /* print redchain */
struct p_data *p;
struct stface *handle;
{
	int count=0;
	struct stface *hold;
	extern int nghb_tri();

	hold=handle;
	fprintf(stderr,"Red: %d  ",hold->face);
	hold=hold->next;
	while (hold!=handle && count<=2*p->facecount)
	 {
		if (nghb_tri(p,hold->face,hold->prev->face)<0)
			fprintf(stderr,"B ");
		fprintf(stderr,"%d  ",hold->face);
		hold=hold->next;
		count++;
	 }
	fprintf(stderr,"%d \n",hold->face);	
} /* pred */


int
print_flower(p,v) /* print out flower */
struct p_data *p;
int v;
{
	int j;

	fprintf(stderr,"Flower p%d, v%d, num= %d ",
		pack_num(p),v,p->packK_ptr[v].num);
	for (j=0;j<=p->packK_ptr[v].num && j<100;j++)
		fprintf(stderr," %d, ",p->packK_ptr[v].flower[j]);
} /* print_flower */


int
pack_to_stderr(p) /* debugging help: write comb of p to stderr */
struct p_data *p;
{
	writepack(p,stdout,0);
}

int
pack_face_draw_order(p) /* show order of placing faces */
struct p_data *p;
{
	int nf;

	nf=p->first_face;
	printf("\nCircle-drawing order: %d ",nf);
	while ( (nf=p->faces[nf].next_face)!=p->first_face)
		printf("%d ",nf);
	printf("\n");
}

face_debug(p) /* save face info for debugging */
struct p_data *p;
{
	FILE *fp;
	int nf;

	fp=fopen("/tmp/fbug.txt","w");
	nf=p->first_face;
	fprintf(fp,"\n Face data in drawing order:\n");
	fprintf(fp,"face %d:\t (%d, %d, %d)\n",nf,
		p->faces[nf].vert[p->faces[nf].index_flag],
		p->faces[nf].vert[(p->faces[nf].index_flag+1)%3],
		p->faces[nf].vert[(p->faces[nf].index_flag+2)%3]);
	while ( (nf=p->faces[nf].next_face)!=p->first_face)
	  fprintf(fp,"face %d:\t (%d, %d, %d)\n",nf,
		p->faces[nf].vert[p->faces[nf].index_flag],
		p->faces[nf].vert[(p->faces[nf].index_flag+1)%3],
		p->faces[nf].vert[(p->faces[nf].index_flag+2)%3]);
	fclose(fp);
} /* face_debug */



