# -*- coding: utf-8 -*-
#
#    Copyright (C) 2009 Patrick Ulbrich <zulu99@gmx.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import threading
import os
from stat import *
from xml.etree.ElementTree import ElementTree, Element, SubElement, parse
from email.Utils import parseaddr
from common import *

from utils import log, prepend_config_path

DEFAULT_FILTER_FILE = prepend_config_path("junkfilters.xml")

class CompareType:
	"""CompareType enumeration."""
	STARTS_WITH	= 1
	ENDS_WITH	= 2
	EQUALS		= 3
	CONTAINS	= 4
	
	# Human readable string representations.
	names = {
		STARTS_WITH : _("starts with"),
		ENDS_WITH	: _("ends with"),
		EQUALS		: _("equals"),
		CONTAINS	: _("contains")
	}
	
	from_string = {
		"startswith"	: STARTS_WITH,
		"endswith"		: ENDS_WITH,
		"equals"		: EQUALS,
		"contains"		: CONTAINS
	}


class MailField:
	"""MailField enumeration."""
	FROM	= 1
	SUBJECT	= 2
	
	# Human readable string representations.
	names = {
		FROM	: _("From"),
		SUBJECT	: _("Subject")
	}
	
	from_string =  {
		"from"		: FROM,
		"subject"	: SUBJECT
	}


class JunkFilter:
	"""Represents a junkmail filter."""
	def __init__(self, mail_field, compare_type, string, enabled = True):
		self.mail_field		= mail_field
		self.compare_type	= compare_type
		self.string			= string
		self.enabled		= enabled


class JunkFilterManager:
	"""Manages all junkmail filtering."""
	
	# Default instance.
	__default = None
	
	
	def __init__(self, filename, create = False):
		self.__lock					= threading.Lock()
		self.__filename				= filename
		self.__file_last_modified	= 0
		self.__filters				= []
		
		if create:
			self.__create_filter_file()
		
		self.reload_filters(True)
	
	
	@staticmethod
	def get_default():
		"""
		Returns the default JunkFilterManager instance.
		"""
		if not JunkFilterManager.__default:
			if os.path.exists(DEFAULT_FILTER_FILE):
				create = False
			else:
				create = True
			
			JunkFilterManager.__default = JunkFilterManager(DEFAULT_FILTER_FILE, create)
		
		return JunkFilterManager.__default
	
	
	def filter(self, mails):
		"""
		Filters out emails that match the laoded filters
		and returns the remaining list.
		"""
		return [m for m in mails if (not self.match_filters(m))]
	
	
	def match_filters(self, mail):
		"""
		Matches the loaded filters against a single mail item
		where mail item: [subject, from, msgid].
		"""
		
		def cmp(f, content):
			# Determine comparetype and compare.
			if f.compare_type == CompareType.STARTS_WITH:
				return content.startswith(f.string)
			elif f.compare_type == CompareType.ENDS_WITH:
				return content.endswith(f.string)
			elif f.compare_type == CompareType.EQUALS:
				return (content == f.string)
			elif f.compare_type == CompareType.CONTAINS:
				return (content.find(f.string) > -1)
			else:
				raise NotImplementedError()
				
		for f in self.__filters:
			if not f.enabled:
				continue
			
			# Determine mailfield content to be compared.
			if f.mail_field == MailField.SUBJECT:
				match = cmp(f, mail[0])
			elif f.mail_field == MailField.FROM:
				name, addr = parseaddr(mail[1])
				match = cmp(f, addr) or cmp(f, name)
			else:
				raise NotImplementedError()
			
			if match:
				return True
				
		return False
	
	
	def reload_filters(self, force = True):
		"""
		Loads the filters of the XML file 
		specified in the constructor.
		
		When force is set to False, filters are reloaded 
		only when the filterfile has been changed.
		"""
		modified = os.stat(self.__filename)[ST_MTIME]
		
		if (not force) and (modified == self.__file_last_modified):
			return
		
		self.__file_last_modified = modified
		
		log("loading junkmail filters from '%s'" % self.__filename)
		self.clear_filters()
		tree = parse(self.__filename)
		filter_nodes = tree.findall("/filter")
		for f in filter_nodes:
			enabled = (f.attrib["enabled"].upper() == "TRUE")
			part_nodes = f.findall("part")
			for p in part_nodes:
				mail_field	= MailField.from_string[p.attrib["mailfield"]]
				cmp_type	= CompareType.from_string[p.attrib["comparetype"]]
				string		= p.attrib["string"]

				self.__filters.append(JunkFilter(mail_field, cmp_type, string, enabled))
	
	
	def save_filters(self):
		mailfield_names = dict([(v,k) for k,v in MailField.from_string.iteritems()])
		cmp_type_names = dict([(v,k) for k,v in CompareType.from_string.iteritems()])
		
		root = Element("junkfilters")
		for f in self.__filters:
			filter = SubElement(root, "filter")
			if f.enabled:
				enabled = "true"
			else:
				enabled = "false"
			filter.set("enabled", enabled)
			filter.append(Element("part", 
				mailfield = mailfield_names[f.mail_field],
				comparetype = cmp_type_names[f.compare_type],
				string = f.string))

		log("saving junkmail filters to '%s'" % self.__filename)
		self.__indent(root)
		ElementTree(root).write(self.__filename)
		
	
	def get_filters(self):
		return self.__filters
	
	
	def clear_filters(self):
		del self.__filters[:]
	
	
	def add_filter(self, filter):
		self.__filters.append(filter)
	
	
	def remove_filter(self, filter):
		self.__filters.remove(filter)
		
	
	def lock(self):
		self.__lock.acquire()
	
	
	def unlock(self):
		self.__lock.release()
	
	
	def __create_filter_file(self):
		log("creating junkmail filterfile '%s'" % self.__filename)
		
		# Create path to filterfile if necessary.
		dir_name = os.path.dirname(self.__filename)
		if len(dir_name) and (not os.path.exists(dir_name)):
			os.makedirs(dir_name)
		
		root = Element("junkfilters")		
		ElementTree(root).write(self.__filename)
	
	
	def __indent(self, elem, level = 0):
		"""Indents a tree with each node according to its depth."""
		i = "\n" + level * "\t"
		if len(elem):
			if not elem.text or not elem.text.strip():
				elem.text = i + "\t"
			if not elem.tail or not elem.tail.strip():
				elem.tail = i
			for elem in elem:
				self.__indent(elem, level + 1)
			if not elem.tail or not elem.tail.strip():
				elem.tail = i
		else:
			if level and (not elem.tail or not elem.tail.strip()):
				elem.tail = i


# tests:
if __name__ == "__main__":
	m = JunkFilterManager(os.path.expanduser("~/junkfilters_test.xml"), True);
	print m.match_filters([u"späm!","me@spamer.com","1"])
	print m.match_filters(["none","zulu <zulu99@gmx.net>","3"])
	print m.filter([["nospam","guy@domain.com", "5"], [u"späm!","guy@domain.com","3"], ["subject", "spamer@1und1.de", "8"]])
	print CompareType.names[m.get_filters()[0].compare_type]
	#m.add_filter(JunkFilter(MailField.SUBJECT, CompareType.CONTAINS, "foo"))
	#a,b,c,d = m.get_filters()
	#print m.get_filters()
	#m.remove_filter(d)	
	#print m.get_filters()
	#m.save_filters()
