/*
cdcd - Command Driven CD player
Copyright (C) 1998-99 Tony Arcieri
Copyright (C) 2001 Fabrice BAUZAC

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <errno.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <stdio.h>

#include "cmdline.h"

#include "config.h"

#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include "cdaudio.h"

#include "conf.h"
#include "global.h"
#include "config.h"
#include "str.h"

int
cdcdrc_data_mod (const char *var, const char *value, struct cdcdrc *conf)
{
  if (value == NULL)
    value = "";
	
  if (!strcasecmp (var, cdcdrc_device_var))
    {
      if (!value[0])
	strncpy (conf->device, DEFAULT_DEVICE, CDCDRC_DEVICE_LEN - 1);
      else
	strncpy (conf->device, value, CDCDRC_DEVICE_LEN - 1);
      conf->device[CDCDRC_DEVICE_LEN - 1] = 0;
    }

  if (!strcasecmp(var, cdcdrc_verbose_var))
    {
      if (!strcasecmp (value, "on"))
	conf->verbosity = 1;
      else if (!strcasecmp (value, "off"))
	conf->verbosity = 0;
      else
	printf ("Error reading configuration file:\n"
		"Unknown value `%s' for parameter `%s', ignored.\n",
		*value? "(unspecified)": value,
		cdcdrc_verbose_var);
    }
	
  return 0;
}

void
cdcdrc_manual_configuration (struct cdcdrc *conf)
{
  void cdcdrc_write (const struct cdcdrc *);
  struct cddb_conf cddbconf;
  struct cddb_host proxy_host;
  struct cddb_serverlist list;
  struct cdcdrc writeconf;
  int useproxy = 0;
  char question_buffer[256], inbuffer[256], *proxy;
   
  conf->verbosity = 1;
   
  printf ("Welcome to %s %s.\n\n", PACKAGE, VERSION);
  pprintf (get_width () - 1,
	   "You will now be asked a series of questions which will configure "
	   "cdcd for your platform.  The default answers are given between "
	   "brackets.");
  putchar ('\n');

  snprintf (question_buffer, 256,
	    "What is the name of your CD-ROM device? [%s] ",
	    DEFAULT_DEVICE);
  get_input_text (inbuffer, question_buffer, 256);
   
  if (strlen (inbuffer) != 0)
    strncpy (conf->device, inbuffer, 256);
  else
    strncpy (conf->device, DEFAULT_DEVICE, 256);
   
  while (1)
    {
      get_input_text (inbuffer,
		      "Are you connected to a network (y/n) [y]? ",
		      256);
      if (!strcasecmp (inbuffer, "n"))
	{
	  cddbconf.conf_access = CDDB_ACCESS_LOCAL;
	  cddb_write_serverlist (cddbconf, list, proxy_host.host_server);
	  memcpy (&writeconf, conf, sizeof *conf);
	  cdcdrc_write (&writeconf);
	  return;
	}
      else if (!strcasecmp (inbuffer, "y") || !inbuffer)
	{
	  cddbconf.conf_access = CDDB_ACCESS_REMOTE;
	  break;
	}
      else
	puts ("?");
    }

  putchar ('\n');
  pprintf (get_width () - 1,
	   "There are two methods of retrieving CD track information, CDDB "
	   "and CD Index.  Using the CD Index will be somewhat faster, but "
	   "does not yet facilitate inexact matching.  Both methods may be "
	   "used through HTTP proxy servers.");
  fputs ("\n", stdout);

  while (1)
    {
      fputs ("\n", stdout);
      fputs ("1) CDDB\n"
	     "2) CD Index\n\n",
	     stdout);
      get_input_text (inbuffer, "Choose a method [1]: ", 256);
      if (!strcmp (inbuffer, "2"))
	{
	  list.list_len = 1;
	  strncpy (list.list_host[0].host_server.server_name,
		   "www.cdindex.org", 256);
	  strncpy (list.list_host[0].host_addressing,
		   "cgi-bin/cdi/get.pl", 256);
	  list.list_host[0].host_server.server_port = 80;
	  list.list_host[0].host_protocol = CDINDEX_MODE_HTTP;
	  break;
	}
      else if (!*inbuffer || !strcmp (inbuffer, "1"))
	{
	  list.list_len = 1;
	  strncpy (list.list_host[0].host_server.server_name,
		   "www.freedb.org", 256);
	  strncpy (list.list_host[0].host_addressing,
		   "cgi-bin/cddb.cgi", 256);
	  list.list_host[0].host_server.server_port = 80;
	  list.list_host[0].host_protocol = CDDB_MODE_HTTP;
	  break;
	}
      else
	puts ("?");
    }
   
  if ((proxy = getenv ("http_proxy")))
    {
      cddb_process_url (&proxy_host, proxy);
      printf ("\nHTTP proxy server found: http://%s:%d/\n\n",
	      proxy_host.host_server.server_name,
	      proxy_host.host_server.server_port);
      get_input_text (inbuffer, "Use it? [y] ", 256);
      if (inbuffer[0] == 'n')
	useproxy = 1;
      else
	cddbconf.conf_proxy = 1;
    }
   
  if (proxy == NULL || useproxy == 1)
    {
      get_input_text (inbuffer,
		      "Use a HTTP proxy server (y/n)? [n] ", 256);
      if (inbuffer[0] == 'y')
	{
	  get_input_text (inbuffer, "Enter proxy server URL: ", 256);
	  cddb_process_url (&proxy_host, inbuffer);
	  cddbconf.conf_proxy = 1;
	}
    }
   
  putchar ('\n');
   
  cddb_write_serverlist (cddbconf, list, proxy_host.host_server);
  memcpy (&writeconf, conf, sizeof (struct cdcdrc));
  cdcdrc_write (&writeconf);
   
  return;
}

int
cdcdrc_process_line (char *line, struct cdcdrc *conf)
{
  char *var, *value;
    
  if (! (value = strchr (line, '=')))
    return 0;

  *value++ = 0;

  var = strtok (line, " \t");
  value = strtok (value, " \t\n");

  if (!var || !value || strtok (NULL, "\n"))
    return 0;
  if (cdcdrc_data_mod (var, value, conf) < 0)
    return -1;
  return 0;
}

int
file_exists (char *filename)
{
  struct stat s;
  return !stat (filename, &s);
}

/* Returns a newly-allocated string containing the path of .cdcdrc, or
   NULL if $HOME is not set. */
char *
cdcdrc_location_a ()
{
  char *loc, *home;
  if (! (home = getenv ("HOME")))
    return NULL;
  loc = (char*) xmalloc (strlen (cdcdrc_file) + strlen (home) + 2);
  sprintf (loc, "%s/%s", home, cdcdrc_file);
  return loc;
}

/* Read .cdcdrc.  */
int
cdcdrc_read (struct cdcdrc *conf)
{
  FILE *cdcdrcfile;
  char inbuffer[256];
  char *localconfpath;
  char *p;
  
  if (!(localconfpath = cdcdrc_location_a ()))
    {
      puts ("$HOME is not set!");
      return -1;
    }

  /* This macro frees the resources and exits.  */
#define X(n) { free(localconfpath); return n; }

  if (!file_exists (localconfpath))
    {
      cdcdrc_manual_configuration (conf);
      X (0);
    }
	
  cdcdrcfile = fopen (localconfpath, "r");
  if (cdcdrcfile == NULL)
    {
      fputs ("error: can't read $HOME/.cdcdrc\n", stderr);
      exit (1);
    }

  fgets (inbuffer, 256, cdcdrcfile);
  if (strncmp (inbuffer, cdcdrc_firstline, strlen (cdcdrc_firstline)))
    {
      puts ("Your .cdcdrc file is invalid.\n"
	    "Please remove it and restart cdcd.");
      exit (1);
    }
	
  while (! feof (cdcdrcfile))
    {
      fgets (inbuffer, 256, cdcdrcfile);
      inbuffer[255] = '\0';

      if ((p = strchr (inbuffer, '#')))
	*p = 0;
	   
      if (cdcdrc_process_line (inbuffer, conf) < 0) {
	X (-1);
      }
    }
	
  fclose (cdcdrcfile);
	
  X (0);
#undef X
}

/* Write .cdcdrc.  */
void
cdcdrc_write (const struct cdcdrc *conf)
{
  FILE *cdcdrcfile;
  char *localconfpath;
	
  if (! (localconfpath = cdcdrc_location_a ()))
    {
      puts ("$HOME is not set!");
      return;
    }

  if ((cdcdrcfile = fopen (localconfpath, "w")) == NULL)
    {
      printf ("Unable to save configuration to `%s': %s\n", localconfpath,
	      strerror (errno));
      free (localconfpath);
      return;
    }
	
  fputs ("# .cdcdrc revision 3\n\n", cdcdrcfile);
  fprintf (cdcdrcfile, "%s=%s\n", cdcdrc_verbose_var,
	   conf->verbosity ? "ON" : "OFF");
  fprintf (cdcdrcfile, "%s=%s\n", cdcdrc_device_var, conf->device);
	
  fclose (cdcdrcfile);
  free (localconfpath);
}
