#!/usr/bin/env ruby
#
#    This program can be distributed under the terms of the GNU GPL.
#    See the file COPYING.
#
#    Author: Martin Prtel <martin.partel@gmail.com>
#

require 'fileutils.rb'
include FileUtils

# Set the default umask for all tests
File.umask 0022

EXECUTABLE_PATH = '../src/bindfs'
TESTDIR_NAME = 'tmp_test_bindfs'


# Prepares a test environment with a mounted directory
def testenv(testcase_title, bindfs_args, &block)

    puts "--- #{testcase_title} ---"
    puts "[  #{bindfs_args}  ]"

    begin
        Dir.mkdir TESTDIR_NAME
    rescue Exception => ex
        $stderr.puts "ERROR creating testdir at #{TESTDIR_NAME}"
        exit! 1
    end

    begin
        Dir.chdir TESTDIR_NAME
        Dir.mkdir 'src'
        Dir.mkdir 'mnt'
    rescue Exception => ex
        $stderr.puts "ERROR preparing testdir at #{TESTDIR_NAME}"
        $stderr.puts ex
        exit! 1
    end

    bindfs_pid = nil
    begin
        cmd = "../#{EXECUTABLE_PATH} #{bindfs_args} src mnt"
        bindfs_pid = Process.fork do
            exec cmd
            exit! 127
        end
    rescue Exception => ex
        $stderr.puts "ERROR running bindfs"
        $stderr.puts ex
        Dir.chdir '..'
        system("rm -Rf #{TESTDIR_NAME}")
        exit! 1
    end

    # Wait for bindfs to daemonize itself
    Process.wait bindfs_pid

    # TODO: check that mounting was successful

    testcase_ok = true
    begin
        yield
    rescue Exception => ex
        $stderr.puts "ERROR: testcase `#{testcase_title}' failed"
        $stderr.puts ex
        $stderr.puts ex.backtrace
        testcase_ok = false
    end

    begin
        unless system('fusermount -uz mnt')
            raise Exception.new("fusermount -uz failed with status #{$?}")
        end
    rescue Exception => ex
        $stderr.puts "ERROR: failed to umount"
        $stderr.puts ex
        $stderr.puts ex.backtrace
        testcase_ok = false
    end

    begin
        Dir.chdir '..'
    rescue Exception => ex
        $stderr.puts "ERROR: failed to exit test env"
        $stderr.puts ex
        $stderr.puts ex.backtrace
        exit! 1
    end

    unless system "rm -Rf #{TESTDIR_NAME}"
        $stderr.puts "ERROR: failed to clear test directory"
        exit! 1
    end

    if testcase_ok
        puts "OK"
    else
        exit! 1
    end
end


def assert
    raise Exception.new('test failed') unless yield
end

def assert_exception(ex)
    begin
        yield
    rescue ex
        return
    end
    raise Exception.new('expected exception ' + ex.to_s)
end


