/*
 * (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Progress dialog class implementation
 *
 *	by Tony Sideris	(05:54AM Aug 12, 2001)
 *================================================*/
#include "arson.h"

#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qregexp.h>
#include <qfontmetrics.h>

#include <kfiledialog.h>
#include <klocale.h>
#include <kapp.h>

#include <dcopclient.h>

#include "progressdlg.h"
#include "configdlg.h"
#include "processmgr.h"
#include "logwnd.h"
#include "progressbar.h"
#include "layout.h"

/*========================================================*/
/*	ArsonProgress-specific implemetation of the
 *	ArsonWriterUI interface
 *========================================================*/

class progressDlgUI : public ArsonProcessUI
{
public:
	progressDlgUI (ArsonProgress *pDlg)
		: m_pDlg(pDlg) {}

	virtual void setMaxProgress (long maxp)
	{
		ArsonProcessUI::setMaxProgress(maxp);
		m_pDlg->setMaxProgress(maxp);
	}

	virtual void setProgress (long prog)
	{
		ArsonProcessUI::setProgress(prog);
		m_pDlg->setProgress(prog);
	}

	virtual void output (const QString &str, bool error) {
		m_pDlg->output(str, error);
	}

	virtual void setText (const QString &text) {
		m_pDlg->setText (text);
	}

	virtual void begin (void) {
		m_pDlg->beginWrite();
	}

	virtual void end (void) {
		m_pDlg->endWrite();
	}

private:
	ArsonProgress *m_pDlg;
};

/*========================================================*/

class pushButtonLogUI : public ArsonLogState
{
public:
	pushButtonLogUI (QPushButton *pb) : m_pb(pb) { }

	virtual void setState (bool enable) {
		m_pb->setOn(enable);
	}

protected:
	QPushButton *m_pb;
};

/*========================================================*/
/*	Progress dialog class implementation
 *========================================================*/

ArsonProgress::ArsonProgress (QWidget *parent, const char *name)
	: ArsonProgressBase(parent ? parent : kapp->mainWidget(), name),
	m_pWriter(NULL), m_pBtnLayout(NULL), m_pUI(NULL),
	m_pBtnGroup(NULL), m_baseCaption(i18n("Progress"))
{
	ArsonLogWindow::Wnd()->addUiItem((m_pLogUI = new pushButtonLogUI(cmdlog)));

	setProgressMode(arsonProgressNormal);
	out->setSelectionMode(QListBox::NoSelection);
	cmdlog->setToggleButton(true);

	const QFontMetrics fm (fontMetrics());
	out->setMinimumHeight((fm.height() + fm.leading()) * 6);

#ifdef ARSONDBG
	m_DEBUG_SETUP = false;
#endif

	setMinimumWidth(420);
}

ArsonProgress::~ArsonProgress (void)
{
	ArsonLogWindow::Wnd()->delUiItem(m_pLogUI);
	delete m_pWriter;

	Trace("ArsonProgress widget destroyed\n");
}

/*========================================================*/

ArsonProcessUI *ArsonProgress::ui (void)
{
	if (!m_pUI)
		m_pUI = new progressDlgUI(this);

	return m_pUI;
}

/*========================================================*/

void ArsonProgress::setCaption (const QString &text)
{
	m_baseCaption = text;
	_setCaption();
}

void ArsonProgress::_setCaption (void)
{
	const int max = progress->totalSteps();

	if (progress->mode() != arsonProgressNormal)
		emit setParentCaption(m_baseCaption);

	else if (max != 0)
	{
		const double percent =
			double(progress->progress()) /
			double(max);
		const int n = int(percent * 100);

		emit setParentCaption(
			i18n("%1 - %2%")
			.arg(m_baseCaption)
			.arg((n < 0) ? 0 : n));
	}
}

/*========================================================*/

void ArsonProgress::enableControls (bool enable)
{
	int index;

	for (index = 0; index < m_widgets.count(); ++index)
		m_widgets[index]->setEnabled(enable);
}

/*========================================================*/

void ArsonProgress::start_clicked (void)
{
	Assert(m_pWriter == NULL);
	Assert(m_DEBUG_SETUP == true);

	try
	{
		if ((m_pWriter = createProcessMgr()))
		{
			ArsonProcessOpts opts;
			ArsonCtrlState state (this);

			emit saveState(state);
			processOpts(opts);

			m_pWriter->begin(opts);
		}
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/

void ArsonProgress::setup (void)
{
	ArsonCtrlState state (this);
	emit restoreState(state);
	reconfigure();

#ifdef ARSONDBG
	m_DEBUG_SETUP = true;
#endif
}

/*========================================================*/
/*	Write the contents of the output window to file.
 *========================================================*/

void ArsonProgress::save_output_clicked (void)
{
	arsonSaveLogWindow(out);
}

/*========================================================*/

void ArsonProgress::show_cmd_log (void)
{
	if (!ArsonLogWindow::Wnd()->isVisible())
		ArsonLogWindow::Wnd()->show();
	else
		ArsonLogWindow::Wnd()->hide();
}

/*========================================================*/
/**
 *	Return TRUE if it's OK to close the parent window.
 *	This should only be true if we're not writing
 *	anything.
 */

bool ArsonProgress::cancel (void)
{
	if (!m_pWriter || !m_pWriter->isWriting())
		return true;

	else if (cancelWarning())
		m_pWriter->abort();

	return false;
}

/*========================================================*/

bool ArsonProgress::cancelWarning (void)
{
	return KMessageBox::questionYesNo(kapp->mainWidget(),
		i18n("Interrupting a read/write operation on some CDR(W) devices can lead to system instability. Are you sure you wish to cancel?"),
		i18n("Read/Write Abort?")) == KMessageBox::Yes;
}

/*========================================================*/

void ArsonProgress::setText (const QString &text)
{
	message->setText(text);
}

void ArsonProgress::output (const QString &str, bool error)
{
	QString text (str);
	const int index = out->count();
	QScrollBar *ps = out->verticalScrollBar();

	text.replace(QRegExp("[\t]"), "  ");
	out->insertItem(text);

	//	Auto scroll the log window
	if (!ps || (ps->value() == ps->maxValue()))
	{
		out->setSelected(index, true);
		out->setCurrentItem(index);
		out->setBottomItem(index);
	}
}

/*========================================================*/

void ArsonProgress::setMaxProgress (long maxp)
{
	switch (maxp)
	{
	case -1:
		setProgressMode(arsonProgressInfinite);
		break;

	case 0:
		setProgressMode(arsonProgressBlank);
		break;

	default:
		setProgressMode(arsonProgressNormal);
		progress->setTotalSteps(maxp);
	}
}

/*========================================================*/

void ArsonProgress::setProgress (long prog)
{
	if (prog > progress->totalSteps())
		prog = progress->totalSteps();

	progress->setProgress(prog);
	_setCaption();
}

/*========================================================*/

void ArsonProgress::setProgressMode (int mode)
{
	if (progress->mode() != mode)
	{
		progress->setMode(mode);
		_setCaption();
	}
}

/*========================================================*/
/*	Enable/disable the scrensaver
 *========================================================*/

bool isSSEnabled (void)
{
	QByteArray res;
	QCString repl;
	int enabled;

	if (!kapp->dcopClient()->call("kdesktop", "KScreensaverIface",
			"isEnabled()", QByteArray(), repl, res, true))
	{
		Trace("WARNING: Failed to query screensaver setting\n");
		return false;
	}

	QDataStream ds (res, IO_ReadOnly);

	ds >> enabled;
	return enabled ? true : false;
}

void ArsonProgress::enableScreenSaver (bool enable)
{
	if (!ACONFIG.is(ArsonConfig::flagScreenSaver))
		return;

	if (!enable)
	{
		m_bScreenSaver = isSSEnabled();

		Trace("[Dis]abling screensaver, was: %d\n", m_bScreenSaver);
	}

	if (m_bScreenSaver)
	{
		QByteArray ba;
		QDataStream ds (ba, IO_WriteOnly);

		ds << enable;

		kapp->dcopClient()->send("kdesktop",
			"KScreensaverIface",
			"enable(bool)", ba);
	}

#ifdef ARSONDBG
	Trace("%sabled screensaver\n", isSSEnabled() ? "En" : "Dis");
#endif	//	ARSONDBG
}

/*========================================================*/

void ArsonProgress::beginWrite (void)
{
	enableControls(false);
	emit enableUI(Writing);

	enableScreenSaver(false);
}

void ArsonProgress::endWrite (void)
{
	setProgressMode(arsonProgressBlank);
	emit enableUI(Complete);

	enableScreenSaver(true);
}

/*========================================================*/
/*	Adds and manages the widgets in the center
 *========================================================*/

QCheckBox *ArsonProgress::addCheckbox (const QString &text, const char *name, bool state)
{
	QCheckBox *ptr = new ArsonProgressCheckbox(text, ctrlParent(), name);

	ptr->setChecked(state);
	widgetMgr() << ptr;
	return ptr;
}

bool ArsonProgress::isCheckboxChecked (const char *name)
{
	if (QCheckBox *ptr = (QCheckBox *) child(name))
		return ptr->isChecked();

	Assert(0 && "Checkbox not found");
	return false;
}

int ArsonProgress::addWidget (QWidget *widget, bool align)
{
	m_widgets.append(widget);

	if (align)
		ctrlLayout()->addWidget(widget);

	return m_widgets.count();
}

/*========================================================*/

QBoxLayout *ArsonProgress::ctrlLayout (void)
{
	if (!m_pBtnLayout)
	{
		m_pBtnGroup = new QGroupBox(i18n("Options"), this);
		m_pBtnGroup->setColumnLayout(0, Qt::Vertical);
		m_pBtnGroup->layout()->setSpacing(0);
		m_pBtnGroup->layout()->setMargin(0);

		m_pBtnLayout = new QVBoxLayout(m_pBtnGroup->layout());
		m_pBtnLayout->setAlignment(Qt::AlignTop);
		m_pBtnLayout->setSpacing(6);
		m_pBtnLayout->setMargin(11);
		
		ctrl_b0x->add(m_pBtnGroup);
	}

	return m_pBtnLayout;
}

QWidget *ArsonProgress::ctrlParent (void)
{
	ctrlLayout();
	return m_pBtnGroup;
}

/*========================================================*/
/*	Geometry Helperz
 *========================================================*/

ArsonProgress::WidgetMgr::WidgetMgr (ArsonProgress *pd)
	: m_pd(pd)
{
	//	Nothing...
}

ArsonProgress::WidgetMgr &ArsonProgress::WidgetMgr::operator<< (QWidget *pw)
{
	m_pd->addWidget(pw, true);
	return *this;
}

/*========================================================*/

ArsonProgress::LayoutRow::LayoutRow (ArsonProgress *pd, bool indent)
	: m_pd(pd), m_row(NULL)
{
	m_row = m_pd->makeLayoutRow(indent);
}

ArsonProgress::LayoutRow &ArsonProgress::LayoutRow::operator<< (QWidget *pw)
{
	m_row->addWidget(pw);
	m_pd->addWidget(pw, false);

	return *this;
}

ArsonProgress::LayoutRow &ArsonProgress::LayoutRow::operator<< (QLayoutItem *pl)
{
	m_row->addItem(pl);
	return *this;
}

QLayoutItem *ArsonProgress::spacer (int width)
{
	return new QSpacerItem(width, 12,
		QSizePolicy::Fixed,
		QSizePolicy::Minimum);
}

/*========================================================*/

QHBoxLayout *ArsonProgress::makeLayoutRow (bool indent)
{
	QHBoxLayout *ph = new QHBoxLayout(ctrlParent());
	ph->setSpacing(1);

	if (indent)
		ph->addItem(
			new QSpacerItem(14, 12,
				QSizePolicy::Fixed,
				QSizePolicy::Minimum));

	ctrlLayout()->addLayout(ph);
	return ph;
}

/*========================================================*/
/*	The actual dialog class that holds the progress widget
 *========================================================*/

ArsonProgressDlg::ArsonProgressDlg (QWidget *parent, const char *name)
	: QDialog(parent, name, true),
	m_pProgressWidget(NULL)
{
	ArsonLayout vl (new QVBoxLayout(this, 11, 6));
	ArsonLayout hl (new QHBoxLayout(this, 0, 6));
	QSpacerItem *ps = new QSpacerItem(0, 2,
		QSizePolicy::Expanding, QSizePolicy::Fixed);
	QFrame *pf = new QFrame(this);

	//	This will hold the main progress area
	m_pMainLayout = new QVBoxLayout(this, 0, 0);
	vl << m_pMainLayout;

	//	Setup the bottom button row
	m_pStart = new QPushButton(i18n("&Start"), this);
	m_pConfig = new QPushButton(i18n("Con&figure..."), this);
	m_pCancel = new QPushButton(i18n("&Close"), this);

	pf->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	m_pStart->setEnabled(false);

	hl << ps << m_pStart << m_pConfig << m_pCancel;
	vl << pf << hl;

	QObject::connect(m_pConfig, SIGNAL(clicked()), this, SLOT(config_clicked()));

	setCaption(i18n("Progress"));
}

/*========================================================*/

void ArsonProgressDlg::setProgressWidget (ArsonProgress *pw)
{
	m_pMainLayout->addWidget((m_pProgressWidget = pw));
	new ArsonProgressEventFilter(pw, this, m_pStart, m_pCancel);

	QObject::connect(pw, SIGNAL(setParentCaption(const QString &)),
		this, SLOT(setCaption(const QString &)));
	QObject::connect(pw, SIGNAL(enableUI(ArsonProgress::Status)),
		this, SLOT(slotEnableUI(ArsonProgress::Status)));

	m_pStart->setEnabled(true);

	pw->setup();
	adjustSize();
}

/*========================================================*/

void ArsonProgressDlg::slotEnableUI (ArsonProgress::Status status)
{
	m_pConfig->setDisabled(true);
}

/*========================================================*/

void ArsonProgressDlg::config_clicked (void)
{
	ArsonConfigDlg dlg (ACONFIG, ArsonConfigDlg::BURNER, this);

	if (dlg.exec() == QDialog::Accepted)
		m_pProgressWidget->reconfigure();
}

/*========================================================*/

ArsonProgress *ArsonProgress::findProgressWidget (QWidget *widget)
{
	QObject *ptr = widget;

	while (ptr)
	{
		if (ptr->inherits("ArsonProgress"))
			break;

		ptr = ptr->parent();
	}

	return (ArsonProgress *) (ptr ? ptr : widget);
}

/*========================================================*/
/*	The progress event filter, so all parent classes(many
 *	may have completely different base classes) don't
 *	have to re-implement the same start/cancel/close
 *	logic.
 *========================================================*/

ArsonProgressEventFilter::ArsonProgressEventFilter (ArsonProgress *pp,
	QDialog *parent, QPushButton *start, QPushButton *cancel)
	: QObject(parent, "pf"),
	m_progress(pp), m_start(start), m_cancel(cancel)
{
	/*	Block the parent's CloseEvent while
	 *	processes are active.
	 */
	parent->installEventFilter(this);

	start->disconnect();
	start->setText(i18n("&Start"));
	QObject::connect(start, SIGNAL(clicked()), pp, SLOT(start_clicked()));

	cancel->disconnect();
	cancel->setText(i18n("&Close"));
	QObject::connect(cancel, SIGNAL(clicked()), this, SLOT(slotCancel()));

	QObject::connect(pp, SIGNAL(enableUI(ArsonProgress::Status)),
		this, SLOT(slotEnableUI(ArsonProgress::Status)));

	QObject::connect(this, SIGNAL(complete()), parent, SLOT(reject()));
}

ArsonProgressEventFilter::~ArsonProgressEventFilter (void)
{
	Trace("progress event filter destroyed\n");
}

/*========================================================*/

bool ArsonProgressEventFilter::eventFilter (QObject *po, QEvent *pe)
{
	if (pe->type() == QEvent::Close)
		return !m_progress->cancel();

	return po->eventFilter(po, pe);
}

/*========================================================*/

void ArsonProgressEventFilter::slotEnableUI (ArsonProgress::Status status)
{
	switch (status)
	{
	case ArsonProgress::Writing:
		m_start->setEnabled(false);
		m_cancel->setText(i18n("&Cancel"));
		break;
		
	case ArsonProgress::Complete:
		m_cancel->setText(i18n("&Close"));
		break;
	}
}

/*========================================================*/

void ArsonProgressEventFilter::slotCancel (void)
{
	QDialog *pd = (QDialog *) parent();
	
	if (m_progress->cancel())
		emit complete();
}

/*========================================================*/
/*	Simple progress dialog with no additional controls
 *========================================================*/

ArsonSimpleProgress::ArsonSimpleProgress (const QString &text,
	const QString &caption, QWidget *parent, const char *name)
	: ArsonProgress(parent, name)
{
	setCaption(caption);
	message->setText(text);
}

/*========================================================*/
/*	Used to save, and restore the state of controls
 *========================================================*/

ArsonCtrlState::ArsonCtrlState (ArsonProgress *pd)
	: m_pk(kapp->config())
{
	QString group ("progress:");
	const char *name = pd->name();

	Assert(name && *name);
	
	group.append(pd->name());
	m_pk->setGroup(group);
}

/*========================================================*/

ArsonCtrlState &ArsonCtrlState::operator>> (QCheckBox *pc)
{
	if (pc->name(0) && m_pk->readEntry(pc->name()) != QString::null)
		pc->setChecked(m_pk->readBoolEntry(pc->name()));

	return *this;
}

ArsonCtrlState &ArsonCtrlState::operator>> (QComboBox *pc)
{
	if (const char *name = pc->name(0))
	{
		if (pc->editable())
		{
			const QString val = m_pk->readEntry(name);

			if (!val.isEmpty())
				pc->setEditText(val);
		}
		else
		{
			int index;

			if ((index = m_pk->readNumEntry(name, -1)) != -1)
				pc->setCurrentItem(index);
		}
	}

	return *this;
}

ArsonCtrlState &ArsonCtrlState::operator>> (QLineEdit *pe)
{
	QString str;
	
	if (pe->name(0) && (str = m_pk->readEntry(pe->name())) != QString::null)
		pe->setText(str);

	return *this;
}

/*========================================================*/

ArsonCtrlState &ArsonCtrlState::operator<< (QCheckBox *pc)
{
	if (pc->name(0))
		m_pk->writeEntry(pc->name(), (bool) pc->isChecked());

	return *this;
}

ArsonCtrlState &ArsonCtrlState::operator<< (QComboBox *pc)
{
	if (const char *name = pc->name(0))
	{
		if (pc->editable())
			m_pk->writeEntry(name, pc->currentText());
		else
			m_pk->writeEntry(name, pc->currentItem());
	}

	return *this;
}

ArsonCtrlState &ArsonCtrlState::operator<< (QLineEdit *pe)
{
	//	We don't wanna save password boxes
	if (pe->name(0) && pe->echoMode() == QLineEdit::Normal)
		m_pk->writeEntry(pe->name(), pe->text());

	return *this;
}

/*========================================================*/
/*	Fancy widget types that support save/restore
 *========================================================*/

ArsonProgressCheckbox::ArsonProgressCheckbox (const QString &label, QWidget *parent, const char *name)
	: QCheckBox(label, parent, name)
{
	ARSON_CONNECT_PROGRESSCTRL(parent, this);
}

ArsonProgressCombobox::ArsonProgressCombobox (QWidget *parent, const char *name)
	: QComboBox(parent, name)
{
	ARSON_CONNECT_PROGRESSCTRL(parent, this);
}

ArsonProgressLineedit::ArsonProgressLineedit (QWidget *parent, const char *name)
	: QLineEdit(parent, name)
{
	ARSON_CONNECT_PROGRESSCTRL(parent, this);
}

IMPLEMENT_PROGRESSCTRL_SLOTS(ArsonProgressCheckbox)
IMPLEMENT_PROGRESSCTRL_SLOTS(ArsonProgressCombobox)
IMPLEMENT_PROGRESSCTRL_SLOTS(ArsonProgressLineedit)

/*========================================================*/
/*	Control group for cdrdao progress widgets
 *========================================================*/

ArsonCdrdaoCtrlGrp::ArsonCdrdaoCtrlGrp (QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	ArsonLayout hl (new QHBoxLayout(this, 0, 6));
	QLabel *pl = new QLabel(i18n("Read Sub-Channel:"), this);
	const QString subs[] = {
		i18n("None"),
		i18n("R-W"),
		i18n("Raw R-W"),
	};

	m_pRaw = new ArsonProgressCheckbox(i18n("Read &Raw"), this, "raw");
	m_pSubChan = new ArsonProgressCombobox(this, "subchan");

	pl->setSizePolicy(
		QSizePolicy(
			QSizePolicy::Maximum,
			QSizePolicy::Maximum));

	for (int index = 0; index < SubChan_Count; ++index)
		m_pSubChan->insertItem(subs[index]);

	m_pSubChan->setCurrentItem(None);

	hl << m_pRaw << ArsonProgress::spacer() << pl << m_pSubChan;
}

/*========================================================*/

void ArsonCdrdaoCtrlGrp::applyTo (ArsonProcessOpts &opts) const
{
	opts.addLong("subchan", subChan());
	opts.addBool("raw", readRaw());
}

/*========================================================*/

bool ArsonCdrdaoCtrlGrp::readRaw (void) const { return m_pRaw->isChecked(); }

ArsonCdrdaoCtrlGrp::SubChan ArsonCdrdaoCtrlGrp::subChan (void) const
{ return (SubChan) m_pSubChan->currentItem(); }

/*========================================================*/
