/*  Main.cpp - main() function and other gtk+ blurbs
 *  Copyright (C) 1998 Andy Lo A Foe <andy@alsa-project.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

//#define FUNKY_STUFF
//#define TESTING

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <assert.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <math.h>
#include <dlfcn.h>
#ifdef FUNKY_STUFF
#include <X11/Xlib.h>
#endif

#include "config.h"

#include "SampleBuffer.h"
#include "CorePlayer.h"
#include "Playlist.h"
#include "Effects.h"
#include "EffectsWindow.h"
#include "ScopesWindow.h"
#include "Main.h"
#include "input_plugin.h"
#include "output_plugin.h"

#include <gtk/gtk.h>
#include "gladesrc.h"
#include "pixmaps/f_play.xpm"
#include "pixmaps/r_play.xpm"
#include "pixmaps/pause.xpm"
#include "pixmaps/next.xpm"
#include "pixmaps/prev.xpm"
#include "pixmaps/stop.xpm"
#if 0
#include "pixmaps/eject.xpm"
#include "pixmaps/effects.xpm"
#endif
#include "pixmaps/play.xpm"
#include "pixmaps/playlist.xpm"
#include "pixmaps/cd.xpm"

#if 0
#include "gtk_bscale.h"
#endif

//#define NEW_SCALE
#define UPDATE_TIMEOUT	200000	
#define BAL_CENTER	100	
#define MIN_BAL_TRESH	BAL_CENTER-10	// Center is a special case
#define MAX_BAL_TRESH	BAL_CENTER+10	// so we build in some slack
#define ZERO_PITCH_TRESH 2 

#ifdef FUNKY_STUFF
Display *display = NULL;
Window root;
#endif

#ifdef SUBSECOND_DISPLAY
#define INDICATOR_WIDTH 80
#else
#define INDICATOR_WIDTH 64
#endif

#if 1 
extern scope_plugin monoscope_plugin;
extern scope_plugin fftscope_plugin;
#endif

int oldX, oldY;
int leftX, leftY, rightX, rightY;

static pthread_t playlist_thread;
static pthread_t indicator_thread;

gint global_bal = 100;
gint global_vol_left = 100;
gint global_vol_right = 100;
gint global_vol = 100;
gint global_update = 1;
//gint global_ind_timeout = 0;
gint global_speed = 0;
gint global_playlist_show = 0;
gint global_effects_show = 0;
gint global_scopes_show = 0;
gint global_reverb_on = 0;
gint global_reverb_delay = 2;
gint global_reverb_feedback = 0;
gint global_playing = 0;
static GtkWidget *playlist_window = NULL;
static GtkWidget *play_dialog;
static GtkWidget *play_pix;
static GdkPixmap *val_ind = NULL;
static gint global_rb = 1;


typedef struct  _update_struct {
	gpointer data;
	GtkWidget *drawing_area;
	GtkWidget *vol_scale;
	GtkWidget *bal_scale;
	GtkWidget *pos_scale;
	GtkWidget *speed_scale;
} update_struct;	

update_struct global_ustr;

/* These are used to contain the size of the window manager borders around
   our windows, and are used to show/hide windows in the same positions. */
gint windows_x_offset = -1;
gint windows_y_offset = -1; 

static gint main_window_x = 150;
static gint main_window_y = 175;

const char *default_input_addons[] = {
	{ "libmpg123.so" },
	{ "libcdda.so" },
        { "libmod.so" },
	{ "libaf.so" },
#ifdef HAVE_SOCKMON	
	{ "libsockmon.so" },
#endif
	NULL };

const char *default_output_addons[] = {
	{ "libalsa.so" },
	{ "liboss.so" },
	{ "libsparc.so" },
	{ "libesound.so" },
	{ "libsgi.so" },
	NULL };


gint indicator_callback(gpointer data);
gint alsaplayer_button_press(GtkWidget *widget, GdkEvent *event);


void load_output_addons(AlsaDAC *dac, char *module = NULL)
{
	void *handle;
	char path[1024];
	struct stat statbuf;

	output_plugin_info_type output_plugin_info;

	if (module) {
		sprintf(path, "%s/output/lib%s.so", ADDON_DIR, module);
#ifdef DEBUG
		printf("Loading output plugin: %s\n", path);
#endif
		if (stat(path, &statbuf) != 0) // Error reading object
			return;		
		if ((handle = dlopen(path, RTLD_NOW))) {
			output_plugin_info = (output_plugin_info_type) dlsym(handle, "output_plugin_info");
			if (output_plugin_info) {
				dac->RegisterPlugin(output_plugin_info());
				return;
			} else {
				fprintf(stderr, "symbol error in shared object: %s\n", path);
				fprintf(stderr, "Try starting up with \"-o\" to load a\n"
								"specific output module (alsa, oss, or esd)\n");
				dlclose(handle);
				return;
			}
		} else {
			printf("%s\n", dlerror());
		}
	}

	else
	
	for (int i=0; default_output_addons[i]; i++) {
		sprintf(path, "%s/output/%s", ADDON_DIR, default_output_addons[i]);
		if (stat(path, &statbuf) != 0)
			continue;
		if ((handle = dlopen(path, RTLD_LAZY))) {
			output_plugin_info = (output_plugin_info_type) dlsym(handle, "output_plugin_info");
			if (output_plugin_info) {
#ifdef DEBUG
				printf("Loading output plugin: %s\n", path);	
#endif
				if (dac->RegisterPlugin(output_plugin_info()))
					return; // Return as soon as we load one successfully!
			} else {
				printf("could not find symbol in shared object\n");
				dlclose(handle);
			}
		} else {
			printf("%s\n", dlerror());
		}
	}	
}



void load_scope_addons()
{
	char path[1024];
	struct stat buf;

	scope_plugin_info_type scope_plugin_info;

	sprintf(path, "%s/scopes", ADDON_DIR);

	DIR *dir = opendir(path);
	dirent *entry;

	if (dir) {
		while ((entry = readdir(dir)) != NULL) { // For each file in scopes
			if (strcmp(entry->d_name, ".") == 0 ||
				strcmp(entry->d_name, "..") == 0) {
				continue;
			}
			sprintf(path, "%s/scopes/%s", ADDON_DIR, entry->d_name);
			lstat(path, &buf);
			if (S_ISREG(buf.st_mode)) {
				void *handle;

				char *ext = strrchr(path, '.');
				if (!ext)
					continue;
				ext++;
				if (strcasecmp(ext, "so"))
					continue;
				if ((handle = dlopen(path, RTLD_NOW))) {
					scope_plugin_info = (scope_plugin_info_type) dlsym(handle, "scope_plugin_info");
					if (scope_plugin_info) { 
#ifdef DEBUG					
						fprintf(stderr, "Loading scope addon: %s\n", path);
#endif
						apRegisterScopePlugin(scope_plugin_info());
					} else {
						dlclose(handle);
					}
				} else {
					printf("%s\n", dlerror());
				}
			}
		}
		closedir(dir);
	}	
}


void load_input_addons(CorePlayer *p)
{
	void *handle;
	char path[1024];

	input_plugin_info_type input_plugin_info;

	for (int i=0; default_input_addons[i]; i++) {
		sprintf(path, "%s/input/%s", ADDON_DIR, default_input_addons[i]);
		if ((handle = dlopen(path, RTLD_NOW))) {
			input_plugin_info = (input_plugin_info_type)dlsym(handle, "input_plugin_info__Fv");

			if (input_plugin_info) {
#ifdef DEBUG
				printf("Loading input plugin: %s\n", path);
#endif
				p->RegisterPlugin(input_plugin_info());
			} else {
				printf("could not find symbol in shared object\n");
				dlclose(handle);
			}	
		} else {
			printf("%s\n", dlerror());
		}
	}	
}

/*Threads and usleep does not work, select is very portable*/
void dosleep(unsigned int msec)
{
	struct timeval time;
    time.tv_sec=0;
	time.tv_usec=msec;

    select(0,NULL,NULL,NULL,&time);
}

		

void playlist_looper(void *data)
{
	CorePlayer *p = (CorePlayer *)data;
#ifdef DEBUG
	printf("THREAD-%d=playlist thread\n", getpid());
#endif
	while (p) {
		if (!p->IsActive() && global_playing) {
			gdk_threads_enter();
			playlist_next(NULL, playlist_window);
			gdk_threads_leave();
		}
		dosleep(50000);
	}			
}


void indicator_looper(void *data)
{
#ifdef DEBUG
	printf("THREAD-%d=indicator thread\n", getpid());
#endif
	while (global_update >= 0) {
		gdk_threads_enter();
		if (global_update == 1) {
			indicator_callback(data);
		}
		gdk_threads_leave();
		dosleep(UPDATE_TIMEOUT);
	}	
}


GtkWidget *xpm_label_box(gchar * xpm_data[], GtkWidget *to_win)
{
	GtkWidget *box1;
	GtkWidget *pixmapwid;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkStyle *style;

	box1 = gtk_hbox_new(false, 0);
	gtk_container_border_width(GTK_CONTAINER(box1), 0);

	style = gtk_widget_get_style(to_win);

	pixmap = gdk_pixmap_create_from_xpm_d(to_win->window, &mask, &style->bg[
		GTK_STATE_NORMAL], xpm_data);
	pixmapwid = gtk_pixmap_new(pixmap, mask);

	gtk_box_pack_start(GTK_BOX(box1), pixmapwid, true, false, 1);

	gtk_widget_show(pixmapwid);

        return (box1);
}


void on_expose_event (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  gint x, y;

  if (windows_x_offset == -1)
    {
      gdk_window_get_origin (widget->window, &x, &y);
      windows_x_offset = x - main_window_x;
      /* Make sure offset seems reasonable. If not, set it to -2 so we don't
         try this again later. */
      if (windows_x_offset < 0 || windows_x_offset > 50)
        windows_x_offset = -2;
      else
        windows_y_offset = y - main_window_y;
    }
}

gint pixmap_expose(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	GdkPixmap *the_pixmap = (GdkPixmap *)data;
	gdk_draw_pixmap(widget->window,
		widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
		the_pixmap,
		event->area.x, event->area.y,
		event->area.x, event->area.y,
		event->area.width, event->area.height);
	return false;
}


gint val_area_configure(GtkWidget *widget, GdkEventConfigure *event, gpointer data)
{
	if (val_ind) {
		//gtk_timeout_remove(global_ind_timeout);
		global_update = 0;
		gdk_pixmap_unref(val_ind);
	}
	val_ind = gdk_pixmap_new(widget->window,
		widget->allocation.width,
		32, -1);
	gdk_draw_rectangle(val_ind,
                        widget->style->fg_gc[GTK_STATE_NORMAL],
                        true, 
                        0, 0,
                        widget->allocation.width,
                        32);
	// Set up expose event handler 
	gtk_signal_connect(GTK_OBJECT(widget), "expose_event",
                (GtkSignalFunc) pixmap_expose, val_ind);
	//global_ind_timeout = gtk_timeout_add(UPDATE_TIMEOUT,
	//		indicator_callback, NULL);	
	global_update = 1;	
	return true;
}

// CALLBACK FUNCTIONS


void fader(void *data)
{
	printf("In fader\n");
	for (int i=global_vol; i >= 0; i-=10) {
        	global_vol = i;
		printf("global_vol = %d\n", global_vol);
                dosleep(200000);
	}
}


gboolean main_window_delete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;
	global_update = -1;
#ifdef TESTING
	pthread_t bla;

	pthread_create(&bla, NULL, (void * (*)(void *))fader, data);
	pthread_detach(bla);
#else
	p->Stop();
        gtk_main_quit();
#endif
	return FALSE;
}


void press_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	//gtk_timeout_remove(global_ind_timeout);
	global_update = 0;
}

void draw_volume();

void volume_move_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	draw_volume();
}

void draw_title(char *title)
{
        update_struct *ustr = &global_ustr;
        GdkRectangle update_rect;
		static char old_title[128] = "";

		if (strcmp(old_title, title) == 0)
			return;
		else {
			if (strlen(title) > 127) {
				strncpy(old_title, title, 126);
				old_title[127] = 0;
			} else
				strcpy(old_title, title);
		}		
        update_rect.x = 82;
        update_rect.y = 0;
        update_rect.width = ustr->drawing_area->allocation.width - 82;
        update_rect.height = 18;

        // Clear area
        gdk_draw_rectangle(val_ind,
                ustr->drawing_area->style->fg_gc[GTK_STATE_NORMAL],
                true, update_rect.x, update_rect.y, update_rect.width,
                update_rect.height);
        // Draw string
        gdk_draw_string(val_ind, ustr->drawing_area->style->font,
                ustr->drawing_area->style->white_gc, update_rect.x+6,
                update_rect.y+14, title);
        // Do the drawing
        gtk_widget_draw (ustr->drawing_area, &update_rect);	
}

void draw_format(char *format)
{
	update_struct *ustr = &global_ustr;
	GdkRectangle update_rect;
	static char old_format[128] = "";

	if (strcmp(old_format, format) == 0) 
		return;
	else {
		if (strlen(format) > 126) {
			strncpy(old_format, format, 126);
			old_format[127] = 0;
		} else
			strcpy(old_format, format);
	}

	update_rect.x = 82;
	update_rect.y = 16;
	update_rect.width = ustr->drawing_area->allocation.width - 82; 
			// - INDICATOR_WIDTH;  
	update_rect.height = 18;
	
	// Clear area
	gdk_draw_rectangle(val_ind,
                ustr->drawing_area->style->fg_gc[GTK_STATE_NORMAL],
                true, update_rect.x, update_rect.y, update_rect.width,
		update_rect.height);
	// Draw string
	gdk_draw_string(val_ind, ustr->drawing_area->style->font,
		ustr->drawing_area->style->white_gc, update_rect.x+6,
		update_rect.y+12, format);
	// Do the drawing
	gtk_widget_draw (ustr->drawing_area, &update_rect);
}
	
	
void draw_volume()
{
	update_struct *ustr = &global_ustr;
	GtkAdjustment *adj;
	GdkRectangle update_rect;
	char str[60];
	static int old_vol = -1;

	if (!ustr->vol_scale)
		return;
#ifdef NEW_SCALE	
	adj = GTK_BSCALE(ustr->vol_scale)->adjustment;
#else	
	adj = GTK_RANGE(ustr->vol_scale)->adjustment;
#endif
	int val = (int)GTK_ADJUSTMENT(adj)->value;

	if (val == old_vol)
		return;
	else
		old_vol = val;

	val ? sprintf(str, "Volume: %d%%  ", val) : sprintf(str, "Volume: mute");

	update_rect.x = 0;
	update_rect.y = 16;
	update_rect.width = 82;
	update_rect.height = 16;
	gdk_draw_rectangle(val_ind,
		ustr->drawing_area->style->fg_gc[GTK_STATE_NORMAL],
		true, update_rect.x, update_rect.y, update_rect.width, update_rect.height);
	gdk_draw_string(val_ind, ustr->drawing_area->style->font,
	ustr->drawing_area->style->white_gc, update_rect.x+6, update_rect.y+12, str);
	gtk_widget_draw (ustr->drawing_area, &update_rect);
}

void draw_balance();


void balance_move_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	draw_balance();
}

void draw_balance()
{
	update_struct *ustr = &global_ustr;
	GdkRectangle update_rect;
	char str[60];

	if (global_vol_left < MIN_BAL_TRESH) {
		sprintf(str, "Pan: right %d%%", 100-global_vol_left);
	} else if (global_vol_right < MIN_BAL_TRESH) {
		sprintf(str, "Pan: left %d%%", 100-global_vol_right);
	} else {
		sprintf(str, "Pan: center");
	} 
	update_rect.x = 0;
        update_rect.y = 16;
	update_rect.width = 82; 
	update_rect.height = 18;

        gdk_draw_rectangle(val_ind,
        	ustr->drawing_area->style->fg_gc[GTK_STATE_NORMAL],
                true, update_rect.x, update_rect.y, 
				update_rect.width, update_rect.height);
        gdk_draw_string(val_ind,
        	ustr->drawing_area->style->font,
                ustr->drawing_area->style->white_gc,
                update_rect.x+6, update_rect.y+12,
                str);
        gtk_widget_draw (ustr->drawing_area, &update_rect);
}

void draw_speed();

void speed_move_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	draw_speed();
}

void draw_speed()
{
	update_struct *ustr = &global_ustr;
	GtkAdjustment *adj;
	GdkRectangle update_rect;
	char str[60];
	int speed_val;
	static int old_val = -20000;
#if 1
	adj = GTK_RANGE(ustr->speed_scale)->adjustment;
#else
	adj = GTK_BSCALE(ustr->speed_scale)->adjustment;
#endif	

	speed_val = (int)GTK_ADJUSTMENT(adj)->value;
	if (speed_val == old_val) 
		return;
	old_val = speed_val;	
	if (speed_val < ZERO_PITCH_TRESH && speed_val > -ZERO_PITCH_TRESH)
		sprintf(str, "Speed: paused");
	else
		sprintf(str, "Speed: %d%%  ", (int)GTK_ADJUSTMENT(adj)->value);
	update_rect.x = 0; 
	update_rect.y = 0;
	update_rect.width = 82;
	update_rect.height = 16;

	gdk_draw_rectangle(val_ind,
		ustr->drawing_area->style->fg_gc[GTK_STATE_NORMAL],
		true,
		update_rect.x, update_rect.y,
		update_rect.width,
		update_rect.height);
	gdk_draw_string(val_ind,
			ustr->drawing_area->style->font,
			ustr->drawing_area->style->white_gc,
			update_rect.x+6, update_rect.y+14,
			str);
	gtk_widget_draw (ustr->drawing_area, &update_rect);		
}


void val_release_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	update_struct *ustr = &global_ustr;
	GdkRectangle update_rect;

	update_rect.x = 0;
	update_rect.y = 0;
	update_rect.width = 106;
	update_rect.height = 20;

	gdk_draw_rectangle(val_ind,
		ustr->drawing_area->style->fg_gc[GTK_STATE_NORMAL],
                        true,
                        0, 0,
			ustr->drawing_area->allocation.width-64,
                        20);
	gtk_widget_draw (ustr->drawing_area, &update_rect);
}


void release_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	GtkAdjustment *adj;
	update_struct *ustr = &global_ustr;
	CorePlayer *p = (CorePlayer *)ustr->data;

	adj = GTK_RANGE(widget)->adjustment;
	p->Seek((int)adj->value);
	//global_ind_timeout = gtk_timeout_add(UPDATE_TIMEOUT, indicator_callback, NULL);
	global_update = 1;	
}


void move_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	//static int c=0;
	//printf("move event %d\n", c++);
	indicator_callback(data);
}


void speed_cb(GtkWidget *widget, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;
	float val =  GTK_ADJUSTMENT(widget)->value;
	global_speed = (gint) val;	
	if (val < ZERO_PITCH_TRESH && val > -ZERO_PITCH_TRESH)
		val = 0;
	p->SetSpeed(  val / 100.0 );
	draw_speed();
}

#ifdef FADER
void do_fade(int, int); // HACK!!!!!
#endif

void forward_play_cb(GtkWidget *widget, gpointer data)
{
	GtkAdjustment *adj;


	adj = GTK_RANGE(data)->adjustment;
	gtk_adjustment_set_value(adj, 100.0);
#ifdef FADER
	do_fade(100, 5);
#endif
}


void reverse_play_cb(GtkWidget *widget, gpointer data)
{
        GtkAdjustment *adj;

        adj = GTK_RANGE(data)->adjustment;
        gtk_adjustment_set_value(adj, -100.0);
#ifdef FADER
	do_fade(-100, 5);
#endif
}


void pause_cb(GtkWidget *widget, gpointer data)
{
        GtkAdjustment *adj;
		float new_val;
		
		static float val = 100.0;

		adj = GTK_RANGE(data)->adjustment;

		if (adj->value == 0.0) { // paused
			new_val = val;
		} else {
			new_val = 0.0;
			val = adj->value;
		}

        gtk_adjustment_set_value(adj, new_val);
}


void stop_cb(GtkWidget *widget, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;

	if (p && p->IsPlaying()) {
		global_playing = 0;
		p->Stop();
		clear_buffer();
	}	
}

void eject_cb(GtkWidget *, gpointer);

void play_cb(GtkWidget *widget, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;
	if (p) {
		global_playing = 1;
		eject_cb(widget, data);
	}	
}


void eject_cb(GtkWidget *wdiget, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;
	if (p) {
		gtk_widget_show(play_dialog);
		gdk_window_raise(play_dialog->window);
	}
}	


void volume_cb(GtkWidget *widget, gpointer data)
{
	GtkAdjustment *adj = (GtkAdjustment *)widget;

	global_vol = (int)adj->value;
}


void balance_cb(GtkWidget *widget, gpointer data)
{
        GtkAdjustment *adj = (GtkAdjustment *)widget;

        global_bal = (int)adj->value;
	if (global_bal > MIN_BAL_TRESH && global_bal < MAX_BAL_TRESH) global_bal = BAL_CENTER;
	global_vol_left = global_bal > BAL_CENTER ? (BAL_CENTER << 1) - global_bal : BAL_CENTER;
	global_vol_right = global_bal < BAL_CENTER ? global_bal : BAL_CENTER;
}


//#define SUBSECOND_DISPLAY 

gint indicator_callback(gpointer data)
{
	update_struct *ustr;
	CorePlayer *p;
	GtkAdjustment *adj;
	GdkDrawable *drawable;
	GdkRectangle  update_rect;
	stream_info info;
	char str[60];
	unsigned long slider_val, t_min, t_sec;
	unsigned long c_hsec, secs, c_min, c_sec;
	unsigned long sr;
	static char old_str[60] = "";

	ustr = &global_ustr;
	p = (CorePlayer *)ustr->data;
	drawable = ustr->drawing_area->window;

	adj = GTK_RANGE(ustr->pos_scale)->adjustment;
	adj->lower = 0;
	adj->upper = p->GetFrames() - 32; // HACK!!
	memset(&info, 0, sizeof(stream_info));

	sr = p->GetSampleRate();
	if (p->IsActive()) { 
		int pos;
		pos = global_update ? p->GetPosition() : (int) adj->value;
		slider_val = pos;
		secs = global_update ? 
			p->GetCurrentTime() : p->GetCurrentTime((int) adj->value);
		c_min = secs / 6000;
		c_sec = (secs % 6000) / 100;
#ifdef SUBSECOND_DISPLAY		
		c_hsec = secs % 100;
#endif		
		secs = p->GetCurrentTime(p->GetFrames());
		t_min = secs / 6000;
		t_sec = (secs % 6000) / 100;
		gtk_adjustment_set_value(adj, pos);
		p->GetStreamInfo(&info);
	} else {
		t_min = 0;
		t_sec = 0;
		c_sec = 0;
		c_min = 0;
		c_hsec = 0;
		sprintf(info.title, "No stream");
	}
	if (t_min == 0 && t_sec == 0) {
		sprintf(str, "No time data");
	} else {
#ifdef SUBSECOND_DISPLAY	
		sprintf(str, "%02ld:%02ld.%02d/%02d:%02d", c_min, c_sec, c_hsec, t_min, t_sec);
#else
		sprintf(str, "%02ld:%02ld/%02ld:%02ld", c_min, c_sec, t_min, t_sec);
#endif
	}
	if (strcmp(old_str, str) != 0) {
		strcpy(old_str, str);
		// Painting in pixmap here
		update_rect.x = ustr->drawing_area->allocation.width-INDICATOR_WIDTH;
 		update_rect.y = 16;
		update_rect.width = INDICATOR_WIDTH;
		update_rect.height = 18;	
		gdk_draw_rectangle(val_ind, 
			ustr->drawing_area->style->fg_gc[GTK_STATE_NORMAL],
			true,
			update_rect.x,
			update_rect.y,
			update_rect.width,
			update_rect.height);
	 
		gdk_draw_string(val_ind,
				ustr->drawing_area->style->font,
				ustr->drawing_area->style->white_gc,
				update_rect.x + 2, 
				update_rect.y + 12,
				str);	
		gtk_widget_draw (ustr->drawing_area, &update_rect);
	}
	draw_format(info.stream_type);
	draw_title(info.title);
	draw_speed();
	draw_volume();
	gdk_flush();
	return true;
}


void cd_cb(GtkWidget *widget, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;

	if (p) {
		global_playing = 0;
		p->Stop();
		p->SetFile("CD.cdda");
		p->Start();
		global_playing = 1;
	}
}


void lt_cb(GtkWidget *widget, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;

	if (p) {
		global_playing = 0;
		p->Stop();
		p->SetFile("http://linuxtoday.com/volt/streams/volt_1999-06-28.mp3");
		p->Start();
		global_playing = 1;
	}	
}

#ifdef HAVE_SOCKMON
void sock_cb(GtkWidget *widget, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;

	if (p) {
		global_playing = 0;
		p->Stop();
		p->SetFile("\001\001\001localhost:6001");
		p->Start();
		global_playing = 1;
	}
}
#endif

void playlist_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *win = (GtkWidget *)data;
	int x, y;

	if (global_playlist_show) {
		 gdk_window_get_origin(win->window, &x, &y);
        	if (windows_x_offset >= 0) {
                	x -= windows_x_offset;
                	y -= windows_y_offset;
        	}
        	gtk_widget_hide(win);
		gtk_widget_set_uposition(win, x, y); 
	} else {
		gtk_widget_show(win);
	}
	global_playlist_show = 1 - global_playlist_show;
}


void exit_cb(GtkWidget *widget, gpointer data)
{
	//gtk_timeout_remove(global_ind_timeout);
	gtk_main_quit();	
}

void scopes_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *win = (GtkWidget *)data;
	int x, y;
        if (global_scopes_show) {
                gdk_window_get_origin(win->window, &x, &y);
                if (windows_x_offset >= 0) {
                        x -= windows_x_offset;
                        y -= windows_y_offset;
                }
                gtk_widget_hide(win);
                gtk_widget_set_uposition(win, x, y);
        } else {
                gtk_widget_show(win);
        }
        global_scopes_show = 1 - global_scopes_show;	
}


void effects_cb(GtkWidget *widget, gpointer data)
{
        GtkWidget *win = (GtkWidget *)data;
        int x, y;

        if (global_effects_show) {
                 gdk_window_get_origin(win->window, &x, &y);
                if (windows_x_offset >= 0) {
                        x -= windows_x_offset;
                        y -= windows_y_offset;
                }
                gtk_widget_hide(win);
                gtk_widget_set_uposition(win, x, y);
        } else {
                gtk_widget_show(win);
        }
        global_effects_show = 1 - global_effects_show;
}


void play_file_ok(GtkWidget *widget, gpointer data)
{
	CorePlayer *p = (CorePlayer *)data;

	if (p) {
		global_playing = 0;
		p->Stop();
		p->SetFile(gtk_file_selection_get_filename(GTK_FILE_SELECTION(play_dialog)));
		p->Start();
		global_playing = 1;
	}
	gtk_widget_hide(GTK_WIDGET(play_dialog));
}

void play_file_cancel(GtkWidget *widget, gpointer data)
{
	gint x,y;

	gdk_window_get_root_origin(GTK_WIDGET(data)->window, &x, &y);
        gtk_widget_hide(GTK_WIDGET(data));
	gtk_widget_hide(GTK_WIDGET(data));
        gtk_widget_set_uposition(GTK_WIDGET(data), x, y);
}


gboolean play_file_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gint x, y;

	gdk_window_get_root_origin(widget->window, &x, &y);
	gtk_widget_hide(widget);
	gtk_widget_set_uposition(widget, x, y);

	return TRUE;
}


void init_main_window(CorePlayer *p)
{
	GtkWidget *root_menu;
	GtkWidget *menu_item;
	GtkWidget *main_window;
	//GtkWidget *playlist_window;
	GtkWidget *effects_window;
	GtkWidget *scopes_window;
	GtkWidget *working;
	GtkWidget *speed_scale;
	GtkWidget *pix;
	GtkWidget *val_area;
	GtkStyle *style;
	GdkFont *smallfont;
	GtkAdjustment *adj;

	main_window = create_main_window();
	gtk_window_set_policy(GTK_WINDOW(main_window), false, false, false);
	gtk_window_set_title(GTK_WINDOW(main_window), "AlsaPlayer "VERSION);
	gtk_widget_show(main_window); // Hmmm
	
	
	playlist_window = init_playlist_window();
	gtk_object_set_data(GTK_OBJECT(playlist_window), "target", p);
	effects_window = init_effects_window();	
	scopes_window = init_scopes_window();

	// FIX!
	chdir("/pub/mp3");


	play_dialog = gtk_file_selection_new("Play file");
	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (play_dialog));

	gtk_signal_connect(GTK_OBJECT(
		GTK_FILE_SELECTION(play_dialog)->cancel_button), "clicked",
                GTK_SIGNAL_FUNC(play_file_cancel), play_dialog);
        gtk_signal_connect(GTK_OBJECT(play_dialog), "delete_event",
                GTK_SIGNAL_FUNC(play_file_delete_event), play_dialog);
        gtk_signal_connect(GTK_OBJECT(
		GTK_FILE_SELECTION(play_dialog)->ok_button),
                "clicked", GTK_SIGNAL_FUNC(play_file_ok), p);


	gtk_signal_connect (GTK_OBJECT (main_window), "expose_event",
                      GTK_SIGNAL_FUNC (on_expose_event), NULL);


	speed_scale = get_widget(main_window, "pitch_scale"); 

	smallfont = gdk_font_load("-adobe-helvetica-medium-r-normal--10-*-*-*-*-*-*-*");


	style = gtk_style_new();
	style = gtk_style_copy(gtk_widget_get_style(main_window));
 	gdk_font_unref(style->font);
	style->font = smallfont;
	gdk_font_ref(style->font); 	

#if 0
	working = get_widget(main_window, "eject_button");
	if (working) {
		pix = xpm_label_box(eject_xpm, main_window);
		gtk_widget_show(pix);
		gtk_container_add(GTK_CONTAINER(working), pix);
		gtk_signal_connect(GTK_OBJECT(working), "clicked",
				GTK_SIGNAL_FUNC(eject_cb), p);
		gtk_button_set_relief(GTK_BUTTON(working),
			global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);
	}										
#endif

	working = get_widget(main_window, "stop_button");
	pix = xpm_label_box(stop_xpm, main_window);
	gtk_widget_show(pix);
	gtk_container_add(GTK_CONTAINER(working), pix);
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
			GTK_SIGNAL_FUNC(stop_cb), p);
	gtk_button_set_relief(GTK_BUTTON(working), global_rb ? GTK_RELIEF_NONE :
		GTK_RELIEF_NORMAL);
	
	working = get_widget(main_window, "reverse_button");
	pix = xpm_label_box(r_play_xpm, main_window);
	gtk_widget_show(pix);
	gtk_container_add(GTK_CONTAINER(working), pix);
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
		GTK_SIGNAL_FUNC(reverse_play_cb), speed_scale);
	gtk_button_set_relief(GTK_BUTTON(working), global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);

        working = get_widget(main_window, "forward_button");
        pix = xpm_label_box(f_play_xpm, main_window);
        gtk_widget_show(pix);
        gtk_container_add(GTK_CONTAINER(working), pix); 
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
		GTK_SIGNAL_FUNC(forward_play_cb), speed_scale);
	gtk_button_set_relief(GTK_BUTTON(working), 
		global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);

	working = get_widget(main_window, "pause_button");
        pix = xpm_label_box(pause_xpm, main_window);
        gtk_widget_show(pix);
        gtk_container_add(GTK_CONTAINER(working), pix);
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
		GTK_SIGNAL_FUNC(pause_cb), speed_scale);
	gtk_button_set_relief(GTK_BUTTON(working), 
		global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);

	working = get_widget(main_window, "prev_button");
	pix = xpm_label_box(prev_xpm, main_window);
	gtk_widget_show(pix);
	gtk_container_add(GTK_CONTAINER(working), pix);
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
		GTK_SIGNAL_FUNC(playlist_prev), playlist_window);
	gtk_button_set_relief(GTK_BUTTON(working), 
                global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);

	working = get_widget(main_window, "next_button");
	pix = xpm_label_box(next_xpm, main_window);
	gtk_widget_show(pix);
	gtk_container_add(GTK_CONTAINER(working), pix); 
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
		GTK_SIGNAL_FUNC(playlist_next), playlist_window);
	gtk_button_set_relief(GTK_BUTTON(working), 
                global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);

    working = get_widget(main_window, "play_button");
    play_pix = xpm_label_box(play_xpm, main_window);
	gtk_widget_show(play_pix);
	gtk_container_add(GTK_CONTAINER(working), play_pix);
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
		GTK_SIGNAL_FUNC(play_cb), p);
	gtk_button_set_relief(GTK_BUTTON(working), 
                global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);

	working = get_widget(main_window, "vol_scale");
	if (working) {
#ifdef NEW_SCALE	
		adj = GTK_BSCALE(working)->adjustment;
#else
		adj = GTK_RANGE(working)->adjustment;
#endif	
		gtk_adjustment_set_value(adj, 100.0);
		gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
   	                             GTK_SIGNAL_FUNC(volume_cb), NULL);
	}
	
	val_area = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(val_area), 204, 32);
	gtk_widget_show(val_area);

	global_ustr.vol_scale = working;
    global_ustr.drawing_area = val_area;
    global_ustr.data = NULL;
	if (working) {	
		gtk_signal_connect (GTK_OBJECT (working), "motion_notify_event",
                GTK_SIGNAL_FUNC(volume_move_event), &global_ustr);
        gtk_signal_connect (GTK_OBJECT (working), "button_press_event",
                GTK_SIGNAL_FUNC(volume_move_event), &global_ustr);
	}
	gtk_signal_connect(GTK_OBJECT(val_area), "configure_event",
                (GtkSignalFunc) val_area_configure, NULL);
	
	working = get_widget(main_window,  "bal_scale");
	if (working) {		
		adj = GTK_RANGE(working)->adjustment;
        gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                                GTK_SIGNAL_FUNC(balance_cb), NULL);
        global_ustr.bal_scale = working;
        gtk_signal_connect (GTK_OBJECT (working), "motion_notify_event",
                GTK_SIGNAL_FUNC(balance_move_event), &global_ustr);
        gtk_signal_connect (GTK_OBJECT (working), "button_press_event",
                GTK_SIGNAL_FUNC(balance_move_event), &global_ustr);
	}

	working = get_widget(main_window, "playlist_button");
	pix = xpm_label_box(playlist_xpm, main_window);
        gtk_widget_show(pix);
        gtk_container_add(GTK_CONTAINER(working), pix);
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
				GTK_SIGNAL_FUNC(playlist_cb), playlist_window); 
	gtk_button_set_relief(GTK_BUTTON(working), 
                global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);
#if 0
	working = get_widget(main_window, "scope_button");
	if (working) {
		GtkWidget *label = gtk_label_new("alsaplayer");
		style = gtk_style_copy(gtk_widget_get_style(label));
		gdk_font_unref(style->font);
		style->font = smallfont;
   	     gdk_font_ref(style->font);
		gtk_widget_set_style(GTK_WIDGET(label), style);
		gtk_container_add(GTK_CONTAINER(working), label);
		gtk_widget_show(label);
		gtk_button_set_relief(GTK_BUTTON(working),
			global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_HALF);
	}
#endif

	working = get_widget(main_window, "cd_button");
	pix = xpm_label_box(cd_xpm, main_window);
	gtk_widget_show(pix);
	gtk_container_add(GTK_CONTAINER(working), pix);
	//gtk_signal_connect(GTK_OBJECT(working), "clicked",
	//			GTK_SIGNAL_FUNC(cd_cb), p);
	gtk_button_set_relief(GTK_BUTTON(working),
		global_rb ? GTK_RELIEF_NONE : GTK_RELIEF_NORMAL);
	
	working = get_widget(main_window, "info_box"); 
	
	gtk_widget_set_style(val_area, style);

	gtk_box_pack_start (GTK_BOX (working), val_area, true, true, 0);
	
	global_ustr.data = p;
	working = get_widget(main_window, "pos_scale");
	global_ustr.pos_scale = working;
	//global_ind_timeout = gtk_timeout_add(UPDATE_TIMEOUT, indicator_callback, &global_ustr);

	working = get_widget(main_window, "pos_scale");
	gtk_signal_connect(GTK_OBJECT(working), "button_release_event",
		 GTK_SIGNAL_FUNC(release_event), &global_ustr);
	gtk_signal_connect (GTK_OBJECT (working), "button_press_event",
                 GTK_SIGNAL_FUNC(press_event), NULL);
	gtk_signal_connect (GTK_OBJECT (working), "motion_notify_event",
		 GTK_SIGNAL_FUNC(move_event), &global_ustr);


	global_ustr.speed_scale = speed_scale;
#if 1
	gtk_signal_connect (GTK_OBJECT (speed_scale), "motion_notify_event",
		GTK_SIGNAL_FUNC(speed_move_event), &global_ustr);
	gtk_signal_connect (GTK_OBJECT (speed_scale), "button_press_event",
		GTK_SIGNAL_FUNC(speed_move_event), &global_ustr);
	
	adj = GTK_RANGE(speed_scale)->adjustment;
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                                GTK_SIGNAL_FUNC(speed_cb), p);
	gtk_adjustment_set_value(adj, 100.0);
#endif
	gtk_signal_connect(GTK_OBJECT(main_window), "delete_event", GTK_SIGNAL_FUNC(main_window_delete), p);

	// Create root menu
	root_menu = gtk_menu_new();
	
	// Preferences
#if 0
	menu_item = gtk_menu_item_new_with_label("Preferences...");
	gtk_menu_append(GTK_MENU(root_menu), menu_item);
	gtk_widget_show(menu_item);
#endif
	// Scopes
	menu_item = gtk_menu_item_new_with_label("Scopes...");
        gtk_menu_append(GTK_MENU(root_menu), menu_item);
	gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		GTK_SIGNAL_FUNC(scopes_cb), scopes_window);
        gtk_widget_show(menu_item);
	// Effects
	menu_item = gtk_menu_item_new_with_label("Effects...");
        gtk_menu_append(GTK_MENU(root_menu), menu_item);
	gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		GTK_SIGNAL_FUNC(effects_cb), effects_window);
        gtk_widget_show(menu_item);
	gtk_widget_set_sensitive(menu_item, false);

	// About
	menu_item = gtk_menu_item_new_with_label("About...");
	gtk_menu_append(GTK_MENU(root_menu), menu_item);
	gtk_widget_show(menu_item);
	gtk_widget_set_sensitive(menu_item, false);
   
#if 1	
	// Separator
	menu_item = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(root_menu), menu_item);
	gtk_widget_show(menu_item);
				
    // CD playback
	menu_item = gtk_menu_item_new_with_label("Play CD (using CDDA)");
	gtk_widget_show(menu_item);
	gtk_menu_append(GTK_MENU(root_menu), menu_item);
	gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		GTK_SIGNAL_FUNC(cd_cb), p);
#endif

	
	// Separator
	menu_item = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(root_menu), menu_item);
	gtk_widget_show(menu_item);
	// Exit
	menu_item = gtk_menu_item_new_with_label("Exit");
	gtk_menu_append(GTK_MENU(root_menu), menu_item);
	gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		GTK_SIGNAL_FUNC(exit_cb), NULL);
	gtk_widget_show(menu_item);
#if 0	
	// Connect popup menu
	working = get_widget(main_window, "scope_button");
	if (working) {
		gtk_signal_connect_object (GTK_OBJECT (working), "event",
			GTK_SIGNAL_FUNC(alsaplayer_button_press), GTK_OBJECT(root_menu));
	}
	// Create CD button menu
	GtkWidget *cd_menu = gtk_menu_new();	
	menu_item = gtk_menu_item_new_with_label("Play CD (using CDDA)");
	gtk_widget_show(menu_item);
	gtk_menu_append(GTK_MENU(cd_menu), menu_item);
	gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		GTK_SIGNAL_FUNC(cd_cb), p);
#ifdef HAVE_SOCKMON
	menu_item = gtk_menu_item_new_with_label("Monitor socket (experimental)");
	gtk_widget_show(menu_item);
	gtk_menu_append(GTK_MENU(cd_menu), menu_item);
	gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		GTK_SIGNAL_FUNC(sock_cb), p);
#endif
#endif
	working = get_widget(main_window, "cd_button");
	gtk_signal_connect_object (GTK_OBJECT (working), "event",
		GTK_SIGNAL_FUNC(alsaplayer_button_press), GTK_OBJECT(root_menu)); // cd

	gdk_flush();
}


gint alsaplayer_button_press(GtkWidget *widget, GdkEvent *event)
{
	if (event->type == GDK_BUTTON_PRESS) {
		GdkEventButton *bevent = (GdkEventButton *) event; 
		gtk_menu_popup (GTK_MENU (widget), NULL, NULL, NULL, NULL,
			bevent->button, bevent->time);
		return true;
	}
	return false;
}


void quit_handler(int sig)
{
	printf("Got signal %d\n", sig);
}


extern input_plugin mpg123_plugin;
extern input_plugin aplay_plugin;
extern input_plugin	cdda_plugin;
extern int init_reverb();
extern bool reverb_func(void *arg, void *data, int size);

static char *copyright_string = PACKAGE" "VERSION", (C) 1999 Andy Lo A Foe <andy@alsa-project.org>";

static void help()
{
	fprintf(stdout,
"%s\n"
"\n"
"Usage: alsaplayer [options] [--nogui filename <filename> ...]\n"
"\n"	
"Available options:\n"
"\n"
"  -h,--help                   print this help message\n"
"  -f,--fragsize #             fragment size in bytes [default=4096]\n"
"  -g,--fragcount #            fragment count [default=8]\n"
"  -c,--card #,#               use card number [default=0,0]\n"
"  -n,--nogui                  start in command line mode\n"
"  -q,--quiet                  quiet operation. no output\n"
"  -v,--version                print version of this program\n"
"\n"
"Testing options:\n"
"\n"
"  -r,--reverb                 use reverb function (CPU intensive!)\n"
"  -o,--output [alsa|oss|esd]  Use ALSA, OSS or ESD for output\n"
"\n", copyright_string
	);
}

static void version()
{
	fprintf(stdout,
"%s version %s\n\n", PACKAGE, VERSION); 	
}


int main(int argc, char **argv)
{
	int use_card = 0;
	int use_device = 0;
	int use_fragsize = 4096;
	int use_fragcount = 8;
	int use_reverb = 0; // TEMP!
	int be_quiet = 0;
	int use_gui = 1;
	int use_alsa = 1;
	int use_oss = 0;
	int use_sgi = 0;
	int use_esd = 0;
	int use_sparc = 0;
	int use_user = 0;
	int last_arg = 0;
	int arg_pos = 0;
	CorePlayer *p;
	char path[256], *home;

#ifdef FUNKY_STUFF
	// Do funky stuff
	display = XOpenDisplay(NULL);
	if (display)
		root = RootWindow(display, DefaultScreen(display));
#endif
	oldX = -1; oldY = -1;

	g_thread_init(NULL);
	if (!g_thread_supported()) {
		fprintf(stderr, "Woops! This program requiers working threads.\n");
		return 1;
	}

	init_effects();

	for (arg_pos=1; arg_pos < argc; arg_pos++) {
		if (strcmp(argv[arg_pos], "--help") == 0 ||
			strcmp(argv[arg_pos], "-h") == 0) {
			help();
			return 0;
		} else if (strcmp(argv[arg_pos], "--nogui") == 0 ||
				strcmp(argv[arg_pos], "-n") == 0) {
			use_gui = 0;
			last_arg = arg_pos;
		} else if (strcmp(argv[arg_pos], "--version") == 0 ||
				   strcmp(argv[arg_pos], "-v") == 0) {
			version();
			return 0;
		} else if (strcmp(argv[arg_pos], "--reverb") == 0 ||
				    strcmp(argv[arg_pos], "-r") == 0) {
			use_reverb = 1;
			last_arg = arg_pos;
		} else if (strcmp(argv[arg_pos], "--output") == 0 ||
					strcmp(argv[arg_pos], "-o") == 0) {
			//printf("Use define output method!\n");		
			if (argc < arg_pos+1) {
				fprintf(stderr, "argument expected for %s\n",
					argv[arg_pos]);
				return 1;
			}
			use_alsa = use_sparc = use_esd = use_oss = 0;
			if (strcmp(argv[arg_pos+1], "alsa") == 0) {
				//printf("using ALSA output method\n");
				use_alsa = 1;
				last_arg = ++arg_pos;
			} else if (strcmp(argv[arg_pos+1], "sgi") == 0) {
				use_sgi = 1;
				last_arg = ++arg_pos;
			} else if (strcmp(argv[arg_pos+1], "oss") == 0) {
				//printf("Using OSS output method\n");
				use_oss = 1;
				last_arg = ++arg_pos;
			} else if (strcmp(argv[arg_pos+1], "esd") == 0) {
				//printf("Using ESD output method\n");
				use_esd = 1;
				last_arg = ++arg_pos;
			} else if (strcmp(argv[arg_pos+1], "sparc") == 0) {
				use_sparc = 1;
				last_arg = ++arg_pos;
			} else {
				fprintf(stderr, "Unsupported output module: %s\n",
					argv[arg_pos+1]);
				return 1;
			}
			use_user = 1;
		} else if (strcmp(argv[arg_pos], "--quiet") == 0 ||
				strcmp(argv[arg_pos], "-q") == 0) {
			be_quiet = 1;
			last_arg = arg_pos;
		} else if (strcmp(argv[arg_pos], "--fragsize") == 0 ||
				   strcmp(argv[arg_pos], "-f") == 0) {
			if (sscanf(argv[++arg_pos], "%d", &use_fragsize) != 1) {
				fprintf(stderr, "numeric argument expected for --fragsize\n");
				return 1;
			}
			if (!use_fragsize || (use_fragsize % 32)) {
				fprintf(stderr, "invalid fragment size, must be multiple of 32\n");
				return 1;
			}
			last_arg = arg_pos;
		} else if (strcmp(argv[arg_pos], "--fragcount") == 0 ||
				   strcmp(argv[arg_pos], "-g") == 0) {
			if (sscanf(argv[++arg_pos], "%d", &use_fragcount) != 1) {
				fprintf(stderr, "numeric argument expected for --fragcount\n");
				return 1;
			}
			if (!use_fragcount) {
				fprintf(stderr, "fragment count cannot be 0\n");
				return 1;
			}
			last_arg = arg_pos;
		} else if (strcmp(argv[arg_pos], "--card") == 0 ||
		           strcmp(argv[arg_pos], "-c") == 0) {
			if (sscanf(argv[++arg_pos], "%d,%d", &use_card, &use_device) != 2) {
				fprintf(stderr, "numeric arguments expected for --card\n");
				return 1;
			}
			last_arg = arg_pos;
		}
	}	

	if (!be_quiet)
		fprintf(stdout, "%s\n", copyright_string);	
	
	AlsaDAC *dac = new AlsaDAC(use_card, use_device);

	if (!dac) {
		printf("Bwah!!\n");
		return 1;
	}

	if (use_user) {
		if (use_alsa) 
			load_output_addons(dac, "alsa");
		else if (use_oss) 
			load_output_addons(dac, "oss");
		else if (use_esd) 
			load_output_addons(dac, "esound");
		else if (use_sparc)
			load_output_addons(dac, "sparc");
		else if (use_sgi)
			load_output_addons(dac, "sgi");
	} else
		load_output_addons(dac);

	if (!dac->ReadyToRun()) {
		printf("alsaplayer: error accessing DAC\n");
		return 1;
	}	
	dac->SetSamplingRate(44100);
	dac->SetStreamBuffers(use_fragsize, use_fragcount);
	
#ifdef USE_EXCEPTIONS
	try {
#endif	
		p = new CorePlayer(dac);
#ifdef USE_EXCEPTIONS		
	}
	catch(char *str) {
		printf("Exception: %s\n", str);
		return 1;
	}
#endif

	load_input_addons(p);

	if (use_gui) {
		// GTK calls start here
		gtk_set_locale();
		gtk_init(&argc, &argv);

		home = getenv("HOME");
		if (home && strlen(home) < 128) {
			sprintf(path, "%s/.aprc", home);
			gtk_rc_parse(path);
		} else {
			sprintf(path, "%s/.gtkrc", home);
			gtk_rc_parse(path);
		}

		gdk_threads_enter();

		init_main_window(p);

		// Scope addons
		// Monoscope and fftscope are compiled in for now
		apRegisterScopePlugin(&monoscope_plugin);
		apRegisterScopePlugin(&fftscope_plugin);
		load_scope_addons();

		// Scope functions
		AlsaSubscriber *scopes = new AlsaSubscriber();
		scopes->Subscribe(dac);
		scopes->EnterStream(scope_feeder_func, p);
		
		// Reverb effect
		AlsaSubscriber *reverb = NULL;
		
		if (use_reverb) {
			init_reverb();
			reverb = new AlsaSubscriber();
			reverb->Subscribe(dac);
			reverb->EnterStream(reverb_func, p);
		}
		// Playlist thread
		pthread_create(&playlist_thread, NULL,		
			(void * (*)(void *))playlist_looper, p);
		// Indicator thread
		pthread_create(&indicator_thread, NULL,
			(void * (*)(void *))indicator_looper, p);
#ifdef DEBUG
		printf("THREAD-%d=main thread\n", getpid());
#endif
		gtk_main();
		gdk_threads_leave();
		// GTK calls end here

		delete scopes;
		if (use_reverb && reverb) delete reverb;
	} else {
		for (int i=last_arg+1; i < argc; i++) {
			unsigned long secs, t_min, t_sec, c_min, c_sec;
			t_min = t_sec = c_min = c_sec = 0;
			if (!be_quiet)
				printf("Playing: %s\n", argv[i]);
			p->Stop();
			p->SetFile(argv[i]);
			p->Start();
			if (!be_quiet) {	
				secs = p->GetCurrentTime(p->GetFrames());
				t_min = secs / 6000;
				t_sec = (secs % 6000) / 100;
			}
			while (p->IsActive() || p->IsPlaying()) {
				if (!be_quiet) {
					secs = p->GetCurrentTime();
					c_min = secs / 6000;
					c_sec = (secs % 6000) / 100;	
					fprintf(stdout, "\r[%02ld:%02ld/%02ld:%02ld]       ",
						c_min, c_sec, t_min, t_sec);
					fflush(stdout);
				}
				dosleep(100000);
			}
			if (!be_quiet)
				fprintf(stdout, "\n");
		}
		if (!be_quiet)
			printf("done playing\n");
	}
	delete p;
	delete dac;
#ifdef FUNKY_STUFF	
	XCloseDisplay(display);
#endif	
	return 0;	
}
