/*
    ras - Redundant Archive System
    Copyright (C) 1998  Nick Cleaton

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Nick Cleaton <nc@dial.pipex.com>
*/
/*

Exactly one of USE_GETOPT_LONG, USE_GETOPT and USE_GETOPT_QD must be 
defined when compiling this file.  In decreasing order of preference ..

  USE_GETOPT_LONG       Use the getopt_long library routine, handles both
                        long and short options correctly.

  USE_GETOPT            Use the getopt library routine, handles short options
                        only.

  USE_GETOPT_QD         Use the quick and dirty implementation of getopt
                        found at the bottom of this file.

  If none are defined we choose one here ... try using getopt_long if
  header is there, otherwise go with QD (most portable).
*/

#ifndef USE_GETOPT_LONG
#ifndef USE_GETOPT
#ifndef USE_GETOPT_QD

#ifdef HAVE_GETOPT_H
#define USE_GETOPT_LONG
#else
#define USE_GETOPT_QD
#endif

#endif
#endif
#endif

#include <stdio.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef USE_GETOPT
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif

#ifdef USE_GETOPT_QD
#include <string.h>
#endif

#include <stdlib.h>
#include "common.h"
#include "utils.h"
#include "segfname.h"
#include "options.h"

#ifdef USE_GETOPT_QD
static int qd_optind = 1;
static char *qd_optarg;
int getopt_qd(int argc, char **argv, char *optchars);
#define my_optind qd_optind
#define my_optarg qd_optarg
#else
#define my_optind optind
#define my_optarg optarg
#endif

/*************************************************************************/

int opt_restore;
int opt_create_sums;
int opt_brief;
int opt_verbose;
int opt_silent;
int opt_offset;
int opt_blocksize;
int opt_no_header;
int opt_no_footer;


void print_help()
{
   puts(
"ras usage: ras [-r] [OPTION...] [SUMFILE...]              restore archive\n"
"           ras -c [OPTION...] [SUMFILE[:SUMSPEC]...]      create sumfiles\n"
"\n"
"A sumspec is an integer in the range 0 to (255 - number of segfiles), \n"
"specifying which of the possible redundancy sums should be generated. If\n"
"the sumspec is omitted it defaults to 0 for the first arg, 1 for the\n"
"second and so on.\n"
"\n"
"Mode of operation:\n"
"   -r,--restore       (the default) Restore any missing segfiles using the\n"
"                         sumfiles specified on the command line\n"
"   -c,--create-sums   Create the sumfiles specified on the command line\n"
"\n"
"Verbosity options:\n"
"   -b,--brief        (the default) 1-line summary of actions to stdout\n"
"   -v,--verbose      Verbosely describe what's happening to stdout\n"
"   -s,--silent       Output nothing to stdout\n"
"\n"
"Misc options:\n"
"   -f,--filename <string>  Use repeatedly to set the names to be used for\n"
"                           segfiles, eg \"-fseg1 -fseg2 -fseg3\" for 3 segs.\n"
"   -n,--segcount <int>     Specify the number of segments in the archive\n"
"   -o,--offset <int>       When generating default sumspecs (see above)\n"
"                           start from <int> instead of 0\n"
"   -z,--blocksize <int>    Set the number of bytes that are read and written\n"
"                           at a time, default 1024\n"
"   -H,--no-header          Don't use 5-byte headers on sumfiles to make\n"
"                           regeneration more convenient\n"
"   -F,--no-footer          Don't use 1-byte footers on sumfiles to allow\n"
"                           segfiles of differing lengths\n"
"   -V,--version            Print version information and exit\n"
"   -h,--help               Print this help and exit\n"
   );
}

/*************************************************************************/

int get_options(int argc, char **argv)
{
#ifdef USE_GETOPT_LONG
   static struct option long_options[] = 
   {
      {"restore", 0, 0, 'r'},
      {"create-sums", 0, 0, 'c'},
      {"brief", 0, 0, 'b'},
      {"verbose", 0, 0, 'v'},
      {"silent", 0, 0, 's'},
      {"filename", 1, 0, 'f'},
      {"segcount", 1, 0, 'n'},
      {"offset", 1, 0, 'o'},
      {"blocksize", 1, 0, 'z'},
      {"no-header", 0, 0, 'H'},
      {"no-footer", 0, 0, 'F'},
      {"version", 0, 0, 'V'},
      {"help", 0, 0, 'h'},
      {0, 0, 0, 0}
   };
#endif

   /*  Set up default values  */
   opt_restore = 1;
   opt_create_sums = 0;
   opt_brief = 1;
   opt_verbose = 0;
   opt_silent = 0;
   opt_offset = 0;
   opt_blocksize = 1024;
   opt_no_header = 0;
   opt_no_footer = 0;

   while (1)
   {
      int c;

#ifdef USE_GETOPT_LONG
      int option_index = 0;
      c = getopt_long(argc, argv, "rcbvsf:n:o:z:HFVh", long_options, &option_index);
#endif
#ifdef USE_GETOPT
      c = getopt(argc, argv, "rcbvsf:n:o:z:HFVh");
#endif
#ifdef USE_GETOPT_QD
      c = getopt_qd(argc, argv, "rcbvsf:n:o:z:HFVh");
#endif
   
      if( c == -1 ) break;
   
      switch( c )
      {  
         case 'r':
            opt_restore = 1;
            opt_create_sums = 0;
            break;
         case 'c':
            opt_create_sums = 1;
            opt_restore = 0;
            break;
         case 'b':
            opt_brief = 1;
            opt_verbose = 0;
            opt_silent = 0;
            break;
         case 'v':
            opt_verbose = 1;
            opt_brief = 0;
            opt_silent = 0;
            break;
         case 's':
            opt_silent = 1;
            opt_brief = 0;
            opt_verbose = 0;
            break;
         case 'f':
            change_segment_filenames(my_optarg);
            break;
         case 'n':
            register_segment_count(atoi(my_optarg), "command line option");
            break;
         case 'o':
            opt_offset = atoi(my_optarg);
            break;
         case 'z':
            opt_blocksize = atoi(my_optarg);
            break;
         case 'H':
            opt_no_header = 1;
            break;
         case 'F':
            opt_no_footer = 1;
            break;
         case 'V':
            puts(PROGNAME " version " PROGVERSION "\n");  exit(0);
            break;
         case 'h':
            print_help();
            exit(0);
            break;
	 case '?':
	    /* option error, getopt_long has already printed message */
#ifndef USE_GETOPT_LONG
           fputs("Invalid option", stderr);
#endif
	    exit(1);
	 default:
	    fputs("internal failure in get_options", stderr);
	    exit(1);
      }
   }

   return my_optind;
}

/*************************************************************************/



#ifdef USE_GETOPT_QD

static char *qd_optbuf = "";


int qd_seekchar(char c, char *str)
{
   while( *str && (*str != c) )
      str++;

   if( *str )
      return (str[1] == ':') ? 2 : 1;

   return 0;
}


int getopt_qd(int argc, char **argv, char *optchars)
{
   int c;

   if( *qd_optbuf == '\0' )
   {  if( qd_optind >= argc || argv[qd_optind][0] != '-' )
	 return -1;
      if( !strcmp(argv[qd_optind], "--") )
      {  qd_optind++;
         return -1;
      }
      qd_optbuf = argv[qd_optind++] + 1;
   }

   c = *qd_optbuf++;
   switch (qd_seekchar(c,optchars))
   {  case 0:
         return '?';
      case 1:
         return c;
      case 2:
         if( *qd_optbuf )
	 {  qd_optarg = qd_optbuf;
	    qd_optbuf = "";
	 }
	 else
	 {  if( qd_optind >= argc )
	       return -1;
	    else
	       qd_optarg = argv[qd_optind++];
	 }
	 return c;
   }

   return -1;  /*  control should never get here  */
}

#endif  /*  end USE_GETOPT_QD  */
      


