/**************************************************************************** 
** File: icmp.c
**
** Author: Mike Borella
**
** Comments: Dump ICMP header information
**
*****************************************************************************/

#include "config.h"
#include <stdio.h>
#include "ip_icmp.h"

/* rfc1191 */
struct mtu_discovery 
{
  short unused;
  short nexthopmtu;
};

struct tok 
{
  int v;                  /* value */
  char *s;                /* string */
};

static struct tok icmp2str[] = 
{
  { ICMP_ECHOREPLY,               "echo reply" },
  { ICMP_SOURCEQUENCH,            "source quench" },
  { ICMP_ECHO,                    "echo request" },
  { ICMP_ROUTERSOLICIT,           "router solicitation" },
  { ICMP_TSTAMP,                  "time stamp request" },
  { ICMP_TSTAMPREPLY,             "time stamp reply" },
  { ICMP_IREQ,                    "information request" },
  { ICMP_IREQREPLY,               "information reply" },
  { ICMP_MASKREQ,                 "address mask request" },
  { 0,                            NULL }
};

extern u_char *packet_end;

/*----------------------------------------------------------------------------
**
** dump_icmp()
**
** Parse ICMP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_icmp(u_char *bp, u_char *bp2)
{
  char *cp;
  struct icmp *dp;
  struct ip *ip;
  char *str, *fmt;
  struct ip *oip;
  struct udphdr *ouh;
  u_int hlen, dport, mtu;
  char buf[256];

  dp = (struct icmp *)bp;
  ip = (struct ip *) bp2;
  str = buf;

  printf("----------------------------------------------------------\n");
  printf("                        ICMP Header\n");
  printf("----------------------------------------------------------\n");
  printf("Type:                   %d ", dp->icmp_type);
  
  switch (dp->icmp_type) 
    {
    case ICMP_UNREACH:
      printf("(unreachable)\n");
      printf("Code:                   %d ", dp->icmp_code);

      switch (dp->icmp_code) 
	{
	  
	case ICMP_UNREACH_PROTOCOL:
	  printf("(protocol)\n");
	  printf("Destination IP:         %s\n");
	  break;

	case ICMP_UNREACH_PORT:
	  printf("(port)\n");
	  printf("Destination IP:         %s\n");
	  
	  oip = &dp->icmp_ip;
	  hlen = oip->ip_hl * 4;
	  ouh = (struct udphdr *)(((u_char *)oip) + hlen);
	  dport = ntohs(ouh->dest);
	  switch (oip->ip_p) 
	    {

	    case IPPROTO_TCP:
	      printf("tcp port %s unreachable\n", tcpport_string(dport));
	      break;
	      
	    case IPPROTO_UDP:
	      printf("udp port %s unreachable\n", udpport_string(dport));
	      break;
	      
	    default:
	      printf("Unknown protocol\n");
	      break;
	    }
	  break;
	  
	case ICMP_UNREACH_NEEDFRAG:
	  {
	    struct mtu_discovery *mp;
	    
	    mp = (struct mtu_discovery *)&dp->icmp_void;
	    mtu = EXTRACT_16BITS(&mp->nexthopmtu);

	    printf("(fragmentation needed)\n");
	    printf("Destination IP:         %s\n");
	    
	    if (mtu) printf("MTU = %d\n", mtu);
	  }
	break;
	
	default:
	  printf("(unknown)\n");
	  break;
	}
      
      break;
            
    case ICMP_TIMXCEED:
      switch (dp->icmp_code) 
	{
	case ICMP_TIMXCEED_INTRANS:
	  printf("(time exceeded in transit)\n");
	  break;

	case ICMP_TIMXCEED_REASS:
	  printf("(time exceeded for IP reassembly)\n");
	  break;

	default:
	  printf("Time exceeded-#%d", dp->icmp_code);
	  break;
	}
      break;

    default:
      str = tok2str(icmp2str, "type-#%d", dp->icmp_type);
      break;
    }
  
  printf("icmp: %s", str);
  return;
  
}

  
/*
 * Convert a token value to a string; use "fmt" if not found.
 */
const char *tok2str(register const struct tok *lp, register const char *fmt,
		     register int v)
{
  static char buf[128];
  
  while (lp->s != NULL) 
    {
      if (lp->v == v) return (lp->s);
      ++lp;
    }
  
  if (fmt == NULL) fmt = "#%d";
  sprintf(buf, fmt, v);
  
  return (buf);
}
