/*  main.c */


/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>
#ifdef sgi
#  include <sys/sysmp.h>
#endif
#include "api.h"
#include "gui.h"
#include "vis5d.h"
#include "script.h"



#ifndef MBS
#  define MBS 0
#endif
#ifndef TOPOFILE
#  define TOPOFILE "EARTH.TOPO"
#endif
#ifndef WORLDFILE
#  define WORLDFILE "OUTLSUPW"
#endif
#ifndef USAFILE
#  define USAFILE "OUTLUSAM"
#endif


#define MAX(A,B)  ( (A) > (B) ? (A) : (B) )



/*
 * Print list of options.
 */
static void usage( void )
{
#define P printf
   P("Vis5D version 4.3 Beta  (MBS=%d)\n", MBS);
   P("Usage:\n");
   P("   vis5d file.v5d [options]\n");
   P("Options:\n");
#if defined(SGI_GL) || defined(DENALI) || defined(OPENGL)
   P("   -alpha\n");
   P("      Use alpha blending instead of screen door transparency.\n");
#endif
#ifdef MCIDAS
   P("   -area N\n");
   P("      Specifies the first McIDAS AREA file in a sequence to\n");
   P("      display in the 3-D box.\n");
#endif
   P("   -barbs\n");
   P("      Display wind barbs rather than vectors.\n");
   P("   -box alon alat ahgt\n");
   P("      Set aspect ratio of the 3-D box (longitude:latitude:height).\n");
   P("      Example:  vis5d LAMPS.v5d -box 4 2 1\n");
   P("   -date\n");
   P("      Display date as dd month yy rather than yyddd on clock.\n");
#if defined(SGI_GL) || defined(DENALI)
   P("   -font fontname size\n");
   P("      Set the IRIS GL font and its height used in the viewing window.\n");
   P("      Use the 'listfonts' program to list your system's font names.\n");
   P("      Size of 72 equals one inch (2.54 cm).\n");
   P("      Example:  vis5d LAMPS.v5d -font Helvetica 30\n");
#endif
#if defined(VOGL) || defined(OPENGL)
   P("   -font fontname\n");
   P("      Set the X font used in the viewing window.  Use xlsfonts(1)\n");
   P("      to list your system's fonts.\n");
   P("      Example:  vis5d LAMPS.v5d -font fg-30\n");
#endif
   P("   -full\n");
   P("      Full-screen window; make the 3-D window as large as possible.\n");
   P("   -funcpath pathname\n");
   P("      Specify directory to search for user Fortran functions.\n");
   P("      Example:  vis5d LAMPS.v5d -funcpath /usr/local/vis5d/userfuncs\n");
   P("   -geometry WIDTHxHEIGHT+X+Y  (or WIDTHxHEIGHT or +X+Y)\n");
   P("      Specify the geometry of the 3-D window\n");
   P("   -hirestopo\n");
   P("      Display high resolution topography.  Recommended only for\n");
   P("      fast graphics systems.\n");
   P("   -legend position size\n");
   P("      Set color legend position (1=BOT,2=TOP,3=LFT,4=RT) and size.\n");
   P("   -log [log_scale] [log_exponent]\n");
   P("      Specify a logrithmic vertical scale.\n");
   P("      Default log_scale = %f\n", DEFAULT_LOG_SCALE );
   P("      Default log_exponent = %f\n", DEFAULT_LOG_EXP );
   P("   -map mapfile\n");
   P("      Use a different map outline file instead of OUTLSUPW\n");
   P("      Example:  vis5d LAMPS.v5d -map OUTLUSAL\n");
   P("   -mbs n\n");
   P("      Limit the memory used by vis5d to 'n' megabytes.  When\n");
   P("      the limit is exceeded, the least-recently-viewed graphics\n");
   P("      are deallocated.\n");
   P("   -path pathname\n");
   P("      Specify directory to search for map and topograpy files.\n");
   P("      Example:  vis5d LAMPS.v5d -path /usr3/data\n");
   P("   -projection p\n");
   P("      Set the map projection.  Default is obtained from the datafile.\n");
   P("      p may be one of:\n");
   P("          generic     - rectilinear grid in generic units\n");
   P("          linear      - rectilinear lat/lon grid (cylindrical equidistant)\n");
   P("          lambert     - Lambert Conformal or Polar Stereographic\n");
   P("          stereo      - Azimuthal Stereographic\n");
   P("          rotated     - rotated rectilinear lat/lon grid\n");
   P("          cylindrical - rectilinear lat/lon grid drawn as a cylinder\n");
   P("          spherical   - rectilinear lat/lon grid drawn as a sphere\n");
   P("      Only the first 3 characters of p are needed.\n");
   P("      You will be prompted for additional parameters.\n");
   P("      Example:   vis5d LAMPS.v5d -projection spherical\n");
   P("   -quickstart\n");
   P("      Don't load any data when starting up vis5d, even if the whole\n");
   P("      file will fit into memory.  Useful when reading files by\n");
   P("      NFS.\n");
   P("   -rate ms\n");
   P("      Set the animation rate.  ms is the miniumum time delay in\n");
   P("      milli-seconds between frames.  Default is 100.\n");
   P("   -samescale\n");
   P("      Set the scale for the vertical plot variables to be the same\n");
   P("      for all three variables\n");
   P("   -script filename\n");
   P("      Execute a Tcl script for controlling Vis5D.\n");
#if defined(SGI_GL) || defined(DENALI) || defined(OPENGL)
   P("   -sequence filename\n");
   P("      Specify a sequence of images to texture map over the topography.\n");
   P("   -texture image.rgb\n");
   P("      Specify an image to texture map over the topography.\n");
#endif
   P("   -topo topfile\n");
   P("      Use a different topography file instead of EARTH.TOPO.\n");
   P("      Example:  vis5d LAMPS.v5d -topo MARS.TOPO\n");
   P("   -trajvars uvar vvar [wvar]\n");
   P("      Specify which variables to use for tracing trajectories\n");
   P("      Example:  vis5d dataset -trajvars U-REL V-REL WREL\n");
   P("   -vertical v\n");
   P("      Define the vertical coordinate system.  Default is from datafile.\n");
   P("      v may be one of:\n");
   P("          generic    - equally spaced levels in generic units\n");
   P("          equal      - equally spaced levels in km\n");
   P("          unequal    - unequally spaced levels in km\n");
   P("      Only the first 3 characters of v are needed.\n");
   P("      You will be prompted for additional parameters.\n");
   P("      Example:   vis5d LAMPS.v5d -vertical equal\n");
   P("   -verylarge flag (0 or 1)\n");
   P("      Force VeryLarge flag to be set or cleared.  This controls\n");
   P("      whether graphics generation is sync'd with rendering.\n");
   P("      Default is calculated from ratio of file and memory sizes.\n");
   P("   -wdpy xdisplay\n");
   P("      Make widgets appear on a different display.  Useful in\n");
   P("      combination with -full for making slides and videos.\n");
   P("      Example:  vis5d LAMPS.v5d -full -wdpy pluto:0\n");
   P("   -wide w\n");
   P("      Set width of line segments in pixels (default is 1.0).\n");
   P("   -wind2 uvar vvar [wvar]\n");
   P("      Specify which variables to use when computing the HWIND-2\n");
   P("      and VWIND-2 wind vector graphics.\n");
   P("Keyboard functions (when pointing in 3-D window):\n");
   P("   F1 - Raise/lower control panel window.\n");
   P("   F2 - Toggle display of system info.\n");
   P("   P  - Print window image to PostScript printer.\n");
   P("   F  - Faster animation.\n");
   P("   S  - Slower animation.\n");
#undef P
}



/*
 * Reset the timer or return milliseconds elapsed since last reset.
 * Input:  flag - 0 = reset timer
 *                1 = return elapsed time in ms.
 * Return:  elapsed time in ms.
 */
static int timer( int flag )
{
   struct timeval tp;
   static long start_sec, start_usec, lapsed_usec;

   /* Get the current time */
   gettimeofday(&tp, (struct timezone *) 0);

   if (flag) {
      /* Return elapsed time */
      lapsed_usec = ((tp.tv_sec - start_sec) * 1000000) +
                    tp.tv_usec - start_usec;
      return lapsed_usec / 1000;
   }
   else {
      /* Reset timer */
      lapsed_usec = 0;
      start_sec = tp.tv_sec;
      start_usec = tp.tv_usec;
      return 0;
   }
}



/*
 * This is called by main.  It reads user input and updates the 3-D
 * display until the program terminates.
 */
static int main_loop( int index )
{
   int block;
   int numtimes, verylarge;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_numtimes( index, &numtimes );
   verylarge = vis5d_verylarge_mode(index, VIS5D_GET);

   vis5d_signal_redraw(index, 1);

   while (1) {

      /* Do this when not animating: */
      while (gtx->GoTime==0) {
         int redraw, work;

         while (1) {
            /* Determine if we should block/wait or poll for user input */
            vis5d_check_redraw( index, &redraw );
            vis5d_check_work( &work );
            block = work==0 && redraw==0;
            if (get_user_input(block)!=2) {
               break;
            }
         }

         vis5d_do_work();   /* if not in parallel mode, do a queued job */

         vis5d_check_redraw( index, &redraw );
         if (redraw) {
            vis5d_draw_frame( index, gtx->GoTime );
            vis5d_swap_frame( index );
         }
      }

      /* Do this while animating: */
      while (gtx->GoTime!=0) {
         int time;
         vis5d_get_timestep( index, &time );
         if (gtx->GoTime==1) {
            /* Step forward */
            time++;
            if (time >= numtimes) {
               time = 0;
            }
         }
         else {
            /* Step backward */
            time--;
            if (time<0) {
               time = numtimes-1;
            }
         }
         vis5d_set_timestep( index, time );
         vis5d_signal_redraw(index, 1);
         timer(0);  /* reset timer */

         if (verylarge) {
           /* make graphics */
           vis5d_make_timestep_graphics(index, time);
           vis5d_finish_work();
         }

         /* render */
         vis5d_draw_frame( index, gtx->GoTime );
         vis5d_swap_frame( index );

         vis5d_do_work();   /* if not in parallel mode, do a queued job */

         /* until it's time to advance the clock... */
         do {
            /* process all pending user input */
            while (get_user_input( 0 )==2)
              ;
         } while (timer(1) < gtx->AnimRate);
      }

   }

}

/*
 * Create the 2-D Sounding window
 */
static int make_snd_window(int index, char *title, Window scw, char *wdpy_name )
{
   GuiContext gtx = get_gui_gtx(index);
   int xpos, ypos, height, width, scrwidth, scrheight;
   Display *dpy;
   int scr;
   
   dpy = XOpenDisplay( NULL );
   if (!dpy) {
      printf("Unable to open default display\n");
      exit(1);
   }

   scr = DefaultScreen( dpy );

   scrwidth = DisplayWidth( dpy, scr );
   scrheight = DisplayHeight( dpy, scr );

   height = 10;
   width  = 10;
   xpos = 15;
   ypos = 15;

   return vis5d_init_sndwindow( index, "Skew-T and Vertical Plot Displsy",
                                 xpos, ypos, width, height, scw, wdpy_name);
}



/*
 * Create the 3-D rendering window.
 */
static int make_gfx_window( int index, char *title, int geometry,
                            int xpos, int ypos, int width, int height )
{
   int scrwidth, scrheight;
   Display *dpy;
   int scr;

   dpy = XOpenDisplay( NULL );
   if (!dpy) {
      printf("Unable to open default display\n");
      exit(1);
   }

   scr = DefaultScreen( dpy );

   scrwidth = DisplayWidth( dpy, scr );
   scrheight = DisplayHeight( dpy, scr );


   if (geometry==0) {
      /* compute best pos and size */
#if defined(VOGL) || defined(MESA)
      /* use smaller screen w/ VOGL and Mesa for faster rendering */
      height = width = (scrwidth - 420) * 2 / 3;
#else
      height = width = scrwidth - 420;
#endif
      xpos = 410;
      ypos = 10;
   }
   else if (geometry==1) {
      /* just use arguments as is */
      if (xpos==-1 && ypos==-1) {
         xpos = 410;
         ypos = 10;
      }
      if (width==-1 && height==-1) {
         width = height = scrwidth - 420;
      }
      if (xpos<0)  xpos = scrwidth - width + xpos;
      if (ypos<0)  ypos = scrheight - height + ypos;
   }
   else if (geometry==2) {
      /* maximum window size */
      width = scrwidth;
      height = scrheight;
      xpos = ypos = 0;
   }
   else {
      printf("bad value (%d) for geometry in make_gfx_window()\n", geometry );
      exit(1);
   }

   return vis5d_init_window( index, title, xpos, ypos, width, height );
}



/*
 * Get a floating point value from stdin, don't change the value if
 * a blank line is read.
 */
static void input_float( float *fptr )
{
   char str[1000];

   gets( str );
   if (str[0]!=0 && str[0]!='\n') {
      *fptr = atof( str );
   }
}




/*
 * Prompt the user to enter projection parameters for the specifed projection
 * type.
 * Input:  index - the context index
 *         proj - the projection type, one of PROJ_* symbols
 * Output:  the current and then modified projection parameters
 */
static void prompt_for_projection_args( int index, int proj, float projargs[] )
{
   int current_proj;

   /* get default values */
   vis5d_get_projection( index, &current_proj, projargs );

   /* Prompt user for projection parameters now */
   printf("Enter projection parameters, press Return for (default)\n");

   switch (proj) {
      case PROJ_GENERIC:
         printf("Generic projection (no units):\n");
         printf("   North Bound (%g): ", projargs[0] );
         input_float( &projargs[0] );
         printf("   West Bound (%g): ", projargs[1] );
         input_float( &projargs[1] );
         printf("   Row Increment (%g): ", projargs[2] );
         input_float( &projargs[2] );
         printf("   Column Increment (%g): ", projargs[3] );
         input_float( &projargs[3] );
         break;
      case PROJ_LINEAR:
         printf("Cylindrical Equidistant projection:\n");
         printf("   North Bound Latitude (degrees) (%g): ", projargs[0] );
         input_float( &projargs[0] );
         printf("   West Bound Longitude (degrees) (%g): ", projargs[1] );
         input_float( &projargs[1] );
         printf("   Row Increment (degrees) (%g): ", projargs[2] );
         input_float( &projargs[2] );
         printf("   Column Increment (degrees) (%g): ", projargs[3] );
         input_float( &projargs[3] );
         break;
      case PROJ_LAMBERT:
         if (current_proj!=PROJ_LAMBERT) {
            int nr, nc;
            vis5d_get_size( index, &nr, &nc, NULL,NULL,NULL,NULL,NULL,NULL );
            projargs[0] = 70.0;
            projargs[1] = 20.0;
            projargs[2] = -nr;
            projargs[3] = nc/2;
            projargs[4] = 90.0;
            projargs[5] = 100.0;
         }
         printf("Lambert Conformal projection:\n");
         printf("   Standard Latitude 1 (degrees) (%g): ", projargs[0] );
         input_float( &projargs[0] );
         printf("   Standard Latitude 2 (degrees) (%g): ", projargs[1] );
         input_float( &projargs[1] );
         printf("   North/South Pole row (%g): ", projargs[2] );
         input_float( &projargs[2] );
         printf("   North/South Pole column (%g): ", projargs[3] );
         input_float( &projargs[3] );
         printf("   Central longitude (degrees) (%g): ", projargs[4] );
         input_float( &projargs[4] );
         printf("   Column increment (km) (%g): ", projargs[5] );
         input_float( &projargs[5] );
         break;
      case PROJ_STEREO:
         if (current_proj!=PROJ_STEREO) {
            /* default is north pole polar stereographic */
            int nr, nc;
            vis5d_get_size( index, &nr, &nc, NULL,NULL,NULL,NULL,NULL,NULL );
            projargs[0] = 90.0;
            projargs[1] = 0.0;
            projargs[2] = nr/2.0;
            projargs[3] = nc/2.0;
            projargs[4] = 500.0;
         }
         printf("Stereographic:\n");
         printf("  Center Latitude (degrees) (%g): ", projargs[0] );
         input_float( &projargs[0] );
         printf("  Center Longitude (degrees) (%g): ", projargs[1] );
         input_float( &projargs[1] );
         printf("  Center Row (%g): ", projargs[2] );
         input_float( &projargs[2] );
         printf("  Center Column (%g): ", projargs[3] );
         input_float( &projargs[3] );
         printf("  Column spacing (km) (%g): ", projargs[4] );
         input_float( &projargs[4] );
         break;
      case PROJ_ROTATED:
         printf("Rotated Equidistant projection:\n");
         printf("  Latitude on rotated globe of grid row 0 (degrees) (%g): ",
                projargs[0] );
         input_float( &projargs[0] );
         printf("  Longitude on rotated globe of grid column 0 (degrees)");
         printf(" (%g): ", projargs[1] );
         input_float( &projargs[1] );
         printf("   Row Increment on rotated globe (degrees) (%g): ",
                projargs[2] );
         input_float( &projargs[2] );
         printf("   Column Increment on rotated globe (degrees) (%g): ",
                projargs[3] );
         input_float( &projargs[3] );
         printf("  Earth Latitude of (0, 0) on rotated globe (degrees) (%g): ",
                projargs[4] );
         input_float( &projargs[4] );
         printf("  Earth Longitude of (0, 0) on rotated globe (degrees) (%g): ",
                projargs[5] );
         input_float( &projargs[5] );
         printf("   Clockwise rotation of rotated globe (degrees) (%g): ",
                projargs[6] );
         input_float( &projargs[6] );
         break;

      case PROJ_CYLINDRICAL:
      case PROJ_SPHERICAL:
         if (proj==PROJ_CYLINDRICAL) {
            printf("Cylindrical projection of rectangular grid:\n");
         }
         else {
            printf("Spherical projection of rectangular grid:\n");
         }
         printf("   North Bound Latitude (degrees) (%g): ", projargs[0] );
         input_float( &projargs[0] );
         printf("   West Bound Longitude (degrees) (%g): ", projargs[1] );
         input_float( &projargs[1] );
         printf("   Row Increment (degrees) (%g): ", projargs[2] );
         input_float( &projargs[2] );
         printf("   Column Increment (degrees) (%g): ", projargs[3] );
         input_float( &projargs[3] );
         break;
      default:
         /* should never allow a bad projection number to get this far! */
         abort();
   }
}



/*
 * Prompt the user to enter parameters for the given vertical coordinate
 * sytem type.
 * Input:  index - the context index number
 *         vcs - the vertical coordinate system number, one of VERT_*
 * Output:  vertargs - the vcs arguments
 */
static void prompt_for_vertical_args( int index, int vcs, float vertargs[] )
{
   int current_vcs;
   int maxnl, i;
   float pressure;

   vis5d_get_size( index, NULL, NULL, NULL, NULL, &maxnl, NULL, NULL, NULL );

   /* get default values */
   vis5d_get_vertical( index, &current_vcs, vertargs );

   switch (vcs) {
      case VERT_GENERIC:
         printf("Generic linear vertical coordinate system:\n");
         printf("  Bottom Bound (%g): ", vertargs[0] );
         input_float( &vertargs[0] );
         printf("  Level Increment (%g): ", vertargs[1] );
         input_float( &vertargs[1] );
         break;
      case VERT_EQUAL_KM:
         printf("Linear, equally spaced levels in km:\n");
         printf("  Bottom Bound (km) (%g): ", vertargs[0] );
         input_float( &vertargs[0] );
         printf("  Level Increment (km) (%g): ", vertargs[1] );
         input_float( &vertargs[1] );
         break;
      case VERT_NONEQUAL_KM:
         printf("Linear, non-equally spaced levels in km:\n");
         for (i=0;i<maxnl;i++) {
            printf("  Level %d Height (km) (%g): ", i, vertargs[i] );
            input_float( &vertargs[i] );
         }
         break;
      case VERT_NONEQUAL_MB:
         printf("Linear, non-equally spaced levels in mb:\n");
         for (i=0;i<maxnl;i++) {
            printf("  Level %d Pressure (km) (%g): ", i,
                   height_to_pressure(vertargs[i]) );
            input_float( &pressure );
            vertargs[i] = pressure_to_height(pressure);
         }
         break;
      default:
         /* should never get this far with a bad vcs type! */
         abort();
   }
}



main( int argc, char *argv[] )
{
   int i;
   struct rlimit rl;
   int cpus;
   GuiContext gtx;
   char windowname[100];
   char swindowname[100];
   int volflag;
   int index;
   int workers = 0;

   char *v5dfile = NULL;

   /* command line arguments with default values: */
   int first_area = 0;                          /* -area */
   int alpha = 0;                               /* -alpha */
   int barbs = 0;                               /* -barbs */
   float boxx = 0.0, boxy = 0.0, boxz =0.0;     /* -box */
   char *fontname = NULL;                       /* -font */
   int fontsize = 0;
   char *funcpath = NULL;                       /* -funcpath */
   int geometry = 0;                            /* -geometry */
   int width, height, xpos, ypos;
   int hirestopo = 0;                           /* -hirestopo */
   int julian = 1;                              /* -date */
   char mapname[100] = "";                      /* -map */
   int mbs = MBS;                               /* -mbs */
   char *path = NULL;                           /* -path */
   int user_projection = -1;                    /* -projection */
   int preload = 1;                             /* -quick */
   int animrate = ANIMRATE;                     /* -rate */
   char *script = NULL;                         /* -script */
   char *sequence = NULL;                       /* -sequence */
   char *texture = NULL;                        /* -texture */
   char *toponame = TOPOFILE;                   /* -topo */
   char traju[20], trajv[20], trajw[20];        /* -trajvars */
   int user_vertsys = -1;                       /* -vertical */
   int log_vcs = 0;                             /* -log */
   float scale = DEFAULT_LOG_SCALE;
   float exponent = DEFAULT_LOG_EXP;
   int verylarge = -1;                          /* -verylarge (-1=don't care)*/
   char *wdpy_name = NULL;                      /* -wdpy */
   float linewidth = 1.0;                       /* -wide */
   char u2[20], v2[20], w2[20];                 /* -wind2 */
   int samescale = 0;                         
   int legend_position = VIS5D_BOTTOM;
   int legend_size = 128;

   /* these variables help keep track of new variables during scripting */
   int number_of_vars_before, number_of_vars_after; 

   /* If no arguments, print help */
   if (argc==1) {
      usage();
      exit(0);
   }

#ifndef hp
   /* disable core dumps */
   rl.rlim_cur = rl.rlim_max = 0;
   setrlimit( RLIMIT_CORE, &rl );
#endif

#ifdef sgi
   cpus = sysmp( MP_NPROCS );  /* get number of CPUs present */
#  ifdef SINGLE_TASK
   workers = 0;
   vis5d_workers( 0 );  /* for debugging */
#  else
   workers = MAX(1,cpus-1);
#  endif
#else
   /* Non-SGI systems -> no parallelism */
   workers = 0;
#endif

   /* default wind var names */
   strcpy( traju, "U" );   strcpy( trajv, "V" );   strcpy( trajw, "W" );
   strcpy( u2, "U" );      strcpy( v2, "V" );      strcpy( w2, "W" );

   /* Parse command line options */
   for (i=1;i<argc;i++) {
      if (strcmp(argv[i],"-area")==0 && i+1<argc) {
         first_area = atoi(argv[i+1]);
         i++;
      }
#if defined(SGI_GL) || defined(DENALI) || defined(OPENGL) || defined(PEX)
      else if (strcmp(argv[i],"-alpha")==0) {
         /* alpha transparency flag */
         alpha = 1;
      }
#endif
      else if (strcmp(argv[i],"-barbs")==0) {
         /* use wind barbs rather than vectors */
         barbs = 1;
      }
      else if (strcmp(argv[i],"-box")==0 && i+3<argc) {
         boxx = atof(argv[i+1]);
         boxy = atof(argv[i+2]);
         boxz = atof(argv[i+3]);
         i+=3;
      }
#if defined(SGI_GL) || defined(DENALI)
      else if (strcmp(argv[i],"-font")==0 && i+2<argc) {
         fontname = argv[i+1];
         fontsize = atoi( argv[i+2] );
         i+=2;
      }
#endif
#if defined(XFDI) || defined(VOGL) || defined(OPENGL)
      else if (strcmp(argv[i],"-font")==0 && i+1<argc) {
         fontname = argv[i+1];
         i++;
      }
#endif
      else if (strcmp(argv[i],"-full")==0) {
         /* make 3-D window as large as possible */
         geometry = 2;
      }
      else if (strcmp(argv[i],"-funcpath")==0 && i+1<argc) {
         funcpath = argv[i+1];
         i++;
      }
      else if (strcmp(argv[i],"-geometry")==0 && i+1<argc) {
         /* 3-D window geometry */
         geometry = 1;
         if (sscanf( argv[i+1], "%dx%d%d%d", &width, &height,
                    &xpos, &ypos)==4) {
            /* size and position given */
         }
         else if (sscanf( argv[i+1], "%dx%d", &width, &height )==2) {
            /* only size given */
            xpos = ypos = -1;
         }
         else if (sscanf( argv[i+1], "%d%d", &xpos, &ypos )==2) {
            width = height = -1;
         }
         else {
            printf("Bad geometry specificiation: %s\n", argv[i+1] );
         }
         i++;
      }
      else if (strcmp(argv[i],"-hirestopo")==0) {
         hirestopo = 1;
      }
      else if (strcmp(argv[i],"-date")==0) {
         /* use non-Julian date for clock */
         julian = 0;
      }
      else if (strcmp(argv[i],"-map")==0 && i+1<argc) {
         strcpy( mapname, argv[i+1] );
         i++;
      }
      else if (strcmp(argv[i],"-mbs")==0 && i+1<argc) {
         mbs = atoi( argv[i+1] );
         i++;
      }
      else if (strcmp(argv[i],"-samescale")==0){
         samescale = 1;
      }
      else if (strcmp(argv[i],"-legend")==0 && i+2<argc) {
        int pos;
        pos = atoi( argv[i+1] );
        if (pos == 1) legend_position = VIS5D_BOTTOM;
        else if (pos == 2) legend_position = VIS5D_TOP;
        else if (pos == 3) legend_position = VIS5D_LEFT;
        else if (pos == 4) legend_position = VIS5D_RIGHT;
        legend_size = atoi( argv[i+2] );
        i += 2;
      }
      else if (strcmp(argv[i],"-path")==0 && i+1<argc) {
         path = argv[i+1];
         i++;
      }
      else if (strcmp(argv[i],"-projection")==0 && i+1<argc) {
         /* User-specified map projection */
         if (strncmp(argv[i+1],"gen",3)==0) {
            user_projection = PROJ_GENERIC;
         }
         else if (strncmp(argv[i+1],"lin",3)==0) {
            user_projection = PROJ_LINEAR;
         }
         else if (strncmp(argv[i+1],"lam",3)==0) {
            user_projection = PROJ_LAMBERT;
         }
         else if (strncmp(argv[i+1],"ste",3)==0) {
            user_projection = PROJ_STEREO;
         }
         else if (strncmp(argv[i+1],"rot",3)==0) {
            /* rotated equidistant projection */
            user_projection = PROJ_ROTATED;
         }
         else if (strncmp(argv[i+1],"sph",3)==0) {
            user_projection = PROJ_SPHERICAL;
         }
         else if (strncmp(argv[i+1],"cyl",3)==0) {
            /* cylindrical projection */
            user_projection = PROJ_CYLINDRICAL;
         }
         else {
            printf("Bad projection option: %s\n", argv[i+1] );
         }
         i++;
      }
      else if (strncmp(argv[i],"-quick",6)==0) {
         preload = 0;
      }
      else if (strcmp(argv[i],"-rate")==0 && i+1<argc) {
         animrate = atoi( argv[i+1] );
         if (animrate<1)
           animrate = 1;
         i++;
      }
      else if (strcmp(argv[i],"-sequence")==0 && i+1<argc) {
         sequence = argv[i+1];
         i++;
      }
      else if (strcmp(argv[i],"-texture")==0 && i+1<argc) {
         texture = argv[i+1];
         i++;
      }
      else if (strcmp(argv[i],"-topo")==0 && i+1<argc) {
         toponame = argv[i+1];
         i++;
      }
      else if (strcmp(argv[i],"-trajvars")==0) {
         /* wind trajectory variables */
         if (i+3<argc && argv[i+3][0]!='-') {
            strcpy( traju, argv[i+1] );
            strcpy( trajv, argv[i+2] );
            strcpy( trajw, argv[i+3] );
            i += 3;
         }
         else {
            strcpy( traju, argv[i+1] );
            strcpy( trajv, argv[i+2] );
            i += 2;
         }
      }
      else if (strncmp(argv[i],"-vert",5)==0 && i+1<argc) {
         if (strncmp( argv[i+1], "gen", 3)==0) {
            user_vertsys = VERT_GENERIC;
         }
         else if (strncmp( argv[i+1], "equ", 3)==0) {
            user_vertsys = VERT_EQUAL_KM;
         }
         else if (strncmp( argv[i+1], "une", 3)==0) {
            user_vertsys = VERT_NONEQUAL_KM;
         }
         else {
            printf("Bad vertical coordinate option: %s\n", argv[i+1] );
         }
         i++;
      }
      else if (strcmp( argv[i], "-log")==0) {
         /* This is after "-vert" because it must override */
         float x, y;
         if (i+2<argc && (x=atof(argv[i+2])) && (y=atof(argv[i+1]))) {
            exponent = x;
            scale = y;
            i += 2;
         }
         else if (i+1<argc && (x=atof(argv[i+1]))) {
            scale = x;
            i++;
         }
         log_vcs = 1;
      }
      else if (strcmp(argv[i],"-script")==0) {
         script = argv[i+1];
         i++;
      }
      else if (strcmp(argv[i],"-verylarge")==0 && i+1<argc) {
        /* set flag for very large data set */
        verylarge = atoi(argv[i+1]);
        i++;
      }
      else if (strcmp(argv[i],"-wdpy")==0 && i+1<argc) {
         /* widget display */
         wdpy_name = argv[i+1];
         i++;
      }
      else if (strcmp(argv[i],"-wide")==0 && i+1<argc) {
         linewidth = atof( argv[i+1] );
         i++;
      }
      else if (strcmp(argv[i],"-wind2")==0) {
         /* secondary wind variables */
         if (i+3<argc && argv[i+3][0]!='-') {
            strcpy( u2, argv[i+1] );
            strcpy( v2, argv[i+2] );
            strcpy( w2, argv[i+3] );
            i += 3;
         }
         else {
            strcpy( u2, argv[i+1] );
            strcpy( v2, argv[i+2] );
            i += 2;
         }
      }
      else if (argv[i][0]!='-') {
         v5dfile = argv[i];
      }
      else {
         printf("unknown option: %s\n", argv[i] );
      }
   }

   if (!v5dfile) {
      printf("Error: no v5d file specified\n");
      exit(0);
   }

   /* Initialize the Vis5D API */
   vis5d_initialize( 0 /*NO CAVE*/);   /* Do once, independent of contexts */
   vis5d_workers( workers );


   /* Initialize context #1 */
   index = vis5d_alloc_context();
   if (index<0) {
      printf("Error: couldn't allocate Vis5D context!\n");
      vis5d_terminate(1);
      exit(1);
   }

   vis5d_init_begin( index );

   vis5d_init_memory( index, mbs );
   if (first_area) {
      vis5d_init_firstarea( index, first_area );
   }
   vis5d_alpha_mode( index, alpha );
   if (boxx && boxy && boxz) {
      vis5d_init_box( index, boxx, boxy, boxz );
   }
   if (fontname) {
      vis5d_font( index, fontname, fontsize );
   }
   vis5d_init_map( index, mapname );
   if (path) {
      vis5d_init_path( index, path );
   }
   if (sequence) {
      vis5d_init_sequence( index, sequence );
   }
   if (samescale) {
      vis5d_init_samescale( index );
   }
   if (texture) {
      vis5d_init_texture( index, texture );
   }
   vis5d_init_topo( index, toponame, hirestopo );
#ifdef JUNK
   if (traju && trajv) {
      vis5d_init_trajvars( index, traju, trajv, trajw );
   }
#endif
   vis5d_linewidth( index, linewidth );
#ifdef JUNk
   if (u2 || v2) {
      vis5d_init_wind2( index, u2, v2, w2 );
   }
#endif

   if (vis5d_open_gridfile( index, v5dfile, preload )<0) {
      vis5d_terminate(1);
      exit(0);
   }

   if (user_projection>-1) {
      float projargs[100];
      prompt_for_projection_args( index, user_projection, projargs );
      vis5d_init_projection( index, user_projection, projargs );
   }
   if (log_vcs) {
      vis5d_init_log( index, scale, exponent );
   }
   else if (user_vertsys>-1) {
      float vertargs[MAXLEVELS];
      prompt_for_vertical_args( index, user_vertsys, vertargs );
      vis5d_init_vertical( index, user_vertsys, vertargs );
   }

   sprintf( windowname, "Vis5D 3-D Display (%s)", v5dfile );
   make_gfx_window( index, windowname, geometry, xpos, ypos, width, height );


   if (vis5d_init_end( index )<0) {
      printf("Error in vis5d_init_end\n");
      vis5d_terminate(1);
      exit(0);
   }


   /* setup trajectory and wind variables */
   vis5d_set_wind_vars( index, vis5d_find_var(index,"U"),
                               vis5d_find_var(index,"V"),
                               vis5d_find_var(index,"W"),
                               vis5d_find_var(index,u2),
                               vis5d_find_var(index,v2),
                               vis5d_find_var(index,w2),
                               vis5d_find_var(index,traju),
                               vis5d_find_var(index,trajv),
                               vis5d_find_var(index,trajw) );

   vis5d_set_sound_vars( index, vis5d_find_var(index,"T"),
                                vis5d_find_var(index,"TD"),
                                vis5d_find_var(index,"U"),
                                vis5d_find_var(index,"V"),
                                -1, -1, -1 );

   if (verylarge == 1) {
      vis5d_verylarge_mode(index, VIS5D_ON);
   }
   else if (verylarge == 0) {
      vis5d_verylarge_mode(index, VIS5D_OFF);
   }

   if (barbs == 1) vis5d_graphics_mode(index, VIS5D_BARBS, VIS5D_ON);
   else vis5d_graphics_mode(index, VIS5D_BARBS, VIS5D_OFF);

   if (julian == 1) vis5d_graphics_mode(index, VIS5D_JULIAN, VIS5D_ON);
   else vis5d_graphics_mode(index, VIS5D_JULIAN, VIS5D_OFF);

   vis5d_check_volume( index, &volflag );
   gtx = create_gui_context(index);
   gtx->AnimRate = animrate;
   if (funcpath) {
      strcpy(gtx->funcpath, funcpath);
   }
   gtx->SoundCtrlWindow = 1;

   if ( wdpy_name != NULL){
      gtx->othersnddpy = 1;
   }
   else{
      gtx->othersnddpy = 0;
   }

   make_gui( index, v5dfile, wdpy_name, volflag );

   sprintf( swindowname," Skew-T and Vertical Plot Display (%s)", v5dfile );
   make_snd_window( index, swindowname, gtx->SoundCtrlWindow, wdpy_name );
   if (gtx->othersnddpy ){
      vis5d_resize_sounding_window( index, 630, 600,
                                    gtx->cpx, 15);
   }


   /* execute script specified by -script command line option */
   vis5d_get_numvars( index, &number_of_vars_before);
   if (script) {
      execute_script( index, script );
      vis5d_get_numvars( index, &number_of_vars_after);
      make_new_buttons_after_interp(index, number_of_vars_after-
                                             number_of_vars_before);
   }

   vis5d_set_legends(index, legend_position, legend_size);

   main_loop( index );

   vis5d_terminate(1);
   return 0;
}


