/* *INDENT-OFF* */
!/* PSPP - computes sample statistics.
!   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
!   Written by Ben Pfaff <blp@gnu.org>.
!
!   This program is free software; you can redistribute it and/or
!   modify it under the terms of the GNU General Public License as
!   published by the Free Software Foundation; either version 2 of the
!   License, or (at your option) any later version.
!
!   This program is distributed in the hope that it will be useful, but
!   WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
!   General Public License for more details.
!
!   You should have received a copy of the GNU General Public License
!   along with this program; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
!   02111-1307, USA. */
!
!/*
! MEANS procedure
!*/
!

$ means:
$   *tables=custom;
$   +variables=custom;
$   +crossbreak=custom;
$   +format=lab:!labels/nolabels/nocatlabs,
$          name:!names/nonames,
$          val:!values/novalues,
$          fmt:!table/tree;
$   +missing=miss:!table/include/dependent;
$   +cells[cl_]=default,count,sum,mean,stddev,variance,all;
$   +statistics[st_]=anova,linearity,all,none.
/* *INDENT-ON* */

/* Main declarations. */

#include <config.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "avl.h"
/* (headers) */

#undef DEBUGGING
#define DEBUGGING 1
#include "debug-print.h"

/* (decls global) */

#if DEBUGGING
static void debug_print (void);
#endif

/* TABLES: Variable lists for each dimension. */
int n_dim;		/* Number of dimensions. */
int *nv_dim;		/* Number of variables in each dimension. */
variable ***v_dim;	/* Variables in each dimension.  */

/* VARIABLES: List of variables. */
int n_var;
variable **v_var;

/* Parses and executes the T-TEST procedure. */
int
cmd_means (void)
{
  int success = 0;
  
  n_dim = 0;
  nv_dim = NULL;
  v_dim = NULL;
  v_var = NULL;
  
  /* (parser) */

  if (sbc_cells)
    {
      int i;
      for (i = 0; i < CL_count; i++)
	if (cl_a_cells[i])
	  break;
      if (i >= CL_count)
	cl_a_cells[CL_ALL] = 1;
    }
  else
    cl_a_cells[CL_DEFAULT] = 1;
  if (cl_a_cells[CL_DEFAULT] || cl_a_cells[CL_ALL])
    cl_a_cells[CL_MEAN] = cl_a_cells[CL_STDDEV] = cl_a_cells[CL_COUNT] = 1;
  if (cl_a_cells[CL_ALL])
    cl_a_cells[CL_SUM] = cl_a_cells[CL_VARIANCE] = 1;

  if (sbc_statistics)
    {
      if (!st_a_statistics[ST_ANOVA] && !st_a_statistics[ST_LINEARITY])
	st_a_statistics[ST_ANOVA] = 1;
      if (st_a_statistics[ST_ALL])
	st_a_statistics[ST_ANOVA] = st_a_statistics[ST_LINEARITY] = 1;
    }

  if (!sbc_tables)
    {
      msg (SE, _("Missing required subcommand TABLES."));
      goto free;
    }

#if DEBUGGING
  debug_print ();
#endif
  
  success = 1;

free:
  {
    int i;
    
    for (i = 0; i < n_dim; i++)
      free (v_dim[i]);
    free (nv_dim);
    free (v_dim);
    free (v_var);
  }
  
  return success;
}

/* Returns nonzero only if value V is valid as an endpoint for a
   dependent variable in integer mode. */
int
validate_dependent_endpoint (double V)
{
  return V == (int) V && V != LOWEST && V != HIGHEST;
}

/* Parses the TABLES subcommand. */
int
custom_tables (void)
{
  dictionary *dict;
  dictionary temp_dict;
  
  if (!match_id (TABLES) && (token != ID || !is_varname (tokstr))
      && token != ALL)
    return 2;
  match_tok ('=');

  if (sbc_tables || sbc_crossbreak)
    msg (SE, _("TABLES or CROSSBREAK subcommand may not appear more "
	       "than once."));

  if (sbc_variables)
    {
      dict = &temp_dict;
      temp_dict.var = v_var;
      temp_dict.nvar = n_var;
      
      {
	int i;
      
	temp_dict.var_by_name = avl_create (NULL, cmp_variable, NULL);
	for (i = 0; i < temp_dict.nvar; i++)
	  avl_force_insert (temp_dict.var_by_name, temp_dict.var[i]);
      }
    }
  else
    dict = &default_dict;

  do
    {
      int nvl;
      variable **vl;
	
      if (!parse_variables (dict, &vl, &nvl, PV_NO_DUPLICATE | PV_NO_SCRATCH))
	return 0;
      
      n_dim++;
      nv_dim = xrealloc (nv_dim, n_dim * sizeof (int));
      v_dim = xrealloc (v_dim, n_dim * sizeof (variable **));

      nv_dim[n_dim - 1] = nvl;
      v_dim[n_dim - 1] = vl;

      if (sbc_variables)
	{
	  int i;

	  for (i = 0; i < nv_dim[0]; i++)
	    {
	      means_proc *v_inf = &v_dim[0][i]->p.mns;

	      if (v_inf->min == SYSMIS)
		return msg (SE, _("Variable %s specified on TABLES or "
				  "CROSSBREAK, but not specified on "
				  "VARIABLES."),
			    v_dim[0][i]->name);
	      
	      if (n_dim == 1)
		{
		  v_inf->min = (int) v_inf->min;
		  v_inf->max = (int) v_inf->max;
		} else {
		  if (v_inf->min == LOWEST || v_inf->max == HIGHEST)
		    return msg (SE, _("LOWEST and HIGHEST may not be used "
				      "for independent variables (%s)."),
				v_dim[0][i]->name);
		  if (v_inf->min != (int) v_inf->min
		      || v_inf->max != (int) v_inf->max)
		    return msg (SE, _("Independent variables (%s) may not "
				      "have noninteger endpoints in their "
				      "ranges."),
				v_dim[0][i]->name);
		}
	    }
	}
    }
  while (match_tok (BY));

  /* Check for duplicates. */
  {
    int i;
    
    for (i = 0; i < default_dict.nvar; i++)
      default_dict.var[i]->foo = 0;
    for (i = 0; i < dict->nvar; i++)
      if (dict->var[i]->foo++)
	{
	  msg (SE, _("Variable %s is multiply specified on TABLES "
		     "or CROSSBREAK."),
	       dict->var[i]->name);
	  return 0;
	}
  }
  
  if (sbc_variables)
    avl_destroy (temp_dict.var_by_name, NULL);

  return 1;
}

/* Parse CROSSBREAK subcommand. */
int
custom_crossbreak (void)
{
  return custom_tables ();
}

/* Parses the VARIABLES subcommand. */
int
custom_variables (void)
{
  if (sbc_tables)
    return msg (SE, _("VARIABLES must precede TABLES."));

  if (sbc_variables == 1)
    {
      int i;
      
      for (i = 0; i < default_dict.nvar; i++)
	default_dict.var[i]->p.mns.min = SYSMIS;
    }
  
  do
    {
      int orig_n = n_var;
      
      double min, max;
      
      if (!parse_variables (&default_dict, &v_var, &n_var,
			    PV_APPEND | PV_NO_DUPLICATE | PV_NO_SCRATCH))
	return 0;

      force_match ('(');

      /* Lower value. */
      if (token == ID && (streq (tokstr, "LO") || id_match ("LOWEST", tokstr)))
	min = LOWEST;
      else
	{
	  force_num ();
	  min = tokval;
	}
      get_token ();

      match_tok (',');

      /* Higher value. */
      if (token == ID
	  && (streq (tokstr, "HI") || id_match ("HIGHEST", tokstr)))
	max = HIGHEST;
      else
	{
	  force_num ();
	  max = tokval;
	}
      get_token ();

      force_match (')');

      /* Range check. */
      if (max < min)
	return msg (SE, _("Upper value (%g) is less than lower value "
			  "(%g) on VARIABLES subcommand."), max, min);
      
      {
	int i;

	for (i = orig_n; i < n_var; i++)
	  {
	    means_proc *v_inf = &v_var[i]->p.mns;

	    v_inf->min = min;
	    v_inf->max = max;
	  }
      }
    }
  while (token != '/' && token != '.');
  
  return 1;
}

#if DEBUGGING
static void
debug_print (void)
{
  int i;
  
  printf ("MEANS");

  if (sbc_variables)
    {
      int j = 0;
      
      printf (" VARIABLES=");
      for (i = 0; i < default_dict.nvar; i++)
	{
	  variable *v = default_dict.var[i];
	  
	  if (v->p.mns.min == SYSMIS)
	    continue;
	  if (j++)
	    printf (" ");
	  printf ("%s(", v->name);
	  if (v->p.mns.min == LOWEST)
	    printf ("LO");
	  else
	    printf ("%g", v->p.mns.min);
	  printf (",");
	  if (v->p.mns.max == HIGHEST)
	    printf ("HI");
	  else
	    printf ("%g", v->p.mns.max);
	  printf (")");
	}
      printf ("\n");
    }

  printf (" TABLES=");
  for (i = 0; i < n_dim; i++)
    {
      int j;

      if (i)
	printf (" BY");

      for (j = 0; j < nv_dim[i]; j++)
	{
	  if (i || j)
	    printf (" ");
	  printf (v_dim[i][j]->name);
	}
    }
  printf ("\n");
}
#endif /* DEBUGGING */
      
/* Local Variables: */
/* mode:c */
/* End: */
