/*--------------------------------------------------------------------------
  ----- File:        t1trans.c 
  ----- Author:      Rainer Menzner (rmz@neuroinformatik.ruhr-uni-bochum.de)
  ----- Date:        04/22/1998
  ----- Description: This file is part of the t1-library. It contains
                     functions for transforming fonts and setting
		     line-parameters.
  ----- Copyright:   t1lib is copyrighted (c) Rainer Menzner, 1996-1998. 
                     As of version 0.5, t1lib is distributed under the
		     GNU General Public Library Lincense. The
		     conditions can be found in the files LICENSE and
		     LGPL, which should reside in the toplevel
		     directory of the distribution.  Please note that 
		     there are parts of t1lib that are subject to
		     other licenses:
		     The parseAFM-package is copyrighted by Adobe Systems
		     Inc.
		     The type1 rasterizer is copyrighted by IBM and the
		     X11-consortium.
  ----- Warranties:  Of course, there's NO WARRANTY OF ANY KIND :-)
  ----- Credits:     I want to thank IBM and the X11-consortium for making
                     their rasterizer freely available.
		     Also thanks to Piet Tutelaers for his ps2pk, from
		     which I took the rasterizer sources in a format
		     independent from X11.
                     Thanks to all people who make free software living!
--------------------------------------------------------------------------*/
  
#define T1TRANS_C


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>

#include "../type1/ffilest.h"
#include "../type1/types.h"
#include "parseAFM.h" 
#include "../type1/objects.h"
#include "../type1/spaces.h"
#include "../type1/util.h"
#include "../type1/fontfcn.h"
#include "../type1/regions.h"


#include "t1types.h"
#include "t1extern.h"
#include "t1trans.h"
#include "t1base.h"


/* T1_ExtendFont(): Extend the font referenced by FontID by the factor
   extend. This is only allowed if no size dependent data exists.
   Of course, the font must already have been loaded.
   Returns 0 for success and -1 otherwise.
   */
int T1_ExtendFont( int FontID, double extend)
{

  /* First, check for font residing in memory: */
  if (CheckForFontID(FontID)!=1){
    T1_errno=T1ERR_INVALID_FONTID;
    return(-1);
  }
  
  /* Second, check whether size-dependent data exists: */
  if (pFontBase->pFontArray[FontID].pFontSizeDeps != NULL){
    T1_errno=T1ERR_OP_NOT_PERMITTED;
    return(-1); 
  }
  
  pFontBase->pFontArray[FontID].extend=extend;
  pFontBase->pFontArray[FontID].FontTransform[0] = extend;
  return(0);
}



/* T1_SlantFont(): Slant the font referenced by FontID by the factor
   extend. This is only allowed if no size dependent data exists.
   Of course, the font must already have been loaded.
   Returns 0 for success and -1 otherwise.
   */
int T1_SlantFont( int FontID, double slant)
{
  
  /* First, check for font residing in memory: */
  if (CheckForFontID(FontID)!=1){
    T1_errno=T1ERR_INVALID_FONTID;
    return(-1);
  }
  
  /* Second, check whether size-dependent data exists: */
  if (pFontBase->pFontArray[FontID].pFontSizeDeps != NULL){
    T1_errno=T1ERR_OP_NOT_PERMITTED;
    return(-1); 
  }
  
  pFontBase->pFontArray[FontID].slant=slant;
  pFontBase->pFontArray[FontID].FontTransform[2] = slant;
  return(0);
}



/* Functions for setting line-parameters:
   linetypeis expected to be an OR'ed combination of
   T1_UNDERLINE, T1_OVERLINE and T1_OVERSTRIKE.
 */
int T1_SetLinePosition( int FontID, int linetype, float value)
{

  if (CheckForFontID(FontID)!=1){
    T1_errno=T1ERR_INVALID_FONTID;
    return(-1);
  }

  if (linetype & T1_UNDERLINE){
    pFontBase->pFontArray[FontID].UndrLnPos=value;
    return( 0);
  }
  if (linetype & T1_OVERLINE){
    pFontBase->pFontArray[FontID].OvrLnPos=value;
    return( 0);
  }
  if (linetype & T1_OVERSTRIKE){
    pFontBase->pFontArray[FontID].OvrStrkPos=value;
    return( 0);
  }
    
  /* The linetype was bad */
  T1_errno=T1ERR_INVALID_PARAMETER;
  return( -1);
  
}



int T1_SetLineThickness( int FontID, int linetype, float value)
{

  if (CheckForFontID(FontID)!=1){
    T1_errno=T1ERR_INVALID_FONTID;
    return(-1);
  }

  if (linetype & T1_UNDERLINE){
    pFontBase->pFontArray[FontID].UndrLnThick=value;
    return( 0);
  }
  if (linetype & T1_OVERLINE){
    pFontBase->pFontArray[FontID].OvrLnThick=value;
    return( 0);
  }
  if (linetype & T1_OVERSTRIKE){
    pFontBase->pFontArray[FontID].OvrStrkThick=value;
    return( 0);
  }
    
  /* The linetype was bad */
  T1_errno=T1ERR_INVALID_PARAMETER;
  return( -1);
  
}


float T1_GetLinePosition( int FontID, int linetype)
{

  if (CheckForFontID(FontID)!=1){
    T1_errno=T1ERR_INVALID_FONTID;
    return(0.0);
  }

  if (linetype & T1_UNDERLINE)
    return( pFontBase->pFontArray[FontID].UndrLnThick);
  if (linetype & T1_OVERLINE)
    return( pFontBase->pFontArray[FontID].OvrLnThick);
  if (linetype & T1_OVERSTRIKE)
    return( pFontBase->pFontArray[FontID].OvrStrkThick);
  
  /* The linetype was bad */
  T1_errno=T1ERR_INVALID_PARAMETER;
  return( 0.0);
  
}



float T1_GetLineThickness( int FontID, int linetype)
{

  if (CheckForFontID(FontID)!=1){
    T1_errno=T1ERR_INVALID_FONTID;
    return(0.0);
  }

  if (linetype & T1_UNDERLINE)
    return( pFontBase->pFontArray[FontID].UndrLnThick);
  if (linetype & T1_OVERLINE)
    return( pFontBase->pFontArray[FontID].OvrLnThick);
  if (linetype & T1_OVERSTRIKE)
    return( pFontBase->pFontArray[FontID].OvrStrkThick);
  
  /* The linetype was bad */
  T1_errno=T1ERR_INVALID_PARAMETER;
  return( 0.0);
  


}
