/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/*
 * parser_inst.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <constants.h>
#include <defaults.h>
#include <pm.h>

#include "manifold.h"
#include "parser.h"
#include "parser_defs.h"

#include "utilities.h"

static void parser_go(void); 
static void parser_save(void); 
static void parser_writec(void);
static void parser_add_text(void); 
static void parser_init(void);
static void parser_new_ds_check(void);

static char *PARSER_OBJ_NAME = "Parser";

static char *PARSER[] = {
  "Parser.Filename", 
  "Parser.Tempfile", 
  "Parser.Modelname", 
  "Parser.Directory",
  "Parser.Mapping_Flag",
  "Parser.Text",
  "Parser.Text_Max_Length",
  "Parser.Add_Text",
  "Parser.Init",
  "Parser.Save",
  "Parser.Go",
  "Parser.WriteC",
  "Parser.New_DS",
  "Parser.Old_DS",
  "Parser.New_DS_Check",
  "Parser.New_DS_Status"
};

typedef enum {
  FILENAME=0, TEMPFILE, MODELNAME, 
  DIRECTORY, MAPPING_FLAG, TEXT, TEXT_MAX_LENGTH, ADD_TEXT, INITP,
  SAVE, GO, WRITEC, NEW_DS, OLD_DS, NEW_DS_CHECK, NEW_DS_STATUS
} PARSER_t;


/*********************************************************************************
 *
 * PM initialization calls
 *
 *********************************************************************************/

void parser_install()
{
  char dirname[SIZE_OF_DIR];

  pm(CREATE_OBJ, PARSER_OBJ_NAME,
     CREATE_ELEM, PARSER[FILENAME], STRNG,
     CREATE_ELEM, PARSER[TEMPFILE], STRNG,
     CREATE_ELEM, PARSER[MODELNAME], STRNG,
     CREATE_ELEM, PARSER[DIRECTORY], STRNG,
     CREATE_ELEM, PARSER[MAPPING_FLAG], INT,
     CREATE_ELEM, PARSER[TEXT], STRNG,
     CREATE_ELEM, PARSER[TEXT_MAX_LENGTH], INT,
     CREATE_ELEM, PARSER[ADD_TEXT], FNCT,
     CREATE_ELEM, PARSER[INITP], FNCT,
     CREATE_ELEM, PARSER[GO], FNCT,
     CREATE_ELEM, PARSER[SAVE], FNCT,
     CREATE_ELEM, PARSER[WRITEC], FNCT,
     CREATE_ELEM, PARSER[NEW_DS], ADDRS,
     CREATE_ELEM, PARSER[OLD_DS], ADDRS,
     CREATE_ELEM, PARSER[NEW_DS_CHECK], FNCT,
     CREATE_ELEM, PARSER[NEW_DS_STATUS], INT,
     NULL);

  pm(INIT, PARSER[FILENAME], SIZE_OF_FNAME,
     INIT, PARSER[TEMPFILE], SIZE_OF_FNAME,
     INIT, PARSER[MODELNAME], MAX_LONG_STR,
     INIT, PARSER[DIRECTORY], SIZE_OF_DIR,
     INIT, PARSER[TEXT], MAX_LONG_STR,
     INIT, PARSER[TEXT_MAX_LENGTH],
     INIT, PARSER[ADD_TEXT],
     INIT, PARSER[INITP],
     INIT, PARSER[GO],
     INIT, PARSER[SAVE],
     INIT, PARSER[WRITEC],
     INIT, PARSER[NEW_DS],
     INIT, PARSER[OLD_DS],
     INIT, PARSER[NEW_DS_CHECK],
     NULL);

  get_dstool_path(dirname, DSTOOL_DATA_DIR);

  pm(PUT, PARSER[DIRECTORY], dirname,
     PUT, PARSER[FILENAME], "model.def",
     PUT, PARSER[TEMPFILE], "",
     PUT, PARSER[MODELNAME], "parsed_model",
     PUT, PARSER[MAPPING_FLAG], 0,   /* 0 = vf, 1=mapping */
     PUT, PARSER[ADD_TEXT], parser_add_text,
     PUT, PARSER[TEXT_MAX_LENGTH], MAX_LONG_STR,
     PUT, PARSER[INITP], parser_init,
     PUT, PARSER[GO], parser_go,
     PUT, PARSER[SAVE], parser_save,
     PUT, PARSER[WRITEC], parser_writec,
     PUT, PARSER[NEW_DS_CHECK], parser_new_ds_check,
     PUT, PARSER[NEW_DS_STATUS], 0,
     NULL);
}


/*********************************************************************************
 *
 * Go initializes parser object, adds text and builds parsed model. 
 *
 *********************************************************************************/

static void
parser_init()
{
  void* new_ds = NULL;

  new_ds = parser_create_ds();


  /*
  fprintf(stdout,"\nparser_init()");
  if( new_ds == NULL ) {
    fprintf(stdout,"\n\tds object is NULL, init failed");
  }
  else {
    fprintf(stdout,"\n\tds object is not NULL");
    parser_print_ds(stdout,new_ds);
  }
  */

  pm(INIT, PARSER[NEW_DS],
     PUT, PARSER[NEW_DS], new_ds,
     PUT, PARSER[NEW_DS_STATUS], 0,
     NULL);
}

static void
parser_add_text()
{
  int status;
  void* new_ds = NULL;
  char* buffer = (char*)calloc( MAX_LONG_STR, sizeof(char) );

  new_ds = (void* )pm( GET, PARSER[NEW_DS], NULL );
  pm(PUT, PARSER[NEW_DS_STATUS], 0, NULL);

  if( new_ds == NULL ) {
   /*
    fprintf(stdout,"\nds object is NULL, parser_add_text failed");
    */
    return;
  }

  pm( GET, "Parser.Text", buffer, NULL );

 /*
  fprintf(stdout,"\nparser_add_text()");
  fprintf(stdout,"\n\tParser.Text:\n");
  fprintf(stdout,"%s", buffer );
  */

  status = parser_add_string(new_ds, buffer ); 
  free( buffer );

  if( status != PARSER_OK) {
   /*
    fprintf(stdout,"\ncould not parse string, destroying ds");
    */
    parser_destroy_ds(new_ds);
    pm(INIT, PARSER[NEW_DS], NULL);
  }

}

static void
parser_go( void )
{
 /*
  fprintf(stdout,"\nparser_go()");
  */
  pm(PUT, "Model.Load_Number", -1, NULL); /* -1 = parsed dynamical system */
  pm(EXEC, "Model.Load", NULL );
}

/*
static void
parser_go_old( void )
{ */
 /*
  fprintf(stdout,"\nparser_go()");
  */

  /* check that new dynamical system is valid */
/*
  pm(EXEC, PARSER[NEW_DS_CHECK], NULL);

  if ( *( (int *) pm(GET, PARSER[NEW_DS_STATUS], NULL)) ) {
  */
    /* new dynamical system is valid and built */
/*
    pm(PUT, "Model.Load_Number", -1, NULL); */ /* -1 = parsed dynamical system */
/*
    pm(EXEC, "Model.Load", NULL ); 

  } else {
*/
    /* new dynamical system build failed 
     * fprintf(stdout,"\nbuild ds failed");
     */
/*
  }

}
*/

static void
parser_new_ds_check()
{
  void *new_ds;

  new_ds = pm(GET, PARSER[NEW_DS], NULL);

  if (!new_ds || (PARSER_OK != parser_build_ds(new_ds)))
    pm(PUT, PARSER[NEW_DS_STATUS], 0, NULL);
  else
    pm(PUT, PARSER[NEW_DS_STATUS], 1, NULL);
}

/*********************************************************************************
 *
 * Write C code 
 *
 *********************************************************************************/

static void
parser_writec()
{
  char tmpfile[SIZE_OF_FNAME];
  FILE* fp = NULL;
  void* ds = NULL;

  int n_varb, n_param, n_funct, n_temp;
  int i, manifold_type=EUCLIDEAN, flag;
  int ds_type;
  double defaultd, temp;
  char name[30];
  char *auxfcn;		/* 8/20/97 BAM */

  /* the new ds has already been built */
  ds = (void*)pm( GET, "Parser.New_DS", NULL );

  if( ds == NULL ) {
    /* error in parsed model */
    fprintf(stdout, "\nds is NULL (not built), write c code aborted");
    return;
  }

  /* new dynamical system is valid and built */
  pm( GET, "Parser.Tempfile", tmpfile, NULL );
  fp = fopen( tmpfile, "w" ); 

  pm( GET, "Parser.Modelname", name, NULL );
  ds_type = *(int*)pm( GET, "Parser.Mapping_Flag", NULL);

  n_varb = parser_get_n_vars(ds);
  n_param = parser_get_n_pars(ds);
  n_funct = parser_get_n_aux_fns(ds);
  n_temp = parser_get_n_temps(ds);
  
  for(i=0; i<n_varb; i++)
    if(parser_get_var_periodic(ds,i,&defaultd,&temp)==PARSER_OK) 
      manifold_type = PERIODIC;
  
  fprintf(fp, "/*\n * %s - automatically generated model file\n * %s\n * %s\n */\n", 
	    "dstool_tk", get_user_info(), get_the_time());
  fprintf(fp, "#include <model_headers.h>\n\n");
  
  /* write out definition with translation table  */
  fprintf(fp, "/*\n");
  parser_write_ds_def(ds, fp);
  parser_write_aux_fn(ds, fp);
  if (n_temp>0) fprintf(fp,"\n   where\n");
  parser_write_temp(ds,fp);
  fprintf(fp,"\n\n\tTranslation table:\n\n");

  for (i=0; i<n_varb; i++)
    fprintf(fp, "\t\tx[%d] <--> %s\n", i, parser_get_var_name(ds, i));
  for (i=0; i<n_param; i++)
    fprintf(fp, "\t\tp[%d] <--> %s\n", i, parser_get_par_name(ds, i));
  for (i=0; i<n_temp; i++)
    fprintf(fp, "\t\tTEMP[%d] <--> %s\n", i, parser_get_temp_name(ds, i));
  fprintf(fp, "*/\n\n");

  fprintf(fp, "/* function used to define dynamical system */\n");
  fprintf(fp, "int %s_ds_func( double* f, double* x, double* p)\n{\n\n", name);
  parser_writeC_ds_def(ds, fp);
  fprintf(fp, "\n\treturn 0;\n}\n\n");

  /* write out auxiliary functions */
  fprintf(fp, "/* function used to define aux functions */\n");
  if (n_funct == 0) fprintf(fp,"/*\n");
  fprintf(fp, "int %s_aux_func(double* f,double* x,double* p)\n{\n", name);
  parser_writeC_aux_fn(ds, fp);
  fprintf(fp, "\n\treturn 0;\n}\n");
  if (n_funct == 0) fprintf(fp,"*/\n");

  /* write out jacobian stub */
  fprintf(fp, "\n/* function used to define jacobian. NOT AUTOMATICALLY GENERATED.\n");
  fprintf(fp, "\tinput explicit jacobian in the  form\n");
  fprintf(fp, "\tm[i][j] = d f_i / d x_j; (starting with 0)\n");
  fprintf(fp, "*/\n/*\nint %s_jac( double** m, double* x, double* p)\n{\n", name);
  fprintf(fp, "\n\treturn 0;\n}\n*/\n\n");

  /* write out init routine */
  fprintf(fp, "/* function used to define default data */\n");
  fprintf(fp, "int %s_init()\n{\n", name);

  fprintf(fp, "\tint n_varb=%d;\n",n_varb);
  fprintf(fp, "\tstatic char *variable_names[]={\"%s\"", parser_get_var_name(ds,0));
  for(i=1; i<n_varb; i++) 
    fprintf(fp, ",\"%s\"", parser_get_var_name(ds,i));
  defaultd = ds_type ? VAR_IC_MAP : VAR_IC_VF;
  parser_get_var_ic(ds,0,&defaultd);
  fprintf(fp, "};\n\tstatic double variables[]={%g", defaultd);

  for(i=1; i<n_varb; i++) {
    defaultd = ds_type ? VAR_IC_MAP : VAR_IC_VF;
    parser_get_var_ic(ds, i, &defaultd);
    fprintf(fp, ",%g",defaultd);
  }
  defaultd = ds_type ? VAR_MIN_MAP : VAR_MIN_VF;
  parser_get_var_range(ds,0,&defaultd, &temp);
  fprintf(fp, "};\n\tstatic double variable_min[]={%g", defaultd);

  for(i=1; i<n_varb; i++) {
    defaultd = ds_type ? VAR_MIN_MAP : VAR_MIN_VF;
    parser_get_var_range(ds, i, &defaultd, &temp);
    fprintf(fp, ",%g",defaultd);
  }
  defaultd = ds_type ? VAR_MAX_MAP : VAR_MAX_VF;
  parser_get_var_range(ds,0, &temp, &defaultd);
  fprintf(fp, "};\n\tstatic double variable_max[]={%g", defaultd);

  for(i=1; i<n_varb; i++) {
    defaultd = ds_type ? VAR_MAX_MAP : VAR_MAX_VF;
    parser_get_var_range(ds, i, &temp, &defaultd);
    fprintf(fp, ",%g",defaultd);
  }
  fprintf(fp,"};\n\n");

  fprintf(fp,"\tstatic char *indep_varb_name=\"%s\";\n",ds_type ? INDEP_MAP : INDEP_VF);
  fprintf(fp,"\tdouble indep_varb_min=%g;\n",ds_type ? INDEP_MIN_MAP : INDEP_MIN_VF);
  fprintf(fp,"\tdouble indep_varb_max=%g;\n\n",ds_type ? INDEP_MAX_MAP : INDEP_MAX_VF);

  fprintf(fp, "\tint n_param=%d;\n",n_param);
  fprintf(fp, "\tstatic char *parameter_names[]={\"%s\"", parser_get_par_name(ds,0));
  for(i=1; i<n_param; i++) 
    fprintf(fp, ",\"%s\"", parser_get_par_name(ds,i));
  defaultd = ds_type ? PAR_IC_MAP : PAR_IC_VF;
  parser_get_param_ic(ds,0,&defaultd);
  fprintf(fp, "};\n\tstatic double parameters[]={%g", defaultd);

  for(i=1; i<n_param; i++) {
    defaultd = ds_type ? PAR_IC_MAP : PAR_IC_VF;
    parser_get_param_ic(ds, i, &defaultd);
    fprintf(fp, ",%g",defaultd);
  }
  defaultd = ds_type ? PAR_MIN_MAP : PAR_MIN_VF;
  parser_get_param_range(ds,0,&defaultd, &temp);
  fprintf(fp, "};\n\tstatic double parameter_min[]={%g", defaultd);

  for(i=1; i<n_param; i++) {
    defaultd = ds_type ? PAR_MIN_MAP : PAR_MIN_VF;
    parser_get_param_range(ds, i, &defaultd, &temp);
    fprintf(fp, ",%g",defaultd);
  }
  defaultd = ds_type ? PAR_MAX_MAP : PAR_MAX_VF;
  parser_get_param_range(ds,0, &temp, &defaultd);
  fprintf(fp, "};\n\tstatic double parameter_max[]={%g", defaultd);

  for(i=1; i<n_param; i++) {
    defaultd = ds_type ? PAR_MAX_MAP : PAR_MAX_VF;
    parser_get_param_range(ds, i, &temp, &defaultd);
    fprintf(fp, ",%g",defaultd);
  }
  fprintf(fp,"};\n\n");
  
  fprintf(fp, "\tint n_funct=%d;\n",n_funct);

/* Modified 8/20/97  BAM  (to stop segmentation faults) */
  auxfcn = parser_get_aux_fn_name(ds,0);
  if (auxfcn == NULL) {
      fprintf(fp, "\tstatic char *funct_names[]={\"\"");
  } else {
      fprintf(fp, "\tstatic char *funct_names[]={\"%s\"", parser_get_aux_fn_name(ds,0));
      }


  for(i=1; i<n_funct; i++)  {
    fprintf(fp, ",\"%s\"", parser_get_aux_fn_name(ds,i));
    printf(",\"%s\"", parser_get_aux_fn_name(ds,i));
    }

  defaultd = ds_type ? FN_MIN_MAP : FN_MIN_VF;
  parser_get_aux_fn_range(ds,0,&defaultd, &temp);
  fprintf(fp, "};\n\tstatic double funct_min[]={%g", defaultd);

  for(i=1; i<n_funct; i++) {
    defaultd = ds_type ? FN_MIN_MAP : FN_MIN_VF;
    parser_get_aux_fn_range(ds, i, &defaultd, &temp);
    fprintf(fp, ",%g",defaultd);
  }

  defaultd = ds_type ? FN_MAX_MAP : FN_MAX_VF;
  parser_get_aux_fn_range(ds,0, &temp, &defaultd);
  fprintf(fp, "};\n\tstatic double funct_max[]={%g", defaultd);

  for(i=1; i<n_funct; i++) {
    defaultd = ds_type ? FN_MAX_MAP : FN_MAX_VF;
    parser_get_aux_fn_range(ds, i, &temp, &defaultd);
    fprintf(fp, ",%g",defaultd);
  }
  fprintf(fp,"};\n\n");

  fprintf(fp, "\tint manifold_type=%s;\n", 
    ((manifold_type==EUCLIDEAN) ? "EUCLIDEAN" : "PERIODIC"));
  flag = parser_get_var_periodic(ds, 0, &defaultd, &temp);
  fprintf(fp, "\tstatic int periodic_varb[]={%s", 
    ((flag==PARSER_OK) ? "TRUE" : "FALSE"));

  for(i=1; i<n_varb; i++) {
    flag = parser_get_var_periodic(ds, i, &defaultd, &temp);
    fprintf(fp, ",%s", ((flag==PARSER_OK) ? "TRUE" : "FALSE"));
  }

  flag = parser_get_var_periodic(ds, 0, &defaultd, &temp);
  fprintf(fp, "};\n\tstatic double period_start[]={%g", 
    ((flag==PARSER_OK) ? defaultd : 0.0));

  for(i=1; i<n_varb; i++) {
    flag = parser_get_var_periodic(ds, i, &defaultd, &temp);
    fprintf(fp, ",%g", ((flag==PARSER_OK) ? defaultd : 0.0));
  }

  flag = parser_get_var_periodic(ds, 0, &temp, &defaultd);
  fprintf(fp, "};\n\tstatic double period_end[]={%g", 
    ((flag==PARSER_OK) ? defaultd : 1.0));

  for(i=1; i<n_varb; i++) {
    flag = parser_get_var_periodic(ds, i, &temp, &defaultd);
    fprintf(fp, ",%g", ((flag==PARSER_OK) ? defaultd : 0.0));
  }

  fprintf(fp,"};\n\n");

  fprintf(fp, "\tint mapping_toggle=%s;\n", (ds_type==0 ? "FALSE" : "TRUE"));
  fprintf(fp, "\tint inverse_toggle=FALSE;\n\n");

  fprintf(fp, "\tint (*def_name)()=%s_ds_func;\n", name);
  fprintf(fp, "\tint (*jac_name)()=NULL;\n");
  fprintf(fp, "\tint (*aux_func_name)()=");

  if (n_funct==0) 
    fprintf(fp,"NULL;\n");
  else 
    fprintf(fp,"%s_aux_func;\n",name);

  fprintf(fp, "\tint (*inv_name)()=NULL;\n");
  fprintf(fp, "\tint (*dfdt_name)()=NULL;\n");
  fprintf(fp, "\tint (*dfdparam_name)()=NULL;\n\n");

  fprintf(fp, "/*\n" );
  fprintf(fp, " * Uncomment and customize for Geomview animation\n" );
  fprintf(fp, " *\n");
  fprintf(fp, "#define GV\n\n" );

  fprintf(fp,"\tint (*gv_funct)()=%s_gv;\n", name);
  fprintf(fp,"\tint (*gv_custom_funct)()=NULL;\n");
  fprintf(fp,"\tint gv_n=1;\n");

  fprintf(fp,"\tstatic char *gv_filename = \"%s.oogl\";\n", name);
  fprintf(fp, " */\n\n" );

  fprintf(fp, "/* display this file for help with this system */\n");
  fprintf(fp, "\tc_filename = __FILE__;\n\n" );

  fprintf(fp, "#include <ds_define.c>\n\n");
  fprintf(fp, "\treturn 0;\n}\n\n");

  fclose(fp);

  parser_destroy_ds(ds);
  pm(INIT, "Parser.New_DS",
     PUT, "Parser.New_DS_Status", 0,
     NULL);
}


static void
parser_save()
{
}

