/*
 * Copyright (c) 1997 Stanford University
 *
 * The use of this software for revenue-generating purposes may require a
 * license from the owners of the underlying intellectual property.
 * Specifically, the SRP protocol may not be used for revenue-generating
 * purposes without license.
 *
 * Within that constraint, permission to use, copy, modify, and distribute
 * this software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notices and this permission
 * notice appear in all copies of the software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "t_defines.h"
#include "t_sha.h"

#ifndef NULL
#define NULL 0
#endif

/* Only use if the weak RNG is needed.  This sometimes causes problems
 * with systems that don't have random(). */

/* #define WEAKRAND */

/*
 * t_envhash - Generate a 160-bit SHA hash of the environment
 *
 * This routine performs an SHA hash of all the "name=value" pairs
 * in the environment concatenated together and dumps them in the
 * output.  While it is true that anyone on the system can see
 * your environment, someone not on the system will have a very
 * difficult time guessing it, especially since some systems play
 * tricks with variable ordering and sometimes define quirky
 * environment variables like $WINDOWID or $_.
 */
extern char ** environ;

static void
t_envhash(out)
     unsigned char * out;
{
  char ** ptr;
  char ebuf[256];
  SHA1_CTX ctxt;

  SHA1Init(&ctxt);
  for(ptr = environ; *ptr; ++ptr) {
    strncpy(ebuf, *ptr, 255);
    ebuf[255] = '\0';
    SHA1Update(&ctxt, ebuf, strlen(ebuf));
  }
  SHA1Final(out, &ctxt);
}

/*
 * t_fshash - Generate a 160-bit SHA hash from the file system
 *
 * This routine climbs up the directory tree from the current
 * directory, running stat() on each directory until it hits the
 * root directory.  This information is sensitive to the last
 * access/modification times of all the directories above you,
 * so someone who lists one of those directories injects some
 * entropy into the system.  Obviously, this hash is very sensitive
 * to your current directory when the program is run.
 *
 * For good measure, it also performs an fstat on the standard input,
 * usually your tty, throws that into the buffer, creates a file in
 * /tmp (the inode is unpredictable on a busy system), and runs stat()
 * on that before deleting it.
 *
 * The entire buffer is run once through SHA to obtain the final result.
 */
static void
t_fshash(out)
     unsigned char * out;
{
  char dotpath[128];
  struct stat st;
  SHA1_CTX ctxt;
  int i, pinode;
  dev_t pdev;

  SHA1Init(&ctxt);
  if(stat(".", &st) >= 0) {
    SHA1Update(&ctxt, (unsigned char *) &st, sizeof(st));
    pinode = st.st_ino;
    pdev = st.st_dev;
    strcpy(dotpath, "..");
    for(i = 0; i < 40; ++i) {
      if(stat(dotpath, &st) < 0)
	break;
      if(st.st_ino == pinode && st.st_dev == pdev)
	break;
      SHA1Update(&ctxt, (unsigned char *) &st, sizeof(st));
      pinode = st.st_ino;
      pdev = st.st_dev;
      strcat(dotpath, "/..");
    }
  }

  if(fstat(0, &st) >= 0)
    SHA1Update(&ctxt, (unsigned char *) &st, sizeof(st));

  sprintf(dotpath, "/tmp/rnd.$$", getpid());
  if(creat(dotpath, 0600) >= 0 && stat(dotpath, &st) >= 0)
    SHA1Update(&ctxt, (unsigned char *) &st, sizeof(st));
  unlink(dotpath);

  SHA1Final(out, &ctxt);
}	

/*
 * Generate a high-entropy seed for the strong random number generator.
 * This uses a wide variety of quickly gathered and somewhat unpredictable
 * system information.  The 'preseed' structure is assembled from:
 *
 *   The system time in seconds
 *   The system time in microseconds
 *   The current process ID
 *   The parent process ID
 *   A hash of the user's environment
 *   A hash gathered from the file system
 *   Input from a random device, if available
 *   Timings of system interrupts
 *
 * The entire structure (60 bytes on most systems) is fed to SHA to produce
 * a 160-bit seed for the strong random number generator.  It is believed
 * that in the worst case (on a quiet system with no random device versus
 * an attacker who has access to the system already), the seed contains at
 * least about 80 bits of entropy.  Versus an attacker who does not have
 * access to the system, the entropy should be slightly over 128 bits.
 */
static char initialized = 0;

static struct {
  unsigned int trand1;
  time_t sec;
  time_t usec;
  short pid;
  short ppid;
  unsigned char envh[SHA_DIGESTSIZE];
  unsigned char fsh[SHA_DIGESTSIZE];
  unsigned char devrand[8];
  unsigned int trand2;
} preseed;

static unsigned char randkey1[SHA_DIGESTSIZE], randkey2[SHA_DIGESTSIZE];
static unsigned char randpool[SHA_DIGESTSIZE];
static int inpool = 0;

static void
t_initrand()
{
  SHA1_CTX ctxt;
  struct timeval t;
  long r1, r2;
  int i;

  if(initialized)
    return;

  preseed.trand1 = raw_truerand();

  gettimeofday(&t, NULL);

  preseed.sec = t.tv_sec;
  preseed.usec = t.tv_usec;
  preseed.pid = getpid();
  preseed.ppid = getppid();
  t_envhash(preseed.envh);
  t_fshash(preseed.fsh);

  i = open("/dev/random", O_RDONLY);
  if(i > 0) {
    read(i, preseed.devrand, sizeof(preseed.devrand));
    close(i);
  }
  preseed.trand2 = raw_truerand();

  SHA1Init(&ctxt);
  SHA1Update(&ctxt, (unsigned char *) &preseed, sizeof(preseed));
  SHA1Final(randkey1, &ctxt);
  memcpy(randkey2, randkey1, sizeof(randkey1));
  memset(randpool, 0, sizeof(randpool));
  inpool = 0;

  i = ((preseed.trand2 & 0xff) << 24) | (preseed.usec & 0xffff00) |
    (preseed.pid & 0xff);

#ifdef WEAKRAND
  srandom(i);
  srand48(i);
#endif

  initialized = 1;
}

#define NUM_RANDOMS 12

void
t_stronginitrand()
{
  SHA1_CTX ctxt;
  unsigned int rawrand[NUM_RANDOMS];
  int i;

  if(!initialized)
    t_initrand();
  for(i = 0; i < NUM_RANDOMS; ++i)
    rawrand[i] = raw_truerand();
  SHA1Init(&ctxt);
  SHA1Update(&ctxt, (unsigned char *) rawrand, sizeof(rawrand));
  SHA1Final(randkey2, &ctxt);
  memset(rawrand, 0, sizeof(rawrand));
}

#ifdef WEAKRAND
#if defined (hpux) || defined (__alpha__)
/* HPUX lacks random().  DEC Alpha's random() returns a double.  */
static inline long
urandom ()
{
  return mrand48 ();
}
#else
long random ();
 
static inline long
urandom ()
{
  /* random() returns 31 bits, we want 32.  */
  return random() ^ (random() << 1);
}
#endif

/*
 * The weak random number generator.  This uses random() on most
 * systems and has no more than 32 bits of entropy.  Its seed is
 * independent of the strong generator, so breaking this generator
 * will not affect the security of the strong generator.
 */
void
t_rand(data, size)
     unsigned char * data;
     unsigned size;
{
  long rbuf = 0;
  int bytes = 0;

  if(!initialized)
    t_initrand();

  while(size-- > 0) {
    if(bytes <= 0) {
      rbuf = urandom();
      bytes = 4;
    }
    *data++ = rbuf & 0xff;
    rbuf >>= 8;
    --bytes;
  }
}
#endif

/*
 * The strong random number generator.  This uses two 160-bit seeds
 * and uses SHA in a feedback configuration to generate successive
 * outputs.  Let A[0] = 0'160 (160 zero bits).  Then, the outputs
 * become:
 *		A[i+1] = SHA(K1, A[i], K2)
 * where K is the seed and the outputs start with A[1].  Each cycle
 * generates 20 bytes of new output.
 */
void
t_random(data, size)
     unsigned char * data;
     unsigned size;
{
  SHA1_CTX ctxt;

  if(!initialized)
    t_initrand();

  while(size > inpool) {
    memcpy(data, randpool + sizeof(randpool) - inpool, inpool);
    data += inpool;
    size -= inpool;

    /* Recycle */
    SHA1Init(&ctxt);
    SHA1Update(&ctxt, randkey1, sizeof(randkey1));
    SHA1Update(&ctxt, randpool, sizeof(randpool));
    SHA1Update(&ctxt, randkey2, sizeof(randkey2));
    SHA1Final(randpool, &ctxt);
    inpool = sizeof(randpool);
  }

  if(size > 0) {
    memcpy(data, randpool + sizeof(randpool) - inpool, size);
    inpool -= size;
  }
}

/*
 * The interleaved session-key hash.  This separates the even and the odd
 * bytes of the input (ignoring the first byte if the input length is odd),
 * hashes them separately, and re-interleaves the two outputs to form a
 * single 320-bit value.
 */
unsigned char *
t_sessionkey(key, sk, sklen)
     unsigned char * key;
     unsigned char * sk;
     unsigned sklen;
{
  unsigned i, klen;
  unsigned char * hbuf;
  unsigned char hout[SHA_DIGESTSIZE];
  SHA1_CTX ctxt;

  while(sklen > 0 && *sk == 0) {	/* Skip leading 0's */
    --sklen;
    ++sk;
  }

  klen = sklen / 2;
  if((hbuf = malloc(klen * sizeof(char))) == 0)
    return 0;

  for(i = 0; i < klen; ++i)
    hbuf[i] = sk[sklen - 2 * i - 1];
  SHA1Init(&ctxt);
  SHA1Update(&ctxt, hbuf, klen);
  SHA1Final(hout, &ctxt);
  for(i = 0; i < sizeof(hout); ++i)
    key[2 * i] = hout[i];

  for(i = 0; i < klen; ++i)
    hbuf[i] = sk[sklen - 2 * i - 2];
  SHA1Init(&ctxt);
  SHA1Update(&ctxt, hbuf, klen);
  SHA1Final(hout, &ctxt);
  for(i = 0; i < sizeof(hout); ++i)
    key[2 * i + 1] = hout[i];

  memset(hout, 0, sizeof(hout));
  memset(hbuf, 0, klen);
  free(hbuf);
  return key;
}
