#                              -*- Mode: Perl -*- 
# Package.pm|pkg-order-0.01/Debian/Package --- 
# Author           : Manoj Srivastava ( srivasta@tiamat.datasync.com ) 
# Created On       : Wed Jan 29 17:32:40 1997
# Created On Node  : tiamat.datasync.com
# Last Modified By : Manoj Srivastava
# Last Modified On : Sat May 15 05:26:58 1999
# Last Machine Used: glaurung.green-gryphon.com
# Update Count     : 191
# Status           : Unknown, Use with caution!
# HISTORY          : 
# Description      : 
# 
# 

use strict;
use Carp;
require 5.001;

use Debian::Package::Dependency_List;

=head1 NAME

  Debian::Package::Package - Information about a debian package
 
=cut

=head1 SYNOPSIS

  use Debian::Package::Package;
  # Initialize the package details with a snippet of a Packages file
  $pkg = Debian::Package::Package->new('Package_desc' => $description);
  # Print it back again
  $pkg->print();

  # Exercise new and print
  Debian::Package::Package->test();

=cut


=head1 DESCRIPTION


This module implements a Debian::Package::Package object. 

=cut

=head2 new

This is the constructor for the package. It takes a required argument,
which is a snippet from the Packages file.

=cut



package Debian::Package::Package;

sub new {
  my $this = shift;
  my %params = @_;
  my $class = ref($this) || $this;
  my $self = {};
  
  croak("Need argument Package_desc") unless defined $params{'Package_desc'};
  
  bless $self => $class;
  $self->initialize(%params);
  $self->{' _Debug'} = 0;
  return $self;
}

=head2

Internally, new uses the method B<initialize>. This fixes continuation
lines, and sets up key value pairs (escaping and unescaping newlines
as it does so.)

=cut

sub initialize {
  my $self = shift;
  my %params = @_;

  croak("Need argument Package_desc") unless 
    defined $params{'Package_desc'};
  
  chomp($params{'Package_desc'});
  
  $params{'Package_desc'} =~ s/\n\s+/\376\377/g;  # fix continuation lines
  $params{'Package_desc'} =~ s/\376\377\s*\376\377/\376\377/og;
  
  while ($params{'Package_desc'} =~ m/^(\S+):\s*(.*)\s*$/mg) {
    my $key = $1;
    my $value = $2;
    
    # Field names are not case-sensitive. Guy Maor <maor@ece.utexas.edu>
    $key =~ s/(.)(.*)/\u$1\L$2\E/;
    $key =~ s/([^-]+)-(.)(.*)/$1-\u$2\L$3\E/;
    $value =~ s/\376\377/\n /g;
    # Ignore this if asked to
    if (defined $params{'Ignored Headers'} && $params{'Ignored Headers'}) {
      next if $params{'Ignored Headers'} =~ m/^\s*\Q$key\E\s*$/i;
    }
    $self->{$key} = $value;
  }
  # old style packages which have a separate version. 
  #                                Guy Maor <maor@ece.utexas.edu>
  $self->{' _Mark'} = '';
  $self->{'Version'} .= '-' . $self->{'Revision'} if $self->{'Revision'};
  $self->{' _Results'} = Debian::Package::Results->new($self->{'Package'});
}

=head2 init_depends

Creates a Dependency List object. Takes a named parameter
field. Stored in internal fields whose names start with a blank
followed by an underscore " _")

=cut

sub init_depends {
  my $self = shift;
  my %params = @_;
  
  croak("Need argument 'Type'") unless defined $params{'Type'};
  croak("Package has no name ") unless defined $self->{'Package'};
  return unless defined $self->{$params{'Type'}};
  
  $self->{" _$params{'Type'}"} = 
    Debian::Package::Dependency_List->new
      ('string'  => $self->{$params{'Type'}},
       'Type'    => $params{'Type'},
       'Package' => $self->{'Package'});
}



=head2 print

This method takes the key value pairs that constitute a Package
definition and prints it out.

=cut
 
sub print {
  my $self = shift;
  for (sort keys %$self) {
    next if /^\s+_/og;		# Skip internal fields
    print "$_ => $self->{$_}\n";
  }
}

=head2 order_string

This routine takes the ordering information accumulated in a package
and returns it as a string.

=cut

sub order_string {
  my $self = shift;
  my $string = '';
  my $results = $self->{' _Results'};
  my $errors = 
    $results->check('Type' => "Critical",
		    'Category' => 'Failed')     + 
    $results->check('Type' => "Critical",
		    'Category' => 'Conflict')   + 
    $results->check('Type' => "Critical",
		    'Category' => 'Unknown')    ;
  if (($errors) > 0) {
    print STDERR 
      "Not including $self->{'Package'} $errors Errors \n";
    return '';
  }
  else {
    print STDERR 
      "\tDEBUG:Package.pm:180: including $self->{'Package'} $errors Errors \n"
	if $self->{' _Debug'};
  }
  
  $string = "START $self->{'Package'}\n";
  for (@{$self->{' _Order'}}) {
    $string .= "$_\n";
  }
  $string .= "$self->{'Package'} END\n";
}


=head2 check_relations 

This routine performs the dependency check for the package. For each
package in the list, it constructs an complex associative array to be
used as an argument to the depend method of one of the dependency
lists of the package.

It takes a required named argument Field, whose value should be one of
Pre-Depends, Depends, Recommends, Suggests, Conflicts, or Replaces.
It takes a second required argument, I<New>, which contains a pointer
to the new packages list.

It takes optional named arguments Consistent, (true if inconsistency
in the new list should cause a warning), Installed, (a package list of
installed packages), and I<Warn>, which just sends a warning, and
does not exclude a Package.

=cut

sub check_relations {
  my $self = shift;
  my %params = @_;
  my %Results;
  my %Arguments;
  my $arg = \%Arguments;
  my $ret = -1;
  
  croak("Missing Parameter 'Field'") unless $params{'Field'};
  croak("Missing Parameter 'New'") unless $params{'New'};
  return unless $self->{$params{'Field'}};
  return unless $self->{" _$params{'Field'}"};
  
  if (!defined $self->{'Version'}) {
    warn "Current Package has no version" .
      " :($self->{'Package'}): Skipping\n";
    return;
  }
  $Arguments{'Package'}    = $self;
  $Arguments{'New'}        = $params{'New'};
  $Arguments{'Consistent'} = $params{'Consistent'};
  $Arguments{'Installed'}  = $params{'Installed'};
  $Arguments{'Warn'}       = $params{'Warn'};

  # print qq(DEBUG: |$self->{" _$params{'Field'}"}|\n);
  
  $ret = $self->{" _$params{'Field'}"}->depend($arg);
  print STDERR "\tDEBUG: (package.pm):188 $self->{'Package'} checked as $ret\n" 
    if $self->{' _Debug'};
  $ret;
}

=head2 reset_result 

This routine returns recalculates the results field.

=cut

sub reset_result {
  my $self = shift;
  
  delete $self->{' _Results'};
  $self->{' _Results'} = Debian::Package::Results->new($self->{'Package'});
}

=head2 mark

This routine takes arequired parameter B<Mark> and marks the package
with B<Mark>. This is usually used to mark packages in some fashion so
they will not satisfy dependencies (for packages in the installed list
marked for removal, for example)

=cut

sub mark {
  my $self = shift;
  my %params = @_;

  croak("Need Mark") unless defined $params{'Mark'};
  if ($params{'Mark'}) {	# Do nothing unless we have a mark
    if ($self->{' _Mark'} !~ m/\Q$params{'Mark'}\E/i ) {
      # If not already so marked
      $self->{' _Mark'} .= " $params{'Mark'} ";
      $self->{' _Mark'} =~ s/\s+/ /og;
    }
  }
}
    

=head2 unmark

This routine takes a required parameter, B<Mark> and removes B<Mark>
from the Package.

=cut

sub unmark {
  my $self = shift;
  my %params = @_;
  
  croak("Need Mark") unless defined $params{'Mark'};
  if ($params{'Mark'}) {	# Do nothing unless we have a mark
    if ($self->{' _Mark'} =~ m/\Q$params{'Mark'}\E/i ) {
      # if we are so marked
      $self->{' _Mark'} =~ s/\Q$params{'Mark'}\E/ /iog;
      $self->{' _Mark'} =~ s/\s+/ /og;
    }
  }
}
    

=head2 test_mark

This routine takes two required parameters, B<Package>, and B<Mark>
and tests to see if the B<Package> is marked with B<Mark>.

=cut

sub test_mark {
  my $self = shift;
  my %params = @_;

  croak("Need Mark") unless defined $params{'Mark'};
  
  if ($params{'Mark'}) {	# Do nothing unless we have a mark
    return ($self->{' _Mark'} =~ m/$params{'Mark'}/i);
  }
  else {
    return undef;
  }
}
    

=head2 show_mark

This routine takes one required parameter, B<Package>, and shows all
marks on it.

=cut

sub show_mark {
  my $self = shift;
  
  return $self->{' _Mark'};
}
    


=head2 test

A class  method has been provided to test this class.  Ths uses a
dummy package description string, passes it to new, and prints it back
out.

=cut

sub test {
  my $self = shift;
  my $dummy_package1 = <<'EOF';
Package: Test-package
Essential: yes
Status: install ok installed
Priority: optional
Section: devel
Installed-Size: 33
Maintainer: Manoj Srivastava <srivasta@debian.org>
Source: poeigl
Version: 2.1.3.5-4a
Pre-Depends: libc5 (>= 5.2.18-1)
Replaces: test0
Provides: test1
Depends: xlib6
Recommends: emacs
Suggests: nvi
Conflicts: xemacs
Conffiles: 
 /dev/null
 /etc/another
Description: Short description
 Long description.
 .
 Contains two paragraphs.
EOF
  ;
  my $var = Debian::Package::Package->new('Package_desc' => $dummy_package1);
  $var->print();
}

package Debian::Package::Results;
use Carp;
use strict;

=head1 DESCRIPTION


This module handles storing and retrieving information about the
package ascertained by dependency check runs. 

=cut

=head2 new

This is the constructor for the package. 

=cut

sub new {
  my $this = shift;
  my $class = ref($this) || $this;
  my $package = shift;
  my $self = {};
  
  
  bless $self => $class;
  $self->{'Show Stoppers'} = "Pre-Depends Depends Conflicts";
  $self->{'Special'} = "Critical All";    
  $self->{'Package'} = $package;
  $self->{' _Debug'} = 0;
  
  return $self;
}

=head2 set_fatal_failure_on_types

This routine adds to the current set of I<Show stopper> B<Type>s,
taking one required named argument, I<Type List>.

=cut


sub set_fatal_failure_on_types{
  my $self = shift;
  my %params = @_;
  

  croak("Missing required paramater 'Type List'")
    unless $params{'Type List'};
  $self->{'Show Stoppers'} = $params{'Type List'};
  $self->{'Show Stoppers'} =~ s/^\s*//og;
  $self->{'Show Stoppers'} =~ s/\s*$//og;
  return $self;
}

=head2 add

This routine adds to the current set of dependency results for
this package. It takes 3 named arguments, all of them required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict Unknown> (B<note>, no
trailing s) 

=cut

sub add {
  my $self = shift;
  my %params = @_;		

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 
  croak("Required parameter 'Report' absent") 
    unless $params{'Report'};

  printf STDERR "\tDEBUG: 450 $self->{'Package'} add Type=$params{'Type'} Category=$params{'Category'}"
      if $self->{' _Debug'};
  printf STDERR " Report=$params{'Report'}\n" 
    if $self->{' _Debug'};
  
  my $pattern = $params{'Type'};
  $pattern =~ s/(\W)/\\$1/g;
  croak("Can't add to Special Type:$params{'Type'}") 
    if $self->{'Special'} =~ m/$pattern/i;

  # Make it check known types here?

  # eliminate duplicates
  my $array =
    $self->{'Results'}->{$params{'Category'}}->{$params{'Type'}};
  my $insert = 1;
  $pattern = "\Q$params{'Report'}\E";
  
  for (@{$array}) {
    if (m/^$pattern$/) {
      $insert = 0;		# Ok, result already entered.
      last;			# Could also have returned here
    }
  }
  return $#{$self->{'Results'}->{$params{'Category'}}->{$params{'Type'}}} 
         unless $insert;
  
  push (@{$self->{'Results'}->{$params{'Category'}}->{$params{'Type'}}},
	$params{'Report'});
  return $#{$self->{'Results'}->{$params{'Category'}}->{$params{'Type'}}};
}


=head2 remove

This routine removes a result from the current set of dependency
results for this package. It takes 3 named arguments, all of them
required: B<Type>, one of I<Conflicts Pre-Depends Depends Recommends
Suggests> B<Category>, one of I<Failed Found Warn Conflict Unknown> (B<note>,
no trailing s)

=cut

sub remove {
  my $self = shift;
  my %params = @_;		

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 
  croak("Required parameter 'Report' absent") 
    unless $params{'Report'};

  my $pattern = $params{'Type'};
  $pattern =~ s/(\W)/\\$1/g;
  croak("Can't remove from Special Type:$params{'Type'}") 
    if $self->{'Special'} =~ m/$pattern/i;

  $pattern =  $params{'Report'};
  chomp($pattern);
  $pattern =~ s/(\W)/\\$1/g;
  # Make it check known types here?

  my $array =
    $self->{'Results'}->{$params{'Category'}}->{$params{'Type'}};
  my @tmparray = ();
  
  if ($#{$array} > -1) {
    foreach (@{$array}) {
      if ($self->{' _Debug'}) {
	print STDERR "DEBUG: 521 $self->{'Package'}Considering $_ ($pattern)\n";
	print STDERR "DEBUG: 522 $self->{'Package'}removing $_ ($pattern) from results\n" 
	  if m/^$pattern$/;
	print STDERR "DEBUG: 523 $self->{'Package'}Not removing $_ ($pattern) from results\n"
	  unless m/^$pattern$/;
      }
      push (@tmparray, $_) unless m/^$pattern$/;
    }
    $self->{'Results'}->{$params{'Category'}}->{$params{'Type'}} = ();
    push (@{$self->{'Results'}->{$params{'Category'}}->{$params{'Type'}}},
	  @tmparray);
    return $#{$self->{'Results'}->{$params{'Category'}}->{$params{'Type'}}};
  }
  return 0;
}


=head2 print_result  

This routine prints out the current set of dependency failures for
this package. It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict Unknown> (B<note>, no
trailing s). 

Also, there are two special B<Type>a, I<All Critical>, which print
B<All> and B<Show Stoppers> types respectively. Actually, this
procedure merely handles the Special types, and passes the work to the
next routine.

=cut

sub print_result {
  my $self = shift;
  my %params = @_;	

  
  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 
  my $pattern = $params{'Type'};
  $pattern =~ s/(\W)/\\$1/g;
  if ($self->{'Special'} =~ m/$pattern/io) {
    if ($params{'Type'} =~ m/All/i) {
      for (keys %{$self->{'Results'}->{$params{'Category'}}}) {
	$self->do_print('Type' => "$_",
			'Category' => "$params{'Category'}");
      }
    }
    if ($params{'Type'} =~ m/Critical/i) {
      for (split('', $self->{'Show Stoppers'})) {
	$self->do_print('Type' => "$_",
			'Category' => "$params{'Category'}");
      }
    }
  }
  else {
    $self->do_print('Type' => "$params{'Type'}",
		    'Category' => "$params{'Category'}");
  }
}

=head2 do_print 

This routine prints out the current set of dependency failures for
this package. It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict Unknown> (B<note>, no
trailing s). 

=cut

sub do_print {
  my $self = shift;
  my %params = @_;	
  my $array;

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 

  $array = $self->{'Results'}->{$params{'Category'}}->{$params{'Type'}};
					      
  if ($#{$array} > -1) {
    print STDERR "   Package: $self->{'Package'}\n";
    foreach (@{$array}) {
      print STDERR "      $_";
    }
  }
  return $#{$array};
}


=head2 result_as_string

This routine returns the current set of dependency failures for
this package as a string. It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict Unknown> (B<note>, no
trailing s). 

Also, there are two special B<Type>a, I<All Critical>, which print
B<All> and B<Show Stoppers> types respectively. Actually, this
procedure merely handles the Special types, and passes the work to the
next routine.

=cut

sub result_as_string {
  my $self = shift;
  my $string = '';
  my %params = @_;	

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 

  my $pattern = $params{'Type'};
  $pattern =~ s/(\W)/\\$1/g;
  if ($self->{'Special'} =~ m/$pattern/io) {
    if ($params{'Type'} =~ m/All/i) {
      for (keys %{$self->{'Results'}->{$params{'Category'}}}) {
	$string .= $self->do_string('Type' => "$_",
				    'Category' => "$params{'Category'}");
      }
    }
    if ($params{'Type'} =~ m/Critical/i) {
      for (split(' ', $self->{'Show Stoppers'})) {
	$string .= $self->do_string('Type' => "$_",
				    'Category' => $params{'Category'});
      }
    }
  }
  else {
    $string = $self->do_string('Type' => $params{'Type'},
			       'Category' => "$params{'Category'}");
  }
  return $string;
}


=head2 do_string

This routine returns the current set of dependency failures for
this package as a string. It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict Unknown> (B<note>, no
trailing s). 

=cut

sub do_string {
  my $self = shift;
  my %params = @_;	
  my $array;
  my $string = '';
  
  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 

  $array = $self->{'Results'}->{$params{'Category'}}->{$params{'Type'}};
  if ($#$array > -1) {
    $string .= "   Package: $self->{'Package'}\n";
    foreach (@$array) {
      $string .= "      $_";
    }
  }
  return $string;
}


=head2 check 

This routine returns the number of the  current set of dependency
failures for this package. It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict Unknown> (B<note>, no
trailing s). 

Also, there are two special B<Type>a, I<All Critical>, which print
B<All> and B<Show Stoppers> types respectively. Actually, this
procedure merely handles the Special types, and passes the work to the
next routine.

=cut

sub check {
  my $self = shift;
  my $count = 0;
  my %params = @_;	

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 

  my $pattern = $params{'Type'};
  $pattern =~ s/(\W)/\\$1/g;
  print STDERR "\nPackage.pm:727: $self->{'Package'} Type ",
  "$params{'Type'} Category $params{'Category'} pattern $pattern\n" 
    if $self->{' _Debug'};
  if ("All Critical" =~ m/$pattern/i) {
    if ($params{'Type'} =~ m/All/i) {
      for (keys %{$self->{'Results'}->{$params{'Category'}}}) {
	$count +=  1 +
	  $#{$self->{'Results'}->{$params{'Category'}}->{$_}};
	print STDERR "Package.pm:737: Type $_  ",
	"error: $count\n" 
	  if $self->{' _Debug'};
      }
    }
    if ($params{'Type'} =~ m/Critical/i) {
      for (split(' ', $self->{'Show Stoppers'})) {
	$count += 1 +
	  $#{$self->{'Results'}->{$params{'Category'}}->{$_}};
	print STDERR "Package.pm:746: Type $_ ",
	"error: $count\n" 
	  if $self->{' _Debug'};
      }
    }
  }
  else {
    $count += 1 +
      $#{$self->{'Results'}->{$params{'Category'}}->{$params{'Type'}}};
    print STDERR "Package.pm:755: Category $params{'Category'} ",
    "error: $count\n" 
    if $self->{' _Debug'};
  }
  print STDERR "Package.pm:759: $self->{'Package'} Type ",
  "$params{'Type'} Category $params{'Category'} ",
  "Final error: $count\n" 
    if $self->{' _Debug'};
  
  return $count;
}


=head1 NOTES

=cut

=head1 CAVEATS

This is very inchoate, at the moment, and needs testing.

=cut

=head1 BUGS

None Known so far.

=cut

=head1 AUTHOR

Manoj Srivastava <srivasta@debian.org>

=cut


1;

