/****************************************************************************/
/*                                                                          */
/* ./grid/utm.c   -   Convert to and from UTM Grid Format                   */
/*                                                                          */
/* This file is part of gpstrans - a program to communicate with garmin gps */
/* Parts are taken from John F. Waers (jfwaers@csn.net) program MacGPS.     */
/*                                                                          */
/*                                                                          */
/*    Copyright (c) 1995 by Carsten Tschach (tschach@zedat.fu-berlin.de)    */
/*                                                                          */
/* Permission  to use, copy,  modify, and distribute  this software and its */
/* documentation for non-commercial purpose, is hereby granted without fee, */
/* providing that the  copyright notice  appear in all copies and that both */
/* the  copyright notice and  this permission  notice appear in  supporting */
/* documentation.  I make no representations about  the suitability of this */
/* software  for any  purpose.  It is  provides "as is" without  express or */
/* implid warranty.                                                         */
/*                                                                          */
/****************************************************************************/
#include "defs.h"
#include "Garmin.h"
#include <math.h>


/* define constants */
static const double lat0 = 0.0;	   /* reference transverse mercator latitude */
static const double k0   = 0.9996;


/****************************************************************************/
/* Convert degree to UTM Grid.                                              */
/****************************************************************************/
void DegToUTM(double lat, double lon, char *zone, double *x, double *y)
{
  char   nz;
  double lon0;
  
  if ((lat >= -80.0) && (lat <= 84.0)) {
    nz = 'C'+((short)(lat + 80.0)) / 8;

    /* skip 'I' and 'O' */
    if (nz > 'H') ++nz;
    if (nz > 'N') ++nz;

    lon0 = 6.0 * (double) ((int) (lon / 6.0)) + 3.0;	
    sprintf(zone, "%02d\t%c", ((short)lon0 +183) / 6, nz);

    toTM(lat, lon, lat0, lon0, k0, x, y);

    /* false easting */
    *x = *x + 500000.0;

    /* false northing for southern hemisphere */
    if (lat < 0.0) *y = 10000000.0 - *y;
  } else {
    strcpy(zone, "00\tx");		
    if (lat > 0.0)
      if (lon < 0.0) zone[3] = 'Y';
      else zone[3] = 'Z';
    else
      if (lon < 0.0) zone[3] = 'A';
      else zone[3] = 'B';
    toUPS(lat, lon, x, y);
  }
}


/****************************************************************************/
/* Convert UTM Grid to degree.                                              */
/****************************************************************************/
void UTMtoDeg(short zone, short southernHemisphere, double x, double y, 
	      double *lat, double *lon)
{
  double lon0;
  
  if (zone != 0) {
    lon0 = (double)((-183 + 6 * zone));

    /* remove false northing for southern hemisphere and false easting */
    if (southernHemisphere)
      y = 1.0e7 - y;
    x -= 500000.0;

    fromTM(x, y, lat0, lon0, k0, lat, lon);
  } else
    fromUPS(southernHemisphere, x, y, lat, lon);
}

