// 	$Id: HarmonicsFile.cc,v 1.2 1999/12/05 19:45:42 dave Exp $	
/*  HarmonicsFile  Operations for reading harmonics files.

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

void HarmonicsFile::getline (Dstr &inbuf) {
  assert (!myxmlparsetree);
  Dstr junk1, junk2;
  do {
    do {
      thislinestartpos = ftell (fp);
      inbuf.getline(fp);
      lat_age++;
      lng_age++;
      if (inbuf.isNull())
        return;

      // Ugliness follows.

      // We 'magically' pick up coordinates from the header comments
      // of data sets.
      if ((!strncmp (inbuf.aschar(), "# !longitude:", 13)) ||
          (!strncmp (inbuf.aschar(), "#!longitude:", 12))) {
        lng_age = 0;
        Dstr findit, junk (inbuf);
        do
          junk /= findit;
        while (findit[0] == '#' || findit[0] == '!');
        double lng;
        if (sscanf (findit.aschar(), "%lf", &lng) != 1)
          lng_age = 100;
        else
          mycoords.lng (lng);
      } else if ((!strncmp (inbuf.aschar(), "# !latitude:", 12)) ||
          (!strncmp (inbuf.aschar(), "#!latitude:", 11))) {
        lat_age = 0;
        Dstr findit, junk (inbuf);
        do
          junk /= findit;
        while (findit[0] == '#' || findit[0] == '!');
        double lat;
        if (sscanf (findit.aschar(), "%lf", &lat) != 1)
          lat_age = 100;
        else
          mycoords.lat (lat);
      }

    } while (inbuf[0] == '#');   // Skip comments
    junk1 = inbuf;
    junk1 /= junk2;
  } while (junk2.isNull());      // Skip blank lines
}

void HarmonicsFile::getline_f (Dstr &inbuf) {
  assert (!myxmlparsetree);
  getline (inbuf);
  if (inbuf.isNull()) {
    Dstr details ("The offending file is ");
    details += *myfilename;
    details += ".";
    barf (END_OF_FILE, details);
  }
}

HarmonicsFile::HarmonicsFile (Dstr &filename, Settings *in_settings) {
  settings = in_settings;
  myxmlparsetree = NULL;
  got_past_constituents = 0;
  myfilename = &filename;
  fp = fopen (myfilename->aschar(), "r");
  if (!fp) {
    xperror (myfilename->aschar());
  } else {
    Dstr inbuf;
    getline_f (inbuf);
    if (inbuf[0] == '<') {
      // Parse as XML file
      xmlfilename = filename;
      xmlin = fp;
      fp = NULL;
      rewind (xmlin);
      xmlparsetree = NULL;
      xmlparse();
      fclose (xmlin);
      myxmlparsetree = xmlparsetree;
      xmlparsetree = NULL;
    } else {
      // Not an XML file.
      if (sscanf (inbuf.aschar(), "%u", &mynumconst) != 1) {
	Dstr details (*myfilename);
	details += ": Can't parse number of constituents.  What is '";
	details += inbuf;
	details += "'?";
	barf (CORRUPT_HARMONICS_FILE, details);
      }
      lat_age = lng_age = 100;
    }
  }
}

HarmonicsFile::~HarmonicsFile() {
  if (fp)
    fclose (fp);
  freexml (myxmlparsetree);
}

void
HarmonicsFile::skip_constituents() {
  assert (!myxmlparsetree);
  Dstr inbuf;
  do
    getline_f (inbuf);
  while (inbuf != "*END*");
  do
    getline_f (inbuf);
  while (inbuf != "*END*");
  got_past_constituents = 1;
}

CombinedOffset
HarmonicsFile::getCombinedOffset (struct xmltag *t) {
  CombinedOffset s;
  while (t) {
    if ((*(t->name)) == "timeadd") {
      struct xmlattribute *a = t->attributes;
      while (a) {
        if ((*(a->name)) == "value")
          s.timeAdd = Interval (*(a->value));
        a = a->next;
      }
    } else if ((*(t->name)) == "leveladd") {
      PredictionValue::Unit tu;
      double tv = 0.0;
      struct xmlattribute *a = t->attributes;
      while (a) {
        if ((*(a->name)) == "value")
          tv = (*(a->value)).asdouble();
        else if ((*(a->name)) == "units")
          tu = PredictionValue::Unit (*(a->value));
        a = a->next;
      }
      s.levelAdd = PredictionValue (tu, tv);
    } else if ((*(t->name)) == "levelmultiply") {
      struct xmlattribute *a = t->attributes;
      while (a) {
        if ((*(a->name)) == "value")
          s.levelMultiply ((*(a->value)).asdouble());
        a = a->next;
      }
    } else if ((*(t->name)) == "direction") {
      double tv = 0.0;
      int isnull = 1, kibosh = 0;
      Angle::qualifier tq = Angle::NONE;
      struct xmlattribute *a = t->attributes;
      while (a) {
        if ((*(a->name)) == "value") {
          isnull = 0;
          tv = (*(a->value)).asdouble();
        } else if ((*(a->name)) == "units") {
          if ((*(a->value)) == "degrees")
            ;
          else if ((*(a->value)) == "degrees true")
            tq = Angle::TRU;
          else
            kibosh = 1; // Unrecognized units
        }
        a = a->next;
      }
      if (!(isnull || kibosh))
        s.currentAngle = NullableAngle (Angle::DEGREES, tq, tv);
    }
    t = t->next;
  }
  return s;
}

SimpleOffsets *
HarmonicsFile::getSimpleOffsets (struct xmltag *t) {
  return new SimpleOffsets (getCombinedOffset (t));
}

HairyOffsets *
HarmonicsFile::getHairyOffsets (struct xmltag *t) {
  if (!t)
    return NULL;
  HairyOffsets *s = new HairyOffsets();
  int floodflag=0, ebbflag=0;
  while (t) {
    if ((*(t->name)) == "max")
      s->max = getCombinedOffset (t->contents);
    else if ((*(t->name)) == "min")
      s->min = getCombinedOffset (t->contents);
    else if ((*(t->name)) == "floodbegins") {
      struct xmlattribute *a = t->attributes;
      while (a) {
        if ((*(a->name)) == "value") {
          s->floodbegins.timeAdd = Interval (*(a->value));
          floodflag = 1;
        }
        a = a->next;
      }
    } else if ((*(t->name)) == "ebbbegins") {
      struct xmlattribute *a = t->attributes;
      while (a) {
        if ((*(a->name)) == "value") {
          s->ebbbegins.timeAdd = Interval (*(a->value));
          ebbflag = 1;
        }
        a = a->next;
      }
    }
    t = t->next;
  }
  // This seems evil, but without it, people will get really unexpected
  // results if they omit slack offsets from current stations.
  if (!floodflag)
    s->floodbegins.timeAdd = (s->max.timeAdd + s->min.timeAdd) / 2;
  if (!ebbflag)
    s->ebbbegins.timeAdd = (s->max.timeAdd + s->min.timeAdd) / 2;
  return s;
}

StationRef *
HarmonicsFile::getNextStationRef () {
  while (myxmlparsetree) {
    struct xmltag *temp = myxmlparsetree;
    myxmlparsetree = temp->next;
    temp->next = NULL;
    if ((*(temp->name)) == "subordinatestation") {
      SubordinateStationRef *sr = new SubordinateStationRef();
      sr->rsr = NULL;
      sr->harmonicsFileName = myfilename;
      // Do attributes.
      struct xmlattribute *a = temp->attributes;
      while (a) {
        if ((*(a->name)) == "name")
          sr->name = (*(a->value));
        else if ((*(a->name)) == "latitude")
          sr->coordinates.lat ((*(a->value)).asdouble());
        else if ((*(a->name)) == "longitude")
          sr->coordinates.lng ((*(a->value)).asdouble());
        else if ((*(a->name)) == "timezone")
          sr->timezone = (*(a->value));
        else if ((*(a->name)) == "reference")
          sr->source = (*(a->value));
        a = a->next;
      }
      // Do contents.  There should not be more than one offsets tag.
      struct xmltag *b = temp->contents;
      sr->offsets = NULL;
      while (b) {
        if ((*(b->name)) == "simpleoffsets") {
          sr->offsets = getSimpleOffsets (b->contents);
          break;
        } else if ((*(b->name)) == "offsets") {
          sr->offsets = getHairyOffsets (b->contents);
          break;
        }
        b = b->next;
      }
      freexml (temp);
      return sr;
    }
    freexml (temp);
  }

  if (!fp)
    return NULL;
  Dstr inbuf;
  if (!got_past_constituents)
    skip_constituents();
  getline (inbuf);
  if (inbuf.isNull())
    return NULL;
  ReferenceStationRef *sr = new ReferenceStationRef();
  sr->harmonicsFileName = myfilename;
  sr->name = inbuf;
  sr->filePosition = thislinestartpos;
  if (lat_age < 4 && lng_age < 4)
    sr->coordinates = mycoords;
  // Get time zone (needed for ZoneIndex)
  getline_f (inbuf);
  inbuf /= sr->timezone;
  inbuf /= sr->timezone;
  if (sr->timezone.isNull())
    sr->timezone = "UTC0";
  // Skip remaining crap:  datum, and numconst constituents
  for (unsigned i=0; i<mynumconst+1; i++)
    getline_f (inbuf);
  return sr;
}

StationRef *HarmonicsFile::getStationRef (const Dstr &name) {

  // This function has to be fast to keep the response time down for
  // command line operation.

  // XML.  Skip everything not necessary to finding the desired
  // station.  If the parser gets too slow, the next step will be to
  // defer parsing, skim the XML file itself for the desired location
  // and extract just that section for full parsing.
  while (myxmlparsetree) {
    struct xmltag *temp = myxmlparsetree;
    if ((*(temp->name)) == "subordinatestation") {
      struct xmlattribute *a = temp->attributes;
      while (a) {
        if ((*(a->name)) == "name") {
          if ((*(a->value)) %= name)
            return getNextStationRef();
          else
            break;
        }
        a = a->next;
      }
    }
    myxmlparsetree = temp->next;
    temp->next = NULL;
    freexml (temp);
  }
  if (!fp)
    return NULL;

  // Non-XML.
  // Blaze through the file, but keep tabs on where we need to go back
  // to in order to collect the StationRef properly.  That would be
  // the start of the preceding comment block, or the location name if
  // there is no comment block.

  // This loop was modeled closely on the comparable code in XTide 1.

  char fastbuf[1001]; // Even too cheap to use a Dstr...
  char *lname = name.aschar();
  got_past_constituents = 1; // no need for skip_constituents()
  int prevlineiscomment = 0;
  int thislineiscomment = 0;
  thislinestartpos = ftell (fp);
  long startpos = thislinestartpos;
  while (fgets (fastbuf, 1000, fp)) {
    prevlineiscomment = thislineiscomment;
    if (fastbuf[0] == '#') {
      thislineiscomment = 1;
      if (!prevlineiscomment)
        startpos = thislinestartpos;
    } else {
      thislineiscomment = 0;
      if (!(slackcmp (fastbuf, lname))) {
        // Hit.
        // If there is no comment block preceding this, then get rid of
        // the old pointer.
        if (!prevlineiscomment)
          startpos = thislinestartpos;
        // Get the StationRef and get out.
        assert (!fseek (fp, startpos, SEEK_SET));
        return getNextStationRef();
      }
    }
    thislinestartpos = ftell (fp);
  }
  return NULL;
}

ConstituentSet *HarmonicsFile::get_constituents() {
  assert (!myxmlparsetree);
  Dstr inbuf;
  assert (!got_past_constituents);
  ConstituentSet *c = new ConstituentSet (mynumconst);
  unsigned i;
  for (i=0; i<mynumconst; i++) {
    getline_f (inbuf);
    (*c)[i].readnameandspeed (inbuf);
  }
  getline_f (inbuf);
  if (sscanf (inbuf.aschar(), "%u", &i) != 1) {
    Dstr details (*myfilename);
    details += ": Can't parse starting year.  What is '";
    details += inbuf;
    details += "'?";
    barf (CORRUPT_HARMONICS_FILE, details);
  }
  Year startyear (i);
  getline_f (inbuf);
  if (sscanf (inbuf.aschar(), "%u", &i) != 1) {
    Dstr details (*myfilename);
    details += ": Can't parse number of equilibrium arg years.  What is '";
    details += inbuf;
    details += "'?";
    barf (CORRUPT_HARMONICS_FILE, details);
  }
  Year endyear (startyear + (i-1));
  for (i=0; i<mynumconst; i++)
    (*c)[i].readarg (fp, startyear, endyear);
  getline_f (inbuf);
  if (inbuf != "*END*")
    getline_f (inbuf);  // In case there's trailing whitespace.
  if (inbuf != "*END*") {
    Dstr details (*myfilename);
    details += ": *END* not found after equilibrium args.";
    barf (CORRUPT_HARMONICS_FILE, details);
  }
  getline_f (inbuf);
  if (sscanf (inbuf.aschar(), "%u", &i) != 1) {
    Dstr details (*myfilename);
    details += ": Can't parse number of node factor years.  What is '";
    details += inbuf;
    details += "'?";
    barf (CORRUPT_HARMONICS_FILE, details);
  }
  endyear = startyear + (i-1);
  for (i=0; i<mynumconst; i++)
    (*c)[i].readnod (fp, startyear, endyear);
  getline_f (inbuf);
  if (inbuf != "*END*")
    getline_f (inbuf);  // In case there's trailing whitespace.
  if (inbuf != "*END*") {
    Dstr details (*myfilename);
    details += ": *END* not found after node factors.";
    barf (CORRUPT_HARMONICS_FILE, details);
  }
  return c;
}

Station *HarmonicsFile::getStation (StationRef &in_ref,
				    TideContext *in_context) {
  if (in_ref.is_reference_station()) {
    const ReferenceStationRef *rsr = (const ReferenceStationRef *) &in_ref;
    assert (fp);
    Dstr inbuf;
    ConstituentSet *c = get_constituents();
    assert (!fseek (fp, rsr->filePosition, SEEK_SET));
    getline_f (inbuf);
    if (inbuf != rsr->name)
      barf (CORRUPT_HARMONICS_FILE);
    Dstr meridian;
    getline_f (inbuf);
    inbuf /= meridian;
    if (meridian.isNull()) {
      Dstr details (*myfilename);
      details += ": No meridian for location '";
      details += rsr->name;
      details += "'";
      barf (CORRUPT_HARMONICS_FILE, details);
    }
    Dstr datum, units;
    getline_f (inbuf);
    inbuf /= datum;
    inbuf /= units;
    if (datum.isNull())
      barf (CORRUPT_HARMONICS_FILE);
    if (units.isNull()) {
      Dstr details (*myfilename);
      details += ": Need units after datum for location '";
      details += rsr->name;
      details += "'";
      barf (CORRUPT_HARMONICS_FILE, details);
    }
    PredictionValue::Unit d_units (units);
    double d_datum;
    if (sscanf (datum.aschar(), "%lf", &d_datum) != 1) {
      Dstr details (*myfilename);
      details += ": Invalid datum for location '";
      details += rsr->name;
      details += "'";
      barf (CORRUPT_HARMONICS_FILE, details);
    }
    PredictionValue::Unit a_units;
    if (d_units.mytype == PredictionValue::Unit::KnotsSquared) {
      a_units = PredictionValue::Unit (PredictionValue::Unit::KnotsSquared);
      d_units = PredictionValue::Unit (PredictionValue::Unit::Knots);
    } else
      a_units = d_units;

    ConstantSet *constants = new ConstantSet (mynumconst);
    constants->datum = PredictionValue (d_units, d_datum);

    unsigned i;
    for (i=0; i<mynumconst; i++) {
      Dstr junkname, amp, phase;
      double d_amp, d_phase;
      getline_f (inbuf);
      inbuf /= junkname;
      inbuf /= amp;
      inbuf /= phase;
      if (sscanf (amp.aschar(), "%lf", &d_amp) != 1) {
	Dstr details (*myfilename);
	details += ": Invalid harmonic constant for location '";
	details += rsr->name;
	details += "'";
	barf (CORRUPT_HARMONICS_FILE, details);
      }
      if (sscanf (phase.aschar(), "%lf", &d_phase) != 1) {
	Dstr details (*myfilename);
	details += ": Invalid harmonic constant for location '";
	details += rsr->name;
	details += "'";
	barf (CORRUPT_HARMONICS_FILE, details);
      }
      constants->amplitudes[i] = Amplitude (a_units, d_amp);
      // Note negation of phase
      constants->phases[i] = Degrees (-d_phase);
    }

    // Normalize the meridian to UTC
    // To compensate for a negative meridian requires a positive offset.
    SimpleOffsets fixMeridianOffsets;
    fixMeridianOffsets.timeAdd = -Interval (meridian);
    constants->adjust (fixMeridianOffsets, *c);

    ConstantSetWrapper *csw = new ConstantSetWrapper (c, constants);
    ReferenceStation *s = new ReferenceStation (in_context, csw);
    s->stationRef = &in_ref;
    s->name = rsr->name;
    s->timeZone = rsr->timezone;
    s->coordinates = rsr->coordinates;
    s->myUnits = d_units;
    // FIXME if there are ever units of velocity other than knots
    if (d_units.mytype == PredictionValue::Unit::Knots) {
      s->isCurrent = 1;
      if (a_units.mytype == PredictionValue::Unit::KnotsSquared)
        s->isHydraulicCurrent = 1;
      else
        s->isHydraulicCurrent = 0;
    } else
      s->isCurrent = s->isHydraulicCurrent = 0;

    // FIXME if there are ever units of velocity other than knots
    if (in_context->settings->u != "x" && (!(s->isCurrent)))
      s->setUnits (PredictionValue::Unit (in_context->settings->u));

    return s;
  } else {
    // Can't happen.
    assert (0);
  }
  // Silence bogus SGI compiler warning
  return NULL;
}
