
#include <std.h>
#include <strings.h>
#include <ops.h>

#include <func/apt.h>
#include <func/config.h>
#include <func/keys.h>
#include <func/mouse.h>

#include <list/screens.h>

#include <interface/coreui.h>
#include <interface/acquire.h>
#include <interface/formatting.h>
#include <interface/dialogs.h>

#include <getopt.h>

void initarguments(int, char **);
void InvalidOpt(char *, int);
void DisplayVersion();
void DisplayHelp(char *);
void sighandler(int);
void io_block_loop(void);

int main(int argc, char **argv)
{
	// Set up the signal handlers
	signal(SIGSEGV, sighandler);
	signal(SIGINT, sighandler);
	signal(SIGKILL, sighandler);
	signal(SIGTERM, sighandler);

	// Initialize the apt config structures
	pkgInitialize(*_config);

	// Initialize command line arguments
	initarguments(argc, argv);

	// Parse the capt configuration file
	if (initconfig() == false)
		writeconfig();

#ifdef HAVE_LIBGPM
	// Initialize the mouse
	gpm_open();
#endif

	// Initialize curses data structures
	ui_first_init();

	// Check that capt can actually run on this terminal.
	if (LINES < 15 || COLS < 80)
	{
		capt_clean_exit();
		fprintf(stderr, "Running this program is not recommanded without at least 15 lines and 80 columns.\n");
		exit(-2);
	}

	// Make the internal linked lists
	initialize_cache();

	// Add the first screen
	screen.add();
	screen.boundreset(screen.Current());

	// Draw interface
	ui_initwindows();
	ui_redraw_current();

	// Turn on description window
	if (strcmp(get_config_string("description"), "1") == 0)
	{
		ui_desc_window_show();
		paint_description();
		paint_description_top();
	}

	// Start reading keys
	while (1)
		if (screen->kdriver->inputmgr() == false)
			break;

	return 0;
}

void initarguments(int argc, char **argv)
{
	if (argc == 1)
		return;

	int c;

	int PreparedPkgs = -1;
	TextOp::opType CurrentT = TextOp::none;
	TextOp::CacheInit = false;
	TextOp::opRealCount = 0;

	for (c = 1; c < argc; c++)
	{
		if (*argv[c] == '-')
		{
			if (PreparedPkgs == 0)
			{
				fprintf(stderr, "option '%s' requires at least one argument [line opt %d]\n", argv[c - 1], c - 1);
				endwin();
				exit(255);
			}

			// These ones exit the program

			if (strcmp(argv[c] + 1, "version") == 0 || strcmp(argv[c] + 1, "v") == 0)
				DisplayVersion();
			else if (strcmp(argv[c] + 1, "help") == 0 || strcmp(argv[c] + 1, "h") == 0)
				DisplayHelp(argv[0]);

			// These don't :)

			else if (strcmp(argv[c] + 1, "install") == 0 || strcmp(argv[c] + 1, "i") == 0)
			{
				CurrentT = TextOp::install;
				PreparedPkgs = 0;
			}
			else if (strcmp(argv[c] + 1, "delete") == 0 || strcmp(argv[c] + 1, "d") == 0)
			{
				CurrentT = TextOp::remove;
				PreparedPkgs = 0;
			}
			else if (strcmp(argv[c] + 1, "purge") == 0 || strcmp(argv[c] + 1, "p") == 0)
			{
				CurrentT = TextOp::purge;
				PreparedPkgs = 0;
			}
			else if (strcmp(argv[c] + 1, "update") == 0 || strcmp(argv[c] + 1, "u") == 0)
			{
				CurrentT = TextOp::update;

				if (TextOp::CacheInit == false)
				{
					ui_first_init();
					initialize_cache();
				}

				pkgAcquireInterface *Fetcher = new pkgAcquireInterface(pkgAcquireInterface::TI);
				Fetcher->AcquireUpdate();
				delete Fetcher;

				TextOp::CacheInit = false;
				PreparedPkgs = -1;
			}
			else if (strcmp(argv[c] + 1, "upgrade") == 0 || strcmp(argv[c] + 1, "g") == 0)
			{
				CurrentT = TextOp::upgrade;

				if (TextOp::CacheInit == false)
				{
					ui_first_init();
					initialize_cache();
				}

				pkgAcquireInterface *Fetcher = new pkgAcquireInterface(pkgAcquireInterface::TI);
				Fetcher->AcquireUpgrade();
				delete Fetcher;

				TextOp::CacheInit = false;
				PreparedPkgs = -1;
			}
			else
				InvalidOpt(argv[c] + 1, c);
		}
		else
		{
			switch (CurrentT)
			{
			case TextOp::none:
			case TextOp::update:
			case TextOp::upgrade:
				InvalidOpt(argv[c], c);
				break;
			default:
				TextOp::PrepareOp(CurrentT, (string) argv[c]);
				PreparedPkgs++;
				break;
			}
		}
	}

	if (CurrentT != TextOp::none)
	{
		if (PreparedPkgs > 0 && TextOp::opRealCount > 0)
		{
			if (TextOp::CacheInit == false)
			{
				ui_first_init();
				initialize_cache();
			}

			pkgAcquireInterface *Fetcher = new pkgAcquireInterface(pkgAcquireInterface::TI);
			Fetcher->AcquireRun();
			delete Fetcher;

			endwin();
			exit(0);
		}
		else if (PreparedPkgs == 0)
		{
			fprintf(stderr, "option '%s' requires at least one argument [line opt %d]\n", argv[c - 1], c - 1);
			exit(255);
		}
		else
		{
			endwin();
			exit(10);
		}
	}
}

void InvalidOpt(char *optname, int optnum)
{
	fprintf(stderr, "syntax error at '%s' [line opt %d]\n", optname, optnum);
	exit(255);
}


void DisplayVersion()
{
	printf
		("Console Acquisition Package Tool (capt).\n"
	       "Created and maintained by Patrick Cole (ltd@debian.org)\n" "Compiled on %s %s.\n" "Internal Version String: %s\n", __TIME__, __DATE__, INTERNAL_VERSION);

	exit(0);
}

void DisplayHelp(char *caller)
{
	printf("\nUsage: %s -(op 1) [package1 .. package2 .. packageN] ... -(op N) [pkg1 .. pkgN]\n\n", caller);
	printf(" operation     outcome\n\n");
	printf(" [h]elp        This helpful little screen\n");
	printf(" [v]ersion     Current version of the program\n");
	printf(" [i]nstall     Install package(s)\n");
	printf(" [d]elete      Remove package(s)\n");
	printf(" [p]urge       Remove package(s) including config files\n");
	printf(" [u]pdate      Attempt to get new package listings\n");
	printf(" up[g]rade     Upgrade all packages\n\n");

	exit(0);
}

void sighandler(int sig)
{
	capt_clean_exit();
	fprintf(stderr, "Exit: \"Signal Handler (sig%d)\"\n", sig);
	exit(sig);
}

void capt_clean_exit(void)
{
#ifdef HAVE_LIBGPM
	gpm_close();
#endif
	reset_shell_mode();
	endwin();
	writeconfig();
}
