/*
 * (C) Copyright Keith Visco 1998, 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */
package com.kvisco.xsl;

import com.kvisco.xsl.util.MessageObserver;
import com.kvisco.xsl.util.SimpleMessageObserver;
import com.kvisco.util.List;
import com.kvisco.net.URIUtils;

import java.io.*;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import com.kvisco.xml.*;

import org.w3c.dom.*;

/**
 * XSLProcessor
 * An XSL Processor implementing the XSLT WD 1.0
 * <BR>
 * <B>This application is a work in progress.</B>
 * <BR>
 * see the XSLT WD 1.0 (http://www.w3.org/TR/1999/WD-xslt-19990421.html)
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/

public class XSLProcessor 
    implements com.kvisco.xsl.util.MessageObserver 
{
    
    
    /* <command-line-flags> */
    
    /**
     * The flag directive for the help screen
    **/
    public static final String HELP_FLAG        = "h";
    
    /**
     * The flag directive for the xml input file
    **/
    public static final String INPUT_FLAG       = "i";
    
    /**
     * The flag directive for the formatter to use
    **/
    public static final String FORMATTER_FLAG   = "fo";
    
    /**
     * The flag directive for the result tree output file
    **/
    public static final String OUTPUT_FLAG      = "o";
    /**
     * The flag directive for the stylesheet to use
    **/
    public static final String STYLESHEET_FLAG  = "s";
    
    /**
     * The flag directive for the turning on validation
    **/
    public static final String VALIDATE_FLAG    = "val";
    /**
     * The flag directive for displaying the version
    **/
    public static final String VERSION_FLAG     = "v";
    
    /**
     * The flag directive for the error log file
     * -- added by Mohan Embar
    **/
    public static final String ERR_OUTPUT_FLAG  = "e";  
    /* </command-line-flags> */
    
    private final String ERR_PROPERTIES_NOT_FOUND =
        "unable to load properties file";
       
    /* <properties> */
    
    /**
     * DOM Package property name
    **/
    public static final String DOM_PACKAGE = "dom-package";
    
    /**
     * Indent Size property name
    **/
    public static final String INDENT_SIZE = "indent-size";
    
    /** 
     * The property prefix to define formatters, e.g.
	 * formatter.text=com.acme.MyFormatter
	 * (added by Franck Mangin)
	 */
	private static final String FORMATTER_PREFIX = "formatter.";
	
    /* </properties> */
    
    
    private final String HTML_RESULT_NS   = "http://www.w3.org/TR/REC-html";
    private final String HTML             = "html";
    private final String DEFAULT_NS       = "";

    protected final static String PROPERTIES_FILE  = "xslp.properties";
    
    /**
     * Name of this app 
    **/
    private static final String appName = "XSL:P";
    
    /**
     * Version of this app 
    **/
    private static final String appVersion = "1.0 Beta (19990822)";
    
    
    /**
     * document base for relative url's
    **/
	private String documentBase = null;
	
	/**
	 * PrintWriter to print error messages to
	**/
    private PrintWriter errorWriter = new PrintWriter(System.out, true);
    
    /**
     * The properites for this XSLProcessor
    **/
    private Properties  props = null;
    
    /**
     * The DOMReader to use for Reading in DOM documents
     * as well as creating the Result Document
    **/
    private DOMReader   domReader = null;
    
    private String domPackageName = "DEFAULT";
   
    /**
     * The list of MessageObservers for this processor
    **/
    private List msgObservers = null;
    
    /**
     * The default MessageObserver
    **/
    public static final MessageObserver DEFAULT_MESSAGE_OBSERVER =
        new SimpleMessageObserver();
    
    /**
     *
    **/
    private boolean usingDefaults = false;
    
    /**
     * Flag indicating whether or not to validate when reading an
     * XML document
    **/
    private boolean validate = false;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * creates a new XSLProcessor
    **/
	public XSLProcessor() {
	    super();
	    initialize();
	    //-- set internal properties
	    
    } //-- XSL:Processor
    
    protected InputStream getResourceAsStream(String path) {
        return getClass().getResourceAsStream(path);
    } //-- getResourceAsStream
    
    private void initialize() {
        
        //-- initialize message observers
        msgObservers = new List();
        msgObservers.add(DEFAULT_MESSAGE_OBSERVER);
        
	    // Read in properties for XSL:P
	    props = new Properties();
	    try {
	        InputStream is = null;
	        // look in file system
	        File propsFile = new File(PROPERTIES_FILE);
	        if (propsFile.exists()) {
	            is = new FileInputStream(propsFile);
	        }
	        else is = getResourceAsStream(PROPERTIES_FILE);
	        
	        if (is != null) props.load(is);
	        else usingDefaults = true;
	    }
	    catch(IOException ioEx) {
	        usingDefaults = true;
	    }
	    initializeDOMReader();
	} //-- initialize
	
	private void initializeDOMReader() {
	    
	    // Create DOMReader 
	    domPackageName = props.getProperty(DOM_PACKAGE);
	    
	    if (domPackageName != null) {
	        try {
	            domReader = new DOMReader(props.getProperty(domPackageName));
	        }
	        catch(Exception ex) {
	            errorWriter.print("unable to load DOM Package: ");
	            errorWriter.print(domPackageName);
	            errorWriter.println(" ("+props.getProperty(domPackageName)+")");
	            errorWriter.print(" -- ");
	            errorWriter.println(ex.getMessage());
	        }
	    }
	    else {
	        domPackageName = "DEFAULT";
	        try {
	            domReader = new DOMReader();
  	        }
	        catch(Exception ex) {
	            errorWriter.print("unable to load DOM Package: DEFAULT");
	            errorWriter.print(" (");
	            errorWriter.print(DOMReader.getDefaultDOMPackageClassName());
	            errorWriter.println(")");
	            errorWriter.print(" -- ");
	            errorWriter.println(ex.getMessage());
	        }
	    }
    } //-- initialize
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Adds the given MessageObserver to this processors list
     * of MessageObservers
     * @param msgObserver the MessageObserver to add to this processors
     * list of MessageObservers
    **/
    public void addMessageObserver(MessageObserver msgObserver) {
        msgObservers.add(msgObserver);
    } //-- addMessageObserver
    
    
    /**
	 * Retrieves the name and version of this application
	 * @returns a String with the name and version of this application
	**/
    public static String getAppInfo() {
        return appName + " " + appVersion;
    }

    /**
     * Returns the name of the DOM Package being used
     * @see xslp.properties for more information
    **/
    public String getDOMPackageName() {
        return domPackageName;
    } //-- getDOMPackageName
    
    /**
     * Returns the property value associated with the given String
     * @return the property value associated with the given String
     * <BR>See xslp.properties for for a list of properties
    **/
    public String getProperty(String property) {
        return props.getProperty(property);
    } //-- getProperty
    
	/**
     * Runs this XSLProcessor based on the given arguments. 
     * This method can be called from another Class however,
     * one of the process methods should be more appropriate
     * @param args a list of arguments to this XSLProcessor
     * <BR>
     * Though I do not recommend the following, if you need a static
     * call to XSL:P use the following:<BR>
     * args = {"-i", "xmlfile.xml", "-s", "style.xsl","-o","result.html"}
     * <BR>-- OR --<BR>
     * args = {"-ixmlfile.xml", "-sstyle.xsl","-oresult.html"}
     * <BR>If the Stylesheet is referenced by the <?xml:stylesheet ?>
     * processing instruction use the following<BR>
     * args = {"-ixmlfile.xml","-oresult.html"}
    **/
	public static void main(String args[]) {
	    
        //-- Performance Testing
        /* *
        Runtime rt = Runtime.getRuntime();
        rt.gc();
        System.out.println("Free Memory: " + rt.freeMemory());
        long stime = System.currentTimeMillis();
        /* */
	    
	    
        String xmlFile     = null;
        String outFile     = null;
        String xslFile     = null;
        String fclass      = null;
        String errFile     = null;
        Writer errWriter   = null;
        
	    
	    List flags = new List(8);
	    flags.add(HELP_FLAG);       //-- help screen
	    flags.add(STYLESHEET_FLAG); //-- stylesheet filename
	    flags.add(OUTPUT_FLAG);     //-- output filename
	    flags.add(INPUT_FLAG);      //-- XML input filename
	    flags.add(FORMATTER_FLAG);  //-- Formatter Classname
	    flags.add(VALIDATE_FLAG);   //-- validation
	    flags.add(VERSION_FLAG);    //-- version info
	    flags.add(ERR_OUTPUT_FLAG); //-- error output filename (Mohan)
	    flags.add("d");             //-- Show DOM Package info
	    
	    boolean showDOMPackage = false;
	    
	    Hashtable options = getOptions(args,flags);
	    
	    // -- display usage
	    if ((options == null) || (options.size() == 0)) {
	        printUsage(System.out);
	        return;	        
	    }
	    //-- display help
	    else if (options.containsKey(HELP_FLAG)) {
	        printHelp(System.out);
	        return;
        }
	    //-- display version
	    else if (options.containsKey(VERSION_FLAG)) {
	        System.out.println(appName + " " +appVersion);
	        return;
	    }
	    else {
	        xslFile        = (String)options.get(STYLESHEET_FLAG);
	        xmlFile        = (String)options.get(INPUT_FLAG);
	        outFile        = (String)options.get(OUTPUT_FLAG);
	        fclass         = (String)options.get(FORMATTER_FLAG);
	        showDOMPackage = options.containsKey("d");
	        
	        //-- set error output file (Mohan Embar)
            if ((errFile = (String)options.get(ERR_OUTPUT_FLAG)) != null)
            {
	            try {
	                errWriter = new FileWriter(errFile);
	            }
	            catch (java.io.IOException ioe) {
	                printError("Cannot open error file: " + errFile, false); 
	            }
            }
            
            //-- set XML file
	        if (xmlFile == null) {
    	        printError("XML filename missing.", true);            
	        }	        
	        /* I removed this due to a suggestion from
	         * Warren Hedley (Auckland University)
	         * it now dumps to stdout if no file is given
	         */
	        /*
	        else if (outFile == null) {
    	        printError("Output filename missing.", true);	            
	        }
	        */
	    }
	    

	    OutputStream fileOutput = null;
	    if (outFile == null)
	        fileOutput = System.out; // dump to stdout (WH)
	    else {
	        try {
	            fileOutput = new FileOutputStream(outFile);
	        }
	        catch (java.io.IOException ex) {
	            System.out.println("XSLProcessor error: " + ex.getMessage());
	        }
	    }
	    
	    Formatter formatter = null;
	    if (fclass != null) {
	        try {
	            Class formatterClass = Class.forName(fclass);
	            formatter = (Formatter) formatterClass.newInstance();
	        }
	        catch(ClassNotFoundException cnfe) {
	            System.out.print("Unable to find the specified Formatter: ");
	            System.out.println(fclass);
	            System.out.println("-- processing with default Formatter.");
	        }
	        catch(Exception e) {
	            System.out.print("Unable to instantiate the specified Formatter: ");
	            System.out.println(fclass);
	            System.out.println("-- processing with default Formatter.");
	        }	        	        
	    }
	    
	    
    	XSLProcessor xslp = new XSLProcessor();
    	xslp.setValidation(options.containsKey(VALIDATE_FLAG));
    	if (errWriter != null) xslp.setErrorStream(errWriter);
    	
    	if (showDOMPackage) {
    	    System.out.print(appName + " invoked using ");
    	    System.out.println(xslp.getDOMPackageName());
    	}
    	xslp.process(xmlFile, xslFile, new PrintWriter(fileOutput), formatter);
    	if (errWriter != null) {
    	    try {
	            errWriter.close();
	        }
	        catch(java.io.IOException ioe) {};
	    } 
	   
	    /* <performance-testing> *
	    System.out.println();
	    System.out.print("Total XSL:P/ Time: ");
	    System.out.print(System.currentTimeMillis()-stime);
	    System.out.println(" (ms)\n");
	    /* </performance-testing> */
	}  //-- main

    /**
     * Processes the specified xml file, using the stylesheet specified 
     * by the xml stylesheet PI, and the default Formatter. 
     * All results are sent to the Writer.
     * @param xmlFilename the path to the XML file to process
     * @param out the Writer to print all processing results to.
    **/
    public void process(String xmlFilename, PrintWriter out) {        
        process(xmlFilename, null, out);
    }

    /**
     * Processes the specified xml file, using the stylesheet specified 
     * by the xml stylesheet PI. 
     * All results are sent to the Writer.
     * @param xmlFilename the path to the XML file to process
     * @returns the resulting Document
    **/
    public Document process(String xmlFilename) {
        String xslFile = null;
        return process(xmlFilename, xslFile);
    }
    
    /**
     * Processes the specified xml file, using the specified xsl file, and
     * the default Formatter. 
     * All results are sent to the Writer.
     * @param xmlFilename the href to the XML file to process
     * @param xslFilename the href to the XSL file to use for processing.
     * This stylesheet will supercede any embedded stylesheets in the
     * xsl document. Set to null, to allow xml:stylesheet PI to be processed.
     * @param out the PrintWriter to print all processing results to.
    **/
    public void process
        (String xmlFilename, String xslFilename, PrintWriter out) 
    {
        process(xmlFilename, xslFilename, out, null);
    } //-- process
            
    
    /**
     * Processes the specified xml file, using the specified xsl file, and
     * the desired Formatter. 
     * All results are sent to the PrintWriter.
     * @param xmlFilename the path to the XML file to process
     * @param xslFilename the path to the XSL file to use for processing.
     * This stylesheet will supercede any embedded stylesheets in the
     * xsl document.
     * @param out the PrintWriter to print all processing results to.
     * @param formatter the FlowObjectHandler to use for processing the
     *  Stylesheet
    **/    
    public void process(String xmlFilename, String xslFilename,
                        PrintWriter out, Formatter formatter) {
        
        InputStream xmlInput = null;
        InputStream xslInput = null;
        
        // Create XML InputSream                            
        
        String xmlHref = URIUtils.resolveHref(xmlFilename, documentBase);
        try {
            xmlInput = URIUtils.getInputStream(xmlFilename, documentBase);
        }
        catch(Exception e) {
            errorWriter.println("unable to read XML Document " + xmlFilename);
            errorWriter.println(" -- processing aborted.");
            return;           
        }        
        
        // Create XSL InputStream
        if ((xslFilename != null) && (xslFilename.length() > 0)) {            
            try {
                xslInput = URIUtils.getInputStream(xslFilename, documentBase);
            }
            catch(Exception e) {
                errorWriter.println("Error reading stylesheet");
                errorWriter.println(" -- " + e.getMessage());
                errorWriter.println(" -- processing with default rules");
                errorWriter.flush();
            }
        }
        
		XSLPIHandler piHandler = new XSLPIHandler();		
		piHandler.setDocumentBase(URIUtils.getDocumentBase(xmlHref));
		Document xmlDocument = readXMLDocument(xmlInput,xmlHref);
		parsePIs(xmlDocument, piHandler);
	    XSLStylesheet xsl = readXSLStylesheet(xslInput,xslFilename,piHandler);
	    process(xmlDocument, xsl, out, formatter);
	    
    } //--process
           
    /**
     * Processes the specified xml file, using the specified xsl file. 
     * @param xmlFilename the path to the XML file to process
     * @param xslFilename the path to the XSL file to use for processing.
     * This stylesheet will supercede any embedded stylesheets in the
     * xsl document.
     * @return the resulting Document
    **/    
    public Document process(String xmlFilename, String xslFilename) {
        
        InputStream xmlInput = null;
        InputStream xslInput = null;
        
        // Create XML InputSream                            
        try {
            xmlInput = URIUtils.getInputStream(xmlFilename, documentBase);
        }
        catch(Exception e) {
            errorWriter.println(e.getMessage());
            errorWriter.println("processing aborted.");
            return null;           
        }        
        
        // Create XSL InputStream
        if ((xslFilename != null) && (xslFilename.length() > 0)) {            
            try {
                xslInput = URIUtils.getInputStream(xslFilename, documentBase);
            }
            catch(Exception e) {
                errorWriter.println("Error reading stylesheet");
                errorWriter.println(" -- " + e.getMessage());
                errorWriter.println(" -- processing with default rules");
            }
        }
        
        
		XSLPIHandler piHandler = new XSLPIHandler();		
		String xmlHref = URIUtils.resolveHref(xmlFilename, documentBase);
		piHandler.setDocumentBase(URIUtils.getDocumentBase(xmlHref));
		Document xmlDocument = readXMLDocument(xmlInput,xmlHref);
		parsePIs(xmlDocument, piHandler);
	    XSLStylesheet xsl = readXSLStylesheet(xslInput,xslFilename,piHandler);
	    
	    return process(xmlDocument, xsl);                          
    } //-- process

    /**
     * Processes the specified xml InputStream, using the specified xsl 
     * InputStream. 
     * @param xmlInput the InputStream of the XML to process
     * @param xslInput the InputStream of the XSL to use for processing.
     * This stylesheet will supercede any embedded stylesheets in the
     * xsl document. Set to null, to allow xml:stylesheet PI to be processed.
     * @return the resulting Document
    **/    
	public Document process (InputStream xmlInput, InputStream xslInput) {
	    
		Document xmlDocument = readXMLDocument(xmlInput,null);
		XSLPIHandler piHandler = new XSLPIHandler();
		parsePIs(xmlDocument,piHandler);
	    XSLStylesheet xsl = readXSLStylesheet(xslInput,null,piHandler);	    
        return process(xmlDocument, xsl);
        
	} //--process
	
    /**
     * Processes the specified xml InputStream, using the specified xsl 
     * InputStream, and the default Formatter. 
     * All results are sent to the specified PrintWriter.
     * @param xmlInput the InputStream of the XML to process
     * @param xslInput the InputStream of the XSL to use for processing.
     * This stylesheet will supercede any embedded stylesheets in the
     * xsl document. Set to null, to allow xml:stylesheet PI to be processed.
     * @param out the PrintWriter to print all processing results to.
    **/    
	public void process (InputStream xmlInput, InputStream xslInput, PrintWriter out) {
        process(xmlInput, xslInput, out, null);
	}

    /**
     * Processes the specified xml InputStream, using the specified xsl 
     * InputStream, and the desired Formatter. 
     * All results are sent to the specified PrintWriter.
     * @param xmlInput the InputStream of the XML to process
     * @param xslInput the InputStream of the XSL to use for processing.
     * This stylesheet will supercede any embedded stylesheets in the
     * xsl document. Set to null, to allow xml:stylesheet PI to be processed.
     * @param out the PrintWriter to print all processing results to.
     * @param formatter the Formatter to use for processing the stylesheet
    **/    
	public void process( InputStream xmlInput, InputStream xslInput,
		                 PrintWriter out, Formatter formatter) {
		                    
		Document xmlDocument = readXMLDocument(xmlInput,null);
		XSLPIHandler piHandler = new XSLPIHandler();
		parsePIs(xmlDocument,piHandler);
	    XSLStylesheet xsl = readXSLStylesheet(xslInput,null,piHandler);	    
		          
		process(xmlDocument, xsl, out, formatter);
    } //-- process
    
    
    /**
     * Processes the specified xml (DOM) Document, using the specified 
     * (DOM) xsl stylesheet.
     * @param xmlDocument the XML Document to process
     * @param xslDocument the XSL Document to use for processing.
     * @return the resulting Document
    **/    
    public Document process(Document xmlDocument, Document xslDocument) {
        XSLStylesheet xslStylesheet = null;
        if (xslDocument == null) {
		    XSLPIHandler piHandler = new XSLPIHandler();
		    parsePIs(xmlDocument,piHandler);
		    xslStylesheet = readXSLStylesheet(null, null, piHandler);
        }
        else xslStylesheet = readXSLStylesheet(xslDocument,"XSL Document");
        return process(xmlDocument, xslStylesheet);
    } //--process

    /**
     * Processes the specified xml (DOM) Document, using the specified 
     * xsl (DOM) Document and the default Formatter. 
     * All results are sent to the specified PrintWriter.
     * @param xmlDocument the XML Document to process
     * @param xslDocument the XSL Document to use for processing.
     * @param out the PrintWriter to print all processing results to.
    **/    
    public void process(Document xmlDocument, Document xslDocument,
                        PrintWriter out) {
        process(xmlDocument, xslDocument, out, null);
    } //-- process
    
    /**
     * Processes the specified xml (DOM) Document, using the specified 
     * xsl (DOM) Document and the desired Formatter. 
     * All results are sent to the specified PrintWriter.
     * @param xmlDocument the XML Document to process
     * @param xslDocument the XSL Document to use for processing.
     * @param out the PrintWriter to print all processing results to.
     * @param formatter the desired Formatter to use during processing
    **/        
    public void process(Document xmlDocument, Document xslDocument,
                        PrintWriter out, Formatter formatter) 
    {
        XSLStylesheet xslStylesheet = null;
        if (xslDocument == null) {
            // look for stylesheet PI
		    XSLPIHandler piHandler = new XSLPIHandler();
		    parsePIs(xmlDocument,piHandler);
		    xslStylesheet = readXSLStylesheet(null, null, piHandler);
        }
        else xslStylesheet = readXSLStylesheet(xslDocument,"XSL Document");
        process(xmlDocument, xslStylesheet, out, formatter);
    } //-- process
    
    /**
     * Processes the specified xml (DOM) Document, using the specified xsl 
     * stylesheet.
     * @param xmlDocument the XML Document to process
     * @param stylesheet the XSLStylesheet to use for processing.
     * @return the resulting Document
    **/    
    public Document process(Document xmlDocument, XSLStylesheet stylesheet) {
        RuleProcessor ruleProcessor 
            = new RuleProcessor(stylesheet, domReader.getDOMPackage());
        ruleProcessor.addMessageObserver(this);
		ruleProcessor.setErrorWriter(errorWriter);
		return ruleProcessor.process(xmlDocument);
    } //--process

    /**
     * Processes the specified xml (DOM) Document, using the specified xsl 
     * stylesheet, and the default Formatter. 
     * All results are sent to the specified PrintWriter.
     * @param xmlDocument the XML Document to process
     * @param stylesheet the XSLStylesheet to use for processing.
     * @param out the PrintWriter to print all processing results to.
    **/    
    public void process(Document xmlDocument, XSLStylesheet stylesheet,
                        PrintWriter out) {
        process(xmlDocument, stylesheet, out, null);
    } //-- process
    
    /**
     * Processes the specified xml (DOM) Document, using the specified xsl 
     * stylesheet, and the desired Formatter. 
     * All results are sent to the specified PrintWriter.
     * @param xmlDocument the XML Document to process
     * @param stylesheet the XSLStylesheet to use for processing.
     * @param out the PrintWriter to print all processing results to.
     * @param formatter the desired Formatter to use during processing
    **/        
    public void process(Document xmlDocument, XSLStylesheet stylesheet,
                        PrintWriter out, Formatter formatter) 
    {
        Document resultDoc = process(xmlDocument, stylesheet);
        
        if (resultDoc == null) {
            errorWriter.println("error: an empty document was generated.");
            errorWriter.flush();
            return;
        }
        
        Output output = stylesheet.getOutput();
        
        String method = output.getMethod();
        
        //-- determine default output method
        if ((method == null) || (method.length() == 0)) {
            String name = resultDoc.getDocumentElement().getNodeName();
            if (HTML.equalsIgnoreCase(name)) method = HTML;
            else method = "xml";
        }
        
        if (formatter == null) {
            // Use properites to determine formatter
            // (Franck Mangin)
			String className = props.getProperty(FORMATTER_PREFIX+method);
			if ((className != null) && (className.length() > 0)) {
				try {
					Class formatterClass = Class.forName(className);
					formatter = (Formatter)formatterClass.newInstance();
				} 
	            catch(ClassNotFoundException cnfe) {
	                errorWriter.print("Unable to find the specified Formatter: ");
	                errorWriter.println(className);
	                errorWriter.println("-- processing with default Formatter.");
	            }
	            catch(Exception e) {
	                errorWriter.print("Unable to instantiate the specified Formatter: ");
	                errorWriter.println(className);
	                errorWriter.println("-- processing with default Formatter.");
	            }	        	        
            }
        }
            
        //-- if no formatter was specified, or no valid
        //-- formatter was found use default formatters
        if (formatter == null) {
            if ((method.equals(HTML)) || 
                (method.indexOf(HTML_RESULT_NS) == 0)) {
                formatter = new BasicHTMLFormatter();
                //-- check for indent
                //-- use indent by default for HTML printing 
                //-- I am using the the attribute value to make sure
                //-- one was specified (instead of using Output::getIndent())
                String indentResult = output.getAttribute(Names.INDENT_ATTR);
                formatter.setIndentResult(!Names.NO_VALUE.equals(indentResult));
            }
            else {
                formatter = new DefaultFormatter();
                formatter.setIndentResult(output.getIndent());
            }
        }
        else formatter.setIndentResult(output.getIndent());
            
        //-- Set indentation from Properties file
        //-- (suggested by Aaron Metzger)
        String indentSize = props.getProperty(INDENT_SIZE);
        if (indentSize != null) {
            try {
                formatter.setIndentSize(Integer.parseInt(indentSize));
            }
            catch(NumberFormatException nfe) {};
        }
        formatter.process(resultDoc,out, output);
    } //-- process
    
    /**
     * Recieves a message
     * @param message the message to recieve
     * @see com.kvisco.xsl.util.MessageObserver
    **/
    public void recieveMessage(String message) {
        for (int i = 0; i < msgObservers.size(); i++) {
            ((MessageObserver)msgObservers.get(i)).recieveMessage(message);
        }
    } //-- recieveMessage
    
    /**
     * Removes the given MessageObserver from this processors list
     * of MessageObservers
     * @param msgObserver the MessageObserver to remove from this processors
     * list of MessageObservers
     * @return the given MessageObserver if it was removed from the list,
     * otherwise return null
    **/
    public MessageObserver removeMessageObserver
        (MessageObserver msgObserver) 
    {
        if (msgObservers.remove(msgObserver)) return msgObserver;
        else return null;
    } //-- addMessageObserver

    /**
     * Sets the document base for resolving relative URLs
     * @param documentBase the document base to use while processing.
    **/
    public void setDocumentBase(String documentBase) {
        this.documentBase = documentBase;
    }
    
    /**
     * Sets the DOMReader that will be used to read in XML Documents
     * @param domReader the DOMReader that is to be used to read in
     * XML Documents
    **/
    public void setDOMReader(DOMReader domReader) {
        this.domReader = domReader;
        this.domPackageName = domReader.getDOMPackageClassName();
    } //-- setDOMReader
    
    /**
     * Sets the stream for this XSLProcessor to print errors to.
     * @param errorStream the PrintStream to print all errors to.
    **/
    public void setErrorStream(PrintStream errorStream) {
        this.errorWriter = new PrintWriter(errorStream, true);
    } //-- setErrorStream
    
    /**
     * Sets the stream for this XSLProcessor to print errors to.
     * @param errorWriter the Writer to print all errors to.
    **/
    public void setErrorStream(Writer errorWriter) {
        this.errorWriter = new PrintWriter(errorWriter,true);
    } //-- setErrorStream
    
    /**
     * Sets the property value associated with the given String.
     * @property the property to set
     * @value the value of the property
     * <BR>See xslp.properties for for a list of properties
    **/
    public void setProperty(String property, String value) {
        props.put(property, value);
        if (DOM_PACKAGE.equals(property)) {
            initializeDOMReader();
        }
    } //-- setProperty
    
    /**
     * Sets whether or not to validate when reading an XML document.
     * @param validate the boolean indicating whether to validate or not
     * @since 19990408 
    **/
    public void setValidation(boolean validate) {
        this.validate = validate;
    } //-- setValidation
    
      //-------------------/
     //- Private Methods -/
	//-------------------/
	
	/**
	 * parses the arguments into a hashtable with the proper flag
	 * as the key
	**/
	private static Hashtable getOptions(String[] args, List flags) {
	    Hashtable options = new Hashtable();
	    String flag = null;
	    for (int i = 0; i < args.length; i++) {
	        
	        if (args[i].startsWith("-")) {
	            	
	            // clean up previous flag
	            if (flag != null) {
	                options.put(flag,args[i]);
	                options.put(new Integer(i),args[i]);
	            }
	            // get next flag
	            flag = args[i].substring(1);
	            
	            //-- check full flag, otherwise try to find
	            //-- flag within string
	            if (!flags.contains(flag)) {
	                int idx = 1;
	                while(idx <= flag.length()) {
	                    if (flags.contains(flag.substring(0,idx))) {
	                        if (idx < flag.length()) {
	                            options.put(flag.substring(0,idx),
	                                flag.substring(idx));
	                            break;
	                        }
	                    }
	                    else if (idx == flag.length()) {
	                        printError("invalid option: -" + flag, true);
	                    }
	                    ++idx;
	                }// end while
	            }
	            
	        }// if flag
	        else {
	            // Store both flag key and number key
	            if (flag != null) options.put(flag,args[i]);
	            options.put(new Integer(i),args[i]);
	            flag = null;
	        }
	        
	    }// end for
	    if (flag != null) options.put(flag, "no value");
	    return options;
	} //-- getOptions

    /**
     * Retrieves Processing Instructions from the given document
     * and hands them off to the given XSLPIHandler
    **/
    private void parsePIs
        (Document document, XSLPIHandler piHandler) {
        if ((document == null) || (piHandler == null)) return;
        
        NodeList nl    = document.getChildNodes();
        Node node;
        
        for (int i = 0; i < nl.getLength(); i++) {
            node = nl.item(i);
            if (node.getNodeType() == Node.PROCESSING_INSTRUCTION_NODE) {
                piHandler.handlePI((ProcessingInstruction)node);
            }
        }
    } //-- parsePIs
    
	/**
	 * prints the usage information for this application
	 * @param ps the PrintStream to print the usage information to
	**/
	private static void printUsage(PrintStream ps) {
	    printUsage(ps,true);
	} //-- printUsage
	
	/**
	 * prints the usage information for this application
	 * @param ps the PrintStream to print the usage information to
	 * @param showAppInfo boolean to indicate whether or not to show
	 * the name and version of the application with the usage. If true
	 * the appInfo will be shown.
	**/
	private static void printUsage(PrintStream ps, boolean showAppInfo) {
	    
	    ps.println();
	    if (showAppInfo) ps.println(getAppInfo());
	    ps.println("usage:");
	    ps.print("java com.kvisco.xsl.XSLProcessor -");
	    ps.print(INPUT_FLAG + " xml-file [-");
	    ps.print(STYLESHEET_FLAG + " xsl-file] [-");
	    ps.println(OUTPUT_FLAG + " output-file] [-val]");
	    ps.println();
	    ps.println("for more infomation use the -" + HELP_FLAG + " flag");
	} //-- printUsage
	
	/**
	 * prints the Help for this application
	 * @param ps the PrintStream to print the help information to
	**/
	private static void printHelp(PrintStream ps) {
	    ps.println();
	    ps.println(getAppInfo());
	    ps.print("The following flags are available for use with ");
	    ps.println(appName + " -");
	    ps.println();
	    ps.println("-i  filename  : The XML file to process");
	    ps.println("-o  filename  : The Output file to create");	    
	    ps.println("-s  filename  : The XSL file to use for processing  (Optional)");
	    ps.println("-fo classname : Formatter (for Formatting Objects)  (Optional)");
	    ps.println("-val          : Turns on xml validation if using    (Optional)");
	    ps.println("                a validating DOM parser");
	    ps.println("-h            : This help screen                    (Optional)");
	    ps.println();	    	    
	} // printHelp
	
	/**
	 * Prints an error message to the screen and if specified terminates
	 * the application.
	 * @param message the error message to display
	 * @param exit a boolean indicating whether to exit the application. If
	 * true, the application is terminated.
	**/
	private static void printError(String message, boolean exit) {
	    System.out.println();
	    System.out.println(getAppInfo());
	    System.out.println("#error: " + message);
	    printUsage(System.out, false);
	    if (exit) System.exit(0);
	} //--printError
	
	/**
	 * Reads an XML Document from the given InputStream. 
	 * @param xmlInput the XML InputStream
	 * @param filename the filename to use during error reporting.
	 * @param piHandler the XSLPIHandler to use. If this is set to null, 
	 * no XSLPIHandler will be used.
	**/
	private Document readXMLDocument(InputStream xmlInput, String filename) {
	        
	    /* <performance-testing> *
	    System.out.println ("read XML document: "+filename);
	    long stime = System.currentTimeMillis();
	    /* </performance-testing> */
	    
        // Read in source xml document
        if (filename == null) {
            // TLT: added documentBase to the filename
            if (documentBase != null) 
                filename = documentBase + "XML Document";
            else 
                filename = "XML Document";
        }
        
        BufferedInputStream bin = new BufferedInputStream(xmlInput);
        
		
		return domReader.readDocument(bin, filename, validate, errorWriter);
		    
	    /* <performance-testing> *
		Document document 
		    = domReader.readDocument(bin, filename, validate, errorWriter);
	    System.out.print("- done reading XML document: ");
	    System.out.print(System.currentTimeMillis()-stime);
	    System.out.println(" (ms)");
	    return document;
	    /* </performance-testing> */
	    
	} //-- readXMLDocument

	/**
	 * Reads an XSL Document from the given InputStream. 
	 * @param xslInput the XSL InputStream, if this stream is null,
	 * this method will check the XSLPIHandler for the stylesheet href.
	 * @param filename the filename to use during error reporting.
	 * @param piHandler the XSLPIHandler to use. If this is set to null, 
	 * no XSLPIHandler will be used.
	 * 
	 * <BR>Changes:</BR>
	 * -- Added Warren Hedley's fix for no stylesheet being
	 *    specified or possible error when opening the file
	**/
	private XSLStylesheet readXSLStylesheet
	    (InputStream xslInput, String filename, XSLPIHandler piHandler) 
	{
	    /* <performance-testing> *
	    System.out.println ("read XSL stylesheet: "+filename);
	    long stime = System.currentTimeMillis();
	    /* </performance-testing> */
	    
	    InputStream xslStream = xslInput;
        // look for xml:stylesheet PI. Order of precedence is
        // * xslStream - if it exists
        // * filename - specified with -s on command line
        // * PI in xml document - contained in piHandler
        if (xslStream == null) {
            if (filename == null) {
                if (piHandler != null) {
                    filename = URIUtils.resolveHref
                              ( piHandler.getStylesheetHref(),
                                piHandler.getDocumentBase() );
                }
            }
            if (filename != null) {
                try {
                    xslStream = URIUtils.getInputStream(filename, documentBase);
                }
                catch(ExceptionInInitializerError eiie) {
                    errorWriter.println(eiie.getException());
                    try { 
                        xslStream.close();
                    }
                    catch(IOException ioe) {};
                }
                catch(Exception e) {
                    errorWriter.println("Error reading stylesheet\n -- ");
                    errorWriter.println(e);
                    errorWriter.println(" -- processing with default rules");
                    try { 
                        xslStream.close();
                    }
                    catch(IOException ioe) {};
                }
            }
            else {
                errorWriter.println("No stylesheet specified. Use `-s' option on");
                errorWriter.println("the command line, or `xml:stylesheet' PI");
                errorWriter.println("in XML document");
                errorWriter.println(" -- processing with default rules");
            }
        }

        // create stylesheet
        XSLStylesheet xsl = null;
        if (xslStream != null) {
         
            XSLReader xslReader = new XSLReader(domReader);
            //-- added #setErrorStream 
            //-- Mohan Embar, Michel CASABIANCA
            xslReader.setErrorStream(errorWriter); 
            try {
                xsl = xslReader.readStylesheet(xslStream, filename);
            }
            catch (XSLException xslEx) {
                errorWriter.println(xslEx.getMessage());
                xsl = new XSLStylesheet();
            }
        }
        else xsl = new XSLStylesheet();
        
	    /* <performance-testing> *
	    System.out.print("- done reading XSL stylesheet: ");
	    System.out.print(System.currentTimeMillis()-stime);
	    System.out.println(" (ms)");
	    /* </performance-testing> */
        
        return xsl;
    } //-- readXSLStylesheet

	/**
	 * Reads the given XSL Document into an XSLStylesheet
	 * @param xslDocument the Stylesheet Document to create
	 * an XSLStylesheet from
	 * @param filename the filename to use during error reporting.
	**/
	private XSLStylesheet readXSLStylesheet
	    (Document xslDocument, String filename)
	{
        // create stylesheet
        XSLStylesheet xsl = null;
        if (xslDocument != null) {
            XSLReader xslReader = new XSLReader(domReader);
            //-- added #setErrorStream 
            //-- Mohan Embar, Michel CASABIANCA
            xslReader.setErrorStream(errorWriter); 
            try {
                xsl = xslReader.readStylesheet(xslDocument, filename);
            }
            catch (XSLException xslEx) {
                errorWriter.println(xslEx.getMessage());
                errorWriter.println(" -- processing with default rules");
                xsl = new XSLStylesheet();
            }
        }
        else xsl = new XSLStylesheet();
        return xsl;
  } //-- readXSLStylesheet
  
  
} //-- XSLProcessor