/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.scripting;

import com.kvisco.xsl.*;

import org.w3c.dom.*;
import FESI.jslib.*;
import FESI.Data.ESWrapper;
import FESI.Data.JSWrapper;
import FESI.Data.FunctionPrototype;

import java.util.Hashtable;


/**
 * An implementation of ScriptHandler for ECMAScript
 * @see com.kvisco.xsl.ScriptHandler
 * @author Keith Visco (kvisco@ziplink.net)
**/
public class ECMAScriptHandler implements ScriptHandler {

    private static final String NITIALIATION_ERROR =
        "ECMAScript initialization error";
        
    private static final String UNINITIALIZED_ERROR =
        "ECMAScript environment has not been initialized.";
    
    public static final String ECMASCRIPT = "ECMAScript";

    private static String[] jsExtensions =
    {
        "FESI.Extensions.JavaAccess",
        "FESI.Extensions.BasicIO",
        "FESI.Extensions.OptionalRegExp"
    };        
        
      //----------------------/
     //- Instance Variables -/
    //----------------------/

    JSGlobalObject jsGlobal = null;
    ProcessorCallback pc = null;
    
    Hashtable namespaces = null;
    
      //----------------/
     //- Constructors -/
    //----------------/

    //-- A default Constructor is needed so that Class.newInstance may
    //-- be used
    /**
     * Creates a new ECMAScriptHandler. This handler will not
     * be initialized. A call to #initialize will be needed
     * before using.
    **/
    public ECMAScriptHandler() {
        super();
    }
    
    public ECMAScriptHandler(ProcessorCallback pc) {
        super();
        initialize(pc);
    } //-- ECMAScriptHandler

      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Calls the method with the given name, and set of arguments
     * @param name the name of the method to call
     * @param args the methods arguments
     * @return the result of the method invocation
    **/
    public Object call(String name, Object[] args) {
        return call(name, args, null); 
    } //-- call

    /**
     * Calls the method with the given name, and set of arguments
     * @param name the name of the method to call
     * @param args the methods arguments
     * @param namespace the Namespace to use for evaluation
     * @return the result of the method invocation
    **/
   public Object call(String name, Object[] args, String namespace) {
        if (jsGlobal == null) return UNINITIALIZED_ERROR;
        
        JSGlobalObject nsObject = getNSObject(namespace);
        try {
            return nsObject.call( name, args );
        }
        catch(JSException jse) {
            return jse.getMessage();
        }
    } //-- call
    
    /**
     * Creates a Function Namespace using the given name
     * @param name the name of the Namespace to create
    **/
    public boolean createNamespace(String name) {
        if (name == null) return false;
        JSGlobalObject nsObject = (JSGlobalObject)namespaces.get(name);
        if (nsObject == null) {
            try {
                nsObject = JSUtil.makeEvaluator(jsExtensions);
                namespaces.put(name, nsObject);
            }
            catch (JSException jsException) {
                return false;
            }
        }
        return true;
    } //-- createNamespace
    
    /**
     * Returns the name of the language that this ScriptHandler handles
     * @return the name of the language that this ScriptHandler handles
    **/
    public String getLanguage() {
        return ECMASCRIPT;
    } //-- getLanguage
    
    public boolean hasDefinedFunction(String name, String namespace) {
        if (jsGlobal == null) return false;
        
        JSGlobalObject nsObject = getNSObject(namespace);
        try {
            Object obj = nsObject.getMember(name);
            if (obj instanceof JSWrapper) {
                obj = ((JSWrapper)obj).getESObject();
                return (obj instanceof FunctionPrototype);
            }
            return (obj instanceof JSFunction);
        }
        catch(JSException jsException) {};
        return false;
    } //-- hasDefinedFunction
    
    /**
     * Initializes the scripting environment.
    **/
    public void initialize(ProcessorCallback processorCallback) {
        namespaces = new Hashtable();
        this.pc = processorCallback;
	    // set up scripting environment
        try {
            jsGlobal = JSUtil.makeEvaluator(jsExtensions);
            
            // Create Global XSLP object
            JSObject xslpObj = jsGlobal.makeJSObject();
            jsGlobal.setMember("XSLP",xslpObj);
            
            // Declare API Functions
            // addToResultTree
            xslpObj.setMember("addToResultTree", new JSFunctionAdapter() { 
                public Object doCall(JSObject thisObject, Object args[]) throws
                    JSException { 
                    if (args.length == 0) 
                        throw new JSException("addToResultTree: missing Node argument!");
                    addToResultTree(((ESWrapper)args[0]).toJavaObject());
                    return "";
                } 
            }); 
            // createElement
            xslpObj.setMember("createElement", new JSFunctionAdapter() { 
                public Object doCall(JSObject thisObject, Object args[]) throws
                    JSException { 
                    if (args.length == 0) 
                        throw new JSException("createElement: missing name argument!");
                    return pc.createElement(args[0].toString());
                } 
            }); 
            // createText
            xslpObj.setMember("createText", new JSFunctionAdapter() { 
                public Object doCall(JSObject thisObject, Object args[]) throws
                    JSException { 
                    if (args.length == 0) 
                        throw new JSException("createText: missing name argument!");
                    return pc.createText(args[0].toString());
                } 
            }); 
        }
        catch (JSException e) {
            pc.printError("Unable to create ECMAScript interpreter: " + e);
            pc.printError("-- xsl:script elements will be ignored.");
        }
    } //-- initialize

    /**
     * Evaluates the given XSLScript element using the default namespace
     * @param xslScript the XSLScript to evaluate
     * @param context the current DOM Node that is the context
     * of this evaluation.
     * @return the result of the XSLScript evaluation
    **/
    public Object eval(XSLScript xslScript, Node current) {
        return eval(xslScript, current, null);
    } //-- eval
    
    /**
     * Evaluates the given XSLScript element using the given namespace
     * @param xslScript the XSLScript to evaluate
     * @param context the current DOM Node that is the context
     * of this evaluation.
     * @param namespace the Namespace to use for evaluation
     * @return the result of the XSLScript evaluation
    **/
    public Object eval(XSLScript xslScript, Node current, String namespace) 
    {
        if (jsGlobal == null) return UNINITIALIZED_ERROR;
        
        JSGlobalObject nsObject = getNSObject(namespace);
        
        JSObject jsObj = nsObject.makeObjectWrapper(current);
        try {
            return jsObj.eval( xslScript.getData() );
        }
        catch(JSException jse) {
            return jse.getMessage();
        }
    } //-- eval

    /**
     * Evaluates the given XSLScript element as a function using the
     * default namespace.
     * @param xslScript the XSLScript to evaluate
     * @param context the current DOM Node that is the context
     * of this evaluation.
     * @param namespace the Namespace to use for evaluation
     * @return the result of the XSLScript evaluation
    **/
    public Object evalAsFunction(XSLScript xslScript, Node current) {
        return evalAsFunction(xslScript, current, null);
    } //-- evalAsFunction
    
    /**
     * Evaluates the given XSLScript element as a function using the
     * given namespace.
     * @param xslScript the XSLScript to evaluate
     * @param context the current DOM Node that is the context
     * of this evaluation.
     * @param namespace the Namespace to use for evaluation
     * @return the result of the XSLScript evaluation
    **/
    public Object evalAsFunction
        (XSLScript xslScript, Node current, String namespace) 
    {
        if (jsGlobal == null) return UNINITIALIZED_ERROR;
        
        JSGlobalObject nsObject = getNSObject(namespace);
        
        JSObject jsObj = nsObject.makeObjectWrapper(current);
        try {
            return jsObj.evalAsFunction( xslScript.getData() );
        }
        catch(JSException jse) {
            return jse.getMessage();
        }
    } //-- evalAsFunction
    
      //-------------------/
     //- private methods -/
    //-------------------/

    private JSGlobalObject getNSObject(String namespace) {
        if (namespace == null) return jsGlobal;
        JSGlobalObject nsObject = (JSGlobalObject)namespaces.get(namespace);
        return (nsObject == null) ? jsGlobal : nsObject;
    } //-- getNSObject
 
      //------------------------/
     //- ECMAScript API calls -/
    //------------------------/
    
    /**
     * Adds the given Node to the Result tree
     * @param node the Node to add to the result tree
    **/
    private void addToResultTree(Object obj) {
        if (pc == null) return;
        
        if (obj instanceof Node)
            pc.addToResultTree((Node)obj);
    }
    
} //-- ECMAScriptHandler
