/* $Id: verbose.c,v 1.5 1996/11/12 08:04:35 sverrehu Exp $ */
/**************************************************************************
 *
 *  FILE            verbose.c
 *  MODULE OF       shhmsg - library for displaying messages.
 *
 *  DESCRIPTION     Routines for displaying text according to a verbosity
 *                  level. All messages with a level less than or equal
 *                  to the current level are displayed.
 *
 *  WRITTEN BY      Sverre H. Huseby <sverrehu@ifi.uio.no>
 *
 **************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#include "internal.h"
#include "shhmsg.h"

/**************************************************************************
 *                                                                        *
 *                        P U B L I C    D A T A                          *
 *                                                                        *
 **************************************************************************/

int msgVerboseLevel = 0;                       /* current verbosity level */



/**************************************************************************
 *                                                                        *
 *                   P R I V A T E    F U N C T I O N S                   *
 *                                                                        *
 **************************************************************************/

/*-------------------------------------------------------------------------
 *
 *  NAME          msgLookupVerboseLevel
 *
 *  FUNCTION      Get a default verbose level from the environment.
 *
 *  RETURNS       Default verbosity level.
 *
 *  DESCRIPTION   The environment variable "APPNAME_LEVEL" is checked
 *		  for a numeric value, and that becomes the verbosity
 *		  level, if none has been set in code, or the code
 *		  has not set "default".  The APPNAME is truncated to
 *		  10 bytes, and converted to all upper case.
 */
static int
msgLookupVerboseLevel(void)
{
    char envName[20];
    char *envVal;

    strncpy(envName, msgGetName(), 10);
    envName[10] = '\0';
    
    /* force the string to upper case */
    for (envVal = envName; *envVal; envVal++)
	if (islower(*envVal))
	    *envVal = toupper(*envVal);

    strcat(envName, "_LEVEL");
    envVal = getenv(envName);
    if (envVal == NULL || *envVal == '\0')
	return MSG_VERBOSE_DEFAULT;
    return atoi(envVal);
}



/**************************************************************************
 *                                                                        *
 *                    P U B L I C    F U N C T I O N S                    *
 *                                                                        *
 **************************************************************************/

/*-------------------------------------------------------------------------
 *
 *  NAME          msgSetVerbose
 *
 *  FUNCTION      Set verbosity level.
 *
 *  SYNOPSIS      #include "shhmsg.h"
 *                void msgSetVerbose(int level);
 *
 *  INPUT         level   The new verbosity level. Note that a negative
 *		          value will make the system print nothing, assuming
 *		          all comparisons passed in by the caller to
 *			  msgVerbose are positive. A negative number will
 *			  revert the code to the "Default" state of checking
 *			  the environmental variable for it's verbosity level,
 *			  allowing changes at run time, instead of compile
 *			  time.
 *
 *  DESCRIPTION   Used in conjunction with the msgVerbose() function.
 *                Note that changing the msgVerboseLevel-variable directly
 *                is fully legal.
 */
void
msgSetVerbose(int level)
{
    msgVerboseLevel = level;
}



/*-------------------------------------------------------------------------
 *
 *  NAME          msgVerbose
 *
 *  FUNCTION      Show given message if allowed by the current verb. level.
 *
 *  SYNOPSIS      #include "shhmsg.h"
 *                void msgVerbose(int level, const char *format, ...);
 *
 *  INPUT         level   The verbosity level of the message to display.
 *                format, ...
 *                        Arguments used as with printf().
 *
 *  DESCRIPTION   The given message is displayed on the _msgVerboseStream if
 *                it's level is less than or equal to the current verbosity
 *                level.
 */
void
msgVerbose(int level, const char *format, ...)
{
    int     lev;
    va_list ap;

    if ((lev = msgVerboseLevel) == MSG_VERBOSE_DEFAULT)
	lev = msgLookupVerboseLevel();
    if (level > lev)
        return;
    fflush(stdout);
    if (_msgShowNameAlways)
	fprintf(_msgVerboseStream, "%s: ", msgGetName());
    va_start(ap, format);
    vfprintf(_msgVerboseStream, format, ap);
    va_end(ap);
}
