#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <linux/unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <mntent.h>
#include <assert.h>
#include <linux/cdrom.h>

#include "fdisk.h"

#define SIZE(a) (sizeof(a)/sizeof(a[0]))

/* this array translates partition type numbers to file system types */
static struct fstypes {
    int ptype;
    int fstype;
} ptype2fstype[] = {
#if HAVE_MSDOS_PARTITION
    /* PC */
    { PTYPE_DOS12,		FSTYPE_MSDOS },
    { PTYPE_DOS16S,		FSTYPE_MSDOS },
    { PTYPE_DOSEXT,		FSTYPE_EXTPART },
    { PTYPE_DOS16L,		FSTYPE_MSDOS },
    { PTYPE_MINIX_DR,		FSTYPE_MINIX },
    { PTYPE_SWAP_DR,		FSTYPE_SWAP },
    { PTYPE_LINUX_DR,		FSTYPE_EXT2 },
    { PTYPE_MINIX,		FSTYPE_MINIX },
    { PTYPE_LSWAP,		FSTYPE_SWAP },
    { PTYPE_LINUX,		FSTYPE_EXT2 },
    { PTYPE_LINUXEXT,		FSTYPE_EXTPART },
#endif /* HAVE_MSDOS_PARTITION */
#if HAVE_BSD_DISKLABEL
    { PTYPE_BSD_MSDOS,		FSTYPE_MSDOS },
    { PTYPE_BSD_ADOS,		FSTYPE_AFFS },
    { PTYPE_BSD_HFS,		FSTYPE_HFS },
#endif /* HAVE_BSD_PARTITION */
#if HAVE_SOLARIS_X86_PARTITION
    /* only type PTYPE_SOLARIS_X86 not mountable */
#endif /* HAVE_SOLARIS_X86_PARTITION */
#if HAVE_OSF_PARTITION
    { PTYPE_OSF_SWAP,		FSTYPE_SWAP },
    { PTYPE_OSF_EXT2,		FSTYPE_EXT2 },
    { PTYPE_OSF_OTHER,		FSTYPE_MSDOS },
    { PTYPE_OSF_ADOS,		FSTYPE_AFFS },
    { PTYPE_OSF_HFS,		FSTYPE_HFS },
#endif /* HAVE_OSF_PARTITION */
#if HAVE_SUN_PARTITION
    { PTYPE_SUN_L_MINIX,	FSTYPE_MINIX },
    { PTYPE_SUN_L_SWAP,		FSTYPE_SWAP },
    { PTYPE_SUN_L_NATIVE,	FSTYPE_EXT2 },
    /* other partitions would be ufs, but that's not supported yet */
#endif /* HAVE_SUN_PARTITION */
#if HAVE_AMIGA_PARTITION
    { PTYPE_AMIGA_OFS,		FSTYPE_AFFS },
    { PTYPE_AMIGA_FFS,		FSTYPE_AFFS },
    { PTYPE_AMIGA_OFS_INTL,	FSTYPE_AFFS },
    { PTYPE_AMIGA_FFS_INTL,	FSTYPE_AFFS },
    { PTYPE_AMIGA_OFS_DC,	FSTYPE_AFFS },
    { PTYPE_AMIGA_FFS_DC,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MUFS,		FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_OFS,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_FFS,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_OFS_INTL,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_FFS_INTL,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_OFS_DC,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_FFS_DC,	FSTYPE_AFFS },
    { PTYPE_AMIGA_EXT2,		FSTYPE_EXT2 },
    { PTYPE_AMIGA_LINUX,	FSTYPE_EXT2 },
    { PTYPE_AMIGA_SWAP,		FSTYPE_SWAP },
    { PTYPE_AMIGA_SWP,		FSTYPE_SWAP },
#endif /* HAVE_AMIGA_PARTITION */
#if HAVE_ATARI_PARTITION
    /* Atari */
    { PTYPE_GEMDOSS,		FSTYPE_MSDOS },
    { PTYPE_GEMDOSL,		FSTYPE_MSDOS },
    { PTYPE_ATARI_LINUX,	FSTYPE_EXT2 },
    { PTYPE_ATARI_SWAP,		FSTYPE_SWAP },
    { PTYPE_ATARI_MINIX1,	FSTYPE_MINIX },
    { PTYPE_ATARI_MINIX2,	FSTYPE_MINIX },
    { PTYPE_ATARI_HFS,		FSTYPE_HFS },
#endif /* HAVE_ATARI_PARTITION */
#if HAVE_MAC_PARTITION
    { PTYPE_MAC_HFS,		FSTYPE_HFS },
    { PTYPE_MAC_SCRATCH,	FSTYPE_SWAP },
/*  { PTYPE_MAC_PRODOS,		FSTYPE_MSDOS }, (??) */
    { PTYPE_MAC_SWAP,		FSTYPE_SWAP },
    { PTYPE_MAC_MSDOS,		FSTYPE_MSDOS },
    { PTYPE_MAC_MINIX,		FSTYPE_MINIX },
    { PTYPE_MAC_AFFS,		FSTYPE_AFFS },
    { PTYPE_MAC_EXT2,		FSTYPE_EXT2 },
    /* some people used partitions with names "A/UX ..." for Linux in the past
     * make them the transition easier :-) */
    { PTYPE_MAC_AUX,		FSTYPE_EXT2 },
#endif /* HAVE_MAC_PARTITION */
};

/*
 * List of system Id's, copied from sfdisk 3.05, itself adapted
 * from fdisk 2.0d and <linux/genhd.h> and SFS and several other sources.
 */

static struct systypes {
    unsigned int type;
    char *name;
} sys_types[] = {
#if HAVE_MSDOS_PARTITION
    /* PC */
    {0, "Empty"},
    {1, "DOS 12-bit FAT"},              /* Primary DOS with 12-bit FAT */
    {2, "XENIX /"},                     /* XENIX / filesystem */
    {3, "XENIX /usr"},                  /* XENIX /usr filesystem */
    {4, "DOS 16-bit FAT <32M"},         /* Primary DOS with 16-bit FAT */
    {5, "DOS Extended"},                /* DOS 3.3+ extended partition */
    {6, "DOS 16-bit FAT >=32M"},
    {7, "HPFS / NTFS"},
    {8, "AIX boot or SplitDrive"},
    {9, "AIX data or Coherent"},
    {0x0a, "OS/2 Boot Manager"},
    {0x0b, "Win95 FAT32"},
    {0x0c, "Win95 FAT32 (LBA)"},
    {0x0e, "Win95 FAT16 (LBA)"},
    {0x0f, "Win95 Extended (LBA)"},
    {0x10, "OPUS"},
    {0x11, "Hidden DOS FAT12"},
    {0x12, "Compaq diagnostics"},
    {0x14, "Hidden DOS FAT16"},
    {0x16, "Hidden DOS FAT16 (big)"},
    {0x17, "Hidden HPFS/NTFS"},
    {0x18, "AST Windows swapfile"},
    {0x24, "NEC DOS"},
    {0x3c, "PartitionMagic recovery"},
    {0x40, "Venix 80286"},
    {0x41, "Linux/MINIX (sharing disk with DRDOS)"},
    {0x42, "SFS or Linux swap (sharing disk with DRDOS)"},
    {0x43, "Linux native (sharing disk with DRDOS)"},
    {0x50, "DM (disk manager)"},
    {0x51, "DM6 Aux1 (or Novell)"},
    {0x52, "CP/M or Microport SysV/AT"},
    {0x53, "DM6 Aux3"},
    {0x54, "DM6"},
    {0x55, "EZ-Drive (disk manager)"},
    {0x56, "Golden Bow (disk manager)"},
    {0x5c, "Priam Edisk (disk manager)"},
    {0x61, "SpeedStor"},
    {0x63, "GNU HURD or Mach or Sys V/386 (such as ISC UNIX)"},
    {0x64, "Novell Netware 286"},
    {0x65, "Novell Netware 386"},
    {0x70, "DiskSecure Multi-Boot"},
    {0x75, "PC/IX"},
    {0x77, "QNX4.x"},
    {0x78, "QNX4.x 2nd part"},
    {0x79, "QNX4.x 3rd part"},
    {0x80, "MINIX until 1.4a"},
    {0x81, "MINIX / old Linux"},
    {0x82, "Linux swap"},
    {0x83, "Linux native"},
    {0x84, "OS/2 hidden C: drive"},
    {0x85, "Linux extended"},
    {0x86, "NTFS volume set"},
    {0x87, "NTFS volume set"},
    {0x93, "Amoeba"},
    {0x94, "Amoeba BBT"},               /* (bad block table) */
    {0xa0, "IBM Thinkpad hibernation"}, /* according to dan@fch.wimsey.bc.ca */
    {0xa5, "BSD/386"},                  /* 386BSD */
    {0xa7, "NeXTSTEP 486"},
    {0xb7, "BSDI fs"},
    {0xb8, "BSDI swap"},
    {0xc1, "DRDOS/sec (FAT-12)"},
    {0xc4, "DRDOS/sec (FAT-16, < 32M)"},
    {0xc6, "DRDOS/sec (FAT-16, >= 32M)"},
    {0xc7, "Syrinx"},
    {0xdb, "CP/M or Concurrent CP/M or Concurrent DOS or CTOS"},
    {0xe1, "DOS access or SpeedStor 12-bit FAT extended partition"},
    {0xe3, "DOS R/O or SpeedStor"},
    {0xe4, "SpeedStor 16-bit FAT extended partition < 1024 cyl."},
    {0xf1, "SpeedStor"},
    {0xf2, "DOS 3.3+ secondary"},
    {0xf4, "SpeedStor large partition"},
    {0xfe, "SpeedStor >1024 cyl. or LANstep"},
    {0xff, "Xenix Bad Block Table"},
#endif /* HAVE_MSDOS_PARTITION */
#if HAVE_BSD_DISKLABEL
    { PTYPE_BSD_SWAP, 	"BSD swap" },
    { PTYPE_BSD_V6,	"BSD version 6" },
    { PTYPE_BSD_V7,	"BSD version 7" },
    { PTYPE_BSD_SYSV,	"BSD--System V" },
    { PTYPE_BSD_V71K,	"4.1BSD" },
    { PTYPE_BSD_V8,	"8th edition BSD" },
    { PTYPE_BSD_BSDFFS,	"4.2BSD" },
    { PTYPE_BSD_MSDOS,	"BSD--MSDOS" },
    { PTYPE_BSD_BSDLFS,	"4.4BSD LFS" },
    { PTYPE_BSD_OTHER,	"BSD--unknown" },
    { PTYPE_BSD_HPFS,	"BSD--OS/2 HPFS" },
    { PTYPE_BSD_ISOFS,	"BSD--iso9660" },
    { PTYPE_BSD_BOOT,	"BSD boot" },
    { PTYPE_BSD_ADOS,	"BSD--Amiga AFFS" },
    { PTYPE_BSD_HFS,	"BSD--Apple HFS" },
#endif /* HAVE_BSD_DISKLABEL */
#if HAVE_SOLARIS_X86_PARTITION
    { PTYPE_SOLARIS_X86,"Solaris/x86" },
#endif /* HAVE_SOLARIS_X86_PARTITION */
#if HAVE_OSF_PARTITION
    { PTYPE_OSF_SWAP, 	"swap" },
    { PTYPE_OSF_V6,	"BSD version 6" },
    { PTYPE_OSF_V7,	"BSD version 7" },
    { PTYPE_OSF_SYSV,	"System V" },
    { PTYPE_OSF_V71K,	"4.1BSD" },
    { PTYPE_OSF_V8,	"8th edition BSD" },
    { PTYPE_OSF_BSDFFS,	"4.2BSD" },
    { PTYPE_OSF_EXT2,	"Linux native" },
    { PTYPE_OSF_BSDLFS,	"4.4BSD LFS" },
    { PTYPE_OSF_OTHER,	"unknown (msdos?)" },
    { PTYPE_OSF_HPFS,	"OS/2 HPFS" },
    { PTYPE_OSF_ISOFS,	"iso9660" },
    { PTYPE_OSF_BOOT,	"boot" },
    { PTYPE_OSF_ADOS,	"Amiga AFFS" },
    { PTYPE_OSF_HFS,	"Apple HFS" },
#endif /* HAVE_OSF_PARTITION */
#if HAVE_SUN_PARTITION
    { PTYPE_SUN_EMPTY,		"empty" },
    { PTYPE_SUN_BOOT,		"Boot" },
    { PTYPE_SUN_S_ROOT,		"SunOS root" },
    { PTYPE_SUN_S_SWAP,		"SunOS swap" },
    { PTYPE_SUN_S_USR,		"SunOS usr" },
    { PTYPE_SUN_WHOLE,		"Whole disk" },
    { PTYPE_SUN_S_STAND,	"SunOS stand" },
    { PTYPE_SUN_S_VAR,		"SunOS var" },
    { PTYPE_SUN_S_HOME,		"SunOS home" },
    { PTYPE_SUN_L_MINIX,	"Linux minix" },
    { PTYPE_SUN_L_SWAP,		"Linux swap" },
    { PTYPE_SUN_L_NATIVE,	"Linux native" },
#endif /* HAVE_SUN_PARTITION */
#if HAVE_AMIGA_PARTITION
    { PTYPE_AMIGA_BOOT,		"generic boot" },
    { PTYPE_AMIGA_OFS,		"OFS" },
    { PTYPE_AMIGA_FFS,		"FFS" },
    { PTYPE_AMIGA_OFS_INTL,	"OFS INTL" },
    { PTYPE_AMIGA_FFS_INTL,	"FFS INTL" },
    { PTYPE_AMIGA_OFS_DC,	"OFS DC" },
    { PTYPE_AMIGA_FFS_DC,	"FFS DC" },
    { PTYPE_AMIGA_MUFS,		"MU FFS INTL" },
    { PTYPE_AMIGA_MU_OFS,	"MU OFS" },   
    { PTYPE_AMIGA_MU_FFS,	"MU FFS" },   
    { PTYPE_AMIGA_MU_OFS_INTL,	"MU OFS INTL" },
    { PTYPE_AMIGA_MU_FFS_INTL,	"MU FFS INTL" },
    { PTYPE_AMIGA_MU_OFS_DC,	"MU OFS DC" },
    { PTYPE_AMIGA_MU_FFS_DC,	"MU FFS DC" },
    { PTYPE_AMIGA_LINUX,	"Linux native" },
    { PTYPE_AMIGA_EXT2,		"Linux ext2" },
    { PTYPE_AMIGA_SWAP,		"Swap" },
    { PTYPE_AMIGA_SWP,		"Swap" },
    { PTYPE_AMIGA_AMIX0,	"Amix \\0" },
    { PTYPE_AMIGA_AMIX1,	"Amix \\1" },
    { PTYPE_AMIGA_NETBSD_ROOT,	"NetBSD root" },
    { PTYPE_AMIGA_NETBSD_SWAP,	"NetBSD swap" },
    { PTYPE_AMIGA_NETBSD_OTH,	"NetBSD other" },
    { PTYPE_AMIGA_PFS0,		"Professional FS" },
    { PTYPE_AMIGA_PFS1,		"Professional FS" },
    { PTYPE_AMIGA_AFS0,		"AmiFile Safe" },
    { PTYPE_AMIGA_AFS1,		"AmiFile Safe (exp.)" },
#endif /* HAVE_AMIGA_PARTITION */
#if HAVE_ATARI_PARTITION
    /* Atari */
    { PTYPE_GEMDOSS,		"GEMDOS (< 32M)" },
    { PTYPE_GEMDOSL,		"GEMDOS (> 32M)" },
    { PTYPE_ATARI_LINUX,	"Linux" },
    { PTYPE_ATARI_SWAP,		"Swap" },
    { PTYPE_ATARI_MINIX1,	"Minix" },
    { PTYPE_ATARI_MINIX2,	"Minix" },
    { PTYPE_ATARI_HFS,		"Mac HFS" },
    { PTYPE_ATARI_SYSV,		"Atari SysV Unix" },
#endif /* HAVE_ATARI_PARTITION */
#if HAVE_MAC_PARTITION
    { PTYPE_MAC_PMAP,		"partition map" },
    { PTYPE_MAC_DRIVER,		"Driver" },
    { PTYPE_MAC_DRIVER43,	"Driver 4.3" },
    { PTYPE_MAC_HFS,		"HFS" },
    { PTYPE_MAC_MFS,		"MFS" },
    { PTYPE_MAC_SCRATCH,	"Scratch (swap)" },
    { PTYPE_MAC_PRODOS,		"ProDOS" },
    { PTYPE_MAC_FREE,		"Free" },
    { PTYPE_MAC_SWAP,		"Linux swap" },
    { PTYPE_MAC_AUX,		"A/UX" },
    { PTYPE_MAC_MSDOS,		"MS-DOS" },
    { PTYPE_MAC_MINIX,		"Minix" },
    { PTYPE_MAC_AFFS,		"Amiga AFFS" },
    { PTYPE_MAC_EXT2,		"Linux native" },
#endif /* HAVE_MAC_PARTITION */
};

static char *fstype_name[FSTYPE_MAX] = {
    "ext2",
    "swap",
    "msdos",
    "minix",
    "affs",
    "hfs",
    NULL, /* FSTYPE_EXTPART is not a valid filesystem, so return "UNKNOWN" */
    NULL /* FSTYPE_UNKNOWN */
};

#define SIZE(a) (sizeof(a)/sizeof(a[0]))

/* translate partition type number to description */
char *
sysname(int type) {
    struct systypes *s;

    for( s = sys_types+SIZE(sys_types)-1; s >= sys_types; s-- )
      if (s->type == type)
        return s->name;
    return "Unknown";
}

struct fdisk_disk *fdisk_disks = NULL;

/* internal state */
static struct fdisk_disk *last_disk;
static struct fdisk_partition *last_partition;
static struct fdisk_partition *last_partition_by_disk;
static struct fdisk_partition *last_partitions_by_type[255];
static struct fdisk_partition *last_swapon_partition;

void fdisk_init() {
  int i;
  struct fdisk_disk *d1,*d2;
  struct fdisk_partition *p1,*p2;

  if (NULL != fdisk_disks) {
    d1 = fdisk_disks;
    while ( d1 ) {
      d2 = d1;
      d1 = d1->next;
      free (d2);
    }
    p1 = fdisk_partitions;
    while ( p1 ) {
      p2 = p1;
      p1 = p1->next;
      free (p2);
    }
  }

  for (i=0; i<FSTYPE_MAX; i++) {
    fdisk_partitions_by_type[i] = NULL;
    last_partitions_by_type[i] = NULL;
  }

  fdisk_disks = last_disk = NULL;
  fdisk_partitions = last_partition = last_partition_by_disk = NULL;
  mounted_partitions = NULL;
  swapon_partitions = last_swapon_partition = NULL;
}


struct fdisk_disk *
fdisk_add_disk(char *name, unsigned long size){
  struct fdisk_disk *disk;
  
  disk = (struct fdisk_disk *) malloc(sizeof(struct fdisk_disk));

  disk->name = strdup(name);
  disk->size = size;
  disk->next = NULL;
  disk->partitions = NULL;

  if (NULL == fdisk_disks) {
    fdisk_disks = disk;
  }

  if (NULL != last_disk) {
    last_disk->next = disk;
  }
  last_disk = disk;

  last_partition_by_disk = NULL;

#ifdef DEBUG
  printf( "added disk %s, size %ld\n", name, size );
#endif

  return(0);
}

int
fdisk_fstype_of(unsigned int ptype)
{
    struct fstypes *p;
    
    for( p = ptype2fstype+SIZE(ptype2fstype)-1; p >= ptype2fstype; p-- ) {
	if (ptype == p->ptype)
	return p->fstype;
    }
    return FSTYPE_UNKNOWN;
}

char *
fdisk_fstype_name_of(unsigned int ptype)
{
    int fstype = fdisk_fstype_of(ptype);
    return fstype_name[fstype];
}

struct fdisk_partition *
fdisk_add_partition(char *name, int minor, unsigned int type,
		    unsigned long size){

  struct fdisk_partition *partition;
  int fstype;

  partition = (struct fdisk_partition *) 
    malloc(sizeof(struct fdisk_partition));

  partition->name= malloc(strlen(name)+4);
  sprintf(partition->name,"%s%d", name, minor);
  partition->mount_point= NULL;
  partition->type = type;
  partition->size = size;
  partition->in_use = 0;
  partition->next = NULL;
  partition->next_by_disk = NULL;
  partition->next_by_type = NULL;
  partition->next_in_use = NULL;
  
  assert(last_disk);

  if (NULL == last_disk->partitions) {
    last_disk->partitions = partition;
  }
  if (NULL != last_partition_by_disk) {
    last_partition_by_disk->next_by_disk = partition;
  }
  last_partition_by_disk = partition;

  if (NULL == fdisk_partitions) {
    fdisk_partitions = partition;
  }
  if (NULL != last_partition) {
    last_partition->next = partition;
  }
  last_partition = partition;

  fstype = fdisk_fstype_of(type);
  if (NULL == fdisk_partitions_by_type[fstype]) {
    fdisk_partitions_by_type[fstype] = partition;
  }
  if (NULL != last_partitions_by_type[fstype]) {
    last_partitions_by_type[fstype]->next_by_type = partition;
  }
  last_partitions_by_type[fstype] = partition;

#ifdef DEBUG
  printf( "added partition %s, size %ld, type %08x=\"%s\"\n",
	  partition->name, size, type, sysname(type) );
#endif
  return(0);
}


struct fdisk_disk *
fdisk_find_disk(char *name) {
  struct fdisk_disk *d;

  d = fdisk_disks;

  while (d) {
    if (0 == strcmp(name, d->name)) {
      return (d);
    }
    d = d->next;
  }

  return(NULL);
}


struct fdisk_partition *
fdisk_find_partition(char *name){
  struct fdisk_partition *p;

  p = fdisk_partitions;

  while (p) {
    if ( 0== strcmp(name, p->name)) {
      return(p);
    }
    p = p->next;
  }

  return(NULL);
}

void
mounted_reread(){
  FILE *mtabFile;
  struct mntent *mnt_ent;
  struct fdisk_partition *p; 

  if (NULL == (mtabFile = setmntent("/etc/mtab", "r"))) {
    return ;
  }

  mnt_ent = getmntent(mtabFile);
  while ( NULL != mnt_ent ) {
#ifdef DEBUG
    fprintf(stderr,"mounted partition: %s, on: %s, type: %s ... ", 
	mnt_ent->mnt_fsname, mnt_ent->mnt_dir, mnt_ent->mnt_type);
#endif
    /* If mnt_ent is not a partition on the local disks, then ignore it */
    p = fdisk_find_partition(mnt_ent->mnt_fsname); 
    if ( NULL != p) {
#ifdef DEBUG
      fprintf(stderr,"found on local disks.\n");
#endif
      p->in_use=-1;
      p->mount_point=strdup(mnt_ent->mnt_dir);
      if (mounted_partitions) {
    /* reverse ordered, as this is the order in which they would be
     * un-mounted. */
        p->next_in_use=mounted_partitions;
      }
      mounted_partitions=p; 
    }
#ifdef DEBUG
    else {
      fprintf(stderr,"NOT found on local disks. Ignored.\n");
    }
#endif
    mnt_ent = getmntent(mtabFile);
  }
  endmntent(mtabFile);
}

void
swapon_reread(){
  FILE *swapsFile;
  struct mntent *swap_ent;
  struct fdisk_partition *p;

  if (NULL == (swapsFile = setmntent("/etc/swaps", "r"))) {
    return ;
  }

  swap_ent = getmntent(swapsFile);
  while ( NULL != swap_ent ) {
#ifdef DEBUG
    fprintf(stderr,"Swap partition: %s, type: %s ... ",
        swap_ent->mnt_fsname, swap_ent->mnt_type);
#endif
    /* If swap_ent is not a partition on the local disks, then ignore it */
    p = fdisk_find_partition(swap_ent->mnt_fsname);
    if ( NULL != p) {
#ifdef DEBUG
      fprintf(stderr,"found on local disks.\n");
#endif
      p->in_use=-1;
      if (NULL == swapon_partitions)
        swapon_partitions=p;
      else {
        last_swapon_partition->next_in_use=p;
      }
      last_swapon_partition=p;
    }
#ifdef DEBUG
    else {
      fprintf(stderr,"NOT found on local disks. Ignored.\n");
    }
#endif
    swap_ent = getmntent(swapsFile);
  }
  endmntent(swapsFile);
}

/*
 * sseek: seek to specified sector - return 0 on failure
 *
 * For >4GB disks lseek needs a > 32bit arg, and we have to use llseek.
 * On the other hand, a 32 bit sector number is OK until 2TB.
 * The routines _llseek and sseek below are the only ones that
 * know about the loff_t type.
 *
 * sseek() uses the _llseek system call directly (if available), because of
 * some libc5/libc6 mismatch stuff.
 */

#ifdef __NR__llseek
static _syscall5( int, _llseek, uint, fd, ulong, hi, ulong, lo,
		  loff_t *, res, uint, wh );

/* seek to a sector */
static int sseek(int fd, unsigned long s)
{
    loff_t in, out;

    in = (loff_t)s * 512;
    out = 1;
    if (_llseek (fd, in>>32, in & 0xffffffff, &out, SEEK_SET) != 0) {
	perror( "llseek" );
	return 0;
    }
    if (in != out) {
	fprintf( stderr, "seek error: wanted 0x%08x%08x, got 0x%08x%08x\n",
		 (uint)(in>>32), (uint)(in & 0xffffffff),
		 (uint)(out>>32), (uint)(out & 0xffffffff) );
	return 0;
    }
    return 1;
}
#else
#ifndef __alpha__
/* Better print an error if _llseek isn't available at compile time, the
 * resulting binary wouldn't be able to handle disks > 2GB */
#error No _llseek available
/* Sorry, kernel doesn't know _llseek call :-((( */
#endif
static int sseek(int fd, unsigned long s)
{
    off_t out;

    out = lseek( fd, (off_t)(s*512), SEEK_SET );
    if ((off_t)s*512 != out)
	return 0;
    return 1;
}
#endif

int sread(int fd, unsigned long sector, void *buffer)
{
    if (!sseek( fd, sector ))
	return 0;
    if (read( fd, buffer, 512 ) != 512) {
	fprintf( stderr, "libfdisk: error reading sector %ld: %s\n",
		 sector, strerror(errno) );
	return 0;
    }
    return 1;
}

static struct {
    char *name;
    int (*func)(char *device, int fd);
} parsers[] = {
    { "msdos", parse_msdos_partition },
    { "osf",   parse_osf_partition },
    { "sun",   parse_sun_partition },
    { "amiga", parse_amiga_partition },
    { "atari", parse_atari_partition },
    { "mac",   parse_mac_partition }
};

static void
parse_partition(char *device, int fd)
{
    int i, rv;

    for( i = 0; i < SIZE(parsers); ++i ) {
	rv = parsers[i].func( device, fd );
	if (rv < 0)
	    return;
	if (rv > 0) {
	    struct fdisk_disk *disk;
	    if ((disk = fdisk_find_disk(device)))
		disk->partition_format = parsers[i].name;
	    return;
	}
    }
}

static struct trylist {
    char *dev_pattern;
    char start, end;
} trylist[] = {
    { "/dev/hd", 'a', 'd' },
    { "/dev/sd", 'a', 'h' },
#if #cpu (i386)
    { "/dev/ed", 'a', 'd' },
#endif
#if #cpu (m68k)
    { "/dev/ad", 'a', 'h' },
#endif
#ifdef DEBUG
    { "/dev/loop", '0', '7' },
#endif
};

static int timed_out;

static void alarmed(int signo)
{
    timed_out = 1;
}

void
fdisk_reread(void)
{
    struct trylist *p;
    struct cdrom_volctrl cdvol;
    int i, fd;
    char letter, device[12], *q;
    unsigned size;
    
    fdisk_init();
    
    for( p = trylist, i = 0; i < SIZE(trylist); ++p, ++i ) {
	strcpy( device, p->dev_pattern );
	q = device+strlen(device);
	q[1] = 0;
	for( letter = p->start; letter <= p->end; ++letter ) {
	    *q = letter;
	    signal( SIGALRM, alarmed );
	    timed_out = 0;
	    alarm( 30 );
	    if ((fd = open( device, O_RDONLY )) >= 0 && !timed_out) {
		alarm( 0 );
		if (ioctl(fd, CDROMVOLREAD, &cdvol )!= 0) {
      /* if returns EINVAL or EPERM or EIO, this is not a CD-ROM drive */
      		    if ((errno==EINVAL)||(errno==EPERM)||(errno==EIO)) {
			if (ioctl(fd, BLKGETSIZE, &size))
			    size = 0; /* don't complain, size not really 
					 used yet */
			size >>= 1; /* convert from 512-byte sectors to kB */
			fdisk_add_disk( device, size );

			parse_partition( device, fd );
		    }
		}
		close( fd );
	    }
	    else {
		/* else: don't complain, ignore open errors */
		alarm( 0 );
	    }
	}
    }
    
    mounted_reread();
    swapon_reread();
}
