/* 
 * Help utilities.
 * Copyright (c) 1995 Markku Rossi.
 *
 * Author: Markku Rossi <mtr@iki.fi>
 */

/*
 * This file is part of genscript.
 * 
 * Genscript is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Genscript is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with genscript; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gsint.h"

/*
 * Static variables.
 */

/* 7bit ASCII scands encodings (additions to 7bit ASCII enc). */
static struct
{
  int code;
  char *name;
} enc_7bit_ascii_scands[] =
{
  {'{',		"adieresis"},
  {'|',		"odieresis"},
  {'}',		"aring"},
  {'[',		"Adieresis"},
  {'\\',	"Odieresis"},
  {']',		"Aring"},
  {0, NULL},
};


/*
 * Global functions.
 */

void
message (int verbose_level, char *fmt, ...)
{
  va_list args;

  if (quiet || verbose < verbose_level)
    return;

  va_start (args, fmt);
  vfprintf (stderr, fmt, args);
  va_end (args);
}


void
error (char *fmt, ...)
{
  va_list args;

  fprintf (stderr, "%s: ", program);

  va_start (args, fmt);
  vfprintf (stderr, fmt, args);
  va_end (args);

  fprintf (stderr, "\n");
  fflush (stderr);
}


void
fatal (char *fmt, ...)
{
  va_list args;

  fprintf (stderr, "%s: ", program);

  va_start (args, fmt);
  vfprintf (stderr, fmt, args);
  va_end (args);

  fprintf (stderr, "\n");
  fflush (stderr);

  exit (1);
}

#define GET_TOKEN(from) (strtok (from, " \t\n"))

#define CHECK_TOKEN() 							\
  if (token2 == NULL) 							\
    error ("malformed token \"%s\" in config file \"%s\", line %d", 	\
	   token, fname, line)

int
read_config (char *path, char *file)
{
  FILE *fp;
  char fname[512];
  char buf[4096];
  char *token, *token2;
  int line = 0;

  sprintf (fname, "%s/%s", path, file);
  fp = fopen (fname, "r");
  if (fp == NULL)
    return 0;

  while (fgets (buf, sizeof (buf), fp))
    {
      line++;

      if (buf[0] == '#')
	continue;
      
      token = GET_TOKEN (buf);
      if (token == NULL)
	/* Empty line. */
	continue;

      if (MATCH (token, "Media:"))
	{
	  char *name;
	  int w, h, llx, lly, urx, ury;

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  name = token2;

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  w = atoi (token2);

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  h = atoi (token2);

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  llx = atoi (token2);

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  lly = atoi (token2);

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  urx = atoi (token2);

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  ury = atoi (token2);

	  add_media (name, w, h, llx, lly, urx, ury);
	}
      else if (MATCH (token, "Spooler:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (spooler_command, token2);
	}
      else if (MATCH (token, "DefaultFancyHeader:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (fancy_header_default, token2);
	}
      else if (MATCH (token, "DefaultEncoding:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (encoding_name_buffer, token2);
	  encoding_name = encoding_name_buffer;
	}
      else if (MATCH (token, "DefaultMedia:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (media_name_buffer, token2);
	  media_name = media_name_buffer;
	}
      else if (MATCH (token, "LibraryPath:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (libpath, token2);
	}
      else if (MATCH (token, "AFMPath:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (afm_path_buffer, token2);
	  afm_path = afm_path_buffer;
	}
      else if (MATCH (token, "StatusDict:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  parse_key_value_pair (statusdict, token2);
	}
      else if (MATCH (token, "SetPageDevice:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  parse_key_value_pair (pagedevice, token2);
	}
      else if (MATCH (token, "DefaultOutputMethod:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  if (MATCH (token2, "printer"))
	    output_file = OUTPUT_FILE_NONE;
	  else if (MATCH (token2, "stdout"))
	    output_file = OUTPUT_FILE_STDOUT;
	  else
	    fatal ("illegal value \"%s\" for option %s in config file \
\"%s\", line %d", token2, token, fname, line);
	}
      else if (MATCH (token, "PageLabelFormat:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (page_label_format_buf, token2);
	  page_label_format = page_label_format_buf;
	}
      else if (MATCH (token, "DownloadFont:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strhash_put (download_fonts, token2, strlen (token2) + 1, NULL,
		       NULL);
	}
      else if (MATCH (token, "Queue:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (queue_buf, token2);
	  queue = queue_buf;
	}
      else if (MATCH (token, "QueueParam:"))
	{
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  strcpy (queue_param, token2);
	}
      else
	error ("illegal option \"%s\" in config file \"%s\", line %d", token,
	       fname, line);
    }
  return 1;
}


void
add_media (char *name, int w, int h, int llx, int lly, int urx, int ury)
{
  MediaEntry *entry;

  message (2,
	   "add_media: name=%s, w=%d, h=%d, llx=%d, lly=%d, urx=%d, ury=%d\n",
	   name, w, h, llx, lly, urx, ury);

  entry = xcalloc (1, sizeof (*entry));
  entry->name = xmalloc (strlen (name) + 1);

  strcpy (entry->name, name);
  entry->w = w;
  entry->h = h;
  entry->llx = llx;
  entry->lly = lly;
  entry->urx = urx;
  entry->ury = ury;

  entry->next = media_names;
  media_names = entry;
}


void
do_list_missing_characters ()
{
  int i;
  int count = 0;

  printf ("missing character codes (decimal):\n");

  for (i = 0; i < 256; i++)
    if (missing_chars[i])
      {
	printf ("%3d ", i);
	count++;
	if (count % 15 == 0)
	  printf ("\n");
      }

  if (count % 15 != 0)
    printf ("\n");
}


int
file_existsp (char *name, char *suffix)
{
  FileLookupCtx ctx;

  strcpy (ctx.name, name);
  strcpy (ctx.suffix, suffix ? suffix : "");

  return pathwalk (libpath, file_lookup, &ctx);
}


int
paste_file (char *name, char *suffix)
{
  char buf[512];
  char resources[512];
  FILE *fp;
  FileLookupCtx ctx;
  int pending_comment = 0;

  strcpy (ctx.name, name);
  strcpy (ctx.suffix, suffix ? suffix : "");

  if (!pathwalk (libpath, file_lookup, &ctx))
    return 0;
  fp = fopen (ctx.fullname, "r");
  if (fp == NULL)
    return 0;

  /* Find the end of the header. */
#define HDR_TAG "% -- code follows this line --"
  while ((fgets (buf, sizeof (buf), fp)))
    if (strncmp (buf, HDR_TAG, strlen (HDR_TAG)) == 0)
      break;

  /* Dump rest of file. */
#define RESOURCE_DSC 	"%%DocumentNeededResources:"
#define CONT_DSC 	"%%+"
  while ((fgets (buf, sizeof (buf), fp)))
    {
      if (strncmp (buf, RESOURCE_DSC, strlen (RESOURCE_DSC)) == 0)
	{
	  char *cp, *cp2;

	  strcpy (resources, buf + strlen (RESOURCE_DSC));
	  pending_comment = 1;

	parse_resources:
	  /* Register needed resources. */
	  cp = GET_TOKEN (resources);
	  if (cp == NULL)
	    /* Get the next line. */
	    continue;

	  if (MATCH (cp, "font"))
	    {
	      for (cp = GET_TOKEN (NULL); cp; cp = GET_TOKEN (NULL))
		/* Is this font already known? */
		if (!strhash_get (res_fonts, cp, strlen (cp) + 1,
				  (void **) &cp2))
		  {
		    /* Not it is not,  we must include this resource. */
		    fprintf (ofp, "%%%%IncludeResource: font %s\n", cp);

		    /*
		     * And register that this resource is needed in
		     * this document.
		     */
		    strhash_put (res_fonts, cp, strlen (cp) + 1, NULL, NULL);
		  }

	      /* Do not pass this DSC row to the output. */
	      continue;
	    }
	  else
	    /* Unknown resource, ignore. */
	    continue;
	}
      else if (pending_comment
	       && strncmp (buf, CONT_DSC, strlen (CONT_DSC)) == 0)
	{
	  strcpy (resources, buf + strlen (CONT_DSC));
	  goto parse_resources;
	}
      else
	pending_comment = 0;

      fputs (buf, ofp);
    }

  fclose (fp);

  return 1;
}


int
parse_font_spec (char *spec, char **name_return, double *size_return)
{
  int i;

  i = strlen (spec) - 1;
  if (i <= 0 || !ISNUMBERDIGIT (spec[i]))
    return 0;

  for (i--; i >= 0 && ISNUMBERDIGIT (spec[i]); i--)
    ;
  i++;
  
  *name_return = (char *) xcalloc (1, i + 1);
  strncpy (*name_return, spec, i);
  *size_return = atof (spec + i);

  message (2, "parse_font_spec(): name=%.*s, size=%g\n", i, *name_return,
	   *size_return);
  return 1;
}


void
read_font_info (void)
{
  AFMFont font;
  AFMError error;
  char buf[256];
  int i;

  message (1, "reading AFM info for font \"%s\"\n", Fname);

  /* Open font */

  error = afm_open_font (afm, AFM_I_MINIMUM, Fname, &font);
  if (error != AFM_SUCCESS)
    {
#define COUR "Courier"
      /*
       * Do not report failures for "Courier*" fonts because 
       * AFM library's default font will fix them.
       */
      if (strncmp (Fname, COUR, strlen (COUR)) != 0)
	message (0, "couldn't open AFM file for font \"%s\", using default\n",
		 Fname);
      error = afm_open_default_font (afm, &font);
      if (error != AFM_SUCCESS)
	{
	  afm_error_to_string (error, buf);
	  fatal ("couldn't open AFM file for the default font: %s", buf);
	}
    }

  /* Apply encoding. */
  switch (encoding)
    {
    case ENC_LATIN1:
      (void) afm_font_encoding (font, AFM_ENCODING_LATIN1);
      break;

    case ENC_LATIN2:
      (void) afm_font_encoding (font, AFM_ENCODING_LATIN2);
      break;

    case ENC_ASCII:
      (void) afm_font_encoding (font, AFM_ENCODING_ASCII);
      break;

    case ENC_ASCII_SCANDS:
      /* First apply standard 7bit ASCII encoding. */
      (void) afm_font_encoding (font, AFM_ENCODING_ASCII);

      /* Then add those scand letters. */
      for (i = 0; enc_7bit_ascii_scands[i].name; i++)
	(void) afm_font_encode (font, enc_7bit_ascii_scands[i].code,
				enc_7bit_ascii_scands[i].name);
      break;

    case ENC_IBMPC:
      (void) afm_font_encoding (font, AFM_ENCODING_IBMPC);
      break;

    case ENC_MAC:
      (void) afm_font_encoding (font, AFM_ENCODING_MAC);
      break; 

    case ENC_VMS:
      (void) afm_font_encoding (font, AFM_ENCODING_VMS);
      break; 

    case ENC_PS:
      /* Let's use font's default encoding -- nothing here. */
      break;
    }

  /* Read character widths and types. */
  for (i = 0; i < 256; i++)
    {
      AFMNumber w0x, w0y;

      (void) afm_font_charwidth (font, Fpt, (unsigned int) i, &w0x, &w0y);
      font_widths[i] = w0x;

      if (font->encoding[i] == AFM_ENC_NONE)
	font_ctype[i] = ' ';
      else if (font->encoding[i] == AFM_ENC_NON_EXISTENT)
	font_ctype[i] = '.';
      else
	font_ctype[i] = '*';
    }

  font_is_fixed = font->writing_direction_metrics[0].IsFixedPitch;

  (void) afm_close_font (font);
}


void
download_font (char *name)
{
  AFMError error;
  const char *prefix;
  struct stat stat_st;
  char fname[512];
  char buf[4096];
  FILE *fp;
  int i;
  char *cp;

  /* Get font prefix. */
  error = afm_font_prefix (afm, name, &prefix);
  if (error != AFM_SUCCESS)
    /* Font is unknown, nothing to download. */
    return;

  /* Check if we have a font description file. */

  /* .pfa */
  sprintf (fname, "%s.pfa", prefix);
  if (stat (fname, &stat_st) != 0)
    {
      /* .pfb */
      sprintf (fname, "%s.pfb", prefix);
      if (stat (fname, &stat_st) != 0)
	/* Couldn't find font description file, nothing to download. */
	return;
    }

  /* Ok, fine.  Font was found. */

  message (1, "downloading font \"%s\"\n", name);
  fp = fopen (fname, "r");
  if (fp == NULL)
    {
      message (0, "couldn't open font description file \"%s\": %s\n",
	       fname, strerror (errno));
      return;
    }

  /* Dump file. */
  fprintf (ofp, "%%%%BeginResource: font %s\n", name);
  while ((i = fread (buf, 1, sizeof (buf), fp)) != 0)
    fwrite (buf, 1, i, ofp);
  fprintf (ofp, "%%%%EndResource\n");

  /* Remove font from needed resources. */
  (void) strhash_delete (res_fonts, name, strlen (name) + 1, (void **) &cp);

  fclose (fp);
}


char *
escape_string (char *string)
{
  int i, j;
  int len;
  char *cp;

  /* Count the length of the result string. */
  for (len = 0, i = 0; string[i]; i++)
    switch (string[i])
      {
      case '(':
      case ')':
      case '\\':
	len += 2;
	break;

      default:
	len++;
      }

  /* Create result. */
  cp = xmalloc (len + 1);
  for (i = 0, j = 0; string[i]; i++)
    switch (string[i])
      {
      case '(':
      case ')':
      case '\\':
	cp[j++] = '\\';
	/* FALLTHROUGH */

      default:
	cp[j++] = string[i];
	break;
      }
  cp[j++] = '\0';

  return cp;
}


void
parse_key_value_pair (StringHashPtr set, char *kv)
{
  char *cp;
  char key[256];

  cp = strchr (kv, ':');
  if (cp == NULL)
    {
      if (strhash_delete (set, kv, strlen (kv) + 1, (void **) &cp))
	xfree (cp);
    }
  else
    {
      sprintf (key, "%.*s", cp - kv, kv);
      strhash_put (set, key, strlen (key) + 1, xstrdup (cp + 1),
		   (void **) &cp);
      if (cp)
	xfree (cp);
    }
}


int
count_key_value_set (StringHashPtr set)
{
  int i = 0, got, j;
  char *cp;
  void *value;
  
  for (got = strhash_get_first (set, &cp, &j, &value); got;
       got = strhash_get_next (set, &cp, &j, &value))
    i++;

  return i;
}


int
pathwalk (char *path, PathWalkProc proc, void *context)
{
  char buf[512];
  char *cp;
  char *cp2;
  int len, i;
  
  for (cp = path; cp; cp = strchr (cp, ':'))
    {
      if (cp != path)
	cp++;

      cp2 = strchr (cp, ':');
      if (cp2)
	len = cp2 - cp;
      else
	len = strlen (cp);

      memcpy (buf, cp, len);
      buf[len] = '\0';

      i = (*proc) (buf, context);
      if (i != 0)
	return i;
    }

  return 0;
}


int
file_lookup (char *path, void *context)
{
  int len;
  FileLookupCtx *ctx = context;
  struct stat stat_st;
  int i;

  message (2, "file_lookup(): %s/%s%s\t", path, ctx->name,
	   ctx->suffix);

  len = strlen (path);
  if (len && path[len - 1] == '/')
    len--;

  sprintf (ctx->fullname, "%.*s/%s%s", len, path, ctx->name, ctx->suffix);
  
  i = stat (ctx->fullname, &stat_st) == 0;

  message (2, "#%c\n", i ? 't' : 'f');
  return i;
}


void
tilde_subst (char *from, char *to)
{
  char *cp;
  char user[256];
  int i, j;
  struct passwd *pswd;

  if (from[0] != '~')
    {
    copy_out:
      strcpy (to, from);
      return;
    }

  if (from[1] == '/' || from[1] == '\0')
    {
      cp = getenv ("HOME");
      if (cp == NULL)
	goto copy_out;

      sprintf (to, "%s%s", cp, from + 1);
      return;
    }

  /* Get user's login name. */
  for (i = 1, j = 0; from[i] && from[i] != '/'; i++)
    user[j++] = from[i];
  user[j++] = '\0';

  pswd = getpwnam (user);
  if (pswd)
    {
      /* Found passwd entry. */
      sprintf (to, "%s%s", pswd->pw_dir, from + i);
      return;
    }

  /* No match found. */
  goto copy_out;
}


/*
 * InputStream functions.
 */

void
is_init (InputStream *is)
{
  is->data_in_buf = 0;
  is->bufpos = 0;
  is->nreads = 0;
  is->unget_ch = EOF;
}


int
is_getc (InputStream *is)
{
  int ch;

  if (is->unget_ch != EOF)
    {
      ch = is->unget_ch;
      is->unget_ch = EOF;
      return ch;
    }

 retry:

  /* Do we have any data left? */
  if (is->bufpos >= is->data_in_buf)
    {
      /* At the EOF? */
      if (is->nreads > 0 && is->data_in_buf < sizeof (is->buf))
	/* Yes. */
	return EOF;

      /* Read more data. */
      is->data_in_buf = fread (is->buf, 1, sizeof (is->buf), is->fp);
      is->bufpos = 0;
      is->nreads++;

      goto retry;
    }

  return is->buf[is->bufpos++];
}


int
is_ungetc (int ch, InputStream *is)
{
  is->unget_ch = ch;
  return 1;
}
