changequote(<<, >>)dnl
/* `gf256.h' -- Routines to do arithmetic in the Galois Field (2^8).

   Copyright (C) 1993, 1996 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Developed by Paul Flaherty (paulf@stanford.edu).
   Munged for speed by Fung-Chai Lim (fclim@acm.org).

   This file is part of GNU ecc.  */

#ifndef _gf256_h
#define _gf256_h

#ifdef	__cplusplus
extern "C" {
#endif

/*
#ifdef MAN

gf256(3L)	GNU Library Manual Pages		gf256(3L)

NAME
   gf256 - Routines to do arithmetic in the Galois Field (2^8).

SYNOPSIS
   #include <gf256.h>

   cc ... -lecc

DESCRIPTION
   GF256 x;
   X is an element of GF(2^8).  The representation of Galois Field (2^8) used
   in this library is isomorphic to GF(2)[X] modulo P(x) = x^8 + x^4 + x^3 +
   x^2 + 1.  This representation can be found in many textbooks.  See, for
   instance, Richard Blahut's "Theory and Practice of Error Control Codes".

   r = gf256_log_2 (x)
   GF256 r, x;
   Returns the discrete logarithm of X such that 2^R = X for non-zero X.
   Since the Log of any nonzero element in GF(256) is between 1 and 254,
   the return type is also GF256.

   r = gf256_antilog_2 (x)
   GF256 r, x;
   Returns the discrete antilog of X such that
       gf256_antilog_2 (gf256_log_2 (a)) == a
   and
       gf256_log_2 (gf256_antilog_2 (a)) == a
   for all non-zero A.

   r = gf256_power_of_2 (n)
   GF256 r, n;
   Returns the Nth power of 2, that is, 2^N.  The range of N is restricted
   to GF(256).

   r = gf256_add (a, b)
   GF256 r, a, b;
   Returns the sum of A and B.

   r = gf256_minus (a)
   GF256 r, a;
   Returns the additive inverse of A.  That is, A + R == 0.

   r = gf256_subtract (a, b)
   GF256 r, a, b;
   Returns A - B.

   r = gf256_mul (x, y)
   GF256 r, x, y;
   Returns the product of X and Y.

   r = gf256_inv (x)
   GF256 r, x;
   Returns the multiplicative inverse of non-zero X.  That is, X * R == 1.

   r = gf256_div (x, y)
   GF256 r, x, y;
   Returns X / Y for non-zero Y.

   r = gf256_product (x, y, z)
   GF256 r, x, y, z;
   Returns the product of X, Y, and Z.

   r = gf256_exp (a, n)
   GF256 r, a;
   int n;
   Returns the Nth power of A, that is, A^N.

   r = gf256_term (c, i, n)
   GF256 r, c;
   int i, n;
   Calculates the x^N term which has coefficient C in the evaluation of a
   polynomial at x = 2^I.  In short, it returns C*2^(I*N).

CAVEATS
   (1) Note that most of the above are C-preprocessor macros.  Please, do not
   have side-effects in the arguments.  Avoid functional compositions, instead
   use temporary variables.  For instance, instead of
     y = gf256_add (gf256_mul (s2, s4), gf256_mul (s3, s3));
   write
     y = gf256_mul (s2, s4);
     p = gf256_mul (s3, s3);
     y = gf256_add (y, p);

ifdef(<<CAREFUL_WITH_THAT_X>>, <<dnl
   (2) Any invalid argument value to gf256_log_2(), gf256_inv(), and
   gf256_div() will cause the program to print a message and dump core.
>>, <<dnl
   (2) The values of gf256_log_2(), gf256_inv(), and gf256_div() are
   undefined for invalid argument values; programs may continue to run,
   but the results may be wrong.
>>)dnl

SEE ALSO
   rsff-f9(3L), rs255-249(3L)
#endif MAN
*/

#ifndef GF256
#define GF256 unsigned char
#endif

/* Do the function-declarations before defining the macros.  */
#ifdef __STDC__
GF256 gf256_add (GF256 add1, GF256 add2);
GF256 gf256_minus (GF256 elem);
GF256 gf256_subtract (GF256 elem, GF256 sub);

GF256 gf256_antilog_2 (GF256 elem);
GF256 gf256_power_of_2 (GF256 power);
GF256 gf256_log_2 (GF256 elem);

GF256 gf256_mul (GF256 mul1, GF256 mul2);
GF256 gf256_exp (GF256 mant, int power);
GF256 gf256_inv (GF256 elem);
GF256 gf256_div (GF256 elem, GF256 div);

GF256 gf256_product (GF256 mul1, GF256 mul2, GF256 mul3);
GF256 gf256_term (GF256 coeff, int i, int n);
#endif /* __STDC__ */

#define gf256_add(add1, add2)		((add1) ^ (add2))
#define gf256_minus(elem)		(elem)
#define gf256_subtract(elem, sub)	((elem) ^ (sub))

extern GF256 _gf256_antilog_2_tab [256];
extern GF256 _gf256_log_2_tab [256];

#if notdef
#define gf256_antilog_2(elem)		(_gf256_antilog_2_tab [(elem) % 256])
#define gf256_power_of_2(power)		(_gf256_antilog_2_tab [(power) % 256])
#define gf256_log_2(elem)		(_gf256_log_2_tab [(elem) % 256])
#endif

#define gf256_antilog_2(elem)		(_gf256_antilog_2_tab [elem])
#define gf256_power_of_2(power)		(_gf256_antilog_2_tab [power])
ifdef(<<CAREFUL_WITH_THAT_X>>, <<dnl
#if __STDC__
extern GF256 _f_gf256_log_2 (GF256 elem);
#else
extern GF256 _f_gf256_log_2 ();
#endif
#define gf256_log_2(elem)		_f_gf256_log_2 (elem)
>>, <<dnl
#define gf256_log_2(elem)		(_gf256_log_2_tab [elem])
>>)dnl

#define gf256_mul(mul1, mul2)		\
  (((mul1) == 0 || (mul2) == 0)		\
  ? 0 : gf256_antilog_2 ((gf256_log_2 (mul1) + gf256_log_2 (mul2)) % 255))

#define gf256_exp(mant, power)		\
  (((mant) == 0) ? 0 : gf256_antilog_2 ((gf256_log_2 (mant) * (power)) % 255))

ifdef(<<CAREFUL_WITH_THAT_X>>, <<dnl
#if __STDC__
extern GF256 _gf256_inv (GF256 elem);
extern GF256 _gf256_div (GF256 elem, GF256 div);
#else
extern GF256 _gf256_inv ();
extern GF256 _gf256_div ();
#endif
#define gf256_inv(elem)			_gf256_inv (elem)
#define gf256_div(elem, div)		_gf256_div (elem, div)
>>, <<dnl
#define gf256_inv(elem)			\
  gf256_antilog_2 (255 - gf256_log_2 (elem))

#define gf256_div(elem, div)		\
  (((elem) == 0)			\
  ? 0 : gf256_antilog_2 ((gf256_log_2 (elem) + 255 - gf256_log_2 (div)) % 255))
>>)dnl

#define gf256_product(x, y, z)		\
  (((x) == 0 || (y) == 0 || (z) == 0)	\
  ? 0					\
  : gf256_antilog_2 (			\
      (gf256_log_2 (x) + gf256_log_2 (y) + gf256_log_2 (z)) % 255 \
    ) \
  )

#define gf256_term(coeff, i, n)		\
  (((coeff) == 0)			\
  ? 0 : gf256_antilog_2 ((gf256_log_2 (coeff) + (i)*(n)) % 255))

#ifdef	__cplusplus
}
#endif

#endif /* _gf256_h */
