
/* Written by Peter Ekberg, peda@lysator.liu.se */

#ifdef HAVE_CONFIG_H
#include "../src/config.h"
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <X11/pnm.h>

pixel **picture;
int cols;
int rows;
pixval maxval;

int maxcolors = 256;

pixel *colors;

char *name;

int
extractcolors(pixel **picture, pixel *colors)
{
  int c=0;
  int x=0, y=0;
  int i;
  pixel p;

  while(c<maxcolors && y<rows) {
    p = picture[y][x];
    for(i=0; i<c; i++)
      if(PPM_EQUAL(p, colors[i]))
	break;
    if(i==c) {
      colors[c] = p;
      c++;
    }
    if(c==maxcolors)
      return(0);
    if(++x == cols) {
      ++y;
      x=0;
    }
  }
  return(c);
}

void
write_c(FILE *f, pixel **picture, pixel *colors,
	int cols, int rows, int maxcolors)
{
  int x=0, y=0;
  int i;
  pixel p;

  fprintf(f,
	  "\n/* Automatically generated by the program ppm2c"
	  "\n   written by Peter Ekberg, peda@lysator.liu.se */\n");

  fprintf(f, "\nunsigned int %s_nr_colors = %d;", name, maxcolors);
  fprintf(f, "\nunsigned int %s_cols = %d;", name, cols);
  fprintf(f, "\nunsigned int %s_rows = %d;\n", name, rows);
  fprintf(f, "\nunsigned char %s_colors[] = {\n\t", name);
  for(i=0; i<maxcolors; i++) {
    p = colors[i];
    fprintf(f, "0x%02x, 0x%02x, 0x%02x",
	    PPM_GETR(p), PPM_GETG(p), PPM_GETB(p));
    if(i != maxcolors-1)
      fprintf(f, ",\n\t");
  }
  fprintf(f, "};\n\nunsigned char %s_pixels[] = {\n\t", name);
  while(y<rows) {
    p = picture[y][x];
    for(i=0; i<maxcolors; i++)
      if(PPM_EQUAL(p, colors[i])) {
	fprintf(f, "0x%02x", i);
	break;
      }
    if(++x == cols) {
      ++y;
      x=0;
    }
    if(y<rows) {
      if(x%8==0)
	fprintf(f, ",\n\t");
      else
	fprintf(f, ", ");
    }
  }
  fprintf(f, "};\n");
}

int
string_to_int(int *i, char *s)
{
  int v;
  char *end;

  if(!*s)
    return(-1);

  v = strtol(s, &end, 10);

  if(*end)
    return(-1);

  *i = v;
  return(0);
}

int
main(int argc, char *argv[])
{
  int optc;
  int c;

  do {
#ifdef HAVE_GETOPT_LONG
    static struct option longopts[] = {
      { "version",   0, 0, 'v' },
      { "help",      0, 0, 'h' },
      { "maxcolors", 1, 0, 'm' },
      { "name",      1, 0, 'n' },
      { 0,           0, 0,  0  },
    };

    optc = getopt_long(argc, argv, "vhm:n:", longopts, (int *)0);
#else
    optc = getopt(argc, argv, "vhm:n:");
#endif
    switch(optc) {
    case -1:
      break;
    case 'v':
      printf("pnmfill: version " VERSION "\n");
      exit(0);
    case 'h':
      printf("pnmfill: version " VERSION "\n");
      printf("options:\n"
#ifdef HAVE_GETOPT_LONG
	     "  -v --version   Display the version.\n"
	     "  -h --help      Display this text.\n"
	     "  -m --maxcolors Maximum allowed colors (default 256).\n"
	     "  -n --name      The base name of the C identifiers.\n");
#else
	     "  -v  Display the version.\n"
	     "  -h  Display this text.\n"
	     "  -m  Maximum allowed colors (default 256).\n"
	     "  -n  The base name of the C identifiers.\n");
#endif
      exit(0);
    case 'm':
      if(string_to_int(&maxcolors, optarg) || maxcolors<1 || maxcolors>256) {
	fprintf(stderr, "ppm2c: illegal argument to option '-m'\n");
	exit(1);
      }
      break;
    case 'n':
      name = strdup(optarg);
      if(!name) {
	fprintf(stderr, "ppm2c: insufficient memory\n");
	exit(1);
      }
      break;
    default:
      fprintf(stderr, "ppm2c: bad usage (see 'ppm2c -h')\n");
      exit(1);
    }
  } while(optc != -1);

  picture = ppm_readppm(stdin, &cols, &rows, &maxval);

  colors = (pixel *)malloc(maxcolors * sizeof(pixel));

  if(!colors) {
    fprintf(stderr, "ppm2c: insufficient memory\n");
    exit(1);
  }

  c = extractcolors(picture, colors);

  if(!c) {
    fprintf(stderr, "ppm2c: to many colors in picture\n");
    exit(1);
  }

  write_c(stdout, picture, colors, cols, rows, c);

  return(0);
}
