/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#define _BSD_SOURCE /* strdup() */

#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include "cdw_fs.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_ncurses.h"
#include "cdw_widgets.h"
#include "cdw_window.h"
#include "cdw_debug.h"
#include "cdw_sys.h"
#include "gettext.h"
#include "canonicalize.h"
#include "cdw_file_selector.h"
#include "cdw_form.h"
#include "cdw_main_window.h"





/**
   File picker: widget that allows user to select a single file or
   directory.

   The widget uses cdw_file_selector_t widget to access and browse
   native file system.

   The widget uses safe input line widget to allow user to enter the
   path manually or edit it.

   There are no public _new() or _delete() functions for file
   picker. It is assumed that there can be only one instance of file
   picker at given time.

   This component is built as follows:
   picker->file_selector->fs_browser->display
   \li first there is a list display, providing means for displaying
   and scrolling list of arbitrary items
   \li then there is fs_browser, which browses native file system
   hierarchy and displays contents of currently visited directory in
   the display
   \li then there is file selector, which uses fs_browser to select a
   file
   \li then there is picker, which uses file selector as one of two
   methods to select a file (the second one being text input line)
*/





/* Form field indexes. */
enum {
	f_message_l = 0,
	f_selector_i,
	f_inputline_l,
	f_inputline_i,
	f_ok_b,
	f_cancel_b,

	CDW_FILE_PICKER_N_FIELDS
};





typedef struct {
	int expected_file_type;
	int expected_mode;
	int expected_new_or_existing;

	/* "file selector" and "input line" parts of file picker can
	   be accessed primarily by calling
	   cdw_form_get_WIDGET_NAME (picker->cdw_form, field_index_i).
	   For convenience I'm adding these pointers here. */
	cdw_file_selector_t *file_selector;  /* Here user can select file by browsing native file system. */
	CDW_SAFE_INPUT_LINE *input_line;     /* Here user can enter path manually and edit it. */

	/* Selected path will be stored here during operation of file
	   picker. Every time user changes selection in file selector,
	   or enters new path in input line and then leaves the input
	   line, this fullpath will be updated. */
	char *fullpath;

	/* Form used in main window of file picker. */
	cdw_form_t *cdw_form;

	/* +1 for guard. */
	FIELD *fields[CDW_FILE_PICKER_N_FIELDS + 1];
} cdw_file_picker_t;

cdw_file_picker_t file_picker;





static void     cdw_file_picker_init(cdw_file_picker_t *picker);
static cdw_rv_t cdw_file_picker_synchronize_from_file_selector(cdw_file_picker_t *picker, int key);
static cdw_rv_t cdw_file_picker_synchronize_from_input_line(cdw_file_picker_t *picker, int key, bool include_file_selector);
static int      cdw_file_picker_driver(cdw_file_picker_t *picker);
static void     cdw_file_picker_destroy(cdw_file_picker_t *picker);
static cdw_rv_t cdw_file_picker_build(cdw_file_picker_t *picker, char const * title, char const * message, char const * initial_fullpath);
static cdw_rv_t cdw_file_picker_build_fields(cdw_file_picker_t *picker, char const * message);





/**
   \brief Function that brings up file picker window

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   There are no public _new()/_show()/_delete() functions for file
   picker, just this one function. You call it, user picks one file,
   the function returns.

   Initial path is passed through \p caller_fullpath.  Fullpath to
   selected file is returned through the same \p caller_fullpath.  If
   selecting of path is canceled by user (e.g. by pressing Escape
   key), \p caller_fullpath is not modified.

   \param title
   \param message
   \param caller_fullpath - input/output parameter
   \param expected_file_type - bit sum of CDW_FS_FILE, CDW_FS_DIR
   \param expected_mode - bit sum of R_OK, W_OK, X_OK, F_OK (see "man 2 access")
   \param expected_new_or_existing - bit sum of CDW_FS_NEW, CDW_FS_EXISTING

   \return CDW_OK when a file has been selected
   \return CDW_CANCEL when process of selecting a file has been canceled by user
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_file_picker(char const * title, char const * message, char **caller_fullpath, int expected_file_type, int expected_mode, int expected_new_or_existing)
{
	cdw_file_picker_init(&file_picker);
	file_picker.expected_file_type = expected_file_type;
	file_picker.expected_mode = expected_mode;
	file_picker.expected_new_or_existing = expected_new_or_existing;

	if (CDW_OK != cdw_file_picker_build(&file_picker, title, message, *caller_fullpath)) {
		cdw_vdm ("ERROR: failed to build file picker\n");
		cdw_file_picker_destroy(&file_picker);
		return CDW_ERROR;
	}

	cdw_rv_t retval = CDW_OK;
	int key = cdw_file_picker_driver(&file_picker);
	if (key == CDW_KEY_ENTER) {

		/* Current path is correct, file picker can return it to user. */

		if (file_picker.expected_file_type == CDW_FS_FILE) {
			/* This test is really for non-existing
			   files. Remove ending slashes; in theory
			   there should be no more than one such char
			   at the end of canonical_fullpath. */
			size_t len = strlen(file_picker.fullpath);
			cdw_assert (len > 0, "ERROR: len of canonical fullpath == 0: \"%s\"\n", file_picker.fullpath);
			if (len > 1) {
				if (file_picker.fullpath[len - 1] == '/') {
					file_picker.fullpath[len - 1] = '\0';
				}
				if (len > 2) {
					cdw_assert (file_picker.fullpath[len - 1] != '/',
						    "ERROR: multiple slashes at the end of canonical fullpath \"%s\" (last one already removed)\n",
						    file_picker.fullpath);
				}
			}
		}


		/* Picking a file finished successfully. Save the result. */
		if (CDW_OK != cdw_string_set(caller_fullpath, file_picker.fullpath)) {
			cdw_vdm ("ERROR: failed to set caller's fullpath at return using path \"%s\"\n", file_picker.fullpath);
			retval = CDW_ERROR;
		}
	} else if (key == CDW_KEY_ESCAPE) {
		retval = CDW_CANCEL;
	} else { /* key == -1 */
		retval = CDW_ERROR;
	}

	cdw_file_picker_destroy(&file_picker);

	/* Redraw parent. */
	cdw_main_window_wrefresh();

	return retval;
}





/**
   \brief Initialize fields of file picker data structure

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   \param picker - pointer to existing file picker data structure.
*/
void cdw_file_picker_init(cdw_file_picker_t *picker)
{
	picker->expected_file_type = 0;
	picker->expected_mode = 0;
	picker->expected_new_or_existing = 0;

	picker->file_selector = (cdw_file_selector_t *) NULL;
	picker->input_line = (CDW_SAFE_INPUT_LINE *) NULL;
	picker->fullpath = (char *) NULL;
	picker->cdw_form = (cdw_form_t *) NULL;

	for (int i = 0; i < CDW_FILE_PICKER_N_FIELDS + 1; i++) {
		picker->fields[i] = (FIELD *) NULL;
	}

	return;
}





/**
   \brief Propagate file highlighted in picker's file selector to all relevant elements of file picker

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   On specific events in picker's file selector (a central part of file
   picker window), information about current/last highlighted file
   needs to be propagated to other elements of file picker window (all
   elements need to have the full path to the file synchronized).

   File selector of given \p picker is the origin of the information about the full path.

   Targets of the information are input line (another element of file
   picker window) and picker's internal fullpath buffer.

   \param picker - file picker
   \param key - last key pressed (for debug purposes only)

   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_file_picker_synchronize_from_file_selector(cdw_file_picker_t *picker, int key)
{
	/* Synchronization means that:
	   - picker->file_selector->fs_browser  - origin
	   - picker->input_line                 - target 1
	   - picker->fullpath                   - target 2

	   must have the same value and the value must be taken from
	   picker->fs_browser. */


	cdw_vdm ("INFO: SYNCHRONIZATION POINT: key \"%s\" in file selector\n", cdw_ncurses_key_label(key));


	/* Origin. */
	cdw_file_t * const file = cdw_fs_browser_get_current_file(picker->file_selector->fs_browser);
	char * fullpath = strdup(file->fullpath);
	if (file->is_ref_to_parent_dir) {

		/* Remove trailing ".." from path - the dots shouldn't
		   be displayed in input line. */

		/* Just to be 100% sure that me and the code base
		   agree on contents of the fullpath string. */
		cdw_assert (fullpath[file->name_start] == '.'
			    && fullpath[file->name_start + 1] == '.',
			    "ERROR: fullpath to parent dir does not end with '..': \"%s\"\n", fullpath);

		fullpath[file->name_start] = '\0';
	}
	/* CAN_MISSING, because we will check the path against
	   picker->expected_new_or_existing ourselves later. */
	char * origin = canonicalize_filename_mode(fullpath, CAN_MISSING);
	if (!origin) {
		/* Canonicalization of path failed. */
		cdw_assert (origin, "ERROR: failed to canonicalize path \"%s\"\n", fullpath);
		cdw_vdm ("ERROR: failed to canonicalize path \"%s\"\n", fullpath);
		CDW_STRING_DELETE (fullpath);
		return CDW_ERROR;
	}


	cdw_safe_input_line_set_content(picker->input_line, origin);    /* Target 1. */
	cdw_string_set(&picker->fullpath, origin);                      /* Target 2. */


	cdw_vdm ("INFO: paths synchronized to be \"%s\"\n", origin);


	CDW_STRING_DELETE (fullpath);
	CDW_STRING_DELETE (origin);

	return CDW_OK;
}





/**
   \brief Propagate file entered in input line to relevant elements of file picker window

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   On Enter key pressed in input line, information about file entered
   into the input line needs to be propagated to other elements of
   file picker window (all elements need to have the full path to the
   file synchronized).

   Input line is the origin of the information about the full path.

   Targets of the information are file system browser (another element
   of file picker window) and picker's internal fullpath buffer.

   The function is not very sophisticated. When it comes to updating
   file selector, it just checks if the path is valid, and if it
   is, it asks the file selector to browse to the path.

   Synchronizing to file selector may be skipped, set \p
   include_file_selector to false in that case.

   \param picker - file picker
   \param key - last key pressed (for debug purposes only)
   \param include_file_selector - synchronize to file selector too?

   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_file_picker_synchronize_from_input_line(cdw_file_picker_t *picker, int key, bool include_file_selector)
{
	/* Synchronization means that:
	   - picker->input_line                 - origin
	   - picker->file_selector->fs_browser  - target 1
	   - picker->fullpath                   - target 2

	   must have the same value and the value must be taken from
	   input_line. */

	cdw_vdm ("INFO: SYNCHRONIZATION POINT: key \"%s\" in input line\n", cdw_ncurses_key_label(key));


	/* Origin. */
	char *path = cdw_safe_input_line_get_content(picker->input_line);
	/* CAN_MISSING, because we will check the path against
	   picker->expected_new_or_existing ourselves later. */
	char *origin = canonicalize_filename_mode(path, CAN_MISSING);
	if (!origin) {
		/* Canonicalization of path failed. */
		cdw_assert (origin, "ERROR: failed to canonicalize path \"%s\"\n", path);
		cdw_vdm ("ERROR: failed to canonicalize path \"%s\"\n", path);
		CDW_STRING_DELETE (path);
		return CDW_ERROR;
	}
	CDW_STRING_DELETE (path);


	/* Target 1. */
	int browse = -1;
	if (include_file_selector) {
		/* For now it's only F_OK, we will check against picker->expected_* later. */
		if (0 == cdw_fs_check_existing_path(origin, F_OK, CDW_FS_FILE)) {

			/* In order to browse to file 'origin', we need to be able to
			   enter its dir (R_OK and X_OK). */
			char *dirpath = cdw_fs_shorten_fullpath(origin);

			if (dirpath && dirpath[0] != '\0') {
				if (0 == cdw_fs_check_existing_path(dirpath, R_OK | X_OK, CDW_FS_DIR)) {
					browse = cdw_fs_browser_browse_to_file(picker->file_selector->fs_browser, origin);
				}
			}
			CDW_STRING_DELETE (dirpath);

		} else if (0 == cdw_fs_check_existing_path(origin, F_OK, CDW_FS_DIR)) {
			browse = cdw_fs_browser_browse_into_dir(picker->file_selector->fs_browser, origin);

		} else {
			;
		}
		if (browse == 0) {
			cdw_file_selector_update_window_decoration(picker->file_selector);
		}
	} else {
		browse = 0;
	}


	/* Target 2. */
	if (browse == 0) {
		/* Setting of first target succeeded, it's safe to set second target. */
		cdw_string_set(&picker->fullpath, origin);
		cdw_vdm ("INFO: paths synchronized to be \"%s\"\n", origin);
	} else {
		cdw_vdm ("ERROR: failed to synchronize paths from \"%s\"\n", origin);
	}


	CDW_STRING_DELETE (origin);


	return CDW_OK;
}





/* Constants for layout of elements in file picker window. */
#define first_col                                 1  /* Main message and some labels start in leftmost column of subwindow. */
#define top_label_row                             1  /* Main message is displayed on top of subwindow. */
#define window_n_cols                          COLS
#define window_n_lines                        LINES
#define subwindow_n_cols        (window_n_cols - 2)
#define subwindow_n_lines      (window_n_lines - 2)
#define label_n_cols         (subwindow_n_cols - 2)





/**
   \brief Driver function for file picker

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   The function handles keys pressed in \p picker.
   It returns on Escape/Q/q key.

   \param picker - file picker widget that receives keys

   \return CDW_KEY_ENTER when user has selected a file
   \return CDW_KEY_ESCAPE when user canceled selecting a file
   \return -1 on errors
*/
int cdw_file_picker_driver(cdw_file_picker_t *picker)
{
	/* Initial focus on "Cancel" button. */
	int last_fi = f_cancel_b;
	cdw_form_driver_go_to_field(picker->cdw_form, last_fi);

	int key = 'a'; /* safe initial value */

	while (key != CDW_KEY_ESCAPE && key != 'q' && key != 'Q') {

		key = cdw_form_driver_new(picker->cdw_form, &last_fi);

		if (last_fi == f_selector_i) {
			if (cdw_fs_browser_is_vscroll_key(key)) {
				/* Scrolling up or down in file system
				   view. Propagate information about
				   currently highlighted file to
				   interested parties. */
				cdw_file_picker_synchronize_from_file_selector(picker, key);

			} else if (key == CDW_KEY_TAB || key == CDW_KEY_BTAB) {
				/* cdw_form will move focus to
				   next/previous widget, but we must
				   make sure that info about currently
				   highlighted file is propagated to
				   interested parties. */
				cdw_file_picker_synchronize_from_file_selector(picker, key);

			} else if (cdw_fs_browser_is_fs_navigation_key(key)) {
				cdw_file_picker_synchronize_from_file_selector(picker, key);

			} else {
				;
			}

		} else if (last_fi == f_inputline_i) {

			if (key == CDW_KEY_TAB || key == KEY_DOWN) {
				/* Tab key will move focus to OK
				   button. Synchronize, but to
				   picker->fullpath only. See
				   CDW_KEY_ENTER for synchronizing to
				   file selector too. */
				cdw_file_picker_synchronize_from_input_line(picker, key, false);

			} else if (key == CDW_KEY_BTAB || key == KEY_UP) {
				/* We will go from input line to file
				   selector. Cursor is now positioned
				   in file selector, so synchronize
				   from it. */
				cdw_file_picker_synchronize_from_file_selector(picker, 0);

			} else if (key == CDW_KEY_ENTER) {
				/* User edited a path in input line,
				   and is now confirming it with
				   Enter. */
				cdw_file_picker_synchronize_from_input_line(picker, key, true);

			} else if (key == KEY_EXIT) {
				/* Some error in input line driver. */
				cdw_vdm ("ERROR in input line driver\n");
				key = CDW_KEY_ESCAPE;
			} else {
				;
			}

		} else if (last_fi == f_ok_b) {

			if (key == CDW_KEY_ENTER) {
				/* Moving from file selector to other
				   widget or from input line to other
				   widget should have been followed by
				   assigning current path to
				   picker->fullpath. Evaluate this
				   fullpath and see if we are ready to
				   return it to caller. */
				int e = cdw_fs_check_fullpath(picker->fullpath, picker->expected_file_type, picker->expected_mode, picker->expected_new_or_existing);
				if (e == 0) {
					break;
				} else if (e == EEXIST) {
					; /* Already handled in cdw_fs_check_fullpath(). Pass. */
				} else {
					/* Report and pass. */
					cdw_fs_errno_handler(e);
				}

			} else {
				/* Escape, Tab, or whatever - will be handled by loop condition */
			}

		} else if (last_fi == f_cancel_b) {
			/* Button's driver either returns
			   CDW_KEY_ESCAPE, which here will lead to
			   exiting the while() loop, or other key that
			   will be ignored. */
		} else {
			cdw_vdm ("ERROR: incorrect field index %d\n", last_fi);
		}

	}

	return key;
}





/**
   \brief Destroy file picker

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   Delete/free all members of \p picker, but don't free the \p picker itself.

   \param picker - file picker to destroy
*/
void cdw_file_picker_destroy(cdw_file_picker_t *picker)
{
	/* This order of these four function calls minimizes number of
	   problems reported by valgrind. */
	cdw_form_delete_form_objects(picker->cdw_form);
	cdw_window_delete(&picker->cdw_form->subwindow);
	cdw_window_delete(&picker->cdw_form->window);
	cdw_form_delete(&(picker->cdw_form));

	/* Notice that picker->file_selector and picker->input_line are
	   deleted by cdw_form_delete_form_objects(). */

	CDW_STRING_DELETE (picker->fullpath);

	return;
}





/**
   \brief Create file picker window

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   \p picker must be initialized before passing it to this function.

   \param picker
   \param title - title of picker window
   \param message
   \param initial_fullpath

   \return CDW_OK if file picker has been built successfully
   \return CDW_ERROR otherwise
*/
cdw_rv_t cdw_file_picker_build(cdw_file_picker_t *picker, char const * title, char const * message, char const * initial_fullpath)
{
	if (!initial_fullpath) {
		cdw_vdm ("ERROR: invalid initial fullpath \"%s\"\n", initial_fullpath);
		return CDW_ERROR;
	}

	/* FIXME: what if "initial_fullpath is not canonical? */
	if (CDW_OK != cdw_string_set(&picker->fullpath, initial_fullpath)) {
		cdw_vdm ("ERROR: failed to set picker's initial fullpath \"%s\"\n", picker->fullpath);
		return CDW_ERROR;
	}

	picker->cdw_form = cdw_form_new(CDW_FILE_PICKER_N_FIELDS);
	if (!picker->cdw_form) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_ERROR;
	}

	int begin_y = (LINES - window_n_lines) / 2;
	int begin_x = (COLS - window_n_cols) / 2;
	picker->cdw_form->window = cdw_window_new((WINDOW *) NULL,
						  window_n_lines, window_n_cols,
						  begin_y, begin_x,
						  CDW_COLORS_DIALOG,
						  title,
						  /* 2TRANS: this is tip at the bottom of window - user can
						     switch between window elements using tab key */
						  _("Use 'Tab' key to move"));

	if (!picker->cdw_form->window) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_ERROR;
	}

	picker->cdw_form->subwindow = cdw_window_new(picker->cdw_form->window,
						     subwindow_n_lines, subwindow_n_cols,
						     1, 1,
						     CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);

	if (!picker->cdw_form->subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_ERROR;
	}

	cdw_file_picker_build_fields(picker, message);

	picker->cdw_form->form = cdw_ncurses_new_form(picker->cdw_form->window,
						      picker->cdw_form->subwindow,
						      picker->cdw_form->fields);
	if (!picker->cdw_form->form) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_ERROR;
	}

	cdw_form_redraw_widgets(picker->cdw_form);

	cdw_form_set_button_return_key(picker->cdw_form, f_ok_b, CDW_KEY_ENTER);
	cdw_form_set_button_return_key(picker->cdw_form, f_cancel_b, CDW_KEY_ESCAPE);

	picker->file_selector = cdw_form_get_file_selector(picker->cdw_form, f_selector_i);
	cdw_widget_add_return_keys_pointer(&picker->file_selector->widget,
					   
					   /* For navigating out of file selector widget. */
					   CDW_KEY_TAB, CDW_KEY_BTAB,
					   
					   /* Vertical scroll keys (compare with
					      cdw_fs_browser_is_vscroll_key()). They
					      should already be return keys of
					      browser->display, but just in case
					      add them here as well. */
					   KEY_DOWN,
					   KEY_UP,
					   KEY_NPAGE,
					   KEY_PPAGE,
					   KEY_HOME,
					   KEY_END,
					   
					   0);

	picker->input_line = cdw_form_get_safe_input_line(picker->cdw_form, f_inputline_i);
	cdw_widget_add_return_keys(&picker->input_line->widget, CDW_KEY_ENTER, 0);


	wrefresh(picker->cdw_form->subwindow);
	wrefresh(picker->cdw_form->window);

	return CDW_OK;
}





/**
   \brief Build form fields for file picker

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   \param picker - file picker
   \param message - message displayed in upper part of picker's window

   \return CDW_OK on success
   \return CDW_ERROR otherwise
*/
cdw_rv_t cdw_file_picker_build_fields(cdw_file_picker_t *picker, char const * message)
{
	/* Number of lines for file selector widget. */
	int s_n_lines = subwindow_n_lines
		- 2       /* Static label on top. */
		- 5       /* Dynamic label + input line. */
		- 2;      /* Buttons. */

	if (s_n_lines < 3) { /* 2 for upper and lower border + at least 1 for contents. */
		cdw_vdm ("ERROR: calculated number of lines for file selector is too small: %d\n", s_n_lines);
		return CDW_ERROR;
	}

	cdw_form_descr_t descr[] = {
		/* widget type                                  begin_y          begin_x           n_cols     n_lines      field enum                  data1                  data2 */

		/* 2TRANS: this is label in file picker window */
		{ CDW_WIDGET_ID_STATIC_LABEL,                         1,       first_col,    label_n_cols,          1,   f_message_l,               message,                      0 },

		{ CDW_WIDGET_ID_FILE_SELECTOR,                        3,       first_col,    label_n_cols,  s_n_lines,  f_selector_i,      picker->fullpath,                      0 },

		{ CDW_WIDGET_ID_DYNAMIC_LABEL,    subwindow_n_lines - 5,       first_col,    label_n_cols,          1, f_inputline_l,                    "",                      0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,  subwindow_n_lines - 4,       first_col,    label_n_cols,          1, f_inputline_i,      picker->fullpath,           PATH_MAX - 2 },

		/* 2TRANS: button label, it refers to erasing optical disc */
		{ CDW_WIDGET_ID_BUTTON,           subwindow_n_lines - 2,       first_col,               2,          1,        f_ok_b,               _("OK"),      CDW_COLORS_DIALOG },
		/* 2TRANS: button label */
		{ CDW_WIDGET_ID_BUTTON,           subwindow_n_lines - 2,  first_col + 10,               2,          1,    f_cancel_b,           _("Cancel"),      CDW_COLORS_DIALOG },

		/* guard */
		{ -1,                                                 0,               0,               0,          0,             0,         (void *) NULL,                      0 }};


	picker->cdw_form->n_fields = CDW_FILE_PICKER_N_FIELDS;
	picker->cdw_form->fields = picker->fields;
	/* The function adds guard at the end of fields. */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, picker->cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_bind_input_and_label(picker->cdw_form, f_inputline_i, f_inputline_l);

		cdw_file_selector_t * const file_selector = cdw_form_get_file_selector(picker->cdw_form, f_selector_i);
		file_selector->return_on_navigation_keys = true; /* We need to be able to update e.g. input line on navigation events in file selector. */

		return CDW_OK;
	}
}
