/*
 * Copyright (C) 1999, 2000, 2001  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <string.h>
#include <iostream>
#include <fstream>
#include <stdlib.h>

#include "genfun.h"

#include "messages.h"
#include "tags.h"
#include "keys.h"
#include "textgen.h"
#include "fileutil.h"
#include "docgenerator.h"
#include "lineoutputgenerator.h"

// globals
#include "linenumdigit.h"
#include "cmdlineargs.h"
#include "scanner_map.h"
#include "generators.h"
#include "mainoutputbuffer.h"
#include "mainoutputgenerator.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

//----------------------------------------------------------------------------//
//	Don Stauffer																								June 2003
//----------------------------------------------------------------------------//
#include	"srcpretranslator.h"
#include	"srcuntabifier.h"

Pretranslator	*pretranslator;
//----------------------------------------------------------------------------//

using std::cout;
using std::cerr;
using std::endl;
using std::ofstream;

extern int tabSpaces; // FIXME

static char *get_input_file_name(const char *file_name);

void
processFile( const char *source_language, const char *output_format,
             const char *inputFileName, const char *outputFileName,
             short entire_doc, bool gen_version, bool gen_line, bool gen_ref )
{
  FILE *in = 0;
  bool deleteOStream = false ;

  if ( outputFileName ) {
    sout = new ofstream(outputFileName) ;
    if ( ! (*sout) ) {
      cerr << "Error in creating " << outputFileName << " for output" << endl ;
      exit(1) ;
    }
    deleteOStream = true;
  }

  if (inputFileName)
    {
      unsigned int lines = get_line_count (inputFileName);

      line_num_digit = 0;
      while (lines)
        {
          ++line_num_digit;
          lines /= 10;
        }
    }
  else
    line_num_digit = 5;
  // if we read from stdin, we can't read the file in advance and
  // check how many lines of code it contains.  In this case set
  // the number of digit for the line number to 5.

  if (inputFileName)
    {
      const char *input_file_name = get_input_file_name (inputFileName);

      in = freopen (input_file_name, "r", stdin);
      if (!in)
        {
          cerr << "Error in opening " << inputFileName
               << " for input" << endl ;
          exit(1) ;
        }

      delete input_file_name;
    }

  /*
   * Use default values for any options not provided
   */
  if (sout == 0) {
    sout = &cout;
  }

  if (in == 0) {
    ; /* Well stdin already points to stdin so, .... */
  }

  if (gen_line)
    outputgenerator = new LineOutputGenerator(*outputbuffer, *sout, gen_ref);
  else
    outputgenerator = new OutputGenerator(*outputbuffer, *sout);

  if ( entire_doc ) {
    GlobalDocGenerator->set_gen_version (gen_version);
    GlobalDocGenerator->generate_top ();
  }

//----------------------------------------------------------------------------//
//	Don Stauffer																								June 2003
//----------------------------------------------------------------------------//
  if (tabSpaces)
    pretranslator = new Untabifier (tabSpaces);
  else if (args_info.line_number_given)
    pretranslator = new Untabifier(8);
  else
    pretranslator = new Pretranslator ();
//----------------------------------------------------------------------------//

  printMessage( "translating source code... ", cerr ) ;

  GlobalDocGenerator->generate_start_doc ();

  // decide which lexer to call
  ScannerMap scanner_map;

  if (source_language)
    {
      if (! scanner_map.invoke_scanner (source_language))
        {
          cerr << PACKAGE << ": ";
          cerr << "source language " << source_language
               << " not handled" << endl;
          exit (1);
        }
    }
  else
    {
      if (! inputFileName)
        {
          cerr << PACKAGE << ": ";
          cerr << "when using stdin, please specify a source language"
               << endl;
          exit (1);
        }

      string file_ext = get_file_extension (inputFileName);

      if (file_ext == "")
        {
          cerr << PACKAGE << ": ";
          cerr << "no file extension; please specify a source language"
               << endl;
          exit (1);
        }

      if (! scanner_map.invoke_scanner (file_ext))
        {
          cerr << PACKAGE << ": ";
          cerr << "unknown file extension " << file_ext << endl;
          exit (1);
        }
    }

  outputgenerator->generate();

  GlobalDocGenerator->generate_end_doc ();

  printMessage( "done !", cerr ) ;

  if ( entire_doc ) {
    GlobalDocGenerator->generate_bottom ();
  }

  sout->flush ();

  if ( deleteOStream )
    delete sout ;

  delete outputgenerator;

//----------------------------------------------------------------------------//
//	Don Stauffer																								June 2003
//----------------------------------------------------------------------------//
  delete pretranslator;
//----------------------------------------------------------------------------//

}

char *
get_input_file_name(const char *file_name)
{
  unsigned int length = strlen (CHROOT_INPUT_DIR);

  if (length != 0)
    {
      char *new_name = new char [strlen (file_name) + length + 1];
      strcpy (new_name, strdup (CHROOT_INPUT_DIR));
      return strcat (new_name, file_name);
    }
  else
    return strdup (file_name);
}

//----------------------------------------------------------------------------//
//	Don Stauffer																								June 2003
//----------------------------------------------------------------------------//
void
generate (const char *s)
{
  pretranslator->generate (s);
}

void
generate (const char *s, int start, int	end)
{
  pretranslator->generate (s, start, end);
}

void
generateln (const char *s)
{
  pretranslator->generateln (s);
}

void
generate_preproc ( const char  *s )
{
  pretranslator->generate_preproc (s);
}

void
generate_normal	(const char *s)
{
  pretranslator->generate_normal(s);
}

void
generateNewLine	(const char *s)
{
  if (s)
    {
      int len = strlen (s);
      bool found_carriage_return = false;
      for (int i = 0; i < len; ++i)
        {
          // before generating the real line break we generate all the
          // other previous characters (probably \r)
          if (s[i] == '\n')
            generateln ("");
          else
            {
              bool is_cr = (s[i] == '\r');
              if (! is_cr || ! found_carriage_return)
                generate(s, i, i); // a single character
              if (is_cr)
                found_carriage_return = true;
              // we avoid to generate two consecutive \r that would
              // result ugly (e.g., in html)
            }
        }
    }
  else
    generateln("");
}

void
generateTab(void)
{
  pretranslator->generateTab();
}

void
startComment(const char	*s)
{
  pretranslator->startComment(s);
}

void
endComment(const char *s)
{
  pretranslator->endComment(s);
}

void
generateComment(const char *s)
{
  pretranslator->generateComment(s);
}

void
startString(const char *s)
{
  pretranslator->startString(s);
}

void
endString(const char *s)
{
  pretranslator->endString(s);
}

void
generateString(const char *s)
{
  pretranslator->generateString(s);
}

void
generateKeyWord(const char *s)
{
  pretranslator->generateKeyWord(s);
}

void
generateBaseType(const char *s)
{
  pretranslator->generateBaseType(s);
}

void
generatePreProc(const char *s)
{
  pretranslator->generatePreProc(s);
}

void
generateNumber(const char *s)
{
  pretranslator->generateNumber(s);
}

void
generateSymbol(const char *s)
{
  pretranslator->generateSymbol(s);
}

void
generateFunction(const char *s)
{
  pretranslator->generateFunction(s);
}

void
generateCBracket(const char *s)
{
  pretranslator->generateCBracket(s);
}

void
generate( const char * elem, const char * s )
{
  pretranslator->generate(elem, s);
}

void
generate( const char * elem, const char * s, int start, int end )
{
  pretranslator->generate(elem, s, start, end);
}

void
generate_preproc(const char * elem, const char * s)
{
  pretranslator->generate_preproc(elem, s);
}

void
generateln( const char * elem, const char * s )
{
  pretranslator->generateln(elem, s);
}

void
generateEntire( const char * elem, const char * s )
{
  pretranslator->generateEntire(elem, s);
}

void
beginText( const char * elem, const char * s )
{
  pretranslator->beginText(elem, s);
}

void
endText( const char * elem, const char * s )
{
  pretranslator->endText(elem, s);
}
