//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------
#include "midibus.h"
#include "config.h"
#include "sys/poll.h"

midibus::midibus( int a_localclient,
		  int a_destclient, 
		  int a_destport, 
		  snd_seq_t *a_seq, 
		  const char *a_client_name, 
		  const char *a_port_name,
		  char a_id, int a_queue )
{
 
    /* set members */
    m_local_addr_client = a_localclient;
    m_dest_addr_client = a_destclient;
    m_dest_addr_port   = a_destport;
    m_seq            = a_seq;
    m_queue          = a_queue;

    m_id = a_id;
    m_clocking = false;

    /* copy names */
    char tmp[60];
    snprintf( tmp, 59, "[%d] %d:%d (%s)",
	      m_id,
	      m_dest_addr_client,
	      m_dest_addr_port,
	      a_port_name );

    m_name = tmp;

}



void 
midibus::lock( )
{
   m_mutex.lock();
}


void 
midibus::unlock( )
{ 
    m_mutex.unlock();
}





bool midibus::init_out( )
{
    /* temp return */
    int ret;

    /* create ports */
    ret = snd_seq_create_simple_port(m_seq, 
				     "seq24 out",
			     	     SND_SEQ_PORT_CAP_NO_EXPORT |
				     SND_SEQ_PORT_CAP_READ,
				     SND_SEQ_PORT_TYPE_MIDI_GENERIC |
				     SND_SEQ_PORT_TYPE_APPLICATION );
    m_local_addr_port = ret;

    if ( ret < 0 ){
        printf( "snd_seq_create_simple_port(write) error\n");
        return false;
    }

    /* connect to */
    ret = snd_seq_connect_to( m_seq, 
			      m_local_addr_port, 
			      m_dest_addr_client, 
			      m_dest_addr_port );
    if ( ret < 0 ){
        printf( "snd_seq_connect_to(%d:%d) error\n", m_dest_addr_client, m_dest_addr_port);
        return false;
    }

    return true;
}



bool midibus::init_in( )
{

    /* temp return */
    int ret;

    /* create ports */
    ret = snd_seq_create_simple_port(m_seq, 
                                     "seq24 in",
                                     SND_SEQ_PORT_CAP_NO_EXPORT |
                                     SND_SEQ_PORT_CAP_WRITE,
                                     SND_SEQ_PORT_TYPE_MIDI_GENERIC |
                                     SND_SEQ_PORT_TYPE_APPLICATION );
    m_local_addr_port = ret;

    if ( ret < 0 ){
        printf( "snd_seq_create_simple_port(read) error\n");
        return false;
    }

    snd_seq_port_subscribe_t *subs;
    snd_seq_port_subscribe_alloca(&subs);
    snd_seq_addr_t sender, dest;

    /* the destinatino port is actually our local port */
    sender.client = m_dest_addr_client; 
    sender.port = m_dest_addr_port;   
    dest.client = m_local_addr_client;
    dest.port = m_local_addr_port;

    /* set in and out ports */
    snd_seq_port_subscribe_set_sender(subs, &sender);
    snd_seq_port_subscribe_set_dest(subs, &dest);

    /* use the master queue, and get ticks */
    snd_seq_port_subscribe_set_queue(subs, m_queue);
    snd_seq_port_subscribe_set_time_update(subs, 1);

    /* subscribe */
    ret = snd_seq_subscribe_port(m_seq, subs);

    if ( ret < 0 ){
        printf( "snd_seq_connect_from(%d:%d) error\n", m_dest_addr_client, m_dest_addr_port);
        return false;
    }

    return true;
}



int
midibus::get_id( )
{
    return m_id;
}


void 
midibus::print()
{
    printf( "%s" , m_name.c_str() );
}

string
midibus::get_name()
{
    return m_name;
}

midibus::~midibus()
{

}


/* takes an native event, encodes to alsa event, 
   puts it in the queue */
void 
midibus::play( event *a_e24, unsigned char a_channel )
{
    lock();

  

		snd_seq_event_t ev;
		
		/* alsa midi parser */
		snd_midi_event_t *midi_ev;
		
		/* temp for midi data */
		unsigned char buffer[3];
		
		/* fill buffer and set midi channel */
		buffer[0] = a_e24->get_status();
		buffer[0] += (a_channel & 0x0F);
		
		a_e24->get_data( &buffer[1], &buffer[2] );
		
		snd_midi_event_new( 10, &midi_ev );
		
		/* clear event */
		snd_seq_ev_clear( &ev );
		snd_midi_event_encode( midi_ev,
							   buffer,
							   3,
							   &ev ); 
		
		snd_midi_event_free( midi_ev );
		
		/* set source */
		snd_seq_ev_set_source(&ev, m_local_addr_port );
		snd_seq_ev_set_subs(&ev);
		
		/* set tag unique to each sequence for removal purposes */
		//ev.tag = a_tag;
		
		// its immediate 
		snd_seq_ev_set_direct( &ev );
		
		/* pump it into the queue */
		snd_seq_event_output(m_seq, &ev);
	

    unlock();
}


inline long 
min ( long a, long b ){

  if ( a < b ) 
    return a;
  return b;

}

/* takes an native event, encodes to alsa event, 
   puts it in the queue */
void 
midibus::sysex( event *a_e24 )
{
    lock();

    snd_seq_event_t ev;
   
    /* clear event */
    snd_seq_ev_clear( &ev );
    snd_seq_ev_set_priority( &ev, 1 );

    /* set source */
    snd_seq_ev_set_source(&ev, m_local_addr_port );
    snd_seq_ev_set_subs(&ev);
    
    // its immediate 
    snd_seq_ev_set_direct( &ev );

    
    unsigned char *data = a_e24->get_sysex();
    long data_size =  a_e24->get_size();
    for( long offset = 0;
	 offset < data_size;
	 offset += c_midibus_sysex_chunk ){

      long data_left = data_size - offset;

      snd_seq_ev_set_sysex( &ev, 
			       min( data_left, c_midibus_sysex_chunk), 
			       &data[offset] ); 
      
      /* pump it into the queue */
      snd_seq_event_output_direct(m_seq, &ev);

      usleep(80000);
      
      flush();

    }

    unlock();
}


// flushes our local queue events out into ALSA
void 
midibus::flush()
{
    lock();

    snd_seq_drain_output( m_seq );

    unlock();
} 





/* gets it a runnin */
void 
midibus::start()
{
    m_lasttick = -1;
    
    if ( m_clocking ){ 
	
	snd_seq_event_t ev;
	
	ev.type = SND_SEQ_EVENT_START;
	snd_seq_ev_set_fixed( &ev );
	
	snd_seq_ev_set_priority( &ev, 1 );
	
	/* set source */
	snd_seq_ev_set_source(&ev, m_local_addr_port );
	snd_seq_ev_set_subs(&ev);
	
	// its immediate 
	snd_seq_ev_set_direct( &ev );
	
	/* pump it into the queue */
	snd_seq_event_output(m_seq, &ev);

    }

}


void 
midibus::set_clock( bool a_clocking )
{
    m_clocking = a_clocking;
}


bool 
midibus::get_clock( )
{
    return m_clocking;
}


void 
midibus::stop()
{

    m_lasttick = -1;

    if ( m_clocking ){   
	
	snd_seq_event_t ev;
	
	ev.type = SND_SEQ_EVENT_STOP;
	snd_seq_ev_set_fixed( &ev );
	
	snd_seq_ev_set_priority( &ev, 1 );
	
	/* set source */
	snd_seq_ev_set_source(&ev, m_local_addr_port );
	snd_seq_ev_set_subs(&ev);
	
	// its immediate 
	snd_seq_ev_set_direct( &ev );
	
	/* pump it into the queue */
	snd_seq_event_output(m_seq, &ev);
	
    }

}


// generates midi clock
void
midibus::clock( long a_tick )
{

    lock();

    if ( m_clocking ){

	bool done = false;
	
	long uptotick = a_tick;
	
	if ( m_lasttick == uptotick )
	    done = true;
	
	while ( !done ){
	    
	    m_lasttick++;
	    
	    if ( m_lasttick >= uptotick )
		done = true;
	    
	    /* tick time? */
	    if ( m_lasttick % ( c_ppqn / 24 ) == 0 ){
		
		snd_seq_event_t ev;
		
		ev.type = SND_SEQ_EVENT_CLOCK;
		
		/* set tag to 127 so the sequences
		   wont remove it */
		ev.tag = 127;
		
		snd_seq_ev_set_fixed( &ev );
		
		snd_seq_ev_set_priority( &ev, 1 );
		
		/* set source */
		snd_seq_ev_set_source(&ev, m_local_addr_port );
		snd_seq_ev_set_subs(&ev);
		
		// its immediate 
		snd_seq_ev_set_direct( &ev );
		
		/* pump it into the queue */
		snd_seq_event_output(m_seq, &ev);
	    }
	}
	/* and send out */
	flush();
    }

    unlock();
}

/* deletes events in queue */
/*void 
midibus::remove_queued_on_events( int a_tag )
{
    lock();

    snd_seq_remove_events_t *remove_events;

    snd_seq_remove_events_malloc( &remove_events );
    
    snd_seq_remove_events_set_condition( remove_events, 
					 SND_SEQ_REMOVE_OUTPUT |
					 SND_SEQ_REMOVE_TAG_MATCH |
					 SND_SEQ_REMOVE_IGNORE_OFF );

    snd_seq_remove_events_set_tag( remove_events, a_tag );
    snd_seq_remove_events( m_seq, remove_events );

    snd_seq_remove_events_free( remove_events );

    unlock();
}
*/


void 
mastermidibus::lock( )
{
   // printf( "mastermidibus::lock()\n" );
   m_mutex.lock();
}


void 
mastermidibus::unlock( )
{   
   // printf( "mastermidibus::unlock()\n" );
   m_mutex.unlock();
}



/* gets it a runnin */
void 
mastermidibus::start()
{
    lock();
         
    /* start timer */
    snd_seq_start_queue( m_alsa_seq, m_queue, NULL );
    
    for ( int i=0; i < m_num_out_buses; i++ )
	m_buses_out[i]->start();

     unlock();
}

void 
mastermidibus::stop()
{
    lock();
         
    snd_seq_drain_output( m_alsa_seq );
    snd_seq_sync_output_queue( m_alsa_seq );

    /* start timer */
    snd_seq_stop_queue( m_alsa_seq, m_queue, NULL );

    for ( int i=0; i < m_num_out_buses; i++ )
	m_buses_out[i]->stop();

     unlock();
}


// generates midi clock
void
mastermidibus::clock( long a_tick )
{
    lock();
    
    for ( int i=0; i < m_num_out_buses; i++ )
	m_buses_out[i]->clock( a_tick );
    
    unlock();
}

void 
mastermidibus::set_ppqn( int a_ppqn )
{
    lock();

    m_ppqn = a_ppqn;
    
    /* allocate tempo struct */
    snd_seq_queue_tempo_t *tempo;
    snd_seq_queue_tempo_alloca( &tempo );

    /* fill tempo struct with current tempo info */
    snd_seq_get_queue_tempo( m_alsa_seq, m_queue, tempo );

    /* set ppqn */
    snd_seq_queue_tempo_set_ppq( tempo, m_ppqn );

    /* give tempo struct to the queue */
    snd_seq_set_queue_tempo( m_alsa_seq, m_queue, tempo );

    unlock();
}


void 
mastermidibus::set_bpm( int a_bpm )
{
    lock();

    m_bpm = a_bpm;

    /* allocate tempo struct */
    snd_seq_queue_tempo_t *tempo;
    snd_seq_queue_tempo_alloca( &tempo );

    /* fill tempo struct with current tempo info */
    snd_seq_get_queue_tempo( m_alsa_seq, m_queue, tempo );

    snd_seq_queue_tempo_set_tempo( tempo, 60000000 / m_bpm );

    /* give tempo struct to the queue */
    snd_seq_set_queue_tempo(m_alsa_seq, m_queue, tempo );

    unlock();
}

// flushes our local queue events out into ALSA
void 
mastermidibus::flush()
{
    lock();

    snd_seq_drain_output( m_alsa_seq );

    unlock();
} 



/* fills the array with our buses */
mastermidibus::mastermidibus()
{
    /* temp return */
    int ret;

    /* set initial number buses */
    m_num_out_buses = 0;
    m_num_in_buses = 0;

	for( int i=0; i<c_maxBuses; ++i ){
		m_buses_in_active[i] = false;	
		m_buses_out_active[i] = false;
		m_buses_in_init[i] = false;	
		m_buses_out_init[i] = false;
	}

    /* open the sequencer client */
    ret = snd_seq_open(&m_alsa_seq, "default",  SND_SEQ_OPEN_DUPLEX, 0);

    if ( ret < 0 ){
	printf( "snd_seq_open() error\n");
	exit(1);
    }
  
    /* set our clients name */
    snd_seq_set_client_name(m_alsa_seq, "seq24");
  
    /* set up our clients queue */
    m_queue = snd_seq_alloc_queue( m_alsa_seq );

    /* client info */
    snd_seq_client_info_t *cinfo;
    /* port info */
    snd_seq_port_info_t *pinfo;
	
    int client;
  
    snd_seq_client_info_alloca(&cinfo);
    snd_seq_client_info_set_client(cinfo, -1);


    /* while the next client one the sequencer is avaiable */
    while (snd_seq_query_next_client(m_alsa_seq, cinfo) >= 0){

	/* get client from cinfo */
	client = snd_seq_client_info_get_client(cinfo);
    
	/* fill pinfo */
	snd_seq_port_info_alloca(&pinfo);
	snd_seq_port_info_set_client(pinfo, client);
	snd_seq_port_info_set_port(pinfo, -1);

	/* while the next port is avail */
	while (snd_seq_query_next_port(m_alsa_seq, pinfo) >= 0 ){
			
	    /* get its capability */
	    int cap =  snd_seq_port_info_get_capability(pinfo);
	    
	    if ( snd_seq_client_id( m_alsa_seq ) != snd_seq_port_info_get_client(pinfo) &&
		 snd_seq_port_info_get_client(pinfo) != SND_SEQ_CLIENT_SYSTEM){

		/* the outs */
		if ( (cap & SND_SEQ_PORT_CAP_SUBS_WRITE) != 0 &&
		     snd_seq_client_id( m_alsa_seq ) != snd_seq_port_info_get_client(pinfo)){
		    
		    m_buses_out[m_num_out_buses] = 
			new midibus( snd_seq_client_id( m_alsa_seq ),
				     snd_seq_port_info_get_client(pinfo),
				     snd_seq_port_info_get_port(pinfo),
				     m_alsa_seq,
				     snd_seq_client_info_get_name(cinfo),
				     snd_seq_port_info_get_name(pinfo),
				     m_num_out_buses, m_queue );
		    
		    if ( m_buses_out[m_num_out_buses]->init_out() ){
                m_buses_out_active[m_num_out_buses] = true;		
                m_buses_out_init[m_num_out_buses] = true;
            } else {
                m_buses_out_init[m_num_out_buses] = true;	
            }

		    m_num_out_buses++;
		}	
		
		/* the ins */
		if ( (cap & SND_SEQ_PORT_CAP_SUBS_READ) != 0 &&
		     snd_seq_client_id( m_alsa_seq ) != snd_seq_port_info_get_client(pinfo)){
		    
		    m_buses_in[m_num_in_buses] = 
			new midibus( snd_seq_client_id( m_alsa_seq ),
				     snd_seq_port_info_get_client(pinfo),
				     snd_seq_port_info_get_port(pinfo),
				     m_alsa_seq,
				     snd_seq_client_info_get_name(cinfo),
				     snd_seq_port_info_get_name(pinfo),
				      m_num_in_buses, m_queue);
		    	
			if ( m_buses_in[m_num_in_buses]->init_in() ){
                m_buses_in_active[m_num_in_buses] = true;	
                m_buses_in_init[m_num_in_buses] = true;
            } else {
                m_buses_in_init[m_num_in_buses] = true;	
            }
		    m_num_in_buses++;
		}	
	    }
	}

    } /* end loop for clients */

    set_bpm( c_bpm );
    set_ppqn( c_ppqn );

    /* midi input */
    /* poll descriptors */

    /* get number of file descriptors */
    m_num_poll_descriptors = snd_seq_poll_descriptors_count(m_alsa_seq, POLLIN);

    /* allocate into */
    m_poll_descriptors = new pollfd[m_num_poll_descriptors];

    /* get descriptors */
    snd_seq_poll_descriptors(m_alsa_seq,  
			     m_poll_descriptors, 
			     m_num_poll_descriptors, 
			     POLLIN);
    
    set_sequence_input( false, NULL );

    /* sizes */
    snd_seq_set_output_buffer_size(m_alsa_seq, c_midibus_output_size ); 
    snd_seq_set_input_buffer_size(m_alsa_seq, c_midibus_input_size ); 

		 
	m_bus_announce = 
			new midibus( snd_seq_client_id( m_alsa_seq ),
						 SND_SEQ_CLIENT_SYSTEM,
						 SND_SEQ_PORT_SYSTEM_ANNOUNCE,
						 m_alsa_seq,
						 "system","annouce",
						 0, m_queue);
		    
	m_bus_announce->init_in();


}
      
mastermidibus::~mastermidibus()
{
    for ( int i=0; i<m_num_out_buses; i++ )
	delete m_buses_out[i];

    snd_seq_event_t ev;

    /* kill timer */
    snd_seq_ev_clear(&ev);

    snd_seq_stop_queue( m_alsa_seq, m_queue, &ev );
    snd_seq_free_queue( m_alsa_seq, m_queue );

    /* close client */
    snd_seq_close( m_alsa_seq );

}



void      
mastermidibus::sysex( event *a_ev )
{
	lock();

    for ( int i=0; i<m_num_out_buses; i++ )
      m_buses_out[i]->sysex( a_ev );

    flush();

	unlock();
}


void 
mastermidibus::play( unsigned char a_bus, event *a_e24, unsigned char a_channel )
{
	lock();
	if ( m_buses_out_active[a_bus] && a_bus < m_num_out_buses ){
		m_buses_out[a_bus]->play( a_e24, a_channel );
	}
	unlock();
}


void 
mastermidibus::set_clock( unsigned char a_bus, bool a_clocking )
{
    lock();
    if ( m_buses_out_active[a_bus] && a_bus < m_num_out_buses ){
        m_buses_out[a_bus]->set_clock( a_clocking );
    }
    unlock();
}

bool 
mastermidibus::get_clock( unsigned char a_bus )
{
	if ( m_buses_out_active[a_bus] && a_bus < m_num_out_buses ){
		return m_buses_out[a_bus]->get_clock();
	}
	return false;
}


string
mastermidibus::get_midi_bus_name( int a_bus )
{
	if ( m_buses_out_active[a_bus] && a_bus < m_num_out_buses ){
		return m_buses_out[a_bus]->get_name();
	}
	
	/* copy names */
	char tmp[60];

	if ( m_buses_out_init[a_bus] ){
		snprintf( tmp, 59, "[%d] %d:%d (disconnected)",
				  a_bus,
				  m_buses_out[a_bus]->get_client(),
				  m_buses_out[a_bus]->get_port() );
	} else {
		snprintf( tmp, 59, "[%d] (unconnected)",
				  a_bus );
	}

	string ret = tmp;
	return ret;	
	
}



void 
mastermidibus::print()
{
    printf( "Available Buses\n");
    for ( int i=0; i<m_num_out_buses; i++ ){
	printf( "%s\n", m_buses_out[i]->m_name.c_str() ); 
    }
}


int 
mastermidibus::get_num_out_buses()
{
    return m_num_out_buses;
}


int
mastermidibus::poll_for_midi( )
{
    int ret;

    ret = poll( m_poll_descriptors, 
		 m_num_poll_descriptors, 
		 1000);

    return ret;
}

bool 
mastermidibus::is_more_input( ){
    
    lock();

    int size = snd_seq_event_input_pending(m_alsa_seq, 0);

    unlock();

    return ( size > 0 );
}


void 
mastermidibus::port_start( int a_client, int a_port )
{
	lock();

	bool replacement = false;
	int bus_slot = m_num_out_buses;

	for( int i=0; i< m_num_out_buses; i++ ){

		if( m_buses_out[i]->get_client() == a_client  &&
			m_buses_out[i]->get_port() == a_port &&
			m_buses_out_active[i] == false ){

			replacement = true;
			bus_slot = i;
		}
	}

	/* client info */
    snd_seq_client_info_t *cinfo;
    snd_seq_client_info_alloca(&cinfo);
	snd_seq_get_any_client_info( m_alsa_seq, a_client, cinfo );  

    /* port info */
    snd_seq_port_info_t *pinfo;
	 
	/* fill pinfo */
	snd_seq_port_info_alloca(&pinfo);
	snd_seq_get_any_port_info( m_alsa_seq, a_client, a_port, pinfo );  

			
	/* get its capability */
	int cap =  snd_seq_port_info_get_capability(pinfo);
	    
	if ( snd_seq_client_id( m_alsa_seq ) != snd_seq_port_info_get_client(pinfo)){

		/* the outs */
		if ( (cap & (SND_SEQ_PORT_CAP_SUBS_WRITE | SND_SEQ_PORT_CAP_WRITE ))
			 == (SND_SEQ_PORT_CAP_SUBS_WRITE | SND_SEQ_PORT_CAP_WRITE )
			 && snd_seq_client_id( m_alsa_seq ) != snd_seq_port_info_get_client(pinfo)){
		    
		    m_buses_out[bus_slot] = 
				new midibus( snd_seq_client_id( m_alsa_seq ),
							 snd_seq_port_info_get_client(pinfo),
							 snd_seq_port_info_get_port(pinfo),
							 m_alsa_seq,
							 snd_seq_client_info_get_name(cinfo),
							 snd_seq_port_info_get_name(pinfo),
							 m_num_out_buses, m_queue );
		    
		    m_buses_out[bus_slot]->init_out();
			m_buses_out_active[bus_slot] = true;
			m_buses_out_init[bus_slot] = true;
			
			if ( !replacement ){
				m_num_out_buses++;
			}
		}	
		
		/* the ins */
		if ( (cap & (SND_SEQ_PORT_CAP_SUBS_READ | SND_SEQ_PORT_CAP_READ ))
			 == (SND_SEQ_PORT_CAP_SUBS_READ | SND_SEQ_PORT_CAP_READ )
			 && snd_seq_client_id( m_alsa_seq ) != snd_seq_port_info_get_client(pinfo)){
	    
		    m_buses_in[m_num_in_buses] = 
			new midibus( snd_seq_client_id( m_alsa_seq ),
				     snd_seq_port_info_get_client(pinfo),
				     snd_seq_port_info_get_port(pinfo),
				     m_alsa_seq,
				     snd_seq_client_info_get_name(cinfo),
				     snd_seq_port_info_get_name(pinfo),
				      m_num_in_buses, m_queue);
		    
		    m_buses_in[m_num_in_buses]->init_in();
		    m_buses_in_active[m_num_in_buses] = true;
		    m_buses_in_init[m_num_in_buses] = true;

		    m_num_in_buses++;
		}	
	}
	/* end loop for clients */

 
    /* midi input */
    /* poll descriptors */

    /* get number of file descriptors */
    m_num_poll_descriptors = snd_seq_poll_descriptors_count(m_alsa_seq, POLLIN);

    /* allocate into */
    m_poll_descriptors = new pollfd[m_num_poll_descriptors];

    /* get descriptors */
    snd_seq_poll_descriptors(m_alsa_seq,  
			     m_poll_descriptors, 
			     m_num_poll_descriptors, 
			     POLLIN);

	unlock();
}

void 
mastermidibus::port_exit( int a_client, int a_port )
{
	lock();

	for( int i=0; i< m_num_out_buses; i++ ){

		if( m_buses_out[i]->get_client() == a_client  &&
			m_buses_out[i]->get_port() == a_port ){

			m_buses_out_active[i] = false;
		}
	}

	unlock();
}


bool
mastermidibus::get_midi_event( event *a_in )
{
    lock();
    
    snd_seq_event_t *ev; 

    bool sysex = false;
  
    /* temp for midi data */
    unsigned char buffer[0x1000];

    snd_seq_event_input(m_alsa_seq, &ev);



	bool ret = false;
	switch( ev->type ){ 

	case SND_SEQ_EVENT_PORT_START:
		{   
			//printf("SND_SEQ_EVENT_PORT_START:    addr[%d:%d]\n", 
			//	   ev->data.addr.client, ev->data.addr.port );
			port_start( ev->data.addr.client, ev->data.addr.port );
			ret = true; 
			break;
		}

 	case SND_SEQ_EVENT_PORT_EXIT:     
		{
			//printf("SND_SEQ_EVENT_PORT_EXIT:     addr[%d:%d]\n", 
			//	   ev->data.addr.client, ev->data.addr.port ); 
			port_exit( ev->data.addr.client, ev->data.addr.port );
			ret = true; 
			break;
		}
 
	case SND_SEQ_EVENT_PORT_CHANGE:
		{   
			//printf("SND_SEQ_EVENT_PORT_CHANGE:   addr[%d:%d]\n", 
			//	   ev->data.addr.client, 
			//	   ev->data.addr.port ); 
			ret = true; 
			break;
		}
		
	default: break;
	 		
	}

	if( ret ){
		unlock();
		return false;
	}
    
    /* alsa midi parser */
    snd_midi_event_t *midi_ev;
    snd_midi_event_new( 0x1000, &midi_ev );

    long bytes =
      snd_midi_event_decode( midi_ev,
			     buffer,
			     0x1000,
			     ev ); 
    
    a_in->set_timestamp( ev->time.tick );
    a_in->set_status( buffer[0] );
    a_in->set_size( bytes );

    /* we will only get EVENT_SYSEX on the first 
       packet of midi data, the rest we have
       to poll for */
    if ( buffer[0] == EVENT_SYSEX ){
      
      /* set up for sysex if needed */
      a_in->start_sysex( );
      sysex = a_in->append_sysex( buffer, bytes );
    }
    else {

       a_in->set_data( buffer[1], buffer[2] );
	   
       // some keyboards send on's with vel 0 for off
       if ( a_in->get_status() == EVENT_NOTE_ON &&
			a_in->get_note_velocity() == 0x00 ){
		   a_in->set_status( EVENT_NOTE_OFF );
       }

       sysex = false;
    }

    /* sysex messages might be more than one message */
    while ( sysex ){

      snd_seq_event_input(m_alsa_seq, &ev);

      bytes =
	snd_midi_event_decode( midi_ev,
			       buffer,
			       0x1000,
			       ev ); 
      
      sysex = a_in->append_sysex( buffer, bytes );
  
    }
 
    snd_seq_free_event( ev );
    snd_midi_event_free( midi_ev );

    unlock();

	return true;
}

void 
mastermidibus::set_sequence_input( bool a_state, sequence *a_seq )
{  
    lock();

    m_seq = a_seq;
    m_dumping_input = a_state;

    unlock();
}
  
