// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "convex.h"
#include GL_INCLUDE

#include <boost/python/class.hpp>
#include <boost/python/return_internal_reference.hpp>
#include <boost/python/extract.hpp>
#include "display.h"
#include "slice.h"

#include <cstring>
#include <iostream>

namespace visual { namespace {

inline void
checksum_double( long& sum, const double* d)
{
	const unsigned char* p = (const unsigned char*)d;
	for (unsigned int i = 0; i < sizeof(double); i++) {
		sum ^= *p;
		p++;
		if (sum < 0)
			sum = (sum << 1) | 1;
		else
			sum = sum << 1;
	}
}

// returns a pointer to the ith vector in the array.
double* index( array a, size_t i)
{
	// This is technically an unsafe cast since the alignment requirement
	// goes up for the cast.  It is made safe by padding actions within the Numeric
	// library itself, but the compiler doesn't know that, so I am just using a
	// raw cast vice a static_cast<>.
	return ((double*)data(a)) + i * 3;
}
	
} // !namespace (unnamed)


// Static storage
convex::jitter_table convex::jitter;
	
convex::convex()
	: pos(0), preallocated_size(256), count(0), checksum(0), degenerate(true)
{
	std::vector<int> dims(2);
	dims[0] = 256;
	dims[1] = 3;
	this->pos = makeNum( dims);
	double* i = index(pos, 0);
	i[2] = i[1] = i[0] = 0.0;
	
	// Actually handled by the constructor in Python.
	if (this->display) {
		color = display->fgcolor();
	}
}

convex::convex( const convex& other)
	: DisplayObject( other), pos( other.pos), 
	preallocated_size( other.preallocated_size), count( other.count), 
	checksum(0), degenerate( true)
{
	// Force a refreshCache() call to regenerate the hull.
	write_lock L(mtx);
}

boost::python::object
convex::get_pos()
{
	return pos[visual::python::slice(0, count)];
}
	
void
convex::set_pos( array n_pos)
{
	using namespace boost::python;
	using visual::python::slice;
	
	array_types t = type( n_pos);
	if (t != double_t) {
		n_pos = astype(n_pos, double_t);
	}
	std::vector<int> dims = shape( n_pos);
	if (dims.size() == 1 && count == 0) {
		// perform a single append
		write_lock L(mtx);
		set_length( 1);
		double* pos_data = index(pos, 0);
		pos_data[0] = extract<double>(n_pos[0]);
		pos_data[1] = extract<double>(n_pos[1]);
		pos_data[2] = extract<double>(n_pos[2]);
		return;
	}
	if (dims.size() != 2) {
		throw std::invalid_argument( "pos must be an Nx3 or Nx2 array");
	}
	if (dims[1] == 2) {
		write_lock L(mtx);
		set_length(dims[0]);
		pos[make_tuple(slice(0, count), slice(_,2))] = n_pos;
	}
	else if (dims[1] == 3) {
		write_lock L(mtx);
		set_length(dims[0]);
		pos[slice(0, count)] = n_pos;
		
	}
	else
		throw std::invalid_argument( "pos must be an Nx3 or Nx2 array");
}

void
convex::set_pos_l( const boost::python::list& pos)
{
	this->set_pos( array(pos));
}


void
convex::refreshCache()
{
	if (count < 3) {
		degenerate = true;
		return;
	}
	recalc();
	degenerate = false;
}

double
convex::rayIntersect(const vector &camera, const vector &ray)
{
	if (count < 3)
		return 0.0;

	// xxx This implementation looks like a fish, moves like a fish, steers like a cow
	//     (apologies to Douglas Adams and to your CPU :)

	double inter = 1e300;

	for (std::vector<face>::const_iterator f = hull.begin(); f != hull.end(); ++f) {
		double ndr = f->n.dot(ray);
		if (ndr >= 0.0)
			continue;  // back- or side-facing

		double t = -(f->n.dot(camera) - f->d) / ndr;
		if (t<0 || t > 1e100)
			continue;  // behind camera
		
		vector p = camera + ray*t;
		
		for (int i=0; i<3; ++i) {
			vector v0 = f->v[i];
			vector v1 = f->v[ (i+1)%3 ];
			if ((p-v0).cross(v1-v0).dot(f->n) > 0) {
				t = 1e300;
				break;
			}
		}
		if (t < inter)
			inter = t;
		
	}

	if (inter < 1e100)
		return inter;
	else return 0.0;
}

void
convex::set_length( int length)
{
	size_t npoints = count;
	if (npoints > length) // A shrink operation - never done by VPython.
		npoints = length;
	if (npoints == 0)
		npoints = 1;
		
	if (length > preallocated_size) {
		std::vector<int> dims(2);
		dims[0] = 2 * length;
		dims[1] = 3;
		array n_pos = makeNum( dims);
		std::memcpy( data( n_pos), data( pos), sizeof(double) * 3 * npoints);
		pos = n_pos;
		preallocated_size = dims[0];
	}
	if (length > npoints) {
		// Copy the last good element to the new positions.
		const double* last_element = index( pos, npoints-1);
		double* element_i = index( pos, npoints);
		double* element_end = index( pos, length);
		while (element_i < element_end) {
			element_i[0] = last_element[0];
			element_i[1] = last_element[1];
			element_i[2] = last_element[2];
			element_i += 3;
		}
	}
	count = length;
}


void
convex::append( vector nv_pos)
{
	write_lock L(mtx);
	set_length( count + 1);

	double* pos_data = index(pos, count - 1);
	pos_data[0] = nv_pos.get_x();
	pos_data[1] = nv_pos.get_y();
	pos_data[2] = nv_pos.get_z();
}

void
convex::append_t( const boost::python::object& pos)
{
	this->append( vector(pos));
}

// Used to provide positive assurance that the pos data array is the same as before.
long
convex::computeChecksum() const
{
	long sum = 0;
	const double* pos_data = index(pos, 0);
	const double* const stop = pos_data + count*3;
	while (pos_data < stop) {
		checksum_double( sum, pos_data);
		++pos_data;
	}
	return sum;
}


// Calculate the faces of the object from the raw pos vector_array.
void
convex::recalc()
{
	hull.clear();
	const double* pos_i = index(pos, 0);
	// A face from the first, second, and third vectors.
	hull.push_back( face( vector(pos_i), vector(pos_i+3), vector(pos_i+3*2)));
	// The reverse face from the first, third, and second vectors.
	hull.push_back( face( vector(pos_i), vector(pos_i+3*2), vector(pos_i+3)));
	// The remainder of the possible faces.
	for (int i = 3; i < count; ++i) {
		addpoint( i, vector(pos_i + i*3));
	}

	checksum = computeChecksum();
}

void
convex::addpoint(int n, vector pv)
{
	double m = pv.mag();
	pv.x += m * jitter.v[(n  ) & jitter.mask];
	pv.y += m * jitter.v[(n+1) & jitter.mask];
	pv.z += m * jitter.v[(n+2) & jitter.mask];

	std::vector<edge> hole;
	for (size_t f=0; f<hull.size(); ) {
		if ( hull[f].visibleFrom(pv) ) {
			// hull[f] is visible from pv.  We will never get here if pv is
			//   inside the hull.

			// add the edges to the hole.  If an edge is already in the hole,
			//   it is not on the boundary of the hole and is removed.
			for(int e=0; e<3; ++e) {
				edge E( hull[f].v[e], hull[f].v[(e+1)%3] );

				bool boundary = true;
				for(std::vector<edge>::iterator h = hole.begin(); h != hole.end(); ++h) {
					if (*h == E) {
						*h = hole.back();
						hole.pop_back();
						boundary = false;
						break;
					}
				}

				if (boundary) {
					hole.push_back(E);
				}
			}

			// remove hull[f]
			hull[f] = hull.back();
			hull.pop_back();
		}
		else
			f++;
	}

	// Now add the boundary of the hole to the hull.  If pv was inside
	//   the hull, the hole will be empty and nothing happens here.
	for (std::vector<edge>::const_iterator h = hole.begin(); h != hole.end(); ++h) {
		hull.push_back(face(h->v[0], h->v[1], pv));
	}
}

void
convex::glRender(rView& view)
{
	if (degenerate)
		return;
	if (checksum != computeChecksum()) {
		recalc();
	}


	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glShadeModel(GL_FLAT);
	glEnable(GL_CULL_FACE);

	glBegin(GL_TRIANGLES);
	for (std::vector<face>::const_iterator f = hull.begin(); f != hull.end(); ++f) {
		double illum = view.lights.illuminate( f->n );
		glColor3d( illum*color.r, illum*color.g, illum*color.b );
		for (int v=0; v<3; ++v) {
			vertex vx;
			view.ext_point(f->v[v]);
			view.wct.project(f->v[v], vx );
			glVertex4d( vx.x, vx.y, vx.z, vx.w );
		}
	}
	glEnd();
	glDisable(GL_CULL_FACE);
}


void
convex::set_color( rgb c)
{
	write_lock L(mtx);
	color = c;
}

void
convex_init_type(void)
{	
	using namespace boost::python;
	
	class_<convex, bases<DisplayObject>, boost::shared_ptr<convex> >( "convex")
		.def( init<const convex&>())
		.def( "append", &convex::append, args("pos"),
		 	"Append a point to the surface in O(n) time.")
		.add_property( "color", &convex::get_color, &convex::set_color)
		.def( "_set_pos", &convex::set_pos)
		.def( "_set_pos", &convex::set_pos_l)
		.def( "_get_pos", &convex::get_pos)
		;
}

} // !namespace visual
