// ------------------------------------------------------------------------- //
// $Id: node.h,v 1.47 2004/01/02 15:13:52 pandr Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

// ------------------------------------------------------------------------- //

#ifndef _NODE_H_
#define _NODE_H_

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_VECTOR
# include <vector>
#endif

#if HAVE_STRING
# include <string>
#endif

#if HAVE_GL_GL_H
# ifndef __GNUC__
#  ifdef _WIN32
#   include <windows.h>
#  endif
# endif
# include <GL/gl.h>
#endif

#include "common.h"
#include "image.h"
#include "ftdraw.h"

// ------------------------------------------------------------------------- //

class Node;
class Letter;
class Group;
class Primitive;
class Texture;
class NodeController;

// ------------------------------------------------------------------------- //

//! Node is the base class of every object in the tree of viewable objects
/*!
 *  The tree of nodes represents what can be rendered. The higher level
 *  structure of things, e.g. item-lists, headlines and such, are not
 *  kept here, but rather in the python half of the world.
 *  The Node class is purely abstract.
 */

typedef std::vector<Node*>           NodeList;                                  

class Node
{
friend class Group;
public:
	Node(const std::string& name);
	virtual ~Node();
	virtual std::string name() const { return _name; };
	virtual std::string full_name() const;
	virtual Group* parent() const { return _parent; };
	virtual void   dump_tree(uint indent = 0);
	virtual void   draw_prims(float alpha = 1.0f) = 0;
	virtual void   add_controller(NodeController* cont);
	virtual void   frame_update();
	virtual v3     get_pos() const { return _pos; };
	virtual void   set_pos(v3 pos) { _pos = pos; };
	virtual m33    get_mat() const { return _mat; };
	virtual void   set_mat(m33 mat) { _mat = mat; };
	virtual bool   get_visible() const { return _visible; };
	virtual void   set_visible(bool v) { _visible = v; };
	virtual float  get_alpha() const  { return _alpha; } 
	virtual void   set_alpha(float f) {assert(f>=0.0f && f<=1.0f); _alpha = f;}
	virtual const rgba& get_color() const  { return _color; }
	virtual void   set_color(const rgba &c) { _color = c; }
	virtual size3& get_size() { return _size; }
	virtual void   set_size(const size3& size) { _size = size; }
	virtual	float  get_kerning_with_prev_node(Node* n) { return 0.0f; } 
	virtual	float  get_kerning_with_next_node(Letter* n) { return 0.0f; } 
	static int    _controllers_run;
protected:
	void      _load_GL_matrix() { load_GL_matrix(_mat, _pos); }
	NodeList* _collected;
	rgba      _color;
	float     _alpha;
private:
	Group*        _parent;
	std::string   _name;
	v3            _pos;
	m33           _mat;
	bool          _visible;
	size3         _size;
	std::vector<NodeController*> _controllers;
	typedef       std::vector<NodeController*>::iterator nc_iter;
};

/*!
 *  The Leaf nodes are, surprise, those nodes that are leafs in the tree.
 *  More importantly, however, they can hold geometry primitives.
 */

class Leaf : public Node
{
public:
	Leaf(const std::string &name) : Node(name), _dropshadow(0.0f) {};
	virtual ~Leaf();
	virtual void       add_primitive(Primitive* prim);
	virtual bool       remove_primitive(Primitive* prim);
	virtual Primitive* get_primitive(int i) { return _primitives[i]; }
	virtual void       draw_prims(float alpha);
	void  set_dropshadow(float ds) { _dropshadow = ds; }
private:
	std::vector<Primitive*> _primitives;
	typedef std::vector<Primitive*>::iterator primitive_iter;
	float _dropshadow;
};

/*!
 *  FIXME
 */

class Letter : public Leaf
{
public:
	Letter(const std::string &name, FreeTypeFont::handle font, 
			const Glyph *glyph, unsigned char letter);
	virtual ~Letter();
	const Glyph* get_glyph() const { return _glyph; }
	float get_kerning_with_prev_node(Node* n);
	float get_kerning_with_next_node(Letter* n);
	float get_kerning() const  { return _kern; }
	void  set_kerning(float k) { _kern = k;    }
	virtual void  set_letter(FT_ULong c) { _letter = c; }
	virtual FT_ULong get_letter() { return _letter; }
private:
	FreeTypeFont::handle _font;
	const Glyph*  _glyph;
	float         _kern;
	FT_ULong      _letter;
	bool          _dropshadow;
};

#endif /* _NODE_H_ */

// ------------------------------------------------------------------------- //

