/*
 * Copyright (c) 1998 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the Computer Systems
 *       Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * ---------------------------
 *
 * Filename: srmv2-api.cc
 *   -- Author: Suchitra Raman <suchi@cs.berkeley.edu>
 *
 * @(#) $Header: /usr/mash/src/repository/srmv2/srmv2/srmv2-api.cc,v 1.20 2000/12/07 23:58:18 lim Exp $
 * 
 */

#include "session-srmv2.h"
#include "source-srmv2.h"
#include "ns-srmv2.h"
#include "srmv2.h"
#include "srmv2-api.h"
#include <fcntl.h>


#ifndef NDEBUG
unsigned int srm_trace_flags_ = 0;
#endif


srm_session_t
srm_create_session(unsigned int addr, unsigned int sport,
		   unsigned int rport, unsigned int ttl)
{
	int status;
	/*  SRMv2_Session constructor takes in bps. We give it kbps. */
	srm_session_t session = (srm_session_t) new SRMv2_Session(addr, sport, rport, ttl, 
								  SRMv2_SESSION_KBPS * 1000, &status);
	if (status < 0) {
		srm_destroy_session(session);
		return 0;
	} else {
		return session;
	}
}


void
srm_destroy_session(srm_session_t session)
{
	delete (SRMv2_Session*)session;
}


int
srm_reset_session(srm_session_t session, unsigned int addr, unsigned int sport,
		  unsigned int rport, unsigned int ttl)
{
	return ((SRMv2_Session*)session)->reset(addr, sport, rport, ttl);
}


int
srm_delay_until_full_packet(srm_session_t session, int flag)
{
	return ((SRMv2_Session*)session)->delay_until_full_packet(flag);
}


int
srm_get_delay_until_full_packet(srm_session_t session)
{
	return ((SRMv2_Session*)session)->delay_until_full_packet();
}

#ifndef USE_HFSC_SCHEDULER
void
srm_set_session_bandwidth(srm_session_t session, int bps)
{
	((SRMv2_Session*)session)->bps(bps);
}

int
srm_get_session_bandwidth(srm_session_t session)
{
	return ((SRMv2_Session*)session)->bps();
}
#endif

#ifdef NDEBUG
double
srm_set_drop_probability(srm_session_t /* session */, double /* dropProb */)
{
	return 0.0;
}
#else	
double
srm_set_drop_probability(srm_session_t session, double dropProb)
{
	return ((SRMv2_Session *)session)->drop_probability(dropProb);
}
#endif


#ifdef NDEBUG
double
srm_get_drop_probability(srm_session_t /* session */)
{
	return 0.0;
}
#else	
double
srm_get_drop_probability(srm_session_t session)
{
	return ((SRMv2_Session *)session)->drop_probability();
}
#endif


srm_source_t
srm_create_source(srm_session_t session, const char *srcid)
{
	return (srm_source_t) (((SRMv2_Session*)session)->
			       create_localsrc(srcid));
}


unsigned int 
srm_calloc(srm_source_t source, unsigned int parent, 
	   const unsigned char *container_name, int name_len)
{
	return ((SRMv2_Source*)source)->calloc(parent, SRMv2_ALLOC,
					       container_name, name_len);
}


void srm_set_app_info(srm_source_t source, const unsigned char *info,
		      int info_len)
{
	((SRMv2_Source*)source)->set_app_info(info, info_len);
}


void srm_get_app_info(srm_source_t source, const unsigned char **info,
		      int *info_len)
{
	((SRMv2_Source*)source)->get_app_info(info, info_len);
}


int srm_is_local_source(srm_source_t source)
{
	return ((SRMv2_Source*)source)->islocal();
}


srm_session_t
srm_get_session(srm_source_t source)
{
	return (srm_session_t) ((SRMv2_Source*) source)->session();
}


void
srm_get_container_name(srm_source_t source, unsigned int cid,
		       const unsigned char **name, int *name_len)
{
	SRMv2_NameSpace *ns = ((SRMv2_Source*) source)->name_space();
	NS_Node *node = ns->getnode(cid);
	if (node) {
		if (name)     *name     = node->name();
		if (name_len) *name_len = node->name_len();
	} else {
		if (name) *name = NULL;
		if (name_len) *name_len = 0;
	}
}


unsigned int
srm_get_parent_cid(srm_source_t source, unsigned int cid)
{
	SRMv2_NameSpace *ns = ((SRMv2_Source *)source)->name_space();
	NS_Node *p = ns->getnode(cid)->parent();
	if (p) return p->getcid(); else return 0;
}


const int *
srm_get_source_id(srm_source_t source)
{
	return ((SRMv2_Source *) source)->source_id();
}

void
srm_srcid2str(const int *srcid, char *str)
{
	sprintf(str, "%08x%08x%08x%08x", srcid[0], srcid[1],srcid[2],srcid[3]);
}

const char *
srm_str2srcid(const char *str, int *srcid)
{
	char tmp[10];
	tmp[8] = '\0';
	strncpy(tmp, str, 8);
	srcid[0] = strtoul(tmp, NULL, 16);
	str += 8;
	
	strncpy(tmp, str, 8);
	srcid[1] = strtoul(tmp, NULL, 16);
	str += 8;
	
	strncpy(tmp, str, 8);
	srcid[2] = strtoul(tmp, NULL, 16);
	str += 8;
	
	strncpy(tmp, str, 8);
	srcid[3] = strtoul(tmp, NULL, 16);
	str += 8;

	return str;
}

unsigned int
srm_send(srm_source_t source, unsigned int cid, 
	 unsigned char *data, int len, const srm_adu_info *info)
{
	SRMv2_Source* s = (SRMv2_Source*) source;
	return s->session()->send_adu(s, data, SRMv2_ALLOC, 0, len - 1,
				      len - 1, cid, SRMv2_DATA, 0, 0, info);
}


void srm_recover(srm_source_t source, unsigned int cid, 
		 unsigned int sseq, unsigned int eseq)
{
	((SRMv2_Source*)source)->recover(cid, sseq, eseq);
}



void
srm_set_callbacks(srm_session_t session, const srm_callbacks *callbacks)
{
	srm_callbacks *c = ((SRMv2_Session*) session)->callbacks();
	*c = *callbacks;
}


const srm_callbacks*
srm_get_callbacks(srm_session_t session)
{
	return ((SRMv2_Session*) session)->callbacks();
}


void
srm_set_recv_proc(srm_session_t session, srm_recv_proc proc)
{
	srm_callbacks *c = ((SRMv2_Session*) session)->callbacks();
	c->recv_proc = proc;
}


void
srm_set_should_recover_proc(srm_session_t session,srm_should_recover_proc proc)
{
	srm_callbacks *c = ((SRMv2_Session*) session)->callbacks();
	c->should_recover_proc = proc;
}


void
srm_set_read_adu_proc(srm_session_t session, srm_read_adu_proc proc)
{
	srm_callbacks *c = ((SRMv2_Session*) session)->callbacks();
	c->read_adu_proc = proc;
}


void
srm_set_source_update_proc(srm_session_t session, srm_source_update_proc proc)
{
	srm_callbacks *c = ((SRMv2_Session*) session)->callbacks();
	c->source_update_proc = proc;
}


void
srm_set_recv_cid_proc(srm_session_t session, srm_recv_cid_proc proc)
{
	srm_callbacks *c = ((SRMv2_Session*) session)->callbacks();
	c->recv_cid_proc = proc;
}

#ifdef USE_HFSC_SCHDULER
unsigned int srm_get_session_bandwidth(srm_session_t sess)
{
	return (srm_get_class_bandwidth(sess, ROOT_CLASS));
}
#endif

unsigned int srm_get_class_bandwidth(srm_session_t session, unsigned int tcid)
{
	ClassItem *tc = getClassItem(((SRMv2_Session *) session)->hfscRoot(), tcid);
	return (tc->pService->m2);
}

unsigned int srm_create_tc(srm_session_t sess, unsigned int parent_tcid, unsigned int kbps)
{
	ClassItem *root, *ci;

	root = newClassItem(0, 0, newSC(kbps, 0, kbps));
	unsigned int tcid = ((SRMv2_Session *) sess)->allocate_tcid();
	ci = newClassItem(tcid, 0, newSC(kbps, 0, kbps));
	insertClassItem(root, parent_tcid, ci);
	return tcid;
}

void srm_delete_tc(srm_session_t /* sess */, unsigned int /* tcid */)
{
	/* Not yet implemented. */
}
