/*
 * t_decoder-h261.cc --
 *
 *      This is just a modified version of decoder-h261.cc from the
 *      mash/codec directory.
 *
 * Copyright (c) 2000-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
  Erik Machnicki
  machnick@cs.berkeley.edu

  This is just a modified version of decoder-h261.cc from the mash/codec
  directory.

  There are few changes.  In the constructor, a Tracker object is created.
  When a packet is received, the tracker object is passed to the t_p64
  part of the decoder.  The tracker object is also signaled when the frame
  ends.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
// we need to choose the right directory because mash/audio also has a
//    decoder.h !
#include "../../codec/decoder.h"
#include "renderer.h"
#include "t_p64/t_p64.h"
#include "pktbuf.h"

#include "tclcl.h"
#include "tracker.h"

static int packet_num=0;
static int prev_packet_num = 0;
static int prev_timestamp = 0;
static int timestamp;
static int first_packet = 1;

class t_H261Decoder : public Decoder
{
public:
	t_H261Decoder();
        t_H261Decoder(int argc, const char*const* argv);
	virtual ~t_H261Decoder();
	virtual void info(char* wrk) const;
	virtual void recv(pktbuf*);
	int colorhist(u_int* hist) const;
	virtual void stats(char* wrk);


    protected:
	void decode(const u_char* vh, const u_char* bp, int cc);
  	virtual void redraw();
	t_P64Decoder* codec_;
	int h261_rtp_bug_;

	Tracker* tracker;
};

static class t_H261DecoderClass : public TclClass {
    public:
	t_H261DecoderClass() : TclClass("Module/VideoDecoder/Tracker_H261") {}
	TclObject* create(int argc, const char*const* argv) {
		return (new t_H261Decoder(argc, argv));
	};
} dm_t_h261;

#define STAT_BAD_PSC	0
#define STAT_BAD_GOB	1
#define STAT_BAD_SYNTAX	2
#define STAT_BAD_FMT	3
#define STAT_FMT_CHANGE 4	/* # times fmt changed */
#define STAT_BAD_HEADER 5


t_H261Decoder::t_H261Decoder(int argc, const char*const* argv) : Decoder(4), codec_(0), h261_rtp_bug_(0), tracker(NULL)
{
 //printf("initiing decoder\n");

	stat_[STAT_BAD_PSC].name = "H261-Bad-PSC";
	stat_[STAT_BAD_GOB].name = "H261-Bad-GOB";
	stat_[STAT_BAD_SYNTAX].name = "H261-Bad-Syntax";
	stat_[STAT_BAD_FMT].name = "H261-Bad-fmt";
	stat_[STAT_FMT_CHANGE].name = "H261-Fmt-change";
	stat_[STAT_BAD_HEADER].name = "H261-Bad-Header";
	nstat_ = 6;

	decimation_ = 411;
	/*
	 * Assume CIF.  Picture header will trigger a resize if
	 * we encounter QCIF instead.
	 */
	inw_ = 352;
	inh_ = 288;

	/*FIXME*/

	resize(inw_, inh_);
	/*
	  int x;
	printf("argc is %d\n", argc);
	for(x = 0; x < argc; x++)
	{
	 printf("argv[%d] is %s\n", x, argv[x]);
	}
	*/
	Tcl& tcl = Tcl::instance();
	if(argc > 4)
	{
	  tracker = (Tracker*)tcl.lookup(argv[4]);
	  if(tracker == NULL)
	    {
	      printf("major problem, tracker is NULL\n");
	    }
	  //  printf("tracker is %d", tracker);
	  //tracker->print_test();
	}
	else
	{
	  printf("t_H261Decoder: no tracker specified, no tracking\n");
	}
	//	printf("finished initing decoder\n");
}

t_H261Decoder::~t_H261Decoder()
{
	delete codec_;
}

void t_H261Decoder::info(char* wrk) const
{
	if (codec_ == 0)
		*wrk = 0;
	else
		sprintf(wrk, "[q=%d]", codec_->gobquant());
}

void t_H261Decoder::stats(char* wrk)
{
	/* pull stats out of vic indepdendent t_P64Decoder */
	setstat(STAT_BAD_PSC, codec_->bad_psc());
	setstat(STAT_BAD_GOB, codec_->bad_GOBno());
	setstat(STAT_BAD_SYNTAX, codec_->bad_bits());
	setstat(STAT_BAD_FMT, codec_->bad_fmt());
	Decoder::stats(wrk);
}

int t_H261Decoder::colorhist(u_int* /* hist */) const
{
	return (1);
}




void t_H261Decoder::recv(pktbuf* pb)
{
  int diff;

  rtphdr* rh = (rtphdr*)pb->dp;

  u_int8_t* vh = (u_int8_t*)(rh + 1);
  if (codec_ == 0)
    {
      if ((vh[0] & 2) != 0)
	{
	  //	  printf("decoder is intra\n");
	  codec_ = new Intrat_P64Decoder();
	}
      else
	{
	  //	  printf("decoder is full\n");
	  codec_ = new Fullt_P64Decoder();
	}
      codec_->marks(rvts_);
    }

  /*FIXME*/
  u_int v = ntohl(*(u_int32_t*)vh);
  int sbit = v >> 29;
  int ebit = (v >> 26) & 7;
  int quant = (v >> 10) & 0x1f;
  int mvdh = (v >> 5) & 0x1f;
  int mvdv = v & 0x1f;
  int mba, gob;


  /*
   * vic-2.7 swapped the GOB and MBA fields in the RTP packet header
   * with respect to the spec.  To maintain backward compat, whenever
   * we see an out of range gob, we change our assumption about the
   * stream and continue.
   */
  if (!h261_rtp_bug_)
    {
      mba = (v >> 15) & 0x1f;
      gob = (v >> 20) & 0xf;
      if (gob > 12)
	{
	  h261_rtp_bug_ = 1;
	  mba = (v >> 19) & 0x1f;
	  gob = (v >> 15) & 0xf;
	}
    }
  else
    {
      mba = (v >> 19) & 0x1f;
      gob = (v >> 15) & 0xf;
      if (gob > 12)
	{
	  h261_rtp_bug_ = 0;
	  mba = (v >> 15) & 0x1f;
	  gob = (v >> 20) & 0xf;
	}
    }

  if (gob > 12)
    {
      count(STAT_BAD_HEADER);
      pb->release();
      return;
    }
  int cc = pb->len - (sizeof(*rh) + 4);
  codec_->mark(now_);

  //printf("getting packet %d: timestamp = %u\n", ntohs(rh->rh_seqno), ntohl(rh->rh_ts));

  // timestamp is somehow related to the frame number (i.e., all packets
  //   from the same frame have the same timestamp)
  timestamp = ntohl(rh->rh_ts);
  packet_num = ntohs(rh->rh_seqno);

  if(!first_packet)
    {
      diff = packet_num - prev_packet_num - 1;
      // send how many packets were lost to the tracker object
      // even if diff is 0, we still call lostPackets, because lostPackets
      //   also counts the number of packets successfully received
      tracker->lostPackets(diff);
      //tracker->endFrame();
      if(prev_timestamp != timestamp)
	{
	  if(tracker != NULL)
	    {
	      tracker->endFrame();
	    }
	}
    }
  else
    {
      first_packet = 0;
    }

  prev_packet_num = packet_num;
  prev_timestamp = timestamp;

  if(gob > 0)
    {
      // printf("getting packet %d: timestamp = %u\n", packet_num, timestamp);
    }
  else
    {
      //printf("zero packet %d: timestamp = %u\n", packet_num, timestamp);
    }

  // vh + 4 is the pointer to the stream
  (void)codec_->decode(vh + 4, cc, sbit, ebit, mba, gob, quant, mvdh, mvdv, tracker);


  /*
   * If the stream changes format, issue a resize.
   */
  if (codec_->width() != inw_)
    {
      resize(codec_->width(), codec_->height());
      codec_->marks(rvts_);
      count(STAT_FMT_CHANGE);
    }

  /*FIXME*/
  if (ntohs(rh->rh_flags) & RTP_M)
    {
      codec_->sync();
      ndblk_ = codec_->ndblk();
      // ErikM - commented out
      //		render_frame(codec_->frame());
      codec_->resetndblk();
    }
  pb->release();
}

// this is inherited from Decoder as a virtual function, so must be defined
void t_H261Decoder::redraw()
{
}

