# rf.tcl --
#
#       allows you to specify that the Real Networks video input should switch
#       when one of the RVC video inputs switches
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

import 405Client

# use the Tcl namespace facility to prevent conflicts
namespace eval RealFollower {
    variable self
    variable enabled
    variable followOutput
    variable output
}

Class RealFollower

RealFollower instproc init {base {extendedGUI 1}} {
    $self instvar 405_ statusLabel_ currentInput_

    set 405_ [new 405Client]

    # so that the callback can refer to this object
    set RealFollower::self $self

    # in the future, may want to allow other output sources to be switched,
    #   so use a global variable to make transition to checkbuttons easier
    set RealFollower::output "realNetworks"
    set RealFollower::enabled 0
    set RealFollower::followOutput "htsr"

    set statusLabel_ ""
    set currentInput_ "Erik is a god"

    # FIXME - should provide some getFilter library function that you give a
    #   list of all the things you're interested in, and it gives you an
    #   efficient filter expression to pass to callback_register
    set callback "$self processAmxMsg"
    set filter [$self getFilter $RealFollower::followOutput]
    $405_ callback_register $callback $filter
    $405_ callback_enable $callback

    $self initUI $base $extendedGUI
}

RealFollower instproc initUI {base {extendedGUI 1}} {
    $self instvar statusLabel_ currentInput_ 405_

    frame $base.controls -borderwidth 3
    pack $base.controls -side top -fill x

    checkbutton $base.controls.onOff -text "Enable Real Following" -variable RealFollower::enabled
    pack $base.controls.onOff -side left

    # get the current input
    set result [$405_ matrix_getInputSource $RealFollower::output]
    set currentInput_ $result
    set curFrame "$base.status"
    frame $curFrame
    pack $curFrame -side top -fill x
    set statusLabel_ [label $curFrame.currentState -text "current: $currentInput_"]
    pack $curFrame.currentState -side left

    if {$extendedGUI} {
        frame $base.outputs -borderwidth 3
	pack $base.outputs -side top -fill x

	# I have to give each button its own frame to make it look pretty and
	#  line up correctly
	frame $base.outputs.labelFrame
	pack $base.outputs.labelFrame -side top -fill x
	label $base.outputs.labelFrame.outputLabel -text "Follow Output"
	pack $base.outputs.labelFrame.outputLabel -side left

	frame $base.outputs.htsrBut
	pack $base.outputs.htsrBut -side top -fill x
	radiobutton $base.outputs.htsrBut.htsr -text "htsr" -variable RealFollower::followOutput -value "htsr" -command "$self setOutput htsr"
	pack $base.outputs.htsrBut.htsr -side left

	frame $base.outputs.htsr2But
	pack $base.outputs.htsr2But -side top -fill x
	radiobutton $base.outputs.htsr2But.htsr2 -text "htsr2" -variable RealFollower::followOutput -value "htsr2" -command "$self setOutput htsr2"
	pack $base.outputs.htsr2But.htsr2 -side left
    }

    set RealFollower::followOutput "htsr"

    # if you want the default to be on, have to set RealFollower::enabled
    #   to 1
    set RealFollower::enabled 0
#    set RealFollower::enabled 1
}

RealFollower public enable {} {
    set RealFollower::enabled 1
}

RealFollower public disable {} {
    set RealFollower::enabled 0
}

RealFollower public AMXSwitch { target output } {
    $self instvar 405_

    $405_ matrix_switchVideoStream $target $output
}

RealFollower public processAmxMsg { amxMsg } {
    $self instvar currentInput_ statusLabel_

#    puts stdout "RealFollower::processAmxMsg amxMsg is $amxMsg"
#    puts stdout "RealFollower::output is $RealFollower::output"

    if {[llength $amxMsg] != 0} {
	set eventInfo [lindex $amxMsg 0]
	set type [lindex $eventInfo 0]
	set eventData [lindex $amxMsg 1]
    }

    if {$type == "matrixSwitch"} {
	set input [lindex $eventData 0]
	set output [lindex $eventData 1]
	if {$output == $RealFollower::followOutput} {
	    if {$RealFollower::enabled} {
#		puts stdout "should switch $input to $RealFollower::output"
		$self AMXSwitch $input $RealFollower::output
	    }
	}
	if {$output == $RealFollower::output} {
	    set currentInput_ $input
#	    puts stdout "updating status to $input"
	    $statusLabel_ configure -text "current: $currentInput_"
	}

    }

    return ""
}

RealFollower public setOutput {output} {
    $self instvar 405_

    switch -exact -- $output {
	htsr -
	htsr2 {
	    set RealFollower::followOutput $output

	    # we have to reregister the callback with the updated filter to
	    #   only get the stuff we want
	    set filter [$self getFilter $RealFollower::followOutput]
	    set callback "$self processAmxMsg"
	    $405_ callback_register $callback $filter
	    $405_ callback_enable $callback
	}
	default {
	    return -code error "RealFollower::setOutput: output must be \[htsr, htsr2\]"
	}
    }
}

RealFollower instproc getFilter {watchOutput} {
    # This filters out all events except for matrixSwitch events that have an
    #   output of $watchOutput or $RealFollower::output
    set filter ""
    set filter "$filter set doCallback 0\n"
    set filter "$filter set info \[lindex \$arg 0\]\n"
    set filter "$filter set type \[lindex \$info 0\]\n"
    set filter "$filter set data \[lindex \$arg 1\]\n"
    set filter "$filter if \{\$type == \"matrixSwitch\"\} \{\n"
    set filter "$filter set output \[lindex \$data 1\]\n"
    set filter "$filter if \{(\$output == \"$watchOutput\") || (\$output == \"$RealFollower::output\")\} \{\n"
    set filter "$filter set doCallback 1\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"
    return $filter
}
