# application-tgw.tcl --
#
#       This is the base class for the tgw application, which can transcode
#       video streams to lower bitrates, redirect them to other addresses,
#       transcode them to RealMedia format (if you have the realproducersdk)
#       and combine streams into one RealMedia stream...
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import RTPApplication Configuration AddressBlock CoordinationBus \
	VideoHandler AudioAgent VicUI ExtOut \
	FontInitializer WidgetResourceInitializer TGWUI

# The base class for the TGW application.  Parses the command line
# arguments, creates all the necessary agents and objects, configures
# them, and splices them together.
# Creates a frame using the provided <i>widgetPath</i> and packs the tgw ui in it.
#

Class TGWApplication -superclass RTPApplication

TGWApplication instproc init {widgetPath argv} {
	$self next tgw

	# Initialization of variables and resource db options
	set o [$self options]
	$self init_args $o
	# Search for some good fonts for the user interface and intial
	# the configuration options accordingly.
	new FontInitializer $o
	option add *Font [$o get_option helv12b] startupFile
	$self init_resources $o
	$o load_preferences "rtp vic"
	set argv [$o parse_args $argv]

	# Source the user's hook file if it exists. The function
	# user_hook, which may be defined in this file, will be
	# called at the end of init.
	if {[$o get_option userhookFile] != ""} {
		if {[file isfile [$o get_option userhookFile]] && \
			[file readable [$o get_option userhookFile]]} {
			source [$o get_option userhookFile]
		} else {
			puts stderr "Unable to source \"[$o get_option userhookFile]\". Not a file or not readable."
		}
	}

	set spec [$self check_hostspec [lindex $argv 0] [$self get_option megaVideoSession]]
	$self check_rtp_sdes

	# Backward compat. -B flag is kb/s.
	set t [$self get_option maxbw]
	if { $t > 0 } {
		$o add_option maxbw [expr 1000*$t]
	}

	if { $spec != "" } {
		#XXX need to do this to size the bandwidth slider in the GUI
		set ab [new AddressBlock $spec]
	        $o add_option maxbw [$ab set maxbw_(0)]
		delete $ab
	}

	# Create a Coordination Bus
	$self init_confbus

	# Create a VideoAgent and VideoPipeline
	$self instvar agent_ vpipe_ handler_ scuba_sess_
	set handler_ [new VideoHandler $spec]
	set agent_ [$handler_ agent]
	set vpipe_ [$handler_ vpipe]

	# Create the user interface for this application.
	$self instvar ui_
	set ui_ [$self init_ui $widgetPath $spec]

	$self user_hook
}

# Build the main tgw user interface and pack it into a newly created
# window in the default top-level window.  Call init_confbus before this
# method, so that instvars local_chan_ and glob_chan_ are set.
# Creates a frame using the provided <i>widgetPath</i> and packs the tgw ui in it.
#

TGWApplication instproc init_ui { widgetPath spec } {
	$self instvar agent_ vpipe_
        $self instvar local_chan_ glob_chan_

	frame $widgetPath
	set ui [new TGWUI $widgetPath $local_chan_ $glob_chan_ $agent_ $vpipe_ "$self exit" [$self get_option noUI] [$self get_option realServerProfile] [$self get_option realStreamProfile]]
	pack $widgetPath -expand 1 -fill both
	update idletasks

	return $ui
}

# A hook to gracefully shut down tgw.  Shuts down the
# underlying video agent and terminates the application.
#
TGWApplication instproc exit {} {
	$self instvar agent_
	$agent_ shutdown
	exit 0
}

# Register all of the command line options understood
# by tgw with the generic command-line parser.
#
TGWApplication private init_args o {
	$o register_option -a audioSessionSpec
	$o register_option -B maxbw
	$o register_option -C conferenceName
	$o register_option -c dither
	$o register_option -D device
	$o register_option -f videoFormat
	$o register_option -F maxfps
	$o register_option -I confBusChannel
	$o register_option -K sessionKey
	$o register_option -M colorFile
	$o register_option -m mtu
	$o register_option -o outfile
	$o register_option -q jpegQfactor
	$o register_option -t defaultTTL
	$o register_option -T softJPEGthresh
	$o register_option -U stampInterval
	$o register_option -u userhookFile
	$o register_option -V visual
	$o register_option -N rtpName
	$o register_list_option -map rtpMap

	$o register_boolean_option -H useHardwareDecode
	$o register_boolean_option -P privateColormap

	# cues option
	$o register_boolean_option -useCues useCues

	$o register_boolean_option -useRLM useRLM

	# rendezvous address, currently only for cam ctrls
	$o register_option -rendez rendezSpec

    $o register_boolean_option -noUI noUI
    $o register_option -realServerProfile realServerProfile
    $o register_option -realStreamProfile realStreamProfile
}

#
# Initialize the configuration data base with all of the default
# options assumed by tgw.  These can be overridden.
#
TGWApplication private init_resources o {
        global tcl_platform
	new WidgetResourceInitializer

	# used by TopLevelWindow and for VicUI's window-title
	$o add_default iconPrefix tgw:

	# used when creating new CoordinationBus; get and set by AddressBlock
	$o add_default defaultTTL 16

	# WAS used by init_confbus
	$o add_default confBusChannel 0

	# used by ControlMenu; get and possibly set by AddressBlock
	$o add_default videoFormat h.261

	# used by VideoPipeline; set by ControlMenu
	$o add_default useJPEGforH261 false

	# used by MetworkMananger; only apps that set this are vic and mui.  maybe it should be add_option
	$o add_default useLayersWindow 1

	$o add_default useRLM 0
	$o add_default camctrl 0

	# used by c++ code
	$o add_default jvColors 32
	$o add_default softJPEGthresh -1
	$o add_default softJPEGcthresh 6

	# used by VicApplication and ControlMenu
	# (scuba) default session bandwidth (kb/s).
	$o add_default maxVideoSessionBW 1000000

        # used by the Info window and ControlMenu
        if {$tcl_platform(platform) != "windows"} {
	        option add Vic.background gray85 startupFile
	        option add Vic.foreground black startupFile
	        option add Vic.disabledForeground gray40 startupFile
	        $o add_default background gray85
                $o add_default foreground black
                $o add_default disabledForeground gray40
        } else {
		set b [button .b____dummy____$self]
		$o add_default background [$b cget -background]
                $o add_default foreground [$b cget -foreground]
                $o add_default disabledForeground [$b cget -disabledforeground]
		destroy $b
	}
	
	$o add_default noUI 0
	$o add_default realServerProfile none
	$o add_default realStreamProfile none

}

# Create a global bus for communication between tools
# on different machines, and a local bus for tools on
# the same machine.
#
TGWApplication instproc init_confbus {} {

	set channel [$self get_option confBusChannel]
	if {$channel == ""} {set channel 2}

        $self instvar local_chan_ glob_chan_
        set local_chan_ [new CoordinationBus -channel $channel]

	incr channel
        set ttl [$self get_option defaultTTL]
        set glob_chan_ [new CoordinationBus -channel $channel -ttl $ttl]
}

