# source-sap.tcl --
#
#       Handles creation, and transmission, of a SAP announcement
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import ProgramSource AnnounceListenManager Timer

Class Timer/Adaptive/SAP -superclass Timer/Adaptive

Timer/Adaptive/SAP public init { alm } {
	$self set alm_ $alm
	$self next
}


Timer/Adaptive/SAP private adapt {interval} {
	$self instvar alm_
	return [expr 1000*[$alm_ interval 1]]
}


# Used privately by ProgramSource/SAP.
Class AnnounceListenManager/SAP/Nsdr -superclass AnnounceListenManager/SAP

#
AnnounceListenManager/SAP/Nsdr public init {s mtu scope} {
	set ttl [$self get_option sapTTL]
	set spec "[$scope sapAddr]/none/$ttl"
	$self next $spec $mtu

	$self set bw_ [$scope bw]
	$self set s_ $s
	$self set avgsize_ 500
	$self set nsrcs_ 0
}

#
AnnounceListenManager/SAP/Nsdr public destroy {} {
	$self next
}

#
AnnounceListenManager/SAP/Nsdr private recv_announcement args {
	$self instvar s_
	eval $s_ recv $self $args
}

#
AnnounceListenManager/SAP/Nsdr public sample_size {size} {
	$self instvar avgsize_
	set avgsize_ [expr $avgsize_ + ($size-$avgsize_)>>3]
}

#
AnnounceListenManager/SAP/Nsdr public incrnsrcs {n} {
	$self instvar nsrcs_
	incr nsrcs_ $n
}

#
AnnounceListenManager/SAP/Nsdr public interval {rand} {
	$self instvar avgsize_ nsrcs_ bw_

	set i [expr 8 * $avgsize_ * $nsrcs_ / $bw_]
	if {$rand != 0} {
		# random in U[2/3,4/3] as per SAP spec
		set r1 [expr [random]/double(0x7fffffff)]
		set r2 [expr ($r1*2.0/3.0) + 2.0/3.0]
		set i [expr int($i*$r2)]
	}
	#FIXME
	if {$i < 5} {
		set i 5
	}
	return $i
}

#
AnnounceListenManager/SAP/Nsdr public start {msg} {
	$self instvar nsrcs_
	incr nsrcs_
	$self timer $msg [new Timer/Adaptive/SAP $self]
	$self next $msg
}

#
AnnounceListenManager/SAP/Nsdr private send_announcement {msg} {
	set text [$msg set msgtext_]
	$self sample_size [string length $text]
	$self announce $text
}


# Retrieves program descriptions via the Session Announcement
# Protocol (SAP).
Class ProgramSource/SAP -superclass ProgramSource

# Initializes a new object.  <i>ui</i> is passed on to
# ProgramSource::init.  <i>bw</i> is the aggregate bandwidth
# allocate to the SAP channel (which is used for determining
# the timeout interval as well as for announcing new programs).
# <i>args</i> contains a list of  multicast scope zones
# (e.g., 224.2.128.0/17 for global sap) within which this
# object should listen for SAP announcements.
ProgramSource/SAP public init {ui args} {
	$self next $ui

	$self instvar scopes_ addrs_ cache_ announce_file_
	set scopes_ {}
	set addrs_ {}
	foreach scope $args {
		lappend scopes_ $scope
		set al [new AnnounceListenManager/SAP/Nsdr $self 2048 $scope]
		lappend addrs_ $al
	}

	#FIXME need a way to disable this
	set dir [$self get_option cachedir]
	if {![info exists cache_] && $dir != ""} {
		set o [$self options]
        	set addr [$o get_option SAPaddress]
		set cache_ [file join $dir global-$addr]
	}
	$self readcache
	set write_interval [$self get_option cacheWriteInterval]
	if {$write_interval != ""} {$self periodic-writecache $write_interval }

	#FIXME
	set a [lindex $addrs_ 0]
	if {$a == ""} {
		set announce_file_ ""
	} else {
		set if [[$a set snet_] interface]
		set announce_file_ [file join $dir announce-$if]
	}

	if [file readable $announce_file_] {
		$self instvar sdp_

		set fp [open $announce_file_ r]
		set msgs [$sdp_ parse [read $fp]]
		close $fp

		file delete $announce_file_

		foreach m $msgs {
			set p [new Program $m]
			$self announce $p
		}
	}
}

#
ProgramSource/SAP public destroy {} {
	$self next
	$self instvar addrs_
	foreach a $addrs_ { delete $a }
}

# Used to set the name of the tag for this program source in the
# user interface.
ProgramSource/SAP public name {} {
	return "SAP: Global"
}

# Returns the list of scopes this object knows about
ProgramSource/SAP public scopes {} {
	return [$self set scopes_]
}

# Called by an AnnounceListenManager/SAP/Nsdr object when an
# announcement is received.  Invokes ProgramSource::recv
# and then sets a timeout on this announcement.
ProgramSource/SAP public recv {child addr port data size} {
	# keep nsrcs_ array up to date for this child by watching the
	# change in the size of the progs_ array when the message is
	# processed.
	$self instvar progs_
	set old [array size progs_]
	set objs [$self next $data]
	$child incrnsrcs [expr [array size progs_] - $old]

	# update average announcement size
	$child sample_size $size

	# set timeouts
	$self instvar progs_ timeouts_
	foreach o $objs {
		# calculate timeout with a minimum of 30 mins as per sap spec
		set t [expr 10 * [$child interval 0]]
		if {$t < 1800} { set t 1800 }

		$self timeout $o $t
	}
}

# hack since tcl expr can't do unsigned comparison
ProgramSource/SAP private timestamp-gt {a b} {
	if {$b == 0} { return 1 }
	if {$a > 0 && $b < 0} { return 0 }
	return [expr $a > $b]
}

#
ProgramSource/SAP public announce {prog} {
	$self instvar announce_file_ rcvr_

	foreach msg [$prog set msgs_] {
		set al [$self alof $msg]
		$al start $msg

		if {$announce_file_ != ""} {
			if [catch {set fp [open $announce_file_ a]} m] {
				$self warn "couldn't open announcements file\
						for writing: $m"
				continue
			}
			puts $fp [$msg set msgtext_]
			close $fp
		}
	}

	$rcvr_ addprog $self $prog

	set end 0
	foreach t [[$prog base] set alltimedes_] {
		set newend [$t set endtime_]
		if [$self timestamp-gt $newend $end] { set end $newend }
	}
	if {$end != 0} {
		set wait [expr $end - 2208988800 - [clock seconds]]
		if {$wait <= 0} {
			$self stop-announce $prog
		} else {
			after [expr int($wait * 1000)] "$self stop-announce $prog"
		}
	}
}

#
ProgramSource/SAP public stop-announce {prog} {
	$self instvar announce_file_ sdp_ rcvr_

	foreach msg [$prog set msgs_] {
		set al [$self alof $msg]
		$al stop $msg

		set backup $announce_file_
		append backup "~"
		file delete $backup
		set backup_good 1
		if {[catch {file copy $announce_file_ $backup} m] \
				|| [catch {set fp [open $backup r]} m] \
				|| [catch {set fp2 [open $announce_file_ w]} m]} {
			$self warn "couldn't fix announcement file: $m"
			set backup_good 0
			continue
		}

		set buffer ""
		while { ![eof $fp] } {
			set line [gets $fp]
			if {![eof $fp] && [string compare $line "v=0"] != 0} {
				append buffer $line
				append buffer \n
				continue
			}

			if {[string trim $buffer] != ""} {
				#FIXME
				set msg2 [$sdp_ parse $buffer]
				if {[$msg unique_key] != [$msg2 unique_key]} {
					puts $fp2 $buffer
				}
				delete $msg2
			}

			set buffer $line
			append buffer \n
		}
		close $fp
		close $fp2

		if {![file size $announce_file_]} {
			file delete $announce_file_
		}
		if {$backup_good} {
			file delete $backup
		}
	}

	$rcvr_ removeprog $self $prog
}

# FIXME should be elsewhere
ProgramSource/SAP private alof {msg} {
	$self instvar scopes_ addrs_

	#FIXME
	if [$msg have_field c] {
		set addr [$msg set caddr_]
	} else {
		set media [lindex [$msg set allmedia_] 0]
		set addr [$media set caddr_]
	}
	set addr [lindex [split $addr /] 0]

	set i 0
	set found 0
	set len [llength $scopes_]
	while {$i < $len} {
		set scope [lindex $scopes_ $i]
		if [$scope contains $addr] {
			set found 1
			break
		}
		incr i
	}
	if {$found == 0} {
		$self fatal "Program/SAP got address ($addr) not in any known scope"
	}

	return [lindex $addrs_ $i]
}
