# ui-dc.tcl --
#
#       Basic UI of the DC.  Contains some helper functions.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import VisualFrame
import CDcUIThumbnailFrame
import CDcUIPreviewFrame
import CDcUIBroadcastFrame


Class CDcUI

CDcUI instproc init { appDc } {
    # store away the dc application
    $self instvar m_appDc
    $self instvar m_frameRoot
    $self instvar m_VisualFrame
    $self instvar m_ColorModel
    $self instvar m_uiThumbnailFrame
    $self instvar m_uiPreviewFrame
    $self instvar m_uiBroadcastFrame
    $self instvar m_font

    set m_appDc $appDc

    set m_frameRoot 0
    set m_VisualFrame 0
    set m_ColorModel 0

    set m_uiThumbnailFrame 0
    set m_uiPreviewFrame 0
    set m_uiBroadcastFrame 0

    set m_font [[$self options] get_option smallfont]
}



CDcUI instproc build_ui { framePath } {
    $self instvar m_frameRoot
    $self instvar m_font
    set f $m_font

    set m_frameRoot $framePath

    # create the orignal frame
    pack [frame $m_frameRoot] -fill both -expand 1

    # create the color model
    $self instvar m_VisualFrame m_ColorModel

    set m_VisualFrame [new VisualFrame $m_frameRoot.visual]
    set m_ColorModel [$m_VisualFrame set colorModel_]

    # need to create the major ui layout

    # set the title of the window
    wm title . "Director's Console"

    # ok do the menu bar
    menu $m_frameRoot.menu -type menubar
    $self build_menu $m_frameRoot.menu
    pack $m_frameRoot.menu -side top -fill x

    # now the main frame
    frame $m_frameRoot.frameMain -relief sunken -borderwidth 2
    pack $m_frameRoot.frameMain -side top -fill both

    $self build_main $m_frameRoot.frameMain

    # now the title frame
    frame $m_frameRoot.frameTitle -relief sunken -borderwidth 2
    pack $m_frameRoot.frameTitle -side bottom -fill x

    label $m_frameRoot.frameTitle.title -text "Director's Console" \
	    -relief ridge -font $f
    pack $m_frameRoot.frameTitle.title -side left -fill both -expand 1

    button $m_frameRoot.frameTitle.button -text "Quit" -command "exit" -font $f
    pack $m_frameRoot.frameTitle.button -side left
}

CDcUI private build_menu { menu } {
    $self instvar m_font
    set f $m_font

    $menu add cascade -label "File" -menu $menu.file -font $f
    menu $menu.file
    $menu.file add command -label "Properties..." -font $f

    $menu.file add separator
    $menu.file add command -label "Quit" -command "exit" -font $f

    $menu add cascade -label "Help" -menu $menu.help -font $f
    menu $menu.help
    $menu.help add command -label "Help..." -font $f
    $menu.help add command -label "About Director's Console..." -font $f
}

CDcUI private build_main { frameMain } {
    $self instvar m_font
    $self instvar m_appDc
    $self instvar m_uiThumbnailFrame
    $self instvar m_uiPreviewFrame
    $self instvar m_uiBroadcastFrame
    $self instvar m_framePV m_frameBC

    set frameTN [frame $frameMain.thumbnailFrame -relief ridge -borderwidth 2]
    pack $frameTN -side left -fill y
    set m_uiThumbnailFrame [new CDcUIThumbnailFrame $m_appDc $self $frameTN]

    set framePV [frame $frameMain.previewFrame -relief ridge -borderwidth 2]
    set m_framePV $framePV
    pack $framePV -side left -fill both
    set m_uiPreviewFrame [new CDcUIPreviewFrame $m_appDc $self $framePV]

    set frameBC [frame $frameMain.broadcastFrame -relief ridge -borderwidth 2]
    set m_frameBC $frameBC
    pack $frameBC -side left -fill y
    set m_uiBroadcastFrame [new CDcUIBroadcastFrame $m_appDc $self $frameBC]
}

CDcUI public is_in_window { win x y } {
    set top [winfo rooty $win]
    set left [winfo rootx $win]
    set bottom [expr $top + [winfo height $win] - 1]
    set right [expr $left + [winfo width $win] - 1]

    if { $y <= $bottom && $y >= $top && $x <= $right && $x >= $left } {
	return 1
    }

    return 0
}


#-----------------------------------------------------------------------------
# Method:
#   CDcUI reattach
#
# Arguments:
#   src -- a Source/RTP object we want to show in the UI.
#
# Description:
#   This goes through all the frames that display video, picks those that 
#   display the source, finds out their renderers, and tell the decoder to
#   to send its output to the renderer.
#   
#   NOTE: This works only if the source is already being displayed, and
#   we want to re-target the decoder due to some reasons, such as a format
#   changed.
# 
#-----------------------------------------------------------------------------
CDcUI public reattach { src } {
    $self instvar m_ColorModel
    $self instvar m_uiThumbnailFrame
    $self instvar m_uiPreviewFrame
    $self instvar m_uiBroadcastFrame

    set decoder [$src set decoder_]
    if {$decoder == ""} {
	puts stderr "WARNING: no decoder for $src"
	return
    }

    set thumbnail [$m_uiThumbnailFrame GetUIVideoWithSource $src]
    if {$thumbnail != ""} {
	set video_widget [$thumbnail GetVideoWidget]
	$video_widget attach-decoder $src $m_ColorModel 0
    }
    set preview [$m_uiPreviewFrame GetUIVideoWithSource $src]
    if {$preview != ""} {
	set video_widget [$preview GetVideoWidget]
	$video_widget attach-decoder $src $m_ColorModel 0
    }
    set broadcast [$m_uiBroadcastFrame GetUIVideoWithSource $src]
    if {$broadcast != ""} {
	set video_widget [$broadcast GetVideoWidget]
	$video_widget attach-decoder $src $m_ColorModel 0
    }
}
