# rec_agent.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/pathfinder/rec_agent.tcl,v 1.18 2002/02/03 04:22:06 lim Exp $


import AnnounceListenManager/AS/Client/Platform ArchiveSystem/Record

#
# The Rec_Agent class handles requests to record a particular Program.
#
Class Rec_Agent

Rec_Agent public init { } {

    $self instvar archive_ schedule_ rec_list_ max_duration_

    # Initialize the archive and schedule directories.
    #set archive_ /h/mash/archive/web/
    #set schedule_ ~/mash/tcl/applications/mash_server/schedule/
    #set o [$self options]
    #$o load_preferences "mserver"
    set archive_ [$self get_option archive_root]
    append archive_ [$self get_option archive_dir]
    set schedule_ [$self get_option rec_schedule_dir]

    # Initialize the maximum duration of a recording.
    set max_duration_ [expr [$self get_option max_duration] * 60]


    # Initialize the list of sessions being recorded and scheduled
    # to be recorded.  Read the cache to find out any programs that
    # were scheduled previously.
    set rec_list_ {}
    $self read_cache
}


Rec_Agent private read_cache { } {

    mtrace trcNet "In Rec_Agent::read_cache"

    $self instvar schedule_ rec_list_

    cd $schedule_
    set ctgfiles [glob -nocomplain -- *]

    foreach f $ctgfiles {
	mtrace trcNet "-> Reading file $f"
	set catalog [new SessionCatalog]
	$catalog open $f
	$catalog read

	set msg [lindex [ [new SDPParser 0] parse [$catalog get_sdp]] 0]
	set program [new Program $msg]

	# Add this program to the list of recordings as well as to
	# the list of session lists.
	lappend rec_list_ $f $msg

	set start_time [$catalog get_info rec_start]
	set end_time [$catalog get_info rec_end]

	$self reschedule $program $start_time $end_time

	$catalog destroy
    }
}


Rec_Agent private reschedule { program start_time end_time } {

    mtrace trcNet "In Rec_Agent::reschedule"

    set current [clock seconds]
    set start_offset [expr $start_time - $current]
    set end_offset [expr $end_time - $current]

    if { $end_offset < 0 } {
	$self removeprog $program
	return
    } else {
	$self schedule stop $program $end_offset
    }

    if { $start_offset <= 0 } {
	# For now, begin recording into a new module.
	$self start_recording $program
    } else {
	$self schedule start $program $start_offset
    }
}


#
# The record method schedules a recording of the given program.
#
Rec_Agent public record { program } {

    $self instvar archive_ stop_status_array_ start_status_array_ \
	    max_duration_

    set key [get_key $program]

    set sdp_time [ [$program base] set alltimedes_]
    set repeat [$sdp_time readable_repeat]
    set start_time [ntp_to_unix [$sdp_time set starttime_]]
    set end_time [ntp_to_unix [$sdp_time set endtime_]]
    set orig_end_time $end_time
    set orig_start_time $start_time
    set current [clock seconds]

    # Do time zone adjustment
    set session_zone [$sdp_time readable_zone starttime_]
    set current_zone [clock format $current -format {%Z}]
    if { $session_zone != $current_zone } {
	set offset 3600
	if { [string match *D* $current_zone] } {
	    set offset -3600
	}
	set start_time [expr $start_time + $offset]
	set end_time [expr $end_time + $offset]
    }

    # Find the start_offset and end_offset; i.e. the number of
    # seconds in the future to start and end.
    set start_offset [expr $start_time - $current]
    set end_offset [expr $end_time - $current]
    set max_duration_offset 0

    # These variables will contain the actual start time and end
    # time for the recording.  They will then be passed into the
    # addprog method.
    set act_start_time 0
    set act_end_time 0


    if { $end_offset < 0 && $orig_end_time != 0 } {
	set start_status_array_($key) "Session expired; no recording made"
	set stop_status_array_($key) ""

    } elseif { $repeat == "None" } {

	# If there is no repeat interval, simply base the record
	# time on starttime_ and endtime_.
	if { $start_time <= $current && $current < $end_time || \
		$orig_start_time == 0 && $orig_end_time == 0 || \
		$start_time == 0 && $current < $end_time || \
		$start_time <= $current && $end_time == 0 } {
	    # Begin recording if the current time is in the start/end
	    # interval.
	    $self start_recording $program
	    set act_start_time $current
	    set max_duration_offset $max_duration_

	} elseif { $start_offset >= 0 } {
	    # Otherwise, schedule the recording for the future.
	    $self schedule start $program $start_offset
	    set act_start_time [expr $current + $start_offset]
	    set max_duration_offset [expr $start_offset + $max_duration_]

	} else {
	    puts "Error: Recording not scheduled or started."
	    exit
	}

	# Schedule an end to the recording if possible.
	if { $orig_end_time == 0 || $end_offset >= $max_duration_offset } {
	    # If the end_time is 0, then this session goes on
	    # indefinitely; cap the recording time to max_duration_.
	    # Also cap the recording time if the end_offset is greater
	    # than max_duration_.
	    $self schedule stop $program $max_duration_offset
	    set act_end_time [expr $current + $max_duration_offset]

	} elseif { $end_offset >= 0 && $end_offset < $max_duration_offset } {
	    # Otherwise, have the recordings stopped in the future.
	    $self schedule stop $program $end_offset
	    set act_end_time [expr $current + $end_offset]

	} else {
	    puts "Error: Recording end not scheduled."
	    exit
	}

    } else {
	# Handle the recording of this repeated session.
	set interval [$sdp_time get repeat_interval_]
	set duration [$sdp_time get active_duration_]

	set offset_list [$sdp_time get offlist_]
	# Assume for now, no offsets.  Pain!

	set rep_start $start_time
	set rep_end [expr $start_time + $duration]
	set end_offset [expr $rep_end - $current]
	set start_scheduled 0
	while {	$rep_start < $end_time && $current < $end_time } {

	    # Between repeated sessions or before the first session ->
	    # schedule a recording for the future.
	    if { $current < $rep_start } {
		set start_offset [expr $rep_start - $current]
		$self schedule start $program $start_offset
		set act_start_time [expr $current + $start_offset]
		set max_duration_offset [expr $start_offset + $max_duration_]
		set start_scheduled 1

	    # In the middle of a repeated session -> begin recording now.
	    } elseif { $rep_start <= $current && $current < $rep_end } {
		$self start_recording $program
		set act_start_time $current
		set max_duration_offset $max_duration_
		set start_scheduled 1
	    }

	    # Schedule the end of recording if end_offset has been set
	    # above to a non-zero value.
	    if { $start_scheduled } {
		if { $end_offset < $max_duration_offset } {
		    $self schedule stop $program $end_offset
		    set act_end_time [expr $current + $end_offset]
		} else {
		    $self schedule stop $program $max_duration_offset
		    set act_end_time [expr $current + $max_duration_offset]
		}
		break
	    }

	    set rep_start [expr $rep_start + $interval]
	    set rep_end [expr $rep_start + $duration]
	    set end_offset [expr $rep_end - $current]
	    mtrace trcNet "-> start/current/end: \
		    [$self readable_time $rep_start]/\
		    [$self readable_time $current]/\
		    [$self readable_time $rep_end]"
	}
    }

    # Add the program and the rec_list_ and write it to file.
    # Also, add the archive_sys to the archive_list_ so that
    # it can be stopped later.
    $self addprog $program $act_start_time $act_end_time
}


Rec_Agent private schedule { command program interval } {

    $self instvar stop_status_array_ start_status_array_

    mtrace trcNet "-> Schedule $command recording."
    append full_command $command _recording
    $self schedule_helper "$self $full_command $program" $program $interval

    # Update the right status array
    set key [get_key $program]
    set time_str [$self readable_time [expr [clock seconds] + $interval]]

    if { $command == "start" } {
	set start_status_array_($key) "Scheduled for $time_str"

    } elseif { $command == "stop" } {
	set stop_status_array_($key) "Scheduled for $time_str"

    } else {
	puts "Error: Invalid command to schedule"
	exit
    }
}


Rec_Agent private readable_time { time } {
    return [clock format $time -format {%a %B %d, %Y at %H:%M}]
}


Rec_Agent private schedule_helper { command program interval } {

    $self instvar schedule_array_

    if { $interval < 0 } {
	puts "Error: Parameter interval to proc schedule < 0"
	exit
    }

    mtrace trcNet "-> Scheduling command: $command"

    set interval_ms [expr $interval * 1000]
    set extra 0
    if { $interval_ms < 0 } {
	set extra [expr $extra + 86400]
	set interval [expr $interval - 86400]
	set interval_ms [expr $interval * 1000]
    }

    set new_command ""
    if { $extra == 0 } {
	after $interval_ms $command
	set new_command $command

    } elseif { $extra > 0 } {
	set extra_ms [expr $extra * 1000]
	set new_command "$self schedule_helper $command $program $extra_ms"
	after $interval_ms "$self schedule_helper $command $program $extra_ms"

    } else {
	puts "Error: $extra < 0"
	exit
    }

    mtrace trcNet "-> Command scheduled: $new_command"

    set key [get_key $program]
    set schedule_array_($key) $new_command
}


#
# The addprog method adds a given program to the list of scheduled
# Programs.  Also, the archive_sys is added to the archive_array_ so
# that the Program recording can later be stopped.
#
Rec_Agent private addprog { program start_time end_time } {

    $self instvar schedule_ rec_list_
    global tcl_platform

    # Extract the unique key and message of this Program
    set key [get_key $program]
    set msg [$program base]

    # Add this program to the list of recordings as well as to the list
    # of session lists.
    lappend rec_list_ $key $msg

    # Store the message in a file with the filename key in the cache.
    if {$tcl_platform(platform) == "windows"} {
	    # semi-colons are illegal in windows filenames
	    regsub -all ":" $key "+" newkey
	    append filename $schedule_ $newkey
    } else {
	    append filename $schedule_ $key
    }
    set catalog [new SessionCatalog]
    $catalog open $filename w
    $catalog write_sdp [$msg obj2str]
    $catalog write_info "rec_start=$start_time\nrec_end=$end_time"
    $catalog destroy
}


#
# The return_schedule method returns a list of SDP key and Program
# pairs for scheduled recordings.  This list can then be converted into
# an associated array.
#
Rec_Agent public return_progs { } {
    $self instvar rec_list_
    return $rec_list_
}


Rec_Agent public return_path { key } {

    $self instvar module_array_ archive_
	global tcl_platform

    set path ""
    if { [array names module_array_ $key] != "" } {
	if {$tcl_platform(platform) == "windows"} {
		regsub -all ":" $module_array($key) "+" newmod
		append path $newmod "\cat.ctg"
	} else {
		append path $module_array_($key) "/cat.ctg"
	}
    }
    return $path
}


Rec_Agent public return_status { type key } {

    $self instvar start_status_array_ stop_status_array_ \
	    module_array_ archive_
    global tcl_platform

    if { $type == "start" } {

	set status $start_status_array_($key)

	# Check if there are actually video/audio streams being
	# transmitted and recorded.
	# Need to collect and report local quality as well
        if { $status == "Recording" } {
	    if {$tcl_platform(platform) == "windows"} {
		    # semi-colons are illegal in windows filenames
		    regsub -all ":" $module_array($key) "+" newmod
		    append filename $archive_ $newmod "\cat.ctg"

	    } else {
		    append filename $archive_ $module_array_($key) "/cat.ctg"
	    }
	    if { ![string match *START_STREAM* [read_file $filename]] } {
		set status "Recording (waiting for video/audio stream)."
	    }
	}
    } elseif { $type == "stop" } {
	set status $stop_status_array_($key)

    } else {
	puts "Error: Invalid type to return_status."
	exit
    }

    return $status
}


# start a local recorder
Rec_Agent private start_recording { program } {

	mtrace trcNet "-> Starting recorder"
	$self instvar archive_ archive_array_ start_status_array_ \
			module_array_ schedule_array_

	set key [get_key $program]

	# Use a timestamp for the module name and add the module to the
	# module_array_.
	set module [string trim [clock clicks] -]
	set module_array_($key) $module

	# Use the ArchiveSystem/Record class for recording and write
	# the SDP Program information to the catalog file.
	set archive_sys [new ArchiveSystem/Record]
	$archive_sys open $archive_ $module
	$archive_sys write_announcement $program

	# Write relevant recording information to the catalog file
	set current [$self readable_time [clock seconds]]
	set info "record_start=$current"
	$archive_sys write_info $info

	# Begin recording and add the archive_sys object to the
	# archive_array_ so that the recording can later be stopped.
	$archive_sys record_program $program $module
	set archive_array_($key,local) $archive_sys

	# Update the start-status array.
	set start_status_array_($key) "Recording"

	# Update the schedule array by deleting the entry for this
	# program, if the entry exists.
	# ???
	set exists [array names schedule_array_ $key]
	if { $exists != "" } {
		# ???
	}
}




#
# The stop_recording method stops the recording of the specified
# Program. If the recording has not yet begun, the scheduled recording
# is cancelled.
#
Rec_Agent public stop_recording { program } {

    $self instvar archive_array_ schedule_array_ start_status_array_ \
    end_status_array_ rec_list_ archive_ module_array_

    set key [get_key $program]

    set start_status [$self return_status start $key]
    mtrace trcNet "-> Start status: $start_status"

    set is_recording [string match *Recording* $start_status]
    set is_expired [string match *expired* $start_status]
    set is_scheduled [string match *Scheduled* $start_status]

    if { $is_recording } {

	    # Update the catalog file's info block to include the end time.
	    set current [$self readable_time [clock seconds]]
	    set info "record_end=$current"
	    $archive_array_($key,local) write_info $info

	    # Stop the recording by deleting the archive_sys object.
	    $archive_array_($key,local) close
	    delete $archive_array_($key,local)

	    # Check if any streams were actually recorded, and remove the
	    # archive module if not.
	    append directory $archive_ $module_array_($key)
	    append filename $directory /cat.ctg
	    set rec_started [string match *START_STREAM* [read_file $filename]]
	    if { !$rec_started } {
		    mtrace trcNet "-> Deleting archive directory"
		    file delete $filename
		    # ??? Why can't the directory be deleted
		    file delete $directory
	    }


    } elseif { $is_scheduled } {
	# Stop any future recordings that have been scheduled.
	after cancel $schedule_array_($key)

    } elseif { $is_expired } {

    } else {
	mtrace trcNet "-> Unknown start status"

    }

    # Remove this Program from rec_list_ and delete the corresponding
    # file.
    $self removeprog $program

    # Update the status_array.
    set start_status_array_($key) "stopped expired"
    set end_status_array_($key) ""
}


#
# The removeprog method removes a Program from the list of Programs
# currently and scheduled to be recorded. The information is also
# taken out of the from the schedule directory.
#
Rec_Agent private removeprog { program } {

    $self instvar schedule_ rec_list_
    global tcl_platform

    # Extract the unique key of this Program
    set key [get_key $program]
    mtrace trcNet "-> Removing recording: $key"

    # Remove this program from the list of recordings.
    set index [lsearch -exact $rec_list_ $key]
    if {$index != -1} {
	    set rec_list_ [lreplace $rec_list_ $index [expr $index + 1]]
    }
    # Remove the file in the schedule directory.
    if {$tcl_platform(platform) == "windows"} {
	    # semi-colons are illegal in windows filenames
	    regsub -all ":" $key "+" newkey
	    append filename $schedule_ $newkey

    } else {
	    append filename $schedule_ $key

    }
    file delete $filename
}


# -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=


import HTTP_Agent Rec_Agent/Distrib

#
# The HTTP_Agent/Rec_Agent class is an HTTP-aware agent that can be
# used in the MASH_Server.  This object contains a plain Rec_Agent
# which is used to schedule recordings.
#
Class HTTP_Agent/Rec_Agent -superclass HTTP_Agent

HTTP_Agent/Rec_Agent public init { } {

    $self next
    $self instvar agent_ html_dir_ sessions_dir_ platform_alm_

	if {[$self get_option allow_distrib] == "yes" } {
		set agent_ [new Rec_Agent/Distrib]
		# Fix gspec
		set gspec [$self get_option glob_announce]
		#Listen to the global AS1 platform channel for distributed recording placement
		set platform_alm_ [new AnnounceListenManager/AS/Client/Platform $gspec]
	} else {
		set agent_ [new Rec_Agent]
	}


    # Initialize the html and sessions directories.
    #set html_dir_ ~/mash/tcl/applications/mash_server/html/
    #set o [$self options]
    #$o load_preferences "mserver"
    set html_dir_ [$self get_option html_dir]
    set sessions_dir_ [$self get_option sdp_sessions_dir]

}


#
# The handle_request method is called by the MASH_Server when an
# HTTP request is received.  This method checks for the "magic URLs"
# it cares about and returns the corresponding page.
#
HTTP_Agent/Rec_Agent instproc handle_request { url key source reply_var } {
    upvar $reply_var reply
    $self instvar agent_ html_dir_
    mtrace trcNet "-> Rec_Agent::handle_request called"
    set page ""
    set status 200
    set type "text/html"
    set msg ""

    # Check for the "magic" URLs.
    if { $url == "/record" } {

	mtrace trcNet "-> Request to record program received"

	# Validate that the source can record programs.
	if { [$self validate_source $source] } {

	    set scheduled [$self check_scheduled $key]
	    if { $scheduled == 0 } {


		set program [$self get_program $key]
		if { $program != "" } {
		    $agent_ record $program
		    append html_file $html_dir_ record.html
		    set page [read_file $html_file]

		} else {
		    mtrace trcNet "-> File does not exist in session cache."
		    set msg "Session expired or not in cache."
		    set page [$self get_error_page $msg]
		}

	    } else {
		mtrace trcNet "-> Program already scheduled: $scheduled"
		append html_file $html_dir_ record.html
		set page [read_file $html_file]
	    }

	} else {
	    set msg "Access to recorder not granted."
	    set page [$self get_error_page $msg]
	}

   } elseif { $url == "/record-list.html" } {
	mtrace trcNet "-> Recordings schedule page requested"
	set page [$self get_page recordings_list]

    } elseif { $url == "/stop_record" } {

	mtrace trcNet "-> Request to stop recording received"

	# Validate that the source can record programs.
	if { [$self validate_source $source] } {
	    $agent_ stop_recording [$self get_program $key]
	    set page [$self get_page recordings_list]

	} else {
	    set msg "Access to recorder not granted."
	    set page [$self get_error_page $msg]
	}

    } elseif { $url == "/record-status" } {
	mtrace trcNet "-> Request for recording status received"
	set page [$self get_status_page $key]

    } elseif { $url == "/add-recorder" } {
	mtrace trcNet "-> Request to add a distributed recorder"
	    # pull the platform address out of the key
	    set pk [split $key +]
	    set newkey [lindex $pk 0]
	    set platform [lindex $pk 1]

	    # add recorder with the specified platform
	    set program [$self get_program $newkey]
		if { $program != "" } {
			$agent_ add_recorder $program $platform
		}
	set page [$self get_status_page $newkey]
    }
    if { $page != {} } {
	    set reply(headers) [list content-type $type]
	    set reply(data)    $page
	    set reply(status)  $status
	    return 1
    } else {
	    return 0
    }
}



# Produce another page containing available platforms
HTTP_Agent/Rec_Agent private get_status_page { key } {
	$self instvar platform_alm_ list_ agent_ cur_platform_

	$self update_agent_list
	array set rec_array $list_


	if {[$self get_option allow_distrib] == "yes" } {

		# Use the key to find the program which will then create the
		# HTML to be returned.
		if [info exists rec_array($key)] {
			set status_page [$rec_array($key) create_dynamic_html \
					[DynamicHTMLifier set html_(distrib_recordings)]]
			set start_status [$agent_ return_status start $key]
			set distrib_status [$agent_ return_distrib_status $key]
			if [string match *Recording* $start_status] {
				foreach ALM $distrib_status {
					set p_status([$ALM get_platform]) [$ALM get_status]
				}
			}

			set platforms [$platform_alm_ get_platform_list]
			foreach p $platforms {
				if [info exists p_status($p)] {
					append status_page "$p $p_status($p)<br>"
				} else {
					set cur_platform_ $p
					append status_page "$p <a href=/add-recorder^$key+$p target=\"record-desc\"><img src=\"/images/recbut.gif\" border=0></a><font color=\#ffffff>.</font>
					<br>"
				}
			}

			set stop_status [$agent_ return_status stop $key]
			append status_page "Local recording start status: " $start_status "<br\n"
			append status_page "Stop status: " $stop_status "<br>\n"

			append status_page "</body></html>"
		} else {
			set status_page "<html> No longer recording this session </html>"
		}
		return $status_page
	} else {

		# Use the key to find the program which will then create the
		# HTML to be returned.
		set status_page [$rec_array($key) create_dynamic_html \
				[DynamicHTMLifier set html_(recordings_status)]]
		set start_status [$agent_ return_status start $key]
		set stop_status [$agent_ return_status stop $key]
		set path [$agent_ return_path $key]
		append status_page "Start status: " $start_status "<br>\n"
		append status_page "Stop status: " $stop_status "<br>\n"
		append status_page "Recording location: $path <br>\n"
		append status_page "</body>\n</html>"
		return $status_page
	}
}




#
# The get_program method takes a key and returns the Program
# associated with that key. The Program is found in the SDP_Agent's
# sessions directory.
#
HTTP_Agent/Rec_Agent private get_program { key } {

    $self instvar sessions_dir_
    global tcl_platform

    # Create the filename and find out whether it exists.
    if {$tcl_platform(platform) == "windows"} {
	    # semi-colons are illegal in windows filenames
	    regsub -all ":" $key "+" newkey
		append filename $sessions_dir_ $newkey

    } else {
	    append filename $sessions_dir_ $key

    }
    if { [file exists $filename] } {
	set data [file_dump $filename]
	set msg [lindex [ [new SDPParser 0] parse $data] 0]
	set program [new Program $msg]

    } else {
	mtrace trcNet "-> $filename does not exist in cache."
	set program ""
    }

    return $program
}


#
# The check_scheduled method returns 0 or 1 depending on whether
# the Program specified by key is currently scheduled to be recorded.
#
HTTP_Agent/Rec_Agent private check_scheduled { key } {

    $self instvar list_

    # Update the list of scheduled recordings and check if an element
    # for key is in the list.
    $self update_agent_list
    set exists [lsearch -exact $list_ $key]

    if { $exists == -1 } {
	return 0
    } else {
	return 1
    }
}

