/*
 * mplug.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "mplug.h"
#include "mplug-defs.h"
#include <stdio.h>
#include "mplug-compat.h"


MPlugConfiguration *MPlugInstance::config_=NULL;


/* ------------------------------------------------------------------------ */


PluginInstance *
PluginInstance::CreateInstance(NPMIMEType mimeType,
			       NPP instance,
			       uint16 mode,
			       int16 argc,
			       const char* const *argn,
			       const char* const *argv,
			       NPSavedData * /*savedData*/)
{
	return new MPLUG_INSTANCE_CLASS(mimeType, instance, mode,
					argc, argn, argv);
}



/* ------------------------------------------------------------------------ */


static char *CC_StrDup(const char *str)
{
	char *newStr = new char [strlen(str)+1];
	strcpy(newStr, str);
	return newStr;
}


MPlugInstance::MPlugInstance(NPMIMEType mimeType, NPP instance, uint16 mode,
			     int16 argc, const char * const *argn,
			     const char * const *argv)
	: PluginInstance(instance), mimeType_(NULL), mode_(mode),
	  argc_(argc), argn_(NULL), argv_(NULL), url_(NULL),
	  data_(NULL), dataLen_(0)
{
	LOG(("Creating new plugin instance\n"));
	mimeType_ = CC_StrDup(mimeType);
	argn_ = new char* [argc];
	argv_ = new char* [argc];
	for (int i=0; i < argc; i++) {
		argn_[i] = CC_StrDup(argn[i]);
		argv_[i] = CC_StrDup(argv[i]);
	}
	LOG(("Successfully created new plugin instance\n"));
}


MPlugInstance::~MPlugInstance()
{
	rpi_.DestroyProcess();

	delete [] mimeType_;
	for (int i=0; i<argc_; i++) {
		delete [] argn_[i];
		delete [] argv_[i];
	}

	delete [] argn_;
	delete [] argv_;
	delete [] url_;
	delete [] data_;
}


NPError
MPlugInstance::SetWindow(NPWindow* npWindow)
{
	LOG(("MPlugInstance::SetWindow invoked\n"));
	if (npWindow->window==NULL) {
		return DeleteWindow();
	}

	if (!GetWindowHandle()) {
		return NewWindow(npWindow);
	}

	if (GetWindowHandle() != (unsigned long)npWindow->window) {
		// we can't handle this case; barf
		MPlug_Output("Netscape tried to change the plugin window "
			     "handle after it had already created one\n"
			     "I don't know what to do!!!\n");
		return NPERR_GENERIC_ERROR;
	}

	return ReconfigureWindow(npWindow);
}


NPError
MPlugInstance::NewWindow(NPWindow *npWindow)
{
	LOG(("MPlugInstance::NewWindow invoked\n"));
	// we've got the window, go ahead and create the MASH process
	LOG(("Trying to invoke MPlugInstance::NewProcess\n"));
	if (NewProcess(npWindow)==FALSE) goto error;
	LOG(("Returned from MPlugInstance::NewProcess\n"));

	// notify all plugin status
	{
		char buf[80];
		if (rpi_.Notify(2, -1, "mimetype", -1, mimeType_)==FALSE)
			goto error;
		sprintf(buf, "%u", mode_);
		if (rpi_.Notify(2, -1, "mode", -1, buf)==FALSE) goto error;

		if (rpi_.NotifyInPieces(argc_*2 + 1)==FALSE) goto error;
		if (rpi_.NotifyNextPiece(-1, "arguments")==FALSE) goto error;
		for(int i=0; i < argc_; i++) {
			if (rpi_.NotifyNextPiece(-1, argn_[i])==FALSE)
				goto error;
			if (rpi_.NotifyNextPiece(-1, argv_[i])==FALSE)
				goto error;
		}
	}

	if (url_ != NULL) {
		if (rpi_.Notify(2, -1, "url", -1, url_)==FALSE)
			goto error;
	}

	// notify any stream data that we've already received
	if (data_ != NULL) {
		NPBool retval = NotifyStreamData(dataLen_, data_);
		delete [] data_;
		data_ = NULL;
		dataLen_ = 0;
		if (retval==FALSE) goto error;
	} else if (dataLen_==-1) {
		if (StreamDone(NPRES_USER_BREAK)!=NPERR_NO_ERROR)
			goto error;
	}

	// notify a window resize
	if (NotifyReconfigure(npWindow->width, npWindow->height)==FALSE)
		goto error;

	LOG(("MPlugInstance::NewWindow returning successfully\n"));
	return NPERR_NO_ERROR;

error:
	LOG(("MPlugInstance::NewWindow returning an error\n"));
	return NPERR_MODULE_LOAD_FAILED_ERROR;
}


NPError
MPlugInstance::ReconfigureWindow(NPWindow *npWindow)
{
	if (NotifyReconfigure(npWindow->width, npWindow->height)) {
		return NPERR_NO_ERROR;
	} else {
		return NPERR_GENERIC_ERROR;
	}
}


NPBool
MPlugInstance::NotifyReconfigure(uint32 width, uint32 height)
{
	char buf[80];
	sprintf(buf, "reconfigure %lu %lu", (unsigned long)width,
		(unsigned long)height);
	return rpi_.Notify(buf);
}


NPBool
MPlugInstance::NewProcess(NPWindow *npWindow)
{
	MPlugConfiguration *config = GetConfig();
	RPI::ArgArray args;

	LOG(("Config object is %p\n", config));
	const char *cmdLine = config->GetValue(CFG_EXEC_CMDLINE);
	if (cmdLine==NULL) cmdLine = DEF_EXEC_CMDLINE;
	LOG(("Trying to append cmdline: %s\n", cmdLine));
	args.Append(cmdLine);
	LOG(("Done appending cmdline\n"));

	// figure out the script associated with this mime-type
	{
		char *option;
		option = new char [strlen(mimeType_)+16];
		sprintf(option, "script.%s", mimeType_);
		const char *script = config->GetValue(option);
		if (script!=NULL) {
			args.Insert(script, 1);
		}
	}

	/*
	 * obtain the name of the display on which to start up the
	 * Tcl application. This should be the same as the display
	 * on which the browser is running!
	 */
	{
		const char *displayStr = GetDisplay(npWindow);
		if (displayStr != NULL) {
			args.Append("-display");
			args.Append(displayStr);
		}
	}

	/*
	 * append a -use argument
	 */
	{
		char use[80];
		sprintf(use, "-use 0x%lx", GetWindowHandle());
		args.Append(use);
	}

	LOG(("Trying to invoke RPI::NewProcess\n"));
	if (rpi_.NewProcess(&args, config->GetValue(CFG_STDOUT),
			    config->GetValue(CFG_STDERR))==FALSE) {
		LOG(("RPI::NewProcess returned an error\n"));
		// send error message to output
		char *cmdline;
		args.CreateCmdline(cmdline);
		MPlug_Output("Could not create plugin process: \n%s\n",
			     cmdline);
		delete [] cmdline;
		return FALSE;
	}
	LOG(("RPI::NewProcess returned an successfully\n"));

	return TRUE;
}


long
MPlugInstance::SendData(int32 offset, int32 len, void *buffer)
{
	// first check if the window's been set or not
	// if the window hasn't been set, it means we haven't created
	// the plugin process yet either, we should simply buffer the
	// data until we have the window
	//
	// I'm not sure if the above case will ever happen
	// the plugin documentation is not clear about this

	if (dataLen_ != offset) {
		// we should receive all data in sequence!
		return -1;
	}


	if (GetWindowHandle()==0) {
		// buffer the data
		char *tmp = new char [dataLen_ + len];
		if (data_) memcpy(tmp, data_, dataLen_);
		memcpy(tmp+dataLen_, buffer, len);
		delete [] data_;
		data_ = tmp;
	} else {
		if (NotifyStreamData(len, buffer)==FALSE)
			return -1;
	}

	dataLen_ += len;
	return len;
}


NPBool
MPlugInstance::NotifyURL(const char *url)
{
	LOG(("NotifyURL called: '%s'\n", url));
	if (GetWindowHandle()==0) {
		url_ = CC_StrDup(url);
		LOG(("MASH process not created yet, storing '%s'\n", url_));
		return TRUE;
	} else {
		return rpi_.Notify(2, -1, "url", -1, url);
	}
}


NPBool
MPlugInstance::NotifyStreamData(int32 len, void *buffer)
{
	return rpi_.Notify(2, -1, "recv_stream", len, buffer);
}


NPError
MPlugInstance::StreamDone(NPError reason)
{
	// first check if the window's been set or not
	// if the window hasn't been set, it means we haven't created
	// the plugin process yet either, we should simply throw away the
	// accumulated data
	//
	// I'm not sure if the above case will ever happen
	// the plugin documentation is not clear about this

	if (GetWindowHandle()==0) {
		delete [] data_;
	} else {
		char buf[80];
		sprintf(buf, "stream_done %d", reason);
		if (rpi_.Notify(buf)==FALSE) return NPERR_GENERIC_ERROR;
	}

	data_ = NULL;
	dataLen_ = -1; // -1 indicates that we are actually done

	return NPERR_NO_ERROR;
}

