/*
 * packet-splitter.cc --
 *
 *      Provides split object that when it receives a packet passes it to
 *      1 or 2 targets (depending on how many are attached).
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tclcl.h"
#include "module.h"
#include "pktbuf.h"
#include "rtp.h"
#include "codec/decoder.h"

class PacketSplitter : public PacketModule {
  public:
  PacketSplitter();
  virtual void recv(pktbuf *);
  virtual int command(int argc, const char* const *argv);
  inline Module *target2() { return (target2_); }
  protected:

  Module *target2_;
};

static class PacketSplitterClass : public TclClass {
  public:
  PacketSplitterClass() : TclClass("Module/PacketSplitter") {}
  TclObject* create(int /* argc */, const char*const* /* argv */) {
    return (new PacketSplitter);
  }
} dm_packet_splitter;

PacketSplitter::PacketSplitter() : PacketModule(), target2_(0)
{
}

int PacketSplitter::command(int argc, const char*const* argv)
{
  Tcl& tcl = Tcl::instance();
  int iResult;

  if (argc == 2) {
    if (strcmp(argv[1], "target2") == 0) {
      if (target2_ != 0)
	tcl.result(target2_->name());
      return (TCL_OK);
    }
  }

  if (argc == 3) {
    if (strcmp(argv[1], "target2") == 0) {
      target2_ = (Module *)TclObject::lookup(argv[2]);
      return (TCL_OK);
    }
  }

  // we'll have to dispatch it to the packetmodule object
  iResult = PacketModule::command(argc, argv);

  // if we didn't find a handler for this, then give it to the target
  if (iResult == TCL_ERROR)
    return target_->command(argc, argv);

  return iResult;
}


void PacketSplitter::recv(pktbuf *pkt)
{
  /* Attah so that we can pass this to two targets. */
  pkt->attach();

  if (target_ != NULL) {
    target_->recv(pkt);
  } else {
    pkt->release();
  }

  if (target2_ != NULL) {
    target2_->recv(pkt);
  } else {
    pkt->release();
  }
}

class DummyPacketModule : public PacketModule {
public:
  DummyPacketModule();
  virtual void recv(pktbuf *);
  virtual int command(int argc, const char* const *argv);

protected:
  int count_;
};

static class DummyPacketModuleClass : public TclClass {
  public:
  DummyPacketModuleClass() : TclClass("Module/DummyPacketModule") {}
  TclObject* create(int /* argc */, const char*const* /* argv */) {
    return (new DummyPacketModule());
  }
} dm_dummy_packet;

DummyPacketModule::DummyPacketModule() : PacketModule(), count_(0)
{
}

void DummyPacketModule::recv(pktbuf *pkt)
{
  count_++;
  pkt->release();
}

int DummyPacketModule::command(int argc, const char*const* argv)
{
  Tcl& tcl = Tcl::instance();

  if (argc == 2) {
    if (strcmp(argv[1], "count") == 0) {
      char buffer[128];
      sprintf(buffer, "%d", count_);
      tcl.result(buffer);
      return (TCL_OK);
    }
  }

  return(PacketModule::command(argc, argv));
}

