/*
 * archive-file.h --
 *
 *      Archive file type header
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/archive/archive-file.h,v 1.14 2002/02/03 03:09:26 lim Exp $
 */

#ifndef MASH_ARCHIVE_FILE_H
#define MASH_ARCHIVE_FILE_H

#include <tclcl.h>
#include "misc/all-types.h"


#define DATA_VERSION  "MDAT1.0"
#define INDEX_VERSION "MIDX1.0"

#define PROTO_SRM "SRM"
#define PROTO_RTP "RTP"


struct FileHeader {
	char version[16];	// DATA_VERSION & INDEX_VERSION
	char protocol[8];	// "RTP" or "SRM"
	char media[32];		// the media: "audio", "video", "mediaboard"...
	char cname[128];	// username@host.domain
	char name[128];		// the user's name

  // all of the following values are stored in network form, so require a
  //   ntohl() call when reading from the file
  // why???
	u_int32_t start_sec;    // start timestamp
	u_int32_t start_usec;
	u_int32_t end_sec;      // end timestamp
	u_int32_t end_usec;

	u_int32_t privateLen;	// length of any following private header
};


struct IndexRecord {
	u_int32_t sentTS_sec; // sender timestamp: depends on the media stream
	u_int32_t sentTS_usec;
	u_int32_t recvTS_sec; // timestamp when the packet was rcvd by the
	u_int32_t recvTS_usec;// recorder
	u_int32_t seqno;      // packet sequence number
	u_int32_t filePointer;// location of packet in the data file
};


void net2host(const FileHeader &hdrNet,  FileHeader &hdrHost);
void host2net(const FileHeader &hdrHost, FileHeader &hdrNet);

void net2host(const IndexRecord &idxNet,  IndexRecord &idxHost);
void host2net(const IndexRecord &idxHost, IndexRecord &idxNet);


class ArchiveFile : public TclObject {
public:
	ArchiveFile() : channel_(NULL), headerSize_(0) { };
	~ArchiveFile() {  Close(); }
	void Close() {
		if (channel_!=NULL)
			Tcl_Close(Tcl::instance().interp(), channel_);
		channel_ = NULL;
		headerSize_ = 0;

	}

	Bool IsOpen() { return ((channel_==NULL) ? FALSE : TRUE); }
	Bool Eof() { return ((Tcl_Eof(channel_)==0) ? FALSE : TRUE); }

	int Write(const FileHeader *hdr, const Byte *privateHdr=NULL);
	int Read(FileHeader *hdr, Byte *privateHdr=NULL,
		 u_int32_t privateLen=0);

	int Write(const Byte *buf, int toWrite) {
		return Tcl_Write(channel_, (char*) buf, toWrite);
	}

	int Read(Byte *buf, int toRead) {
		return Tcl_Read(channel_, (char*) buf, toRead);
	}

	int Seek(int offset, int seekMode) {
		return Tcl_Seek(channel_, offset, seekMode);
	}

	int Tell() {
		return Tcl_Tell(channel_);
	}

	int getHeaderSize() {
		if (headerSize_ == 0) {
			// we haven't read the header yet! so read it first
			FileHeader tmp;
			if (Read(&tmp) < 0) return 0;
		}
		return headerSize_;
	}

	int Open(const char *filename, const char *mode, int permissions);
	int open(int argc, const char * const *argv);
	int close(int argc, const char * const *argv);
	int header(int argc, const char * const *argv);
	int private_header(int argc, const char * const *argv);

	static int ts2string(int argc, const char * const *argv);
	char* get_name() { return filename_;}
private:
	Tcl_Channel channel_;
	u_int32_t headerSize_;
	char filename_[100];  // For easier debugging
};


class DataFile : public ArchiveFile {
public:
};


class IndexFile : public ArchiveFile {
public:
	int Write(const IndexRecord *idx) {
		return ArchiveFile::Write((const Byte *)idx,
					  sizeof(IndexRecord));
	}
	int Read(IndexRecord *idx) {
		return ArchiveFile::Read((Byte *)idx, sizeof(IndexRecord));
	}
};


#endif /* MASH_ARCHIVE_FILE_H */
