
import java.io.*;
import java.util.*;


/**
 * This class loads and stores information relating to an Index Generator
 * project.
 *
 * <p>A settings file will follow these rules:</p>
 * <ul>
 *   <li>
 *     A '%' starts a comment; comment text continues to the end of the
 *     line.
 *   </li>
 *   <li>
 *     For a '%' character to be inserted in data, it is escaped with a '\'.
 *   </li>
 *   <li>
 *     A variable is defined as a word starting in the first column.
 *   </li>
 *   <li>
 *     Long variables have a ')' character following their name.
 *     Their value consists of all text, excluding comments, following
 *     the variable declaration up to the string ";;" starting in the first
 *     column.
 *   </li>
 *   <li>
 *     Short variables have a '=' character following their name.
 *     Their value consists of all text from the '=' character to the
 *     end-of-line character or comment character.
 *   </li>
 * </ul>
 *
 * @version $Id: Project.java,v 1.31 2002/09/30 10:59:48 howama Exp $
 * @author Ben Secrest &lt;blsecres@users.sourceforge.net&gt;
 */


public class Project {
    /** A copy of the system line separator */
    private static final String EOL = System.getProperty("line.separator");

    /** The character used to indicate the start of a long variable */
    private static final char LONG_VARIABLE_START = ')';

    /** The string used to indicate the end of a long variable */
    private static final String LONG_VARIABLE_END = ";;";

    /**
     * The character used to separate the key/value pairs of a short
     * variable
     */
    private static final char SHORT_VARIABLE_DELIM = '=';

    /** The comment leader */
    private static final char COMMENT_LEADER = '%';

    /**
     * The width for displaying comments in the settings file.  Two spaces are
     * automatically subtracted for the comment leader and a space
     */
    private static final int COMMENT_WIDTH = 78;

    /** The width for half a section header banners */
    private static final int HEADER_BANNER_WIDTH = 20;

    /** The default logging level for this module */
    private static final int LOGLEVEL = 9;

    /** The oldest supported config file version */
    private static final String oldestSupportedVersion = "0.2";

    /**
     * A HashMap object containing the settings for the current
     * project
     */
    private HashMap variables;

    /** The name of the file the settings were loaded from.  */
    private String settingsFile;

    /** The current locale to use when saving the settings */
    private Locale currentLocale = new Locale("en", "GB");

    /** The resource bundle for setting comments */
    ResourceBundle comments;

    /** The validator for setting names, types, and values */
    private Settings settings;


    /**
     * The log object to use for all comments
     */
    private IGLog log;


    /**
     * The constructor for a project sets up a reference to a IGLog
     * object for logging, establishes a set of valid variables, and
     * loads the defaults for those variables
     * @param locale The locale to use when saving settings
     */
    public Project(IGLog logObj) {
	log = logObj;
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "Project.Project(IGLog)");
	if (LOGLEVEL >= IGLog.VERSION)
	    log.add(IGLog.VERSION, "Project.java $Revision: 1.31 $");

	variables = new HashMap();

	settingsFile = null;
	settings = null;

	// default to using english
	currentLocale = new Locale("en", "");
	comments = null;
    }


    /**
     * Allow access to the variables object
     * @return The current set of settings
     */
    public HashMap getSettings() {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "Hashtable Project.getSettings()");

	return variables;
    }


    /**
     * Retrieve a boolean variable
     * @param var The name of the variable
     * @return The boolean value assigned to the variable
     * @throws IllegalVariableException if name is either not a valid
     *		variable or not a variable of boolean type
     */
    public boolean getBooleanSetting(String var)
	    throws IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "boolean Project.getBooleanSetting(String["
		    + var + "])");

	if (settings.getTypeByVar(var) == Settings.BOOLEAN)
	    return(((Boolean) variables.get(var)).booleanValue());

	log.addError(40, "PROJECT_BAD_ACCESS_ATTEMPT",
		new String[]{log.getString("PROJECT_BOOLEAN"), var});
	throw new IllegalVariableException(currentLocale, var,
					   Settings.BOOLEAN);
    }


    /**
     * Retrieve an integer variable
     * @param var The name of the variable
     * @return The integer value assigned to the variable
     * @throws IllegalVariableException if name is either not a valid
     * 	variable or not a variable of integer type
     */
    public int getIntegerSetting(String var)
	    throws IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "int Project.getIntegerSetting(String[" +
		    var + "])");

	if (settings.getTypeByVar(var) == Settings.INTEGER)
	    return(((Integer) variables.get(var)).intValue());

	log.addError(40, "PROJECT_BAD_ACCESS_ATTEMPT",
		new String[]{log.getString("PROJECT_INTEGER"), var});
	throw new IllegalVariableException(currentLocale, var,
					   Settings.INTEGER);
    }


    /**
     * Retrieve a string value
     * @param var The name of the variable
     * @return The string value assigned to the variable
     * @throws IllegalVariableException if name is either not a valid
     * 	variable or not a variable of string type
     */
    public String getStringSetting(String var)
	    throws IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "String Project.getStringSetting(String["
		    + var + "])");

	if (settings.getTypeByVar(var) == Settings.STRING)
	    return((String) variables.get(var));

	log.addError(40, "PROJECT_BAD_ACCESS_ATTEMPT",
		new String[]{log.getString("PROJECT_STRING"), var});
	throw new IllegalVariableException(currentLocale, var,
					   Settings.STRING);
    }


    /**
     * Retrieve an enumerated value
     * @param var The name of the variable
     * @return The string value assigned to the variable
     * @throws IllegalVariableException if name is either not a valid variable
     *		or not a variable of enum type
     */
    public String getEnumSetting(String var) throws IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "String Project.getEnumSetting(String[" +
		    var + "])");

	if (settings.getTypeByVar(var) == Settings.ENUM)
	    return((String) variables.get(var));

	log.addError(40, "PROJECT_BAD_ACCESS_ATTEMPT",
		new String[]{log.getString("PROJECT_ENUM"), var});
	throw new IllegalVariableException(currentLocale, var, Settings.ENUM);
    }


    /**
     * Retrieve an array value
     * @param var The name of the variable
     * @return The array containing the values for the setting
     * @throws IllegalVariableException if name is either not a valid
     *		variable or not a variable of array type
     */
    public String[] getArraySetting(String var)
	    throws IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "String[] Project.getArraySetting(String["
		    + var + "])");

	if (settings.getTypeByVar(var) == Settings.ARRAY) {
	    // The hashtable must store objects derived from java.lang.Object
	    // so, an ArrayList is used.  A call to ArrayList.toArray() yields
	    // an Object array (Object[]).  This can't simply be cast to
	    // String[] so we have to copy everything over.
	    ArrayList arrayList = (ArrayList) variables.get(var);

	    if (arrayList == null)
		return new String[]{};

	    String[] strings = new String[arrayList.size()];

	    for (int i = 0; i < arrayList.size(); i++)
		strings[i] = arrayList.get(i).toString();

	    return(strings);
	}

	log.addError(40, "PROJECT_BAD_ACCESS_ATTEMPT",
		new String[]{log.getString("PROJECT_ARRAY"), var});
	throw new IllegalVariableException(currentLocale, var, Settings.ARRAY);
    }


    /**
     * Load the settings for an Index Generator project
     * @param filename The file to load settings from
     * @throws java.io.IOException if an error occurs while reading
     * 	the configuration file
     * @throws java.io.FileNotFoundException if the file given to be parsed
     * 	does not exist
     * @throws IllegalVariableException if the configuration file
     * 	contains an illegal variable
     */
    public void load(String filename) throws IllegalVariableException,
	    IOException, FileNotFoundException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "void Project.load(String[" + filename +
		    "])");

	boolean isReading = false, hasLocale = false;
	String curLine, curVariable = "", curValue = "";
	BufferedReader reader = new BufferedReader(
	    new InputStreamReader(new FileInputStream(filename), "UTF8"));
	if (LOGLEVEL >= IGLog.FILE)
	    log.addResource(IGLog.FILE, "PROCESS_FILE", new String[]{filename});

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "FP_BEGIN_PARSE",
			    new String[]{log.getString("PROJECT_SETTINGS")});

	while ((curLine = reader.readLine()) != null) {
	    // ignore comments
	    if (curLine.startsWith("%"))
		continue;
	    // if reading variable data
	    if (isReading) {
		// check for end of data and add it to properties
		if (curLine.startsWith(LONG_VARIABLE_END)) {
		    isReading = false;
		    storeSettingByName(curVariable, curValue);
		    if (LOGLEVEL >= IGLog.PROGRESS)
			log.addResource(IGLog.PROGRESS,
					"PROJECT_STORED_SETTING",
					new String[]{curVariable});
		    curVariable = "";
		    curValue = "";
		// else accumulate data
		} else {
		    // check for a comment
		    int commentIndex = curLine.indexOf(COMMENT_LEADER);

		    // ensure comment character isn't escaped, already
		    // checked for a '%' at index 0, so this should be
		    // safe
		    while (commentIndex != -1 &&
			   curLine.charAt(commentIndex - 1) == '\\')
			commentIndex = curLine.indexOf(COMMENT_LEADER,
						       commentIndex);

		    curValue += (commentIndex == -1 ? curLine :
				 curLine.substring(0, commentIndex)) + EOL;
		}
	    // check for valid settings
	    } else {
		int delimIndex = curLine.indexOf(LONG_VARIABLE_START);

		// check for a long variable
		if (delimIndex != -1) {
		    curVariable = curLine.substring(0, delimIndex).trim();
		    isReading = true;
		// check for a short variable
		} else if ((delimIndex =
			    curLine.indexOf(SHORT_VARIABLE_DELIM)) != -1) {
		    int commentIndex = curLine.indexOf(COMMENT_LEADER);

		    while (commentIndex != -1 &&
			   curLine.charAt(commentIndex - 1) == '\\')
			commentIndex = curLine.indexOf(COMMENT_LEADER,
						       commentIndex);

		    curVariable = curLine.substring(0, delimIndex).trim();
		    curValue = curLine.substring(delimIndex + 1,
						 (commentIndex == -1 ?
						  curLine.length() :
						  commentIndex)).trim();
		    if (hasLocale) {
			storeSettingByName(curVariable, curValue);
			if (LOGLEVEL >= IGLog.PROGRESS)
			    log.addResource(IGLog.PROGRESS,
					    "PROJECT_STORED_SETTING",
					    new String[]{curVariable});
		    } else {
			if (curVariable.equalsIgnoreCase("locale") &&
			    curValue.length() > 1) {
			    currentLocale = new Locale(curValue.substring(0, 2),
						       curValue.length() == 5 ?
						       curValue.substring(3, 5)
						       : "");
			    settings = new Settings(log, currentLocale);
			    storeSettingByName("locale",
				    currentLocale.toString());
			    if (LOGLEVEL >= IGLog.PROGRESS)
				log.addResource(IGLog.PROGRESS,
						"PROJECT_FOUND_LOCALE",
						new String[]{curValue});
			} else {
			    log.addWarning(41, "PROJECT_NO_LOCALE", null);
			}

			hasLocale = true;
		    }

		    curVariable = "";
		    curValue = "";
		}
	    }
	}

	reader.close();

	settingsFile = filename;

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "FP_FINISH_PARSE",
			    new String[]{log.getString("PROJECT_SETTINGS")});

	validateSettings();
    }


    /**
     * Validate the settings collected from a config file or the user
     */
    private void validateSettings() throws IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "void Project.validateSettings()");
	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "PROJECT_BEGIN_VALIDATE", null);

	String version = getStringSetting("SETTINGS_VERSION");

	if (version.compareTo(oldestSupportedVersion) < 0) {
	    log.addError(42, "PROJECT_VERSION_MISMATCH",
		    new String[]{version, oldestSupportedVersion});

	    // TODO throw exception
	    throw new IllegalVariableException(currentLocale,
		    settings.getNameByVar("SETTINGS_VERSION"), version);
	}
    }


    /**
     * Save the settings to a file
     * @param filename The pathname of the file to save to
     * @throws java.io.IOException if an error occurs writing the file
     * @throws IllegalVariableException if an attempt is made to save a
     * 	nonexistant variable
     */
    public void save(String filename) throws IOException,
	    IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "void Project.save(String[" + filename +
		    "])");
	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "PROJECT_BEGIN_SAVE",
		    new String[]{filename});

	validateSettings();

	String[] curVariable;
	BufferedWriter writer = new BufferedWriter(new FileWriter(filename));

	if (comments == null)
	    comments = ResourceBundle.getBundle("Comments", currentLocale);

	//writer.write(formatComment(comments.getString("SETTINGS_HEADER")));

	writeVariable(writer, "LOCALE");
	writeVariable(writer, "SETTINGS_VERSION");

	// file related variables
	writeHeader(writer, "FILES_TO_INDEX_HEADER");
	writeVariable(writer, "GET_METHOD");
	writeVariable(writer, "FILE_SYSTEM_SEARCH");
	writeVariable(writer, "SCAN_ROOT");
	writeVariable(writer, "HTTP_HEADERS");
	writeVariable(writer, "SCAN_INCLUDE_FILTERS");
	writeVariable(writer, "SCAN_EXCLUDE_FILTERS");
	writeVariable(writer, "INDEX_INCLUDE_FILTERS");
	writeVariable(writer, "INDEX_EXCLUDE_FILTERS");
	writeVariable(writer, "OUTPUT_FILENAME");

	// project detail variables
	writeHeader(writer, "PROJECT_DETAILS_HEADER");
	writeVariable(writer, "INDEX_TYPE");
	writeVariable(writer, "FILES_TO_COPY");
	writeVariable(writer, "FILE_COPY_TARGET");

	// sitemap variables
	writeHeader(writer, "SITEMAP_SETTINGS_HEADER");
	writeVariable(writer, "SM_START_TXT");
	writeVariable(writer, "SM_ENTER_FOLDER");
	writeVariable(writer, "SM_LEAVE_FOLDER");
	writeVariable(writer, "SM_LINKS_TXT");
	writeVariable(writer, "SM_END_TXT");

	// ordered list variables
	writeHeader(writer, "ORDERED_LIST_SETTINGS_HEADER");
	writeVariable(writer, "OL_SORT_KEY");
	writeVariable(writer, "OL_ARTICLES");
	writeVariable(writer, "OL_START_TXT");
	writeVariable(writer, "OL_NAV_START");
	writeVariable(writer, "OL_NAV_LINK");
	writeVariable(writer, "OL_NAV_END");
	writeVariable(writer, "OL_SEC_START");
	writeVariable(writer, "OL_SEC_END");
	writeVariable(writer, "OL_LINK");
	writeVariable(writer, "OL_END_TXT");

	// GUI variables
	writeHeader(writer, "GUI_SETTINGS_HEADER");
	writeVariable(writer, "PROJECT_NAME");
	writeVariable(writer, "CREATED");
	writeVariable(writer, "LOAD_AS_THEME");


	writer.close();

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "PROJECT_FINISH_SAVE", null);
    }


    /**
     * Write a section header to a configuration file
     * @param writer The output object to use for writing
     * @param variable The variable to write
     */
    private void writeHeader(BufferedWriter writer, String variable)
	    throws IOException {
	writer.newLine();
	for (int i = 0; i < HEADER_BANNER_WIDTH; i++)
	    writer.write(COMMENT_LEADER);
	writer.write(' ' + comments.getString(variable) + ' ');
	for (int i = 0; i < HEADER_BANNER_WIDTH; i++)
	    writer.write(COMMENT_LEADER);
	writer.newLine();
	writer.newLine();
    }


    /**
     * Write a setting to file
     * @param writer The output object to use for writing
     * @param variable The variable to write
     * @throws IOException if an error occurs writing to file
     * @throws IllegalVariableException if an attempt is made to save a
     * 	non-variable value
     */
    private void writeVariable(BufferedWriter writer, 
	    String variable) throws IOException, IllegalVariableException {
	writer.write(formatComment(comments.getString(variable)));

	switch (settings.getTypeByVar(variable)) {
	    case Settings.BOOLEAN :
		writer.write(settings.getNameByVar(variable) + " "
			+ SHORT_VARIABLE_DELIM + " "
			+ settings.getLocalBoolean(getBooleanSetting(
				variable)));
		break;
	    case Settings.INTEGER :
		writer.write(settings.getNameByVar(variable) + " "
			+ SHORT_VARIABLE_DELIM + " "
			+ getIntegerSetting(variable));
		break;
	    case Settings.STRING :
		{
		    writer.write(settings.getNameByVar(variable));
		    String val = getStringSetting(variable);

		    if (val == null) {
			writer.write(" " + SHORT_VARIABLE_DELIM);
			break;
		    }

		    if (val.indexOf("\r") > 0 || val.indexOf("\n") > 0) {
			writer.write(LONG_VARIABLE_START);
			writer.newLine();
			writer.write(val);
			writer.write(LONG_VARIABLE_END);
		    } else {
			writer.write(" " + SHORT_VARIABLE_DELIM + " " + val);
		    }
		}
		break;
	    case Settings.ENUM :
		writer.write(settings.getNameByVar(variable) + " "
			+ SHORT_VARIABLE_DELIM + " "
			+ settings.getNameByVar(getEnumSetting(variable)));
		break;
	    case Settings.ARRAY :
		{
		    writer.write(settings.getNameByVar(variable)
			    + LONG_VARIABLE_START);
		    writer.newLine();

		    String[] items = getArraySetting(variable);
		    for (int j = 0; j < items.length; j++) {
			writer.write(items[j]);
			writer.newLine();
		    }
		    writer.write(LONG_VARIABLE_END);
		}
		break;
	    case Settings.DNE :
	    default :
		throw new IllegalVariableException(currentLocale, variable);
	}

	writer.newLine();
	writer.newLine();
    }


    /**
     * Save the settings to last file loaded
     * @throws java.io.FileNotFoundException if no filename has been given
     * 	via load()
     * @throws java.io.IOException if an error occurs while writing the file
     * @throws IllegalVariableException if an attempt is made to write a
     * 	nonexistant variable
     * @see #load
     */
    public void save() throws java.io.FileNotFoundException,
	    java.io.IOException, IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "void Project.save()");

	if (settingsFile == null) {
	    log.addWarning(43, "PROJECT_NO_FILE", null);
	    throw new java.io.FileNotFoundException("No filename specified");
	}

	save(settingsFile);
    }


    /**
     * Format a comment string for output
     * <ul>
     *   <li>Replace embedded newlines (&quot;\n&quot;) with the system's
     *     end-of-line string</li>
     *   <li>Ensure that text is &lt;80 columns wide</li>
     *   <li>Ensure each line of text begins with the comment leader</li>
     * </ul>
     * @param comment The comment text
     * @return The comment text properly formatted 
     */
    private String formatComment(String comment) {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "String Project.formatComment(String)");

	int len, curPos, wsPos;
	String leader = "" + COMMENT_LEADER + " ";
	String output = "";
	
	// tokenize the comment based on newlines provided in the resource file
	for (StringTokenizer st = new StringTokenizer(comment, EOL);
	     st.hasMoreTokens(); ) {
	    String curLine = st.nextToken();

	    // if the current line is longer than the desired width, break it
	    // into pieces
	    if ((len = curLine.length()) >= COMMENT_WIDTH) {
		curPos = 0;
		// loop until the remainder of the line is less than the
		// desired width
		while (len - curPos > COMMENT_WIDTH) {
		    // find whitespace to break the line
		    wsPos = curLine.lastIndexOf(" ", curPos + COMMENT_WIDTH);
		    // check for a really big word
		    if (wsPos == -1 || wsPos < curPos)
			wsPos = curLine.indexOf(" ", curPos);
		    // output the comment leader, acceptable piece of line, and
		    // the system end-of-line
		    output += leader + curLine.substring(curPos, wsPos) + EOL;
		    // move up to continue processing the current line
		    curPos = wsPos + 1;
		}
		// output the remainder of the current line
		output += leader + curLine.substring(curPos) + EOL;
	    // else, a line of acceptable length is output
	    } else
		output += leader + curLine + EOL;
	}

	return output;
    }


    /**
     * Store a setting using the appropriate data type
     * @param variable The internal representation of a setting
     * @param value The value to store for the setting
     * @throws IllegalVariableException if the key is not a valid variable or
     * 	it does not have a proper value
     */
    public void storeSettingByVar(String variable, String value)
	    throws IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "void Project.storeSettingByVar(String["
		    + variable + "], String[" + value + "])");

	switch (settings.getTypeByVar(variable)) {
	case Settings.BOOLEAN :
	    {
		String bool = settings.getBoolean(value);
		// confirm proper boolean values
		if (bool != null)
		    variables.put(variable, new Boolean(bool));
		else {
		    log.addError(44, "PROJECT_BAD_ASSIGN_ATTEMPT",
				new String[]{value,
				    log.getString("PROJECT_BOOLEAN"),
				    variable});
		    throw new IllegalVariableException(currentLocale, variable,
			    value);
		}
		break;
	    }
	case Settings.INTEGER :
	    variables.put(variable, new Integer(value));
	    break;
	case Settings.STRING :
	    variables.put(variable, value);
	    break;
	case Settings.ARRAY :
	    {
		// store array settings in an ArrayList
		StringTokenizer st = new StringTokenizer(value);
		ArrayList items = new ArrayList(st.countTokens());

		while (st.hasMoreTokens())
		    items.add(st.nextToken());
		variables.put(variable, items);
	    }
	    break;
	case Settings.ENUM :
	    // confirm the enumerated variable is being set to a valid value
	    if (settings.validEnumeration(variable,
					  value))
		variables.put(variable,
			      value);
	    else {
		log.addError(44, "PROJECT_BAD_ASSIGN_ATTEMPT",
			new String[]{value, log.getString("PROJECT_ENUM"),
			    variable});
		throw new IllegalVariableException(currentLocale, variable,
			value);
	    }
	    break;
	case Settings.DNE :
	    // attempt to store a nonexistant variable
	    log.addWarning(45, "PROJECT_NOT_VARIABLE", new String[]{variable});
	    throw new IllegalVariableException(currentLocale, variable);
	    /* NOTREACHED */
	}
    }


    /**
     * Store a setting using the appropriate data type
     * @param name The localized name to save
     * @param value The value to store for the variable
     * @throws IllegalVariableException if the key is not a valid
     * 	variable or it does not have a proper value
     */
    private void storeSettingByName(String name, String value)
	    throws IllegalVariableException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "void Project.storeSettingByName(String["
		    + name + "], String[" + value + "])");

	switch (settings.getTypeByVar(settings.getVarByName(name))) {
		case Settings.ENUM:
			storeSettingByVar(settings.getVarByName(name), settings.getVarByName(value));
			break;
		default:
			storeSettingByVar(settings.getVarByName(name), value);
	}
    }


    /**
     * Retrieve the localized version of a placeholder
     * @param key The internal representation of the placeholder
     * @return The localized placeholder
     * @throws IllegalVariableException if the key is not a valid placeholder
     */
    public String getPlaceHolder(String key) throws IllegalVariableException {
	String placeholder = settings.getPlaceHolder(key);

	if (placeholder != null)
	    return placeholder;
	else {
	    log.addWarning(50, "PROJECT_NOT_PH", new String[]{placeholder});
	    throw new IllegalVariableException(currentLocale, key);
	}
    }


    /**
     * Returns the name of the setting as will appear in the project file.
     * @param setting The internal representation of a setting
     * @return The localized version of a setting
     */
    public String getLocalName(String setting){
	    return settings.getNameByVar(setting);
    }
}
